#=========================================================================
# Copyright (C) GemTalk Systems 1986-2024.  All Rights Reserved..
#
# Name - environment.sh
# Installed as - environment.sh
#
# Written By: Martin McClure and Norm Green
#
# Purpose -
#
# Setup the UNIX environment for the scripts that create X509 certificates
# and keys.
#
# Requirements -
#
# The following environment variables must be defined to create or print
# certs:
#
# GEMSTONE or OPENSSL_PREFIX_DIR
#
# GEMSTONE_CERT_DIR - A directory where newly created certificates and
#                     subdirectorys will be placed.
#
# To debug, set the env var GEMSTONE_CERT_DEBUG.  This will cause the
# scripts to print debug out including every openssl command executed.
#
#=========================================================================


#### Set up the environment for running scripts
#### This file is sourced by each script in SCRIPTDIR

## scriptDir should already be set by the script

## Set prefixDir to what you put in --prefix when configuring OpenSSL

# GEMSTONE_CERT_DIR is not needed for readcert and readcsr commands.
# Otherwise it is required.
#

debug=0
# Show openssl command out in debug mode
if [ "$GEMSTONE_CERT_DEBUG" != "" ]; then
    showOutput=1
    debug=1
    if [ "$GEMSTONE_CERT_DEBUG" = "verbose" ]; then
	set -xv
    fi
    
fi

cmd=`basename $0`

# Set default requirement booleans here
#
# Most commands need a stone name
needStoneName=1
# Most commands need the cert dir defined
needCertDir=1
#
# Do not show output from openssl commands by default
showOutput=0
#
# Need openssl by default
needOpenSsl=1

# Now check for special cases and adjust booleans
if [[ $cmd = read* ]]; then
    needCertDir=0
    showOutput=1
    needStoneName=0
elif [[ $cmd = rm* ]]; then
    needOpenSsl=0
elif [[ $cmd = new* ]]; then
    true
elif [[ $cmd = revoke* ]]; then
    true    
elif [[ $cmd = ls* ]]; then
    
    if [ "$cmd" = "lsstone" ]; then
	needStoneName=0
    fi
else
    echo "[Internal Error]: Unknown command '$cmd'">&2
    exit 1
fi

errStoneMissing(){
    echo "[Error]: Stone name must be specified">&2
}


if [ $needStoneName -eq 1 -a -z "$stoneName" ]; then
    errStoneMissing
    usage
fi

haveGemstone=0
if [ ! -z "$OPENSSL_PREFIX_DIR" ]; then
    prefixDir=$OPENSSL_PREFIX_DIR
elif [ ! -z "$GEMSTONE" ]; then
    prefixDir=$GEMSTONE
    haveGemstone=1
elif [ $needOpenSsl -eq 1 ]; then
  echo "ERROR: Either GEMSTONE or OPENSSL_PREFIX_DIR environment variable must be defined" >&2
  exit 1
fi

if [ $needOpenSsl -eq 1 ]; then
  openssl="${prefixDir}/bin/openssl"
  if [ ! -x ${openssl} ]; then
      echo "[Error]: Cannot find openssl executable ${openssl}">&2
      exit 1
  fi
fi

if [ $needCertDir -eq 1 -a "$GEMSTONE_CERT_DIR" = "" ]; then
  echo "ERROR: GEMSTONE_CERT_DIR environment variable not defined" >&2
  exit 1
fi

## Or use the system's default OpenSSL if you leave the above commented
unused=${prefixDir:=/usr}

# openssl in $GEMSTONE is statically linked, so no LD_LIBRARY_PATH is needed.
# openssl built from source needs libssl.so and libcrypto.so in LD_LIBRARY_PATH.
if [ $needOpenSsl -eq 1 -a $haveGemstone -eq 0 ]; then
  export LD_LIBRARY_PATH=${prefixDir}/lib:${LD_LIBRARY_PATH}
fi

export configDir=${scriptDir}/config

# scripts write newly created certificates to this directory,
#  creating the directory if needed
# newstone script creates this directory
export stonesDir="${GEMSTONE_CERT_DIR}/stones"

if [ ! -z "$stoneName" ]; then
  export thisStonesDir="${stonesDir}/${stoneName}"
  export stoneCaDir="${thisStonesDir}/stoneCA"
# stoneCA  
  export stoneCaCert="${stoneCaDir}/stoneCA-${stoneName}.cert.pem"
  export stoneCaPrivKey="${stoneCaDir}/stoneCA-${stoneName}.privkey.pem"
  export stoneCaCsr="${stoneCaDir}/stoneCA-${stoneName}.csr.pem"
# user CA  
  export userCaCert="${stoneCaDir}/userCA-${stoneName}.cert.pem"
  export userCaPrivKey="${stoneCaDir}/userCA-${stoneName}.privkey.pem"
  export userCaCsr="${stoneCaDir}/userCA-${stoneName}.csr.pem"
  export userCaCrl="${stoneCaDir}/userCA-${stoneName}.crl.pem"  
# host CA  
  export hostCaCert="${stoneCaDir}/hostCA-${stoneName}.cert.pem"
  export hostCaPrivKey="${stoneCaDir}/hostCA-${stoneName}.privkey.pem"
  export hostCaCsr="${stoneCaDir}/hostCA-${stoneName}.csr.pem"
  export hostCaCrl="${stoneCaDir}/hostCA-${stoneName}.crl.pem"

  # combined CRL
  export combinedHostUserCrl="${stoneCaDir}/hostCA-userCA-combined-${stoneName}.crl.pem"
  export usersDir="${thisStonesDir}/users"
  export hostsDir="${thisStonesDir}/hosts"
fi
		      
export OPENSSL_CONF="${configDir}/openssl.conf"

# Config used to sign CSRs with stoneCA private key
export SIGNCA_CONF="${configDir}/signCA.conf"

# Config used to generate CSRs to be signed by stoneCA
export USERCSR_CONF="${configDir}/userCSR.conf"

# Key size in bits for RSA keys
export rsaKeyBits=2048

# Starting certificate serial number
export certStartSerNum=1000

domkdir () {
    if [ -z "$1" ]; then
	echo "[Internal Error]: Empty arg to domkdir" >&2
	exit 1
    fi

    if [ $debug -eq 1 ]; then
	echo "[Debug]: creating directory $1"
    fi
        
    mkdir -p $1 >/dev/null 2>&1
    if [ $? -ne 0 ]; then
	echo "[Error]: mkdir -p $1 failed" >&2
	exit 1
    fi
}

dotouch () {
    if [ -z "$1" ]; then
	echo "[Internal Error]: Empty arg to dotouch" >&2
	exit 1
    fi

    if [ $debug -eq 1 ]; then
	echo "[Debug]: touching file $1"
    fi    
    touch $1 >/dev/null 2>&1
    if [ $? -ne 0 ]; then
	echo "[Error]: touch $1 failed" >&2
	exit 1
    fi
}

doopenssl () {
    if [ -z "$1" ]; then
	echo "[Internal Error]: Empty arg to doopenssl" >&2
	exit 1
    fi

    if [ $needOpenSsl -eq 0 ]; then
	echo "[Internal Error]: Unexpected call to openssl">&2
	exit 1
    fi
    
    if [ $debug -eq 1 ]; then
	echo "[Debug]: executing 'openssl $*'"
    fi

    # Show command and all output in debug mode
    if [ $showOutput -eq 1 ]; then
        ${openssl} $*
    else
	${openssl} $* >/dev/null 2>&1
    fi
    
    if [ $? -ne 0 ]; then
	echo "[Error]: Command failed: '${openssl} $*'" >&2
	exit 1
    fi
}

checkStoneExists () {
   if [ -z "$1" ]; then
	echo "[Internal Error]: Empty arg to checkStoneExists" >&2
	exit 1
   fi
   
   if [ ! -d ${stonesDir} ]; then
    echo "[Error]: You must first run the newstone script" >&2
    exit 1
  fi

  if [ ! -d "${stonesDir}/${1}" ]; then
    echo "[Error]: Cannot find the certificate for the stone named $1. Use the newstone script to create it." >&2
    exit 1
  fi

  if [ ! -f "${stoneCaCert}" ]; then
      echo "[Error]: Cannot find the stone CA certificate named ${stoneCaCert}. Use the newstone script to create it." >&2
      exit 1
  fi
}


confirmAction () {
read -r -p "Are you sure? [y/N] " response
case "$response" in
    [yY][eE][sS]|[yY]) 
        return 1
        ;;
    *)
        echo "Action not confirmed"
	exit 0
        ;;
esac
}

dorm () {
   if [ -z "$1" ]; then
	echo "[Internal Error]: Empty arg to dorm" >&2
	exit 1
   fi    

   if [ ! -d $1 ]; then
       echo "[Error]: Directory $1 does not exist">&2
       exit 1
   fi

   rm -fr $1 >/dev/null 2>&1
   if [ $? -ne 0 ]; then
       echo "[Error]: Failed to remove directory $1">&2
       exit 1
   fi
}

# Use a different error message for rm operations
verifyStoneExistsForRm () {
   if [ -z "$1" ]; then
	echo "[Internal Error]: Empty arg to verifyStoneExistsForRm" >&2
	exit 1
   fi
   
  if [ ! -d "${stonesDir}/${1}" ]; then
    echo "[Error]: Stone named '$1' not found." >&2
    exit 1
  fi
}

verifyAtLeastOneFileExists() {
    local file1=$1
    local file2=$2
    
    if [ ! -f $file1 -a ! -f $file2 ]; then
	echo "[Error]: At least one file of $file1 and $file2 must exist" >&2
	exit 1
    fi
}

    
verifyFileExists() {
    local fn=${1}
    if [ ! -f ${fn} ]; then
	echo "[Error]: File ${fn} does not exist" >&2
	exit 1
    fi
}

verifyDirExists() {
    local dir=${1}
    if [ ! -d ${dir} ]; then
	echo "[Error]: Directory ${dir} does not exist" >&2
	exit 1
    fi
}

verifyNoFileExists() {
    local fn=${1}
    if [ -f ${fn} ]; then
	echo "[Error]: File ${fn} already exists" >&2
	exit 1
    fi
}

verifyNoDirExists() {
    local dir=${1}
    if [ -d ${dir} ]; then
	echo "[Error]: Directory ${dir} already exists" >&2
	exit 1
    fi
}

renameDir() {
    local oldName=$1
    local newName=$2
    if [ $debug -eq 1 ]; then
	echo "[Debug]: Attempting to rename $oldName to $newName" >&2
    fi
    verifyDirExists $oldName
    verifyNoDirExists $newName
    mv $oldName $newName
    if [ $? -ne 0 ]; then
	echo "[Error]: Failed to rename directory $oldName to $newName" >&2
	exit 1
    fi
}

moveDirToDir() {
    local srcDir=$1
    local destDir=$2
    if [ $debug -eq 1 ]; then
	echo "[Debug]: Attempting to move $srcDir to $destDir" >&2
    fi
    
    verifyDirExists $srcDir
    verifyDirExists $destDir

    mv $srcDir $destDir
    if [ $? -ne 0 ]; then
	echo "[Error]: Failed to move file $fn to directory $destDir" >&2
	exit 1
    fi
}

verifyUserExistsForRm () {
   local user=${1}
   if [ -z "$user" ]; then
	echo "[Internal Error]: Empty arg to verifyUserExistsForRm" >&2
	exit 1
   fi

   if [ -z "$stoneName" ]; then
	echo "[Internal Error]: Empty stoneName to verifyUserExistsForRm" >&2
	exit 1
   fi   
   
  if [ ! -d "${stonesDir}/${stoneName}/users/${user}" ]; then
    echo "[Error]: User named  '$user' not found." >&2
    exit 1
  fi
}

verifyHostExistsForRm () {
   local host=${1}
   if [ -z "$host" ]; then
	echo "[Internal Error]: Empty arg to verifyHostExistsForRm" >&2
	exit 1
   fi

   if [ -z "$stoneName" ]; then
	echo "[Internal Error]: Empty stoneName to verifyHostExistsForRm" >&2
	exit 1
   fi   
   
  if [ ! -d "${stonesDir}/${stoneName}/hosts/${host}" ]; then
    echo "[Error]: Host named  '$host' not found." >&2
    exit 1
  fi
}

errStoneExists(){
  echo "[Error]: Stone name '$stoneName' already exists." >&2
  echo "[Error]: To remove, execute 'rmstone $stoneName'" >&2
  echo "[Error]: Otherwise, please select a new and unique stone name." >&2
  exit 1
}

createStoneDir(){   
  # Make files and directories that the CA will need.
  domkdir ${stoneCaDir}/certs
  dotouch ${stoneCaDir}/index.txt
  dotouch ${stoneCaDir}/index.txt.attr
  echo $certStartSerNum > ${stoneCaDir}/serial
  domkdir ${usersDir}/revoked
  domkdir ${hostsDir}/revoked
}

errInvalidDays(){
  echo "[Error]: -d arg must specifiy a positive integer" >&2
  usage
}

checkDaysValid(){
    days=$1
    if [[ $days =~ ^-?[0-9]+$ ]]; then
	if [ $days -lt 1 ]; then
	    errInvalidDays
	fi
    else
	errInvalidDays
    fi
}

docp() {
    local source=$1
    local target=$2
    verifyFileExists $source
    cp -p $source $target >/dev/null 2>&1
    if [ $? -ne 0 ]; then
	echo "[Error]: file copy failure: cp -p $source $target" >&2
        exit 1
    fi
}

docat() {
    local source=$1
    local target=$2
    verifyFileExists $source
    cat $source >>$target
    if [ $? -ne 0 ]; then
	echo "[Error]: file cat failure: cat $source >>$target" >&2
        exit 1
    fi
}   

# At least one file must exist.  One file may not exist yet.
catFiles() {
    local file1=$1
    local file2=$2
    local resultFile=$3
    verifyAtLeastOneFileExists $file1 $file2

    if [ -z $resultFile ]; then
	echo "[Error]: Missing argument to catFiles" >&2
	exit 1
    fi

    rm -f $resultFile >/dev/null 2>&1
    if [ -f $file1 ]; then	
	docp $file1 $resultFile >/dev/null 2>&1
	if [ -f $file2 ]; then
  	    docat $file2 $resultFile
	fi
    else
	# file1 does not exist yet if we get here
	docp $file2 $resultFile >/dev/null 2>&1
    fi
}


createCrl() {
  local caKey=$1
  local caCert=$2
  local newCrl=$3

  verifyFileExists $caKey
  verifyFileExists $caCert
  verifyNoFileExists $newCrl
  # create CRL
  doopenssl ca -batch -config ${SIGNCA_CONF} -name signca -gencrl -out ${newCrl} -notext \
	    -crldays 36500 -cert $caCert -keyfile $caKey

  # (re)create combined hostCA/userCA CRL file
  catFiles $userCaCrl $hostCaCrl $combinedHostUserCrl
}

revokedDirForCert() {
    local cert=$1
    local certDir=`dirname $cert` # /home/normg/certs_new/stones/norm/users/DataCurator
    local parentCertDir=`readlink -e -n $certDir/../` #/home/normg/certs_new/stones/norm/users
    local revokedDir=$parentCertDir/revoked #/home/normg/certs_new/stones/norm/users/revoked
    echo $revokedDir
}


createCombinedCrl() {

    # Both CRLs may not exist yet, but at least one should if we get here.
    if [ ! -f $userCaCrl -a ! -f $hostCaCrl ]; then
	echo "[Error]: At least one of $userCaCrl and $hostCaCrl must exist" >&2
	exit 1
    fi
    
    #
    if [ -f $userCaCrl -a -f $hostCaCrl ]; then
	catFiles $userCaCrl $hostCaCrl $combinedHostUserCrl
    fi
    
}

# Create a CA (host CA or user CA) cert signed by the stone CA
createIntermediateCa() {
  local key=$1
  local csr=$2
  local cert=$3
  local days=$4
  local kind=$5
  local crl=$6
  local DN="${stoneName}_${kind}"
  export DN

  verifyNoFileExists $key
  verifyNoFileExists $csr
  verifyNoFileExists $cert
  verifyNoFileExists $crl
  
  # create private key
  doopenssl genpkey -out ${key} -algorithm RSA -pkeyopt rsa_keygen_bits:${rsaKeyBits}
  # create CSR
  doopenssl req -config ${USERCSR_CONF} -new -key ${key} -out ${csr} -subj \
/gemstone_CertificateType=${kind}/CN=${DN}/gemstone_StoneName=${stoneName}
  # sign cert
  doopenssl ca -batch -config ${SIGNCA_CONF} -name signca -in ${csr} -out ${cert} -notext \
	    -days ${days} -cert ${stoneCaCert} -keyfile ${stoneCaPrivKey}

  # Create the CRL
  createCrl $key $cert $crl

  # Create the revoked directory
  local revokedDir=$(revokedDirForCert $cert)
  if [ $debug -eq 1 ]; then
      echo "[Debug]: revoked dir for $cert is $revokedDir" >&2
  fi
  
  unset DN
}

revokeCert() {
    local caKey=$1
    local caCert=$2
    local cert=$3
    local crl=$4

   # Tell OpenSSL cert admin that the cert is revoked
  doopenssl ca -batch -config ${SIGNCA_CONF} -name signca -revoke $cert -notext -keyfile $caKey -cert $caCert

  # now re-create intermediate CA CRL
  rm -f $crl >/dev/null 2>&1
  createCrl $caKey $caCert $crl

  # Move the revoked directory and contents to the revoked directory
  # We need to do this so a new cert for the same user/host can be created
  local certDir=`dirname $cert` # /home/normg/certs_new/stones/norm/users/DataCurator
  local revokedDir=$(revokedDirForCert $cert)
  verifyDirExists $revokedDir  #newuserCA or newhostCA should create it!
  local certOwnerName=`basename -z -s .cert.pem $cert` # DataCurator
  local NOW=`date +%h-%d-%Y-%H-%M-%S`
  local newFileName=${certOwnerName}.revokedOn.${NOW}  # DataCurator.revokedOn.Mar-23-2018-14-02-10
  local newDirName=${revokedDir}/${newFileName}
  moveDirToDir ${certDir} ${revokedDir}
  renameDir ${revokedDir}/${certOwnerName} ${newDirName}
  echo "[Info]: Certificate $cert has been revoked." >&2
  echo "[Info]: Combined Certificate Revocation List (CRL) $combinedHostUserCrl has been updated." >&2
  echo "[Info]: Please distribute the new CRL to all appropriate locations." >&2
}

# Functions for validating CIDR addresses

raiseCidrAddressError() {
    echo "[Error]: Invalid CIDR address: $INPUT" >&2
    if [ ! -z "$CIDR_error_reason" ]; then
	echo "Reason: $CIDR_error_reason"
    fi
    exit 1
}

# Check if an integer is in range.
# First arg is the integer
# Second arg is the minimum
# Third arg is the maximum
checkIntegerInRange() {
    local addr=$1
    local min=$2
    local max=$3

    if [ $addr -lt $min -o $addr -gt $max ]; then
	CIDR_error_reason="$addr is out of range. Valid range is $min to $max."
	raiseCidrAddressError 
    fi
}

# Count the number of 0 bits in a 32 bit word starting at the LSB.
# Stop counting after the first non-zero bit is found.
# Result will always be between 0 and 32
uint32CountLsbZeroBits() {
    local arg=$1
    if [ $arg -lt 0 -o $arg -gt 4294967296 ]; then
	echo "Error: invalid arg $arg in uint32CountLsbZeroBits"
	exit 1
    fi

    local result=0
    local idx=0
    while [ $idx -lt 32 ]; do
	local bit=$((($arg>>$idx)&1))
	if [ $bit -eq 0 ]; then
	    result=`expr $result + 1`
	else
	    break
	fi
	idx=`expr $idx + 1`
    done
    return $result
}

# Convert 4 bytes in an array into a 32 bit int.
# Note that we cannot return the result because
# return values cannot exceed 255 in bash.
#
# Result is stored in global var val32
#
compute32BitValue() {
    local idx=0
    local result=0
    while [ $idx -lt 4 ]; do
	local myIdx=`expr 3 - $idx`
	local x=${cidrBytes[$myIdx]}
	local shift=`expr $idx \* 8`
	x=$(($x<<$shift))
	result=$(($result|$x))
	idx=`expr $idx + 1`
    done
    val32=$result
}

# Validate a CIDR IPv4 address.
checkCidrAddress() {
    # Check for correct format with regex.
    # We expect digits separated by 3 dots and 1 slash
    # Digit values are checked later.
    INPUT=$1
    CIDR_error_reason=""
    if [[ ! $INPUT =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\/[0-9]{1,2}$ ]]; then
      CIDR_error_reason="Invalid address format. CIDR address format must be a.b.c.d/e"    
      raiseCidrAddressError
    fi
    
    local saveIFS=$IFS
    IFS='./'
    # ip is NOT local
    cidrBytes[0]=256
    cidrBytes[1]=256
    cidrBytes[2]=256
    cidrBytes[3]=256
    cidrBytes[4]=256
    local idx=0
    
    for aByte in $INPUT
    do
	cidrBytes[$idx]=$aByte
	idx=`expr $idx + 1`
    done
    IFS=$saveIFS

    # Now make sure all bytes are in range
    checkIntegerInRange ${cidrBytes[0]} 0 255
    checkIntegerInRange ${cidrBytes[1]} 0 255
    checkIntegerInRange ${cidrBytes[2]} 0 255
    checkIntegerInRange ${cidrBytes[3]} 0 255
    # Last one is the suffix
    checkIntegerInRange ${cidrBytes[4]} 0 32

    # Number of bits to match is the number after ths slash
    # This many bits must match the IP address of the client, starting at MSB
    local numSigBits=${cidrBytes[4]}
    if [ $numSigBits -eq 0 ]; then
	# All octets must be 0 if suffix is 0
	if [ ${cidrBytes[0]} -ne 0 -o ${cidrBytes[1]} -ne 0 -o ${cidrBytes[2]} \
			     -ne 0 -o ${cidrBytes[3]} -ne 0 ]; then
	    CIDR_error_reason="All bytes must be 0 when a suffix of 0 is specified."
	    raiseCidrAddressError
	fi
    elif [ $numSigBits -ne 32 ]; then
      # No checks needed if suffix is 32
      val32=0	# result from compute32BitValue. Must be global
      compute32BitValue      
      uint32CountLsbZeroBits $val32 # count zero bits up first non-zero bit, starting at LSB
      numZerosFound=$?
      local numExpectedZeros=`expr 32 - $numSigBits`
      # We can have more zeros than expected, but not fewer.
      if [ $numZerosFound -lt $numExpectedZeros ]; then
	  CIDR_error_reason="Expected at least $numExpectedZeros zero bits but only $numZerosFound were found."
	  raiseCidrAddressError
      fi
    fi

    # Good to go if we get here
    return 0
}
