! file  examples/Money.gs
!  install as SystemUser 
run
Published at: #Money put: Special56bit0
%
run
(Published at: #Money) changeNameTo: #'Money'
%
run
Money objectSecurityPolicy: DataCuratorObjectSecurityPolicy.
%
run
Money comment:
'Money is using the class Special56bit0 .
   The bits within the 56bit value are
      16r7F    currency 0..127 
     16r380    3 bits of scale for the amount
   remaining  46 bits of mantissa for the amount
'. 
%
category: 'Private'
classmethod: Money
_amount: aScaledDecimal currency: aCurrencyId
  "aScaledDecimal must be a ScaledDecimal with mantissa representable in 46 bits,
   and scale in the range 0..7 .
   aCurrencyId must be a SmallInteger in the range 0..127 . "
  | bits mant scale min max |
  (0 <= aCurrencyId and:[ aCurrencyId <= 127]) ifFalse:[
    OutOfRange new name:'currencyId' min: 0 max: 127 actual: aCurrencyId; signal.
  ].
  (scale := aScaledDecimal scale) > 7 ifTrue:[
    OutOfRange new name:'scale' min: 0 max: 7 actual: scale; signal.
  ].
  mant := aScaledDecimal mantissa .
  (mant <= (max := 70368744177663) and:[ mant >= (min := -70368744177664) ]) ifFalse:[
    OutOfRange new name:'mantissa' min: min max: max actual: mant; signal.
  ].
  bits := (mant bitShift: 3) bitOr: scale .
  bits := (bits bitShift: 7) bitOr: aCurrencyId . 
  ^ self value: bits .
%
category: 'Currency Types'
classmethod: Money
CAD_currencyType
	^1
%
category: 'Instance Creation'
classmethod: Money
CAD_fromAmount: aNumber
	^self amount: aNumber currency: self CAD_currencyType
%
category: 'Currency Types'
classmethod: Money
EUR_currencyType
	^2
%
category: 'Instance Creation'
classmethod: Money
EUR_fromAmount: aNumber
	^self amount: aNumber currency: self EUR_currencyType
%
category: 'Testing Examples'
classmethod: Money
exampleScripts 
"
	| c e j u|
	c := self CAD_fromAmount: 3.00.
	e := self EUR_fromAmount: 1.00.
	c + e .   
 $4.45 (CAD )
"

"
	self JPY_fromAmount: 3.45.  
 ¥3.
"

"
	[| c e j u|
	c := self USD_fromAmount: 3.00.
	e := self EUR_fromAmount: 1.00.
	c + e ] on: Error do: [:ex | ex printString].
 'a UserDefinedError occurred (error 2318), reason:halt, Conversion not implemented from Euros to US Dollars'
"
%
category: 'Fileout Support'
classmethod: Money
fileOutClassDefinitionOn: stream environmentId: envId
"Overwrite inherited implementation to avoid writing class definition"

"Writes the receiver's preclass, class definition, and comment onto
 the given stream in filein format."

self _fileoutHeaderOn: stream  .
stream nextPutAll:'set compile_env: 0'; lf .
envId == 0 ifTrue:[

 stream    nextPutAll: '! ------------------- Class definition for ' ;
       _fileOutAll: self thisClass name; lf .
  self fileOutPreClassOn: stream .

" stream nextPutAll: 'expectvalue /Class'; lf ;
    nextPutAll: 'doit'; lf ;
    _fileOutAll: self definition; lf ;
    nextPut: $% ; lf .  "

  self fileOutCommentOn: stream .
  self fileOutClassCategoryOn: stream .
  self _fileOutEnvSuperclassesOn: stream .
] ifFalse:[
  self _fileOutEnvSuperclass: envId on: stream .
].
%
category: 'Fileout Support'
classmethod: Money
fileOutMethodsOn: stream environmentId: envId
"Override base implementation to avoid writing the methods in the category 'Base Methods', 
which are reserved for GemStone, and to not delete all methods, which would delete the base
methods. NOTE: with this modification, filein does not remove non-base Money methods from an 
existing older Money definition in the image."

"File out this class's methods, but sort the selectors alphabetically."

| cls sels baseSels  nm |
self _fileoutHeaderOn: stream  .
cls := self thisClass.
nm := cls name.

"self fileOutMethodRemovalOn: stream name: nm environmentId: envId ."
self fileOutPreMethodsOn: stream environmentId: envId .

stream nextPutAll:'set compile_env: '; nextPutAll: envId asString ; lf .
stream nextPutAll: '! ------------------- Class methods for '; _fileOutAll: nm; lf.
sels := SortedCollection withAll: (cls class selectorsForEnvironment: envId ).
baseSels := cls class selectorsIn: 'Base Methods'.
sels removeAllPresent: baseSels.
1 to: sels size do: [:i |
  cls class fileOutMethod: (sels at: i)  environmentId: 0 - envId on: stream .
].
stream nextPutAll: '! ------------------- Instance methods for '; _fileOutAll: nm; lf.
sels := SortedCollection withAll: (cls selectorsForEnvironment: envId ).
baseSels := cls selectorsIn: 'Base Methods'.
sels removeAllPresent: baseSels.
1 to: sels size do: [:i |

  cls fileOutMethod: (sels at: i) environmentId: 0 - envId on: stream .
].
self fileOutPostMethodsOn: stream.
envId ~~ 0 ifTrue:[ stream nextPutAll:'set compile_env: 0' ; lf ].
^stream
%
category: 'Fileout Support'
classmethod: Money
fileOutPreClassOn: stream
"Write out the modifications to Special56bit0 for use as Money"

stream nextPutAll: '
! setup Special 0 to be Money class

run
(Published includesKey: #Money) ifFalse: [
	"Must be SystemUser to execute"
	 Published at: #Money put: Special56bit0. 
	(Published at: #Money) changeNameTo: #''Money''.
	Money objectSecurityPolicy: DataCuratorObjectSecurityPolicy.
	]
', '%

'.
%
category: 'Currency Types'
classmethod: Money
JPY_currencyType
	^3
%
category: 'Instance Creation'
classmethod: Money
JPY_fromAmount: aNumber
	^self amount: aNumber currency: self JPY_currencyType
%
category: 'Currency Types'
classmethod: Money
nameForCurrencyType: aType
	aType = self USD_currencyType ifTrue: [^'US Dollars'].
	aType = self CAD_currencyType  ifTrue: [^'Canadian Dollars'].
	aType = self EUR_currencyType  ifTrue: [^'Euros'].
	aType = self JPD_currencyType  ifTrue: [^'Japanese Yen'].
%
category: 'Instance Creation'
classmethod: Money
amount: aNumber currency: aCurrencyType
	| num |
	(aNumber _isNumber) ifFalse: [self error: 'invalid amount class: ', aNumber class asString].

	"Fractional parts beyond the currency's number of decimal places are not retained"

	num := aNumber asScaledDecimal: (self scaleForCurrencyType: aCurrencyType).
	^self _amount: num currency: aCurrencyType
%
category: 'Currency Types'
classmethod: Money
scaleForCurrencyType: aType
	"Japense Yen do not have decimal divisions"
	aType = self JPY_currencyType 
		ifTrue: [^0].
	^2
%

category: 'Currency Types'
classmethod: Money
USD_currencyType
	^0
%
category: 'Instance Creation'
classmethod: Money
USD_fromAmount: aNumber

	^self amount: aNumber currency: self USD_currencyType
%

! ------------------- Instance methods for Money
category: 'Operations'
method: Money
+ numberOrCurrency
	(numberOrCurrency _isNumber) ifTrue: [
		"the added amount interpreted as in the receiver's currency"
		^self class amount: (self amount + numberOrCurrency) currency: self currency.	
		].
	(numberOrCurrency isKindOf: self class) ifFalse: [
		self error: 'invalid argument to +'
		].
	(numberOrCurrency currency == self currency) ifFalse:
		[ | converted |
		converted := numberOrCurrency convertToCurrencyType: self currency.
		^self + converted
		].
	^self class amount: (self amount + numberOrCurrency amount) currency: self currency
%
category: 'Logical Instvars'
method: Money
amount
  "Returns a ScaledDecimal"
  | v scale |
  scale := ((v := self value) bitShift: -7) bitAnd: 16r7 .
  ^ ScaledDecimal mantissa: (v bitShift: -10) scale: scale .
%
category: 'Conversion'
method: Money
convertToCurrencyType: aType
	"placeholder: this will needs to use an external service"

	| rate |
	"as of 7/8/23, 1 Euro = 1.4546 Candian dollars"
	(self currency = self class EUR_currencyType and: [aType = self class CAD_currencyType]) ifTrue: [
		rate := 1.4546.
		^self class amount: (self amount * rate) currency: aType
		].

	(self currency = self class CAD_currencyType and: [aType = self class EUR_currencyType]) ifTrue: [
		rate := 0.6872.
		^self class amount: (self amount * rate) currency: aType
		].	

	self error: 'Conversion not implemented from ', self nameForCurrency, ' to ', (self class nameForCurrencyType: aType).
%
category: 'Logical Instvars'
method: Money
currency
  "Returns a SmallInteger in the range 0..127"
  ^ self value bitAnd: 16r7F .
%
category: 'Display'
method: Money
currencyCode
	| cur |
	cur := self currency.
	cur = self class USD_currencyType ifTrue: [ ^'USD'].
	cur = self class CAD_currencyType ifTrue: [ ^'CAD' ].
	cur = self class EUR_currencyType ifTrue: [ ^'EUR'].
	cur = self class JPY_currencyType ifTrue: [ ^'JPY' ].
	^$?
%
category: 'Display'
method: Money
currencySymbol
	| cur |
	cur := self currency.
	cur = self class USD_currencyType ifTrue: [ ^$$ ].
	cur = self class CAD_currencyType ifTrue: [ ^$$ ].
	cur = self class EUR_currencyType ifTrue: [ ^Character codePoint: 8364 ].
	cur = self class JPY_currencyType ifTrue: [ ^Character codePoint: 165 ].
	^$?
%
category: 'Display'
method: Money
nameForCurrency
	^self class nameForCurrencyType: self currency.
%
category: 'Display'
method: Money
printOn: aStream
		
	aStream nextPutAll: self currencySymbol.
	self amount printOn: aStream.
	self currencySymbol = $$
		ifTrue: [
			aStream nextPutAll: ' ('.
			aStream nextPutAll: self currencyCode.
			aStream nextPutAll: ' )'.
			].
%

