"
Most class operations done here. 

selectedMethods - client side selection. Used after a method compile.
"
Class {
	#name : 'RowanClassService',
	#superclass : 'RowanService',
	#instVars : [
		'name',
		'comment',
		'instVarNames',
		'classVarNames',
		'classInstVarNames',
		'superclassName',
		'subclassType',
		'poolDictionaryNames',
		'classType',
		'meta',
		'isExtension',
		'version',
		'versions',
		'oop',
		'template',
		'filters',
		'filterType',
		'methods',
		'selectedPackageServices',
		'packageName',
		'definedPackageName',
		'selectedMethods',
		'projectName',
		'hierarchyServices',
		'variables',
		'categories',
		'isTestCase',
		'expand',
		'visibleTests',
		'isNewClass',
		'updateAfterCommand',
		'isInSymbolList',
		'dictionaryName',
		'wasRemoved',
		'renamedName',
		'hasSubclasses',
		'classCategory'
	],
	#category : 'Rowan-Services-Core'
}

{ #category : 'instance creation' }
RowanClassService class >> basicForClassNamed: className [ 
	"Don't get method services. Efficient for classes with many methods"
	^self new basicForClassNamed: className
]

{ #category : 'instance creation' }
RowanClassService class >> forClassNamed: className [ 

	^self new forClassNamed: className

]

{ #category : 'instance creation' }
RowanClassService class >> forClassNamed: className meta: aBoolean [

	| inst |
	inst := self forClassNamed: className subStrings first.
	inst meta: aBoolean.
	^inst

]

{ #category : 'instance creation' }
RowanClassService class >> forClassNamed: className package: packageName [

	| inst |
	inst := self forClassNamed: className.
	inst packageName: packageName.
	^inst

]

{ #category : 'instance creation' }
RowanClassService class >> minimalForClassNamed: className [ 
	"Don't get method services. Efficient for classes with many methods"
	^self new minimalForClassNamed: className
]

{ #category : 'instance creation' }
RowanClassService class >> minimalForClassNamed: className packageNames: packageNames [
	"Don't get method services. Efficient for classes with many methods"
	^self new minimalForClassNamed: className packageNames: packageNames
]

{ #category : 'rsr' }
RowanClassService class >> templateClassName [

	^#RowanClassService
]

{ #category : 'comparing' }
RowanClassService >> = classService [
	(classService class canUnderstand: #isClassService) ifFalse:[^false].
	^classService isClassService
			ifTrue: [  name asString = classService name asString and: [meta = classService meta]]
			ifFalse: [^false]
]

{ #category : 'client commands' }
RowanClassService >> addCategory: string [
	| theClass |
	theClass := self theClass.
	meta
		ifTrue: [ theClass := theClass class ].
	theClass addCategory: string.
	self updateClass.
]

{ #category : 'constants' }
RowanClassService >> addSubclassWarningString [

	^'Superclass is not packaged. Enter the desired package name'
]

{ #category : 'client commands' }
RowanClassService >> allSubclassServices [
	| subclassServices |
	RowanCommandResult addResult: self.
	self updateType: #'updatedFullInClassHierarchy:browser:'.
	subclassServices := self theClass subclasses asArray
		collect: [ :aClass | RowanClassService minimalForClassNamed: aClass name ].
	subclassServices isEmpty ifFalse: [hasSubclasses := true]. 
	hierarchyServices := Dictionary new.
	hierarchyServices at: #'expand' put: subclassServices.
	subclassServices do: [ :classService | classService allSubclassServices ]
]

{ #category : 'Accessing' }
RowanClassService >> allTests [
	| allSelectors theClass |
	self isTestCase
		ifFalse: [ ^ Array new ].
	theClass := self theClass thisClass.
	theClass isAbstract
		ifTrue: [ ^ Array new ].
	allSelectors := self theClass thisClass allTestSelectors.
	^ allSelectors
		collect: [ :selector | 
			RowanMethodService
				forSelector: selector
				class: (theClass whichClassIncludesSelector: selector asString)
				meta: false
				organizer: self organizer ]
]

{ #category : 'testing' }
RowanClassService >> arePackageAndProjectClean [

	^self packageIsDirty not and:[self projectIsDirty not]
]

{ #category : 'initialization' }
RowanClassService >> basicForClassNamed: className [ 

	| theClass |
	self name: className. 
	theClass := self theClass. 
	theClass isNil ifTrue:[oop := nil. ^self].
	self basicRefreshFrom: theClass.
]

{ #category : 'Accessing' }
RowanClassService >> basicRefreshFrom: theClass [
	| classOrMeta theFilters |
	oop := theClass asOop.
	command := nil. 
	commandArgs := nil. 
	superclassName := theClass superClass ifNotNil:[:theSuper | theSuper name asString]. 
	versions := theClass classHistory size.
	version := theClass classHistory indexOf: theClass.
	self setComment.
	template := self classCreationTemplate.
	theFilters := SortedCollection new.
	classOrMeta := meta == true ifTrue:[theClass class] ifFalse:[theClass].
	self initializeVariablesFor: classOrMeta. 
	self initializeCategoriesFor: classOrMeta.
	packageName := definedPackageName := classOrMeta rowanPackageName.
	self setDictionary: classOrMeta.
	projectName := classOrMeta rowanProjectName.
	instVarNames := classOrMeta instVarNames asArray. 
	self setIsTestCase.
	self updateIsExtension.
	hasSubclasses := (self organizer subclassesOf: theClass) notEmpty.
	classCategory := classOrMeta category.
]

{ #category : 'Accessing' }
RowanClassService >> behavior [

	| behavior |
	behavior := self theClass. 
	meta == true ifTrue:[behavior := behavior class].
	^behavior
]

{ #category : 'accessing' }
RowanClassService >> classCategory [
	^classCategory
]

{ #category : 'accessing' }
RowanClassService >> classCategory: object [
	classCategory := object
]

{ #category : 'client commands' }
RowanClassService >> classComment: string [
	| theClass |
	theClass := self theClass. 
	theClass rwComment: string.
]

{ #category : 'rowan' }
RowanClassService >> classCreationTemplate [
	
	^self browserTool classCreationTemplateForClass: self theClass hybridBrowser: true.
]

{ #category : 'rowan' }
RowanClassService >> classCreationTemplateUsing: packageNames [
	"copying RwPrjBrowserToolV2>>classCreationTemplateForClass:hybridBrowser: with one change for performance"

	| result anArray lfsp newByteSubclass civs superClass className thePackageName nonRowanClass |
	result := String new.
	superClass := self theClass superclass.
	className := self theClass name asString.
	superClass
		ifNil: [ result addAll: 'nil' ]
		ifNotNil: [ result addAll: superClass name asString ].
	lfsp := Character lf asString tab.
	newByteSubclass := false.
	thePackageName := self computePackageNameFor: self theClass in: packageNames. "performance improvement here"
	nonRowanClass := thePackageName = Rowan unpackagedName.
	(self theClass isBytes _and: [ superClass isBytes not ])
		ifTrue: [ 
			nonRowanClass
				ifTrue: [ result addAll: ' byteSubclass: ''' ]
				ifFalse: [ result addAll: ' rwByteSubclass: ''' ].
			result
				addAll: className;
				addLast: $'.
			newByteSubclass := true ]
		ifFalse: [ 
			(self theClass isIndexable and: [ superClass isIndexable not ])
				ifTrue: [ 
					nonRowanClass
						ifTrue: [ result addAll: ' indexableSubclass: ''' ]
						ifFalse: [ result addAll: ' rwIndexableSubclass: ''' ].
					result
						addAll: className;
						addLast: $' ]
				ifFalse: [ 
					nonRowanClass
						ifTrue: [ result addAll: ' subclass: ''' ]
						ifFalse: [ result addAll: ' rwSubclass: ''' ].
					result
						addAll: className;
						addLast: $' ] ].
	newByteSubclass
		ifFalse: [ 
			result
				addAll: lfsp;
				addAll: 'instVarNames: #(';
				addAll:
						(self theClass _instVarNamesWithSeparator: lfsp , '                 ');
				add: $) ].
	result
		addAll: lfsp;
		addLast: 'classVars: #('.
	self theClass _sortedClassVarNames
		do: [ :aKey | 
			result addLast: $ .
			(aKey includesValue: $')
				ifTrue: [ result addAll: aKey _asSource ]
				ifFalse: [ result addAll: aKey ] ].
	result addLast: $).
	result
		addAll: lfsp;
		addLast: 'classInstVars: #('.
	civs := self theClass class allInstVarNames.
	civs removeFrom: 1 to: self theClass class superClass instSize.
	civs
		do: [ :civName | 
			result addLast: $ .
			(civName includesValue: $')
				ifTrue: [ result addAll: civName _asSource ]
				ifFalse: [ result addAll: civName ] ].
	result addLast: $).
	result
		addAll: lfsp;
		addAll: 'poolDictionaries: '.
	result addAll: '#()'.	"ignored for now"
	nonRowanClass
		ifTrue: [ 
			"if the class is unpackaged, then we need to provide for the specification of symbol dictionary into which the class will be installed"
			result
				addAll: lfsp;
				addAll: 'inDictionary: '.
			anArray := Rowan image symbolList dictionariesAndSymbolsOf: self theClass.
			anArray isEmpty
				ifTrue: [ result addAll: '''''' ]
				ifFalse: [ result addAll: ((anArray at: 1) at: 1) name asString ] ]
		ifFalse: [ 
			result
				addAll: lfsp;
				addAll: 'category: '.
			result addAll: self theClass category printString.
			(true and: [ thePackageName = self theClass category ])
				ifFalse: [ 
					result
						addAll: lfsp;
						addAll: 'packageName: '.
					result addAll: thePackageName printString ] ].
	self theClass _hasConstraints
		ifTrue: [ 
			result
				add: lfsp;
				add: self theClass _rwDefinitionOfConstraints ].
	result
		add: lfsp;
		add: self theClass _rwOptionsForDefinition.
	result add: Character lf.
	^ result
]

{ #category : 'client commands' }
RowanClassService >> classHierarchy [
	hierarchyServices := self classHierarchy: (Array with: self theClass). 
	RowanCommandResult addResult: self.
]

{ #category : 'Accessing' }
RowanClassService >> classHierarchyNames [

	| names |
	names := Array new. 
	hierarchyServices keys do:[:classService | 
		classService == #nil ifFalse:[names add: classService name]].
	^names
]

{ #category : 'Accessing' }
RowanClassService >> classInstVarNames [
	^classInstVarNames

]

{ #category : 'Updating' }
RowanClassService >> classInstVarNames: newValue [
	classInstVarNames := newValue

]

{ #category : 'Accessing' }
RowanClassService >> classOrMeta [

	^meta 
			ifTrue:[self theClass class] 
			ifFalse: [self theClass].
]

{ #category : 'instance creation' }
RowanClassService >> classServiceFromOop: anOop [
	| theClass className classService |
	theClass := Object _objectForOop: anOop. 
	className := theClass name. 
	classService := RowanClassService new name: className.
	^className asString = name asString ifTrue:[
			className asString = 'Object' 
				ifTrue:[
					classService basicRefreshFrom: theClass]
				ifFalse:[
					classService fastRefresh]]
		ifFalse:[
			classService minimalRefreshFrom: theClass]
]

{ #category : 'instance creation' }
RowanClassService >> classServiceFromOop: anOop packageNames: packageNames [
	| theClass className classService |
	theClass := Object _objectForOop: anOop. 
	className := theClass name. 
	classService := RowanClassService new name: className.
	^classService minimalRefreshFrom: theClass packageNames: packageNames
]

{ #category : 'Accessing' }
RowanClassService >> classType [
	^classType

]

{ #category : 'Updating' }
RowanClassService >> classType: newValue [
	classType := newValue

]

{ #category : 'Accessing' }
RowanClassService >> classVarNames [
	^classVarNames

]

{ #category : 'Updating' }
RowanClassService >> classVarNames: newValue [
	classVarNames := newValue

]

{ #category : 'Accessing' }
RowanClassService >> comment [
	^comment

]

{ #category : 'Updating' }
RowanClassService >> comment: newValue [
	comment := newValue

]

{ #category : 'constants' }
RowanClassService >> compileMethod: methodString behavior: aBehavior symbolList: aSymbolList inCategory: categorySymbol [
	"returns (nil -> anArrayOfErrors) or (aGsNMethod -> compilerWarnings) or (aGsNMethod -> nil)"

	| method warnings |
	
	[ [ [ [ method := aBehavior rwCompileMethod: methodString category: categorySymbol.]
		on: RwExecuteClassInitializeMethodsAfterLoadNotification
		do: [:ex | ex resume: false ]]
			on: CompileError
			do: [:ex | ^nil -> (ex gsArguments at: 1)]]
				on: CompileWarning
				do: 
					[:ex | 
					warnings := ex warningString.
					ex resume]]
					on: RwPerformingUnpackagedEditNotification
					do: [:ex | ex resume ] .
	^[(self compiledMethodAt: method key selector inClass: aBehavior) -> warnings] on: Error
		do: [:ex | ex return: method -> warnings]
]

{ #category : 'client commands' }
RowanClassService >> copyClassTo: newClassName [
	| newTemplate newClass newClassService index |
	(Rowan image symbolList resolveSymbol: newClassName)
		ifNotNil: [ ^ self inform: newClassName , ' already exists' ].
	index := template findPattern: (Array with: name) startingAt: 1.
	newTemplate := template copy.
	newTemplate removeFrom: index to: index + name size - 1.
	newTemplate insertAll: newClassName at: index.
	newClass := GsCurrentSession currentSession execute: newTemplate.
	newClassService := RowanClassService new name: newClassName.
	self theClass thisClass
		methodsDo: [ :selector :gsMethod | 
			newClassService
				compileMethod: gsMethod sourceString
				behavior: newClass
				symbolList: Rowan image symbolList
				inCategory: (self theClass thisClass categoryOfSelector: selector) asSymbol ].
	self theClass thisClass class
		methodsDo: [ :selector :gsMethod | 
			newClassService
				compileMethod: gsMethod sourceString
				behavior: newClass class
				symbolList: Rowan image symbolList
				inCategory:
					(self theClass thisClass class categoryOfSelector: selector) asSymbol ].
	newClassService update.
	(RowanPackageService forPackageNamed: newClassService packageName) update.
	(RowanDictionaryService new name: dictionaryName) update
]

{ #category : 'Updating' }
RowanClassService >> definedPackageName: newValue [

	definedPackageName := newValue

]

{ #category : 'Accessing' }
RowanClassService >> dictionaryName [
	^dictionaryName
]

{ #category : 'rsr' }
RowanClassService >> executeCommand [

	updateAfterCommand ifNil: [updateAfterCommand := true]. 
	^super executeCommand
]

{ #category : 'Updating' }
RowanClassService >> expand: boolean [

	expand := boolean
]

{ #category : 'client commands' }
RowanClassService >> fastRefresh [
	"pushes less information to ston so it's faster"

	| theClass |
	theClass := self theClass. 
	self refreshFrom: theClass. 
	methods do:[:service1 |
			service1 source: nil;
				stepPoints: Array new].
	visibleTests do:[:service2 |
			service2 source: nil;
				stepPoints: Array new.
			].
	RowanCommandResult addResult: self.
]

{ #category : 'client commands' }
RowanClassService >> fileoutCategories: array on: path [
	| ws file |
	ws := WriteStream on: String new.
	self writeFileOutHeaderOn: ws.
	array
		do: [ :category | ws nextPutAll: (self behavior fileOutCategory: category) ].
	(GsFile existsOnServer: path)
		ifTrue: [ 
			(self confirm: 'File exists. File out anyway?')
				ifFalse: [ ^ self ] ].
	file := GsFile openAppendOnServer: path.
	[ file nextPutAll: ws contents ]
		ensure: [ file close ]
]

{ #category : 'client commands' }
RowanClassService >> fileoutClassOn: path [
	| ws |
	ws := WriteStream on: String new.
	self writeFileOutHeaderOn: ws.
	ws nextPutAll: self theClass fileOutClass.
	self fileOut: ws on: path
]

{ #category : 'Accessing' }
RowanClassService >> filters [
	^filters

]

{ #category : 'Updating' }
RowanClassService >> filters: newValue [
	filters := newValue

]

{ #category : 'initialization' }
RowanClassService >> forClassNamed: className [ 

	| theClass |
	self name: className. 
	theClass := self theClass. 
	self refreshFrom: theClass.
]

{ #category : 'client commands' }
RowanClassService >> fullHierarchy [
	| behavior sortedSubclasses |
	behavior := self theClass.
	hierarchyServices := Dictionary new.
	hierarchyServices at: #'expand' put: Array new.
	sortedSubclasses := behavior subclasses
		asSortedCollection: [ :x :y | x name < y name ].
	updateType := #'updatedFullInClassHierarchy:browser:'.
	RowanCommandResult addResult: self.
	sortedSubclasses
		do: [ :subclass | 
			| classService |
			classService := (self classServiceFromOop: subclass asOop) meta: meta.
			(hierarchyServices at: #'expand') add: classService.
			classService allSubclassServices ]
]

{ #category : 'comparing' }
RowanClassService >> hash [
	^self name hash bitXor: meta hash
]

{ #category : 'Accessing' }
RowanClassService >> hasSubclasses [
	^hasSubclasses
]

{ #category : 'Accessing' }
RowanClassService >> hasSubclasses: object [
	hasSubclasses := object
]

{ #category : 'Accessing' }
RowanClassService >> hierarchyServices [

	^hierarchyServices
]

{ #category : 'initialization' }
RowanClassService >> initialize [

	isExtension := false.
	selectedMethods := Array new.
	meta := false. "assume most of our work is on the instance side"
	selectedPackageServices := Array new.
	isNewClass := false.
	methods := Array new.
	isInSymbolList := true.
	categories := Array new.
	updateAfterCommand := true.
	hasSubclasses := false.
]

{ #category : 'initialization' }
RowanClassService >> initializeCategoriesFor: classOrMeta [

	| theFilters |
	theFilters := SortedCollection new.
	classOrMeta env: 0 categorysDo: [:category :selector | theFilters add: category asString].
	categories := theFilters asOrderedCollection.
]

{ #category : 'method history' }
RowanClassService >> initializeMethodHistoryFor: source [
  "about to compile a method. If possible, ensure it's method history is setup."

  | rowanMethodHistory methodHistory selector methodService |
  rowanMethodHistory := self rowanMethodHistory.
  selector := [ (Rowan platform parseSelectorFrom: source) asSymbol ]
    on: CompileWarning
    do: [ :ex | ex resume ].
  selector = #'_____could_not_parse_selector_from_method_source_____'
    ifTrue: [ ^ self	"invalid source, continue and let save method fail" ]
    ifFalse: [ 
      | compiledMethod |
      compiledMethod := (Object _objectForOop: oop)
        compiledMethodAt: selector
        environmentId: 0
        otherwise: nil.
      compiledMethod
        ifNil: [ ^ self	"we'll create history after the method is compiled" ].
      methodService := RowanMethodService
        forSelector: selector
        class: self theClass
        meta: meta
        organizer: self organizer.
      methodHistory := rowanMethodHistory
        at: methodService unregisteredCopy
        ifAbsentPut: [ Array with: methodService ] ]
]

{ #category : 'initialization' }
RowanClassService >> initializeTestMethodsFor: aClass [
	| testSelectors |
	(aClass inheritsFrom: TestCase) ifTrue:[
		aClass isAbstract ifTrue:[^self]. 
		testSelectors := aClass thisClass allTestSelectors.
		methods do:[:methodService | 
			methodService isTestMethod: (testSelectors includes: methodService selector)]].
]

{ #category : 'initialization' }
RowanClassService >> initializeVariablesFor: classOrMeta [

	| theFilters |
	theFilters := SortedCollection new.
	theFilters addAll: (classOrMeta allInstVarNames collect:[:instVar | instVar asString]).
	variables := theFilters asOrderedCollection.
]

{ #category : 'Accessing' }
RowanClassService >> instVarNames [
	^instVarNames

]

{ #category : 'Updating' }
RowanClassService >> instVarNames: newValue [
	instVarNames := newValue

]

{ #category : 'testing' }
RowanClassService >> isClassService [

	^true
]

{ #category : 'Updating' }
RowanClassService >> isExtension: boolean [

	isExtension := boolean

]

{ #category : 'Updating' }
RowanClassService >> isNewClass: boolean [
	isNewClass := boolean
]

{ #category : 'testing' }
RowanClassService >> isPackageClean [

	^self packageIsDirty not
]

{ #category : 'testing' }
RowanClassService >> isProjectClean [
  ^ self projectIsDirty not
]

{ #category : 'Updating' }
RowanClassService >> isTestCase [

	^isTestCase
]

{ #category : 'Updating' }
RowanClassService >> isTestCase: aBoolean [

	isTestCase := aBoolean
]

{ #category : 'testing' }
RowanClassService >> loadedPackageExistsAndIsInSameDictionary: thePackageName [
	| actualName loadedPackage packageDictionaryName |
	actualName := Rowan image packageNames
		detect: [ :loadedName | loadedName asLowercase = thePackageName asLowercase ]
		ifNone: [  ].
	loadedPackage := Rowan image loadedPackageNamed: actualName ifAbsent: [  ].
	^ loadedPackage
		ifNil: [ false ]
		ifNotNil: [ 
			packageDictionaryName := loadedPackage loadedProject
				gemstoneSymbolDictNameForPackageNamed: thePackageName.
			packageDictionaryName = dictionaryName ]
]

{ #category : 'Accessing' }
RowanClassService >> meta [

	^meta

]

{ #category : 'Updating' }
RowanClassService >> meta: anObject [

	meta := anObject

]

{ #category : 'Accessing' }
RowanClassService >> methods [

	"for testing"
	^methods
]

{ #category : 'private' }
RowanClassService >> methodServiceFrom: gsNMethod in: behavior compiltationResult: compilationResult [
	| methodService |

	methodService := RowanMethodService forGsNMethod: gsNMethod organizer: self organizer. 
	methodService compilationWarnings: compilationResult value.
	^methodService
]

{ #category : 'private' }
RowanClassService >> methodServicesFor: classOrMeta organizer: theOrganizer [

	methods addAll: (classOrMeta selectors collect:[:sel | 
			RowanMethodService 
				forSelector: sel class: classOrMeta thisClass meta: meta organizer: theOrganizer])
]

{ #category : 'private' }
RowanClassService >> methodServicesFor: classOrMeta organizer: theOrganizer subclasses: subclasses [
	methods
		addAll:
			(classOrMeta selectors
				collect: [ :sel | 
					RowanMethodService
						forSelector: sel
						class: classOrMeta thisClass
						meta: meta
						organizer: theOrganizer
						subclasses: subclasses])
]

{ #category : 'private' }
RowanClassService >> methodsIn: theClass categories: theCategories [

	| selectors |
	selectors := Array new. 
	theCategories do:[:category |
		selectors addAll: (theClass selectorsIn: category)]. 
	^methods select:[:methodService | selectors includes: methodService selector]
]

{ #category : 'testing' }
RowanClassService >> methodsNamed: selector [
	"For testing. Multiple because class could have both instance and class methods"

	^methods select:[:methodService | methodService selector = selector]
]

{ #category : 'initialization' }
RowanClassService >> minimalForClassNamed: className [ 

	| theClass |
	self name: className. 
	theClass := self theClass. 
	self minimalRefreshFrom: theClass.
]

{ #category : 'initialization' }
RowanClassService >> minimalForClassNamed: className packageNames: packageNames [
	| theClass |
	self name: className.
	theClass := self theClass.
	self minimalRefreshFrom: theClass packageNames: packageNames
]

{ #category : 'initialization' }
RowanClassService >> minimalRefreshFrom: theClass [
	| classOrMeta  |
	command := nil. 
	commandArgs := nil. 
	versions := theClass classHistory size.
	version := theClass classHistory indexOf: theClass.
	oop := theClass asOop.
	classOrMeta := meta == true ifTrue:[theClass class] ifFalse:[theClass].
	packageName := definedPackageName := classOrMeta rowanPackageName.
	self setDictionary: classOrMeta.
	projectName := classOrMeta rowanProjectName.
	instVarNames := classOrMeta instVarNames asArray. 
	template := self classCreationTemplate.
	self initializeVariablesFor: classOrMeta. 
	self initializeCategoriesFor: classOrMeta.
	self setIsTestCase.
]

{ #category : 'initialization' }
RowanClassService >> minimalRefreshFrom: theClass packageNames: packageNames [
	| classOrMeta  |
	command := nil. 
	commandArgs := nil. 
	versions := theClass classHistory size.
	version := theClass classHistory indexOf: theClass.
	oop := theClass asOop.
	classOrMeta := meta == true ifTrue:[theClass class] ifFalse:[theClass].
	packageName := definedPackageName := (self computePackageNameFor: classOrMeta in: packageNames).
	self setDictionary: classOrMeta.
	projectName := classOrMeta rowanProjectName.
	instVarNames := classOrMeta instVarNames asArray. 
	template := self classCreationTemplateUsing: packageNames.
	self initializeVariablesFor: classOrMeta. 
	self initializeCategoriesFor: classOrMeta.
	self setIsTestCase.
]

{ #category : 'client commands' }
RowanClassService >> moveMethods: methodServices to: category [
	"update the dirty flag of the project & package both before and after the move"

	| behavior |
	behavior := self classOrMeta.
	methodServices
		do: [ :methodService | 
			| beforePackageName |
			methodService organizer: self organizer.
			beforePackageName := methodService packageName.
			behavior rwMoveMethod: methodService selector toCategory: category.
			methodService update.
			methodService updatePackageProjectAfterCategoryChange: beforePackageName ].
	self update.
	self selectedMethods: methodServices
]

{ #category : 'client commands' }
RowanClassService >> moveMethodSelectors: methodSelectors toPackageNamed: thePackageName [
	| targetPackageService theClass |
	theClass := meta ifTrue: [self theClass class] ifFalse: [self theClass]. 
	methodSelectors
		do: [ :selector | theClass rwMoveMethod: selector toPackage: thePackageName ].
	self update.
	targetPackageService := (RowanPackageService forPackageNamed: thePackageName)
		update.
	(RowanProjectService new name: targetPackageService projectName) update
]

{ #category : 'client commands' }
RowanClassService >> moveToPackageNamed: thePackageName [
	| sourcePackageService targetPackageService |
	sourcePackageService := RowanPackageService forPackageNamed: packageName. 
	self theClass rwMoveClassToPackage: thePackageName.
	self update.
	sourcePackageService update. 
	(RowanProjectService new name: sourcePackageService projectName) update.
	(targetPackageService := RowanPackageService forPackageNamed: thePackageName) update.
	(RowanProjectService new name: targetPackageService projectName) update
]

{ #category : 'Accessing' }
RowanClassService >> name [
	^name

]

{ #category : 'Updating' }
RowanClassService >> name: newValue [
	name := newValue asString
]

{ #category : 'private' }
RowanClassService >> objectInBaseNamed: aString [

	^Rowan image symbolList objectNamed: aString asSymbol
]

{ #category : 'client commands' }
RowanClassService >> oneLevelClassHierarchy [
	"good for expanding an existing hierarchy quickly"

	| behavior sortedSubclasses |
	behavior := self theClass.
	hierarchyServices := Dictionary new.
	hierarchyServices at: #'expand' put: Array new.
	sortedSubclasses := behavior subclasses
		asSortedCollection: [ :x :y | x name < y name ].
	sortedSubclasses
		do: [ :subclass | 
			| classService |
			classService := (self classServiceFromOop: subclass asOop) meta: meta.
			(hierarchyServices at: #'expand') add: classService.
			(self organizer subclassesOf: subclass) notEmpty
				ifTrue: [ classService hasSubclasses: true ] ].
	updateType := #'updatedOneLevelInClassHierarchy:browser:'.
	RowanCommandResult addResult: self
]

{ #category : 'Accessing' }
RowanClassService >> oop [
	^oop

]

{ #category : 'Updating' }
RowanClassService >> oop: newValue [
	oop := newValue

]

{ #category : 'testing' }
RowanClassService >> packageIsDirty [
	| behavior |
	packageName
		ifNil: [ 
			behavior := self theClass.
			packageName := behavior rowanPackageName ].
	packageName = Rowan unpackagedName
		ifTrue: [ ^ true ].	"avoid a refresh by assuming it's dirty"
	^ (RowanPackageService forPackageNamed: packageName) rowanDirty
]

{ #category : 'Accessing' }
RowanClassService >> packageName [
	
	^packageName

]

{ #category : 'Updating' }
RowanClassService >> packageName: pkgName [
	
	packageName := pkgName

]

{ #category : 'Accessing' }
RowanClassService >> poolDictionaryNames [
	^poolDictionaryNames

]

{ #category : 'Updating' }
RowanClassService >> poolDictionaryNames: newValue [
	poolDictionaryNames := newValue

]

{ #category : 'rsr' }
RowanClassService >> postCommandExecution [
	super postCommandExecution.
	methods
		ifNil: [ 
			methods := Array new	"may not have been a behavior" ]
		ifNotNil: [ methods do: [ :methodService | methodService clearOrganizers ] ]
]

{ #category : 'printing' }
RowanClassService >> printOn: aStream [

	super printOn: aStream. 
	aStream nextPut: $:. 
	aStream nextPutAll: (name ifNil: [nil printString])
]

{ #category : 'testing' }
RowanClassService >> projectIsDirty [

	| behavior |
	behavior := self theClass.
	behavior rowanProjectName =  Rowan unpackagedName ifTrue:[^true]. "avoid a refresh by assuming it's dirty" 
	^(RowanProjectService new name: behavior rowanProjectName) rowanDirty
]

{ #category : 'other' }
RowanClassService >> projectName [

	^projectName
]

{ #category : 'Updating' }
RowanClassService >> projectName: newValue [
	projectName := newValue
]

{ #category : 'initialization' }
RowanClassService >> refreshFrom: theClass [
	| classOrMeta  |
	self basicRefreshFrom: theClass. 
	classOrMeta := meta == true ifTrue:[theClass class] ifFalse:[theClass].
	self refreshMethodsFor: classOrMeta.
	shouldUpdate := false.
]

{ #category : 'initialization' }
RowanClassService >> refreshMethodsFor: classOrMeta [
	| gsNMethods subclasses |
	methods := SortedCollection sortBlock: [ :x :y | x selector < y selector ].
	subclasses := self organizer allSubclassesOf: classOrMeta thisClass.
	self methodServicesFor: classOrMeta organizer: self organizer subclasses: subclasses.
	methods := methods asOrderedCollection.
	classOrMeta allInstVarNames
		do: [ :instVar | 
			gsNMethods :=self  organizer accessorsOf: instVar inClass: classOrMeta.
			gsNMethods
				do: [ :gsNMethod | 
					| service |
					service := methods
						detect: [ :methodService | methodService selector = gsNMethod selector and:[methodService className = gsNMethod inClass name asString] ]
						ifNone: [  ].
					service ifNotNil: [ service accessedInstVars add: instVar asString ] ] ].
	self initializeTestMethodsFor: classOrMeta thisClass.
	self setVisibleTests.	"methods must be available"
	selectedPackageServices
		ifNotNil: [ 
			methods
				do: [ :methodService | 
					methodService
						inSelectedPackage:
							(selectedPackageServices
								detect: [ :selectedPackageService | selectedPackageService name = methodService packageName ]
								ifNone: [  ]) notNil ] ]
]

{ #category : 'client commands' }
RowanClassService >> removeCategories: theCategories [
	| theClass  | 
	self refreshFrom: self theClass. 
	theClass := self theClass.
	meta ifTrue:[theClass := theClass class]. 
	theCategories do: [:category |
		theClass rwRemoveCategory: category.
		].
	shouldUpdate := true.
]

{ #category : 'client commands' }
RowanClassService >> removeInstVar: instVarName [
	"assumes inst var refs were removed"

	| theClass definitionString browserService anonymousMethod |
	self refreshFrom: self theClass.
	theClass := self theClass.
	(self instVarNames includes: instVarName asSymbol)
		ifFalse: [ 
			| superService |
			superService := RowanClassService new
				name: theClass superClass name;
				meta: meta.
			^ superService removeInstVar: instVarName ].
	meta
		ifTrue: [ theClass := theClass class ].
	definitionString := self template.
	definitionString := self
		replaceSubString: ' ' , instVarName , ' '
		in: definitionString
		with: ' '.
	definitionString := self
		replaceSubString: ' ' , instVarName , ')'
		in: definitionString
		with: ')'.
	definitionString := self
		replaceSubString: '(' , instVarName , ' '
		in: definitionString
		with: '('.
	definitionString := self
		replaceSubString: '(' , instVarName , ')'
		in: definitionString
		with: '()'.
	anonymousMethod := definitionString
		_compileInContext: nil
		symbolList: Rowan image symbolList.
	SessionTemps current at: #'jadeiteCompileClassMethod' put: anonymousMethod.
	browserService := RowanBrowserServiceServer new.
	browserService recompileMethodsAfterClassCompilation.
	shouldUpdate := true
]

{ #category : 'client commands' }
RowanClassService >> removeMethods: methodsToRemove [

	| notRemoved |
	notRemoved := Array new. 
	methodsToRemove do: [:methodService |
		self removeSelector: methodService selector ifAbsent:[notRemoved add: methodService].
		(notRemoved includes: methodService) ifFalse:[
			methodService updateType: #removed:.
			RowanCommandResult addResult: methodService.
		]].
	self updateTests.
	notRemoved isEmpty ifFalse:[
		self error: 'These selectors were not removed - ', (notRemoved collect:[:svc | svc selector]) printString].
]

{ #category : 'rowan' }
RowanClassService >> removeSelector: selector [

	self browserTool removeMethod: selector forClassNamed: name asString isMeta: meta

]

{ #category : 'rowan' }
RowanClassService >> removeSelector: selector ifAbsent: absentBlock [
	| theClass |
	theClass := self theClass. 
	meta ifTrue: [theClass := theClass class].
	(theClass compiledMethodAt: selector otherwise: nil) isNil ifTrue:[ ^absentBlock value ].
	[self browserTool removeMethod: selector forClassNamed: name asString isMeta: meta]
		on: RwPerformingUnpackagedEditNotification
		do: [:ex | ex resume ]
]

{ #category : 'client commands' }
RowanClassService >> renameCategoryFrom: old to: new [

	| affectedSelectors behavior |

	self update. 
	self addCategory: new. 
	behavior := self classOrMeta.
	affectedSelectors := behavior selectorsIn: old.
	methods := methods select:[:methodService | affectedSelectors includes: methodService selector].
	self moveMethods: methods to: new.
	self removeCategories: (Array with: old).
	self updateClass.
]

{ #category : 'client commands' }
RowanClassService >> renameClass: oldClassName to: newClassName [
	"needs better class reference replacement than just string replacement"

	| references newMethods newClass oldClass |
	newMethods := Array new.
	oldClass := Rowan image resolveClassNamed: oldClassName.
	newClass := Rowan projectTools browser
		renameClassNamed: oldClassName
		to: newClassName.
	oop := newClass asOop.
	name := newClassName.
	self update.
	renamedName := oldClassName.
	self updateMethodsAfterRenameFrom: oldClassName to: newClassName.
	self updateSubclassesOf: newClass.
	references := self organizer update referencesToObject: oldClass.
	references
		do: [ :method | 
			| newSource compileResult failedCompile methodService oldSource |
			failedCompile := false.
			oldSource := method sourceString.
			newSource := self
				replaceSubString: oldClassName
				in: oldSource
				with: newClassName.
			compileResult := [ 
			method inClass
				rwCompileMethod: newSource
				category: (method inClass categoryOfSelector: method selector) asSymbol ]
				on: CompileError
				do: [ :ex | 
					failedCompile := true.
					method ].
			methodService := RowanMethodService
				forGsNMethod: compileResult
				organizer: self organizer.
			failedCompile
				ifTrue: [ methodService comparisonSource: oldClassName ]
				ifFalse: [ methodService comparisonSource: oldSource ].
			methodService failedCompile: failedCompile.
			methodService renamedName: oldClassName.
			newMethods add: methodService ].
	selectedPackageServices do: [ :ea | ea update ].
	RowanCommandResult addResult: (RowanAnsweringService new answer: newMethods)
]

{ #category : 'private' }
RowanClassService >> replaceSubString: old in: string with: new [
	| offset newSource |
	newSource := string. 
	offset := 1. 	
	[(offset := newSource findString: old startingAt: offset) = 0] whileFalse:[
		newSource := newSource copyReplaceFrom: offset to: offset + old size - 1 with: new. 
		offset := offset + new size. 
	].
	^newSource
]

{ #category : 'rowan' }
RowanClassService >> rowanProjectName [

	^projectName
]

{ #category : 'client commands' }
RowanClassService >> runClassTests: classService [

	"if it errors, the client will handle the error. 
	If it passes, we return true and the client
	will display decent results." 
	| behavior |
	behavior := classService theClass. 
	self refreshFrom: behavior.
	self tests do:[:methodService |
			behavior debug: methodService selector]. 
	RowanCommandResult addResult: (RowanAnsweringService new answer: true).
]

{ #category : 'client commands' }
RowanClassService >> runMethodTests: methodServices [

	| behavior |
	behavior := self theClass.  
	methodServices do:[:methodService |
		(methodService selector asString matchPattern: #('test' $*)) ifTrue:[ 
			behavior debug: methodService selector]].
	RowanCommandResult addResult: (RowanAnsweringService new answer: true).
]

{ #category : 'client commands' }
RowanClassService >> saveMethodSource: source category: category [
	| behavior compilationResult gsNMethod updatedCategory methodService unicodeSource |
	unicodeSource := source. 
	meta
		ifNil: [ 
			behavior := Object _objectForOop: oop.
			meta := behavior isMeta ]
		ifNotNil: [ 
			behavior := meta
				ifTrue: [ self theClass class ]
				ifFalse: [ self theClass ] ].
	oop := behavior asOop.
	self initializeMethodHistoryFor: unicodeSource.
	updatedCategory := category ifNil: [ 'other' ].
	compilationResult := self
		compileMethod: unicodeSource
		behavior: behavior
		symbolList: Rowan image symbolList
		inCategory: updatedCategory asSymbol.
	(gsNMethod := compilationResult key) isNil
		ifTrue: [ 
			System
				signal: 1001
				args: (Array with: compilationResult value)
				signalDictionary: GemStoneError ].
	organizer := ClassOrganizer new. 
	methodService := self
		methodServiceFrom: gsNMethod
		in: behavior
		compiltationResult: compilationResult.
	RowanCommandResult addResult: methodService.
	self refreshFrom: self theClass. "make sure new methods have proper inst var refs"
	RowanQueryService new
		organizer: ClassOrganizer new;
		hierarchyImplementorsOf: methodService selector
			inClass: methodService className.	"this will update hierarchy method indicators for client"
	self selectedMethods: (Array with: methodService).
	self updateDirtyState.
	(methods includes: methodService)
		ifFalse: [ methods add: methodService ].
	methodService isTestMethod
		ifTrue: [ self updateTests ].
	self
		updateSymbols:
			gsNMethod _selectorPool asArray , (Array with: methodService selector).
	methodService addToMethodHistory.
	RowanCommandResult addResult: self
]

{ #category : 'client commands' }
RowanClassService >> saveMethodSources: sources category: category [
	sources do:[:source | 
		self saveMethodSource: source category: category]
]

{ #category : 'other' }
RowanClassService >> selectedMethods [
	"client side selection. Used after a method compile" 
	^selectedMethods
]

{ #category : 'Updating' }
RowanClassService >> selectedMethods: theMethods [
	selectedMethods := theMethods
]

{ #category : 'Accessing' }
RowanClassService >> selectedPackageServices [
	^selectedPackageServices
]

{ #category : 'Updating' }
RowanClassService >> selectedPackageServices: newValue [
	selectedPackageServices := newValue
]

{ #category : 'Accessing' }
RowanClassService >> selectors [

	^methods collect:[:methodService | methodService selector]
]

{ #category : 'perform' }
RowanClassService >> servicePerform: symbol withArguments: collection [
	| wasClean |
	packageName ifNil: [ self update ].	"the class may not have enough information to perform the symbol"
	self isUpdatingButFoundToBeDeleted
		ifTrue: [ ^ self handleDeletedService ].
	wasClean := self isPackageClean.
	super
		servicePerform: symbol
		withArguments: collection
		shouldUpdate: updateAfterCommand.
	wasClean
		ifTrue: [ self updatePackageAndProject ]
]

{ #category : 'Accessing' }
RowanClassService >> setComment [
  comment := self theClass thisClass comment
]

{ #category : 'private' }
RowanClassService >> setDictionary: classOrMeta [
	| dictionaryList |
		dictionaryList := Rowan image symbolList dictionariesAndSymbolsOf: classOrMeta thisClass.
		dictionaryName := dictionaryList isEmpty 
		ifTrue:[String new]
		ifFalse:[dictionaryList first first name asString].
]

{ #category : 'Updating' }
RowanClassService >> setIsTestCase [

	isTestCase := self theClass isSubclassOf: TestCase

]

{ #category : 'client commands' }
RowanClassService >> setIsTestCaseCommand [

	self setIsTestCase.
]

{ #category : 'client commands' }
RowanClassService >> setVisibleTests [
	visibleTests := SortedCollection sortBlock: [:x :y | x selector < y selector]. 
	visibleTests addAll: self allTests.
	visibleTests := visibleTests asArray.
]

{ #category : 'client commands' }
RowanClassService >> subclassCreationTemplate [
  | answerService newClassPackageName |
  answerService := RowanAnsweringService new.
  newClassPackageName := self theClass rowanPackageName = Rowan unpackagedName
    ifTrue: [ self addSubclassWarningString ]
    ifFalse: [ self theClass rowanPackageName ].
  answerService
    answer:
      (self browserTool
        classCreationTemplateForSubclassOf: name
        className: 'NewSubclass'
        category: newClassPackageName).
  RowanCommandResult addResult: answerService
]

{ #category : 'private' }
RowanClassService >> subclassServices: subclasses [

	| sortedSubclasses |

	sortedSubclasses := SortedCollection sortBlock: [:x :y | x name < y name]. 
	sortedSubclasses addAll: subclasses. 
	^(sortedSubclasses collect:[:cls | (self classServiceFromOop: cls asOop) meta: meta]) asArray.
]

{ #category : 'Accessing' }
RowanClassService >> subclassType [
	^subclassType

]

{ #category : 'Updating' }
RowanClassService >> subclassType: newValue [
	subclassType := newValue

]

{ #category : 'Accessing' }
RowanClassService >> superclassName [
	^superclassName

]

{ #category : 'Updating' }
RowanClassService >> superclassName: newValue [
	superclassName := newValue

]

{ #category : 'Accessing' }
RowanClassService >> template [
	^template

]

{ #category : 'Updating' }
RowanClassService >> template: newValue [
	template := newValue

]

{ #category : 'private' }
RowanClassService >> tests [

	^methods select:[:methodService | methodService selector asString matchPattern: #('test' $*)]
]

{ #category : 'instance creation' }
RowanClassService >> theClass [
	| theClass |
	theClass := oop ifNil:[Rowan globalNamed: name] ifNotNil: [Object _objectForOop: oop].
	theClass isMeta ifTrue:[oop := theClass thisClass asOop]. 
	(Rowan globalNamed: name) ifNil:[isInSymbolList := false]. 
	theClass ifNil: [^nil]. 
	^theClass thisClass
]

{ #category : 'updates' }
RowanClassService >> update [ 
	super update.
	self updateClass.
]

{ #category : 'updates' }
RowanClassService >> updateClass [

	"It's possible to have a nil class. For example, if we added & selected
	a class then aborted."

	| theClass |
	theClass := self theClass. 
	theClass isNil ifTrue:[oop := nil. ^self]. 
	theClass isBehavior ifFalse:[oop := theClass asOop. ^self].
	self refreshFrom: theClass.
	RowanCommandResult addResult: self
]

{ #category : 'updates' }
RowanClassService >> updateDirtyState [
	| projectService wasDirty |
	selectedPackageServices
		do: [ :packageService | 
			wasDirty := packageService isDirty.
			wasDirty
				ifFalse: [ 
					packageService update.
					(packageService isDirty and: [ wasDirty not ])
						ifTrue: [ 
							RowanCommandResult addResult: packageService.
							projectService := RowanProjectService
								newNamed: self theClass rowanProjectName.
							RowanCommandResult addResult: projectService ] ] ]
]

{ #category : 'initialization' }
RowanClassService >> updateIsExtension [
  isExtension := ((selectedPackageServices
    collect: [ :packageService | packageService name ])
    includes: definedPackageName) not
]

{ #category : 'updates' }
RowanClassService >> updateLatest [
  oop := ((Rowan image symbolList resolveSymbol: name)
    ifNil: [ 
      wasRemoved := true.
      updateType := #'removedClass:'.
      RowanCommandResult addResult: self.
      ^ self ]) value asOop.
  super updateLatest
]

{ #category : 'private' }
RowanClassService >> updateMethodsAfterRenameFrom: oldClassName to: newClassName [
  methods
    do: [ :methodService | 
      methodService
        renamedName: oldClassName;
        className: newClassName. 
      RowanCommandResult addResult: methodService ]
]

{ #category : 'updates' }
RowanClassService >> updatePackageAndProject [
	| packageService projectService |
	packageName = Rowan unpackagedName ifTrue:[^self]. 
	packageService := RowanPackageService forPackageNamed: packageName.
	packageService update.
	projectService := RowanProjectService new name: projectName.
	projectService update
]

{ #category : 'private' }
RowanClassService >> updateSubclasses [
	self updateSubclassesOf: self theClass
]

{ #category : 'private' }
RowanClassService >> updateSubclassesOf: newClass [
  organizer := ClassOrganizer new.
  (self organizer allSubclassesOf: newClass)
    do: [ :subclass | 
      | subclassService |
      subclassService := RowanClassService minimalForClassNamed: subclass name.
      RowanCommandResult addResult: subclassService ]
]

{ #category : 'updates' }
RowanClassService >> updateSymbols: newSymbols [
  | browserService |
  browserService := RowanBrowserService new.
  browserService newCachedSelectors addAll: newSymbols.
  browserService updateType: #'addCachedSymbols:'. 
  RowanCommandResult addResult: browserService
]

{ #category : 'updates' }
RowanClassService >> updateTests [
	"update the test browsers on certain operations"

	RowanBrowserService new packagesWithTests.
	(RowanPackageService forPackageNamed: packageName) testClasses.
	RowanCommandResult addResult: self update
]

{ #category : 'accessing' }
RowanClassService >> updateType [
	^updateType
]

{ #category : 'accessing' }
RowanClassService >> updateType: object [
	updateType := object
]

{ #category : 'Accessing' }
RowanClassService >> version [
	^version

]

{ #category : 'Updating' }
RowanClassService >> version: newValue [
	version := newValue

]

{ #category : 'Accessing' }
RowanClassService >> versions [
	^versions

]

{ #category : 'Updating' }
RowanClassService >> versions: newValue [
	versions := newValue

]

{ #category : 'Accessing' }
RowanClassService >> visibleTests [

	^visibleTests
]

{ #category : 'testing' }
RowanClassService >> wasDeleted [
	^(Rowan globalNamed: name) isNil
]

{ #category : 'Accessing' }
RowanClassService >> wasRemoved: boolean [

	wasRemoved := boolean
]
