Extension { #name : 'CharacterCollection' }

{ #category : 'Indexing Support' }
CharacterCollection class >> _idxBasicCanCompareWithCharacterCollectionInstance: aCharacterCollection [
  "Returns true if <aCharacterCollection> may be inserted into a basic BtreeNode
   whose #lastElementClass is the receiver (see RangeEqualityIndex
   class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ true

]

{ #category : 'Indexing Support' }
CharacterCollection class >> _idxBasicCanCompareWithUnicodeInstance: aUnicodeString [
  "Returns true if <aUnicodeString> may be inserted into a basic BtreeNode whose
   #lastElementClass is the receiver (see RangeEqualityIndex class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ Unicode16 usingUnicodeCompares

]

{ #category : 'Instance Creation' }
CharacterCollection class >> _newString [

"Returns a new instance of the receiver, or instance of String as appropriate.
 Reimplemented in subclasses as needed to handle canonical symbols."

^ self new

]

{ #category : 'Private' }
CharacterCollection class >> _newString: aSize [

"Returns a new instance of the receiver, or instance of String as appropriate.
 Reimplemented in subclasses as needed to handle canonical symbols."

^ self new: aSize

]

{ #category : 'Formatting' }
CharacterCollection class >> charSize [

"(Subclass responsibility.)
 Returns number of bytes that make up a character for instances of this class."

^ self subclassResponsibility: #charSize

]

{ #category : 'Session Control' }
CharacterCollection class >> disableUnicodeComparisonMode [

  "Causes subsequent sessions, after login, to use Legacy methods for
   #< #> #= and various methods in the indexing
   when a receiver is a kind of String or MultiByteString.

   Signals an Error if the current user is not SystemUser .

   Returns a Boolean , the previous state of
     (Globals at: #StringConfiguration) == Unicode16 ."

   | prev |
   System myUserProfile userId = 'SystemUser' ifFalse:[
     Error signal:'Only SystemUser should execute this method'.
   ].
   prev := (Globals at: #StringConfiguration otherwise: nil) == Unicode16 .
   Globals at: #StringConfiguration put: String .
   System commit .
   ^ prev

]

{ #category : 'Session Control' }
CharacterCollection class >> enableUnicodeComparisonMode [

  "Causes subsequent sessions, after login, to use Unicode methods for
   #< #> #= and various methods in the indexing
   when a receiver is a kind of String or MultiByteString.

   Signals an Error if the current user is not SystemUser .

   Returns a Boolean , the previous state of
     (Globals at: #StringConfiguration) == Unicode16 ."

   | prev |
   System myUserProfile userId = 'SystemUser' ifFalse:[
     Error signal:'Only SystemUser should execute this method'.
   ].
   prev := (Globals at: #StringConfiguration otherwise: nil) == Unicode16 .
   Globals at: #StringConfiguration put: Unicode16 .
   System commit .
   ^ prev

]

{ #category : 'Deprecated' }
CharacterCollection class >> fromServerTextFile: aFileSpec [

self deprecated: 'CharacterCollection class >> fromServerTextFile: deprecated long before v3.0. Use an instance of GsFile
 to access the file system.'.
 "For multi-byte characters, assumes the byte order of the file
 matches the current session's cpu's in-memory byte order.  "

^ self new _fromServerTextFile: aFileSpec

]

{ #category : 'Instance Creation' }
CharacterCollection class >> fromStream: aStream width: anInteger [

"Returns a new instance of the receiver's class that contains the next
 anInteger Characters of aStream."

| result |

self _checkReadStream: aStream forClass: CharacterCollection.
result:= self new: anInteger.
1 to: anInteger do: [ :i | result at: i put: aStream next ].
^ result

]

{ #category : 'Session Control' }
CharacterCollection class >> isInUnicodeComparisonMode [

  "Returns a Boolean, true if Unicode versions of methods
   #< #> #= and various methods in the indexing
   are used when a receiver is a kind of String or MultiByteString"

  ^ Unicode16 usingUnicodeCompares

]

{ #category : 'Instance Creation' }
CharacterCollection class >> new [
"(Subclass responsibility.)"

^ self subclassResponsibility: #new

]

{ #category : 'Instance Creation' }
CharacterCollection class >> new: anInteger [
"(Subclass responsibility.)"

^ self subclassResponsibility: #new:

]

{ #category : 'Instance Creation' }
CharacterCollection class >> withAll: aSequenceableCollection [

"Returns a new instance of the receiver that contains the elements in the
 argument aSequenceableCollection."

| result |
result:= self new.
aSequenceableCollection accompaniedBy: result do: [:res :each | res add: each].
^result

]

{ #category : 'Message Digests' }
CharacterCollection >> _asMessageDigestKind: opCode [

"
opCode   Digest Algorithm   Digest Bits
=======================================
  1          md5              128
  2          sha1             160
  3          sha2-256         256
  4          sha2-512         512
  5          sha3-224         224
  6          sha3-256         256
  7          sha3-384         384
  8          sha3-256         512
=======================================

Postive opCode means return result as a LargeInteger.
Negative opCode means return result as a ByteArray.

"
<primitive: 666>
opCode _validateClass: SmallInteger .
(opCode == 0 or:[(opCode < -8) or:[ opCode > 8]])
  ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ -8 . 8 } ].
^ self _primitiveFailed: #_asMessageDigestKind:

]

{ #category : 'Private' }
CharacterCollection >> _asUtf8WithEol [
  "Return a Utf8 copy of the receiver, with a linefeed
   appended if receiver does not end with a linefeed.  Used by topaz"
  | lastCp sz str |
  sz := self size .
  sz ~~ 0 ifTrue:[ lastCp := self codePointAt: sz ].
  str := self .
  lastCp == 10 ifFalse:[
    (str := self copy) codePointAt: sz + 1 put: 10
  ].
  ^ str encodeAsUTF8

]

{ #category : 'Accessing' }
CharacterCollection >> _at: anIndex [

"Private.  Reimplemented to return a kind of Character."

^ self at: anIndex

]

{ #category : 'Private' }
CharacterCollection >> _charCollCompare: aCharCollection [

"Returns -1 if self < aCharCollection, returns 0 if self = aCharCollection,
 and returns 1 if self > aCharCollection."

| selfSize argSize selfElement argElement |

aCharCollection _validateClass: CharacterCollection.

selfSize:= self size.
argSize:= aCharCollection size.

1 to: (selfSize min: argSize) do: [:i |

  selfElement := self at: i.
  argElement  := aCharCollection at: i.

  (selfElement < argElement) ifTrue: [ ^ -1 ].
  (selfElement > argElement) ifTrue: [ ^ 1 ]
  ].

"All elements of self and argument are equal."

(selfSize < argSize) ifTrue: [ ^ -1 ].
(selfSize > argSize) ifTrue: [ ^ 1 ].

^ 0 "equal"

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _classSortOrdinal [

^ 10

]

{ #category : 'Comparing' }
CharacterCollection >> _coerceToUnicode [
  ^ self asUnicodeString

]

{ #category : 'Private' }
CharacterCollection >> _deepCopyWith: copiedObjDict [

| copy |

copy := copiedObjDict at: self otherwise: nil.
copy ifNotNil:[ ^ copy ].

^ self copy.

]

{ #category : 'Compatibility' }
CharacterCollection >> _encodeAsUTF8intoString [

^ self encodeAsUTF8IntoString

]

{ #category : 'Private' }
CharacterCollection >> _findLastString: subString startingAt: startIndex ignoreCase: aBoolean [

"If a receiver contains subString beginning at some point at or before
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.
 Search is backwards. "

 self  subclassResponsibility: #_findLastString:startingAt:ignoreCase: .

 self _uncontinuableError

]

{ #category : 'Searching' }
CharacterCollection >> _findPattern: aPattern startingAt: anIndex ignoreCase: caseInsens [

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 If caseInsens is true, a case-insensitive search is performed.  Otherwise,
 caseInsens should be false and a case-sensitive search is performed."

| i             "loop counter"
  pattern       "the argument aPattern, converted to an Array"
  selfSize      "the size of the receiver"
  patternSize   "the number of elements in the pattern Array"
  startIndex    "an Array that corresponds to the pattern Array, with each
                 element containing the starting index into the receiver of
                 the corresponding pattern"
  index         "the index into both the pattern and the startIndex Arrays"
  next          "the current pattern (element of the pattern Array) to
                 be matched"
  cursor        "the index into the receiver"
|

aPattern _validateClass: Array.
(anIndex <= 0) ifTrue: [ ^ self _errorIndexOutOfRange: anIndex ].

pattern := Array withAll: aPattern.
patternSize := pattern size.

" First, the pattern Array must be processed so that there are no *? pairs
  it in.  They must all be converted to ?* pairs for the algorithm to work
  correctly."
i := 1.
[ i < patternSize ]
whileTrue:
  [ ( ((pattern at: i) isEquivalent: $*) and:[(pattern at: i+1) isEquivalent: $?])
    ifTrue:
      [ pattern at: i put: $?.
        pattern at: i+1 put: $*.
        i := 1 max: i-1.
      ]
    ifFalse:
      [ i := i + 1 ].
  ].

"initialize"
selfSize := self size.
startIndex := Array new: (patternSize + 1)."last element is set, but not used"
index := 1.
anIndex > selfSize
   ifTrue: [startIndex at: 1 put: (selfSize + 1)] "Fix for bug 15038"
   ifFalse: [startIndex at: 1 put: anIndex].

(patternSize == 0) "no pattern to match"
ifTrue:
  [ startIndex <= selfSize
    ifTrue:
      [ ^ startIndex ]
    ifFalse:
      [ ^ 0 ].
  ].

[index <= patternSize]
whileTrue:
  [ next := pattern at: index.
    cursor := startIndex at: index.

    (next isKindOf: CharacterCollection) "pattern element is a string"
    ifTrue:
      [ ((cursor > selfSize) or:
        [(cursor + next size - 1) > selfSize]) "pattern element too big to "
        ifTrue: "match beginning at cursor location"
          [ ^ 0 ].
        ((index == 1) or: [ (pattern at: index - 1) isEquivalent: $* ])
        ifTrue: "is first pattern or end of *; can skip chars to find match"
          [ cursor := self _findString: next startingAt: cursor
				ignoreCase: caseInsens .
            (cursor == 0)
            ifTrue:
              [ ^ 0 ]
            ifFalse:
              [ startIndex at: index put: cursor.
                startIndex at: index + 1 put: cursor + next size.
              ]
          ]
        ifFalse: "can't skip chars to find match"
          [ (self _at: cursor equals: next ignoreCase: caseInsens )
            ifTrue:
              [ startIndex at: index + 1 put: cursor + next size ]
            ifFalse:
              [
                [ (index := index - 1) < 1
                  ifTrue:
                    [ ^ 0 ].
                  ((pattern at: index) isKindOf: CharacterCollection) or:
                  [ (index == 1) or: [ (pattern at: index - 1) isEquivalent: $* ] ]
                ]
                untilTrue.
                startIndex at: index put: ((startIndex at: index) + 1).
                index := index - 1.
              ].
          ].
      ]
    ifFalse: "pattern element not a string"
      [ (next isEquivalent: $*) "pattern element is *"
        ifTrue:
          [ startIndex at: (index + 1) put: cursor
          ]
        ifFalse:
          [ (next isEquivalent: $?)  "pattern element is ?"
            ifTrue:
              [ cursor > selfSize
                ifTrue:
                  [ ^ 0 ].
                startIndex at: (index + 1) put: (cursor + 1)
              ]
            ifFalse: "found a pattern element other than ?, * or string"
              [ ^ aPattern _error: #rtErrBadPattern
              ].
          ].
      ].

    index := index + 1
  ].

^ startIndex at: 1

]

{ #category : 'Private' }
CharacterCollection >> _findString: subString startingAt: startIndex ignoreCase: aBoolean [

CharacterCollection
  subclassResponsibility: #_findString:startingAt:ignoreCase: .

self _uncontinuableError

]

{ #category : 'Instance Creation' }
CharacterCollection >> _fromServerTextFile: aFileSpec [

"Import the contents of aFileSpec into the receiver.
 For multi-byte characters, assumes the byte order of the file
 matches the current session's cpu's in-memory byte order.
"
<primitive: 302>

aFileSpec _validateByteClass: CharacterCollection .
aFileSpec _error: #hostErrFileImport args: #()

]

{ #category : 'Indexing Support' }
CharacterCollection >> _idxBasicCanCompareWithClass: aClass [
  "Returns true if the receiver may be inserted into a basic BtreeNode whose
   #lastElementClass is <aClass> (see RangeEqualityIndex class>>isBasicClass:)."

  "If using Unicode compares, then *String and Unicode* instances may be compared.
   If not using Unicode compares then *String and Unicode* instances may not be compared."

  ^ aClass _idxBasicCanCompareWithCharacterCollectionInstance: self

]

{ #category : 'New Indexing Comparison - for Compare' }
CharacterCollection >> _idxForCompareCharacterCollectionGreaterThanOrEqualToSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForCompareGreaterThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any >= comparison"

^(aCharacterCollection _idxPrimCompareLessThan: self) not

]

{ #category : 'New Indexing Comparison - for Compare' }
CharacterCollection >> _idxForCompareCharacterCollectionGreaterThanSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForCompareGreaterThan:. Note that aCharacterCollection should be the receiver in any > comparison"

^aCharacterCollection _idxPrimCompareGreaterThan: self

]

{ #category : 'New Indexing Comparison - for Compare' }
CharacterCollection >> _idxForCompareCharacterCollectionLessThanOrEqualToSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForCompareLessThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any <= comparison"

^(aCharacterCollection _idxPrimCompareGreaterThan: self) not

]

{ #category : 'New Indexing Comparison - for Compare' }
CharacterCollection >> _idxForCompareCharacterCollectionLessThanSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForCompareLessThan:. Note that aCharacterCollection should be the receiver in any < comparison"

^aCharacterCollection _idxPrimCompareLessThan: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareEqualTo: arg [
  ""

  ^ arg _idxForCompareEqualToCharacterCollection: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator [
  "treat receiver and aCharacterCollection as Unicode, since anIcuCollator is supplied"

  ^ aCharacterCollection
    _idxForCompareEqualToUnicode: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareGreaterThan: arg [

""

^arg _idxForCompareCharacterCollectionGreaterThanSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareGreaterThan: aCharacterCollection collator: anIcuCollator [
  "treat receiver and aCharacterCollection as Unicode, since anIcuCollator is supplied"

  ^ aCharacterCollection
    _idxForCompareGreaterThanUnicode: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareGreaterThanOrEqualTo: arg [

""

^arg _idxForCompareCharacterCollectionGreaterThanOrEqualToSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareGreaterThanOrEqualTo: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForCompareGreaterThanOrEqualToUnicode: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareLessThan: arg [

""

^arg _idxForCompareCharacterCollectionLessThanSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareLessThan: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForCompareLessThanUnicode: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareLessThanOrEqualTo: arg [

""

^arg _idxForCompareCharacterCollectionLessThanOrEqualToSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareLessThanOrEqualTo: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForCompareLessThanOrEqualToUnicode: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareNotEqualTo: arg [

""

^ (self _idxForCompareEqualTo: arg) not

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForCompareNotEqualTo: aCharacterCollection collator: anIcuCollator [
  "REL_SIG_STRING_SIZE = 900"

  ^ (self _idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator)
    not

]

{ #category : 'New Indexing Comparison - for Sort' }
CharacterCollection >> _idxForSortCharacterCollectionGreaterThanOrEqualToSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForSortGreaterThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any >= comparison"

^(aCharacterCollection _idxPrimCompareLessThan: self) not

]

{ #category : 'New Indexing Comparison - for Sort' }
CharacterCollection >> _idxForSortCharacterCollectionGreaterThanSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForSortGreaterThan:. Note that aCharacterCollection should be the receiver in any > comparison"

^aCharacterCollection _idxPrimCompareGreaterThan: self

]

{ #category : 'New Indexing Comparison - for Sort' }
CharacterCollection >> _idxForSortCharacterCollectionLessThanOrEqualToSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForSortLessThanOrEqualTo:. Note that aCharacterCollection should be the receiver in any <= comparison"

^(aCharacterCollection _idxPrimCompareGreaterThan: self) not

]

{ #category : 'New Indexing Comparison - for Sort' }
CharacterCollection >> _idxForSortCharacterCollectionLessThanSelf: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForSortLessThan:. Note that aCharacterCollection should be the receiver in any < comparison"

^aCharacterCollection _idxPrimCompareLessThan: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortEqualTo: arg [

""

^arg _idxForSortEqualToCharacterCollection: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForSortEqualToCharacterCollection: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison - for Sort' }
CharacterCollection >> _idxForSortEqualToCharacterCollection: aCharacterCollection [
  "second half of a double dispatch call from CharacterCollection>>_idxForSortEqualTo:."

  ^ self _idxPrimCompareEqualTo: aCharacterCollection

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortGreaterThan: arg [

""

^arg _idxForSortCharacterCollectionGreaterThanSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortGreaterThan: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForSortGreaterThanCharacterCollection: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortGreaterThanOrEqualTo: arg [

""

^arg _idxForSortCharacterCollectionGreaterThanOrEqualToSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortGreaterThanOrEqualTo: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForSortGreaterThanOrEqualToCharacterCollection: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortLessThan: arg [

""

^arg _idxForSortCharacterCollectionLessThanSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortLessThan: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForSortLessThanCharacterCollection: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortLessThanOrEqualTo: arg [

""

^arg _idxForSortCharacterCollectionLessThanOrEqualToSelf: self

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortLessThanOrEqualTo: aCharacterCollection collator: anIcuCollator [

  ^ aCharacterCollection
    _idxForSortLessThanOrEqualToCharacterCollection: self
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortNotEqualTo: arg [

""

^(arg _idxForSortEqualTo: self) not

]

{ #category : 'New Indexing Comparison' }
CharacterCollection >> _idxForSortNotEqualTo: aCharacterCollection collator: anIcuCollator [

  ^ (self _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator) not

]

{ #category : 'New Indexing Comparison - for Sort' }
CharacterCollection >> _idxForSortNotEqualToCharacterCollection: aCharacterCollection [

"second half of a double dispatch call from CharacterCollection>>_idxForSortNotEqualTo:."

^ (self _idxPrimCompareEqualTo: aCharacterCollection) not

]

{ #category : 'New Indexing Comparison - prims' }
CharacterCollection >> _idxPrimCompareEqualTo: arg [
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #= .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareEqualTo:

]

{ #category : 'New Indexing Comparison - prims' }
CharacterCollection >> _idxPrimCompareGreaterThan: arg [
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareGreaterThan:

]

{ #category : 'New Indexing Comparison - prims' }
CharacterCollection >> _idxPrimCompareLessThan: arg [
  "This comparison operation is used for the indexing subsystem to
 determine an ordering for insertion into indexing objects.

 This method collates letters AaBb..Zz."

  "The comparison should be compatible with the case-insensitive semantics
 of the String method with selector #< .
 Same primitive as String>>lessThan: "

  self subclassResponsibility: #_idxPrimCompareLessThan:

]

{ #category : 'Converting' }
CharacterCollection >> _isValid32PathTermName [
  "Returns true if the receiver is a valid term in a path expression."

  | first sz maxSize |
  self = '*'
    ifTrue: [ ^ true ].
  maxSize := 64.
  ((sz := self size) > maxSize or: [ sz == 0 ])
    ifTrue: [ ^ false ].
  first := self at: 1.
  (first == $# or: [ first == $_ or: [ first isLetter ] ])
    ifFalse: [ ^ false ].
  2 to: sz do: [ :i |
    | c |
    c := self at: i.
    (c == $_ or: [ c == $| or: [ c isAlphaNumeric ] ])
      ifFalse: [ ^ false ] ].
  ^ true

]

{ #category : 'Indexing Support' }
CharacterCollection >> _isValidPathTermName [
  "Returns true if the receiver is a valid term in a path expression."

  ^ self _isValid32PathTermName

]

{ #category : 'Private' }
CharacterCollection >> _keySizeInBytesForOpCode: opCode [

| code codes128Bit codes192Bit codes256Bit |
code := opCode abs .
codes128Bit := { 1 . 4 . 7 } .
codes192Bit := { 2 . 5 . 8 } .
codes256Bit := { 3 . 6 . 9 . 10 } .

(codes128Bit includesIdentical: code)
  ifTrue:[ ^ 16 ].
(codes192Bit includesIdentical: code)
  ifTrue:[ ^ 24 ].
(codes256Bit includesIdentical: code)
  ifTrue:[ ^ 32 ].

]

{ #category : 'Testing' }
CharacterCollection >> _literalEqual: anotherLiteral [
  "Petit Parser definition:
     For two literals to be _literalEqual, their class must be identical and
     otherwise equal.
     For CharacterCollections equality is sufficent
   In GemStone we allow Unicode7 to compare to String, etc."

  ^ self _unicodeEqual: anotherLiteral

]

{ #category : 'Private' }
CharacterCollection >> _primAsymSignVerifyWithKey: aKey digestKind: opCode signature: aByteArray [

"Signs or verifies the message contained in the receiver using public key
 encryption. aKey must be an instance of GsTlsPrivateKey (which indicates a
 signing operation) or GsTlsPublicKey (which indicates a verify operation).
 For signing keys that require a message digest, anInt indicates one of the
 supported message digests for the specified key type. For signing keys
 that do not require a message digest, anInt must be zero.  See the tables
 below. The same message digest must be used for both signing and
 verifying.

 For signing operations, aByteArray must be a variant instance of ByteArray
 or nil, in which case a new instance of ByteArray will be created to
 contain the signature. For verifying operations, aByteArray must be a
 non-empty instance of ByteArray containing the signature from the signing.

 For secure signing and verifying with RSA keys only, a padding scheme must
 be used. The default RSA padding scheme is RSA_PKCS1_PADDING
 (PKCS #1 v1.5 padding), which is the most common type. However the newer
 PSS (Probabilistic Signature Scheme) is more secure and is recommended
 whenever possible. See RFC 3447 for additional information on PSS padding.
 Other RSA padding schemes, including no padding, are not supported due
 to known security vulnerabilities. The padding scheme selected must be the
 same for the signature and verification else the verification will fail.

 To sign or verify with RSA_PKCS1_PSS_PADDING, negate the digest opCode.
 For example, a message signed with an RSA private key using opCode 2 uses
 SHA1 message digest and (the default) RSA_PKCS1_PADDING. Signing a message
 with an RSA private key using opCode -2 uses SHA1 message digest and
 RSA_PKCS1_PSS_PADDING padding.

 RSA keys of type RSA-PSS may only use RSA_PKCS1_PSS_PADDING. PSS padding
 will be used for such keys in all cases, even if RSA_PKCS1_PADDING is
 requested. RSA-PSS private keys will answer #EVP_PKEY_RSA_PSS when sent
 the #sslAlgorithm message.

 Signing operations return aByteArray containing the signature on success
 and raise a CryptoError exception on error. Verify operations return true
 if the verification succeeds and raise a CryptoError exception if
 verification fails or an error occurs.

 Note that not all private/public key pairs support digital signatures.

 Key Type  Supported Digest Types
 ================================
 DSA       SHA1,SHA2
 ECDSA	   SHA1,SHA2
 RSA	   SHA1,SHA2,SHA3
 EC        None
 Ed25519   None
 Ed448     None
 ==============================


 Digest Type	OpCode
 =====================
 None           0
 SHA1		2
 SHA2-256	3
 SHA2-512	4
 SHA3-224	5
 SHA3-256	6
 SHA3-384	7
 SHA3-512	8
 =====================
"

<primitive: 1089>
opCode _validateClass: SmallInteger .
aKey _validateClasses: { GsTlsPrivateKey . GsTlsPublicKey } .
(aKey isKindOf: GsTlsPrivateKey)
  ifTrue:[ "signing, arg may be nil"
    aByteArray ifNotNil:[ aByteArray _validateClass: ByteArray ].
  ]
  ifFalse:[
     "verifying, arg must be a ByteArray"
    aByteArray _validateClass: ByteArray
].
self _validateKey: aKey withDigestKind: opCode .
^ self _primitiveFailed: #_primAsymSignVerifyWithKey:digestKind:signature:

]

{ #category : 'Private' }
CharacterCollection >> _primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode into: destObjOrNil [
"Private method for invoking non-AEAD encrypt/decrypt modes. See the method:
    #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
    into: destObjOrNil tag: tag extraData: eData
 for more information."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: opCode
       into: destObjOrNil
       tag: nil
       extraData: nil

]

{ #category : 'Private' }
CharacterCollection >> _primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
into: destObjOrNil tag: tag extraData: eData [

"Private method for encrypting or decrypting the receiver.
 The following encryption schemes are currently supported:

 ==================================================
                          Key     Salt    Tag
 opCode Cipher   Mode  bits/Bytes Size   Bytes
 ==================================================
   1     AES     CBC     128/16    16     N/A
   2     AES     CBC     192/24    16     N/A
   3     AES     CBC     256/32    16     N/A
   4     AES     OCB     128/16    12     16
   5     AES     OCB     192/24    12     16
   6     AES     OCB     256/32    12     16
   7     AES     GCM     128/16    12     16
   8     AES     GCM     192/24    12     16
   9     AES     GCM     256/32    12     16
  10   CHACHA20 Poly1305 256/32    12     16
 ==================================================

 AES encryption/decryption (Advanced Encryption Standard) is performed
 using the OpenSSL open source package and the AES specification,
 available at:
   http://csrc.nist.gov/publications/fips/fips197/fips-197.pdf

 CBC is an acronym for cipher block chaining. See the referenced AES
 specification document for further details.

 OCB is an acronym for Offset Cookbook Mode. See RFC 7253 for further
 details.

 GCM is an acronym for Galois Counter Mode. See RFC 5288 for further
 details.

 OCB, GCM and Poly1305 are AEAD modes. AEAD is an acronym for
 Authenticated Encryption with Associated Data. AEAD provides data
 authenticity, confidentiality, and integrity. It also supports
 Additional Authenticated Data (AAD). AAD is not encrypted and therefore
 not kept confidential, however AAD authenticity and integrity are
 guaranteed. AAD is not included in the encrypted payload but must be
 provided in order to decrypt the data. AAD is optional and the eData
 argument may be nil if AAD is not required.

 AEAD encryption/decryption is performed using the OpenSSL open source
 package and is implemented to conform to ISO/IEC standard 19772:2009.
 See https://www.iso.org/standard/46345.html for further information.

 opCode must be an instance of SmallInteger and be one of the values from
 the first column of the above table. A positive value indicates the data
 is to be encrypted and a negative value indicates decryption.

 aKey and aSalt must be instances of ByteArray of the correct size
 per the above table, otherwise an error is raised.

 destObjOrNil must be nil or an instance of a non-invariant byte object.
 If destObjOrNil is nil, the result of the operation will be placed into a
 new instance of ByteArray (encryption) or String (decryption). Otherwise
 the result will be placed into the given byte object starting at offset 1.
 The size of destObjOrNil will be modified to correctly contain all
 encrypted or decrypted data, and may differ from the size of the receiver
 due to the automatic addition or removal of padding by the cipher
 algorithm.

 When encrypting a receiver that has a character size greater than one, data
 is placed into big-endian byte order before encryption.

 When decrypting into a destObjOrNil object that a character size greater
 than one, data is converted to big-endian byte order after decryption.

 During AEAD encryption, a tag is generated which is used during decryption
 to ensure data integrity. The tag data will be stored into the tag
 argument, which must an instance of a variant byte object. During AEAD
 decryption, tag must be a byte object containing the tag bytes returned
 during encryption. For non-AEAD modes, tag must be nil.

 During AEAD encryption, eData must be nil or a byte object with a character
 size of one containing additional data to be used in generating the tag
 value. eData is NOT added to the encrypted payload. During decryption,
 eData must be a byte object with a character size of one containing the
 same bytes provided during encryption or nil if no byte object was
 provided. For non-AEAD modes, eData must be nil.

 Successful encryption or decryption returns encrypted or decrypted data,
 which is stored into destObjOrNil if it was provided or a new byte object
 if it was not. An exception is raised if encryption or decryption fails.
"

<primitive: 953>
self _validateEncryptionOpCode: opCode ;
     _validateEncryptionKey: aKey forOpCode: opCode ;
     _validateEncryptionSalt: aSalt forOpCode: opCode ;
     _validateEncryptionExtraData: eData forOpCode: opCode .
destObjOrNil ifNotNil:[ destObjOrNil _validateIsBytes ; validateIsVariant ] .
self _validateEncryptionTag: tag forOpCode: opCode .
^ self _primitiveFailed: #_primEncryptDecryptWithKey:salt:opCode:into:tag:extraData:

]

{ #category : 'Private' }
CharacterCollection >> _saltSizeForOpCode: opCode [

^ opCode abs >= 4
    ifTrue:[ 12 ]
   ifFalse:[ 16 ]

]

{ #category : 'Deprecated' }
CharacterCollection >> _toServerTextFile: aFileSpec [

 "Writes the receiver to the specified file
 using the current session's cpu's in-memory byte order of any multi-byte characters.

 The argument aFileSpec must be convertable to a Utf8 , otherwise an error is signaled
 by the primitive."

<primitive: 301>
self deprecated: 'CharacterCollection>>_toServerTextFile: deprecated long before v3.0.  Use an instance of GsFile
 to access the file system.'.

aFileSpec _validateByteClass: CharacterCollection .
self _error: #hostErrFileExport args: { aFileSpec }

]

{ #category : 'Private' }
CharacterCollection >> _validate: anObj isSize: expectedSize [

anObj _basicSize == expectedSize
  ifFalse:[ anObj _error: #rtErrBadSize args: { expectedSize . anObj _basicSize } ] .

]

{ #category : 'Private' }
CharacterCollection >> _validateEncryptionExtraData: eData forOpCode: opCode [
opCode abs > 3  "AEAD encrypt/decrypt"
  ifTrue:[ eData ifNotNil:[ eData _validateIsBytes ]]
 ifFalse:[ eData _validateClass: UndefinedObject ]

]

{ #category : 'Private' }
CharacterCollection >> _validateEncryptionKey: aKey forOpCode: opCode [
aKey _validateClass: ByteArray .
self _validate: aKey isSize: (self _keySizeInBytesForOpCode: opCode)

]

{ #category : 'Private' }
CharacterCollection >> _validateEncryptionOpCode: opCode [
opCode _validateClass: SmallInteger .
(((opCode < -10) or:[opCode > 10]) or:[opCode == 0])
  ifTrue:[opCode _error: #rtErrArgOutOfRange args:{ -10 . 10 } ] .
^ true

]

{ #category : 'Private' }
CharacterCollection >> _validateEncryptionSalt: salt forOpCode: opCode [
salt _validateClass: ByteArray .
self _validate: salt isSize: (self _saltSizeForOpCode: opCode)

]

{ #category : 'Private' }
CharacterCollection >> _validateEncryptionTag: tag forOpCode: opCode [
| absCode |
absCode := opCode abs .
absCode > 3 "AEAD encrypt/decrypt"
  ifTrue:[
    tag _validateClass: ByteArray .
    opCode < 3 ifTrue:[ tag validateIsVariant ]			"AEAD Encrypting"
              ifFalse:[ self _validate: tag isSize: 16 ] . 	"AEAD Decrypting"
  ] ifFalse:[ tag _validateClass: UndefinedObject ] "Not AEAD"

]

{ #category : 'Private' }
CharacterCollection >> _validateInteger: anInt inRangeFrom: lowest to: highest [
anInt _validateClass: SmallInteger .
((anInt < lowest) or:[anInt > highest])
  ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ lowest . highest } ] .
^ true

]

{ #category : 'Private' }
CharacterCollection >> _validateKey: aKey withDigestKind: opCode [
|alg|
alg := aKey algorithm . "#RSA, #DSA, #DH, #EC, etc"
alg == #RSA "SHA1,2,3 allowed"
  ifTrue:[ ^ self _validateInteger: opCode abs inRangeFrom: 2 to: 8 ].
alg == #DSA "SHA1 and SHA2 only"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 2 to: 4 ].
alg == #EC "No digest required/allowed"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 0 to: 0 ].

^ CryptoError signal: 'Invalid key kind for signing or verifying'

]

{ #category : 'Formatting' }
CharacterCollection >> _wrapTo: col indentingWith: indentStr [

"Returns a new instance of the class of the receiver.

 Word-wrap the receiver to column col, treating tab Characters as modulo-8.
 Whenever a line-feed is inserted, prepend indentStr to the subsequent line."

| ch curcol linestart wordstart lf tab str sz |

lf := Character lf.
tab := Character tab.
curcol := 1.
wordstart := 0.
linestart := 1.
str := self class _newString.

1 to: (sz := self size) do: [ :i |
  ch := self at: i.
  ch == lf ifTrue: [
    str add: (self copyFrom: linestart to: i).
    linestart := i + 1.
    wordstart := 0.
    curcol := 1.
  ]
  ifFalse: [
    ch isSeparator ifTrue: [
      wordstart := 0
    ]
    ifFalse: [
      wordstart == 0 ifTrue: [
        wordstart := i
      ].
    ].

    ch == tab ifTrue: [
      curcol := (curcol + 8) \\ 8
    ]
    ifFalse: [
      curcol := curcol + 1
    ].

    curcol > col ifTrue: [
      (wordstart == 0 or: [linestart == wordstart]) ifTrue: [
	str add: (self copyFrom: linestart to: i).
	linestart := i + 1.
	curcol := 1.
      ]
      ifFalse: [
	str add: (self copyFrom: linestart to: wordstart - 1).
	linestart := wordstart.
	curcol := i - wordstart + 1.
      ].
      str add: lf.
      str add: indentStr .
      curcol := curcol + indentStr size .
    ].
  ].
].

linestart <= sz ifTrue: [
  str add: (self copyFrom: linestart to: sz )
].

^str.

]

{ #category : 'Concatenating' }
CharacterCollection >> , aCharOrCharCollection [

"Returns a new instance of the receiver's class that contains the elements of
 the receiver followed by the elements of aCharOrCharCollection.

 Warning: Creating a new instance and copying the receiver take time.  If you
 can safely modify the receiver, it can be much faster to use the addAll:
 method.  See the documentation of the Concatenating category of class
 SequenceableCollection for more details."

^ self copy addAll: aCharOrCharCollection; yourself

]

{ #category : 'Comparing' }
CharacterCollection >> < aCharCollection [

"Returns true if the receiver collates before the argument.  Returns false
 otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #<

]

{ #category : 'Comparing' }
CharacterCollection >> <= aCharCollection [

"Returns true if the receiver collates before the argument or if all of the
 corresponding Characters in the receiver and argument are equal.
 Returns false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #<=

]

{ #category : 'Comparing' }
CharacterCollection >> = aCharCollection [

"Returns true if all of the corresponding Characters in the receiver and
 argument are equal.  Returns false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #=

]

{ #category : 'Comparing' }
CharacterCollection >> > aCharCollection [

"Returns true if the receiver collates after the argument.  Returns false
 otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #>

]

{ #category : 'Comparing' }
CharacterCollection >> >= aCharCollection [

"Returns true if the receiver collates after the argument or if all of the
 corresponding Characters in the receiver and argument are equal.  Returns
 false otherwise.
 (Subclass responsibility.)"

^ self subclassResponsibility: #>=

]

{ #category : 'Adding' }
CharacterCollection >> add: aCharOrCharColl [

"Appends all of the elements of aCharOrCharColl to the receiver and returns
 aCharOrCharColl.  Returns aCharOrCharColl"

| index |

(aCharOrCharColl isKindOf: Collection) ifTrue:[
    index := self size.
    aCharOrCharColl accompaniedBy: self do: [:me :aChar |
      index := index + 1.
      me at: index put: aChar.
    ].
    ^ aCharOrCharColl.
].

^ self at: (self size + 1) put: aCharOrCharColl.

]

{ #category : 'Adding' }
CharacterCollection >> addAll: aCharOrCharCollection [

"Equivalent to add: aCharOrCharCollection."

^ self add: aCharOrCharCollection

]

{ #category : 'Adding' }
CharacterCollection >> addCodePoint: aSmallInteger [

 ^ self add: (Character codePoint: aSmallInteger)

]

{ #category : 'Adding' }
CharacterCollection >> addLast: aCharOrCharCollection [

"Equivalent to add: aCharOrCharCollection."

^ self add: aCharOrCharCollection

]

{ #category : 'Updating' }
CharacterCollection >> addLineDelimiters [

"Returns a copy of the receiver that contains each occurrence of the backslash
 Character replaced by the line-feed Character."

^ self copyReplaceAll: '\' with: (self class with: Character lf).

]

{ #category : 'Formatting' }
CharacterCollection >> addLineWith: aCharacterCollection centeredToWidth: anInt [

"Add a centered line of width anInt to the receiver.
 Returns the receiver."

| numSpaces aSpace|
aSpace := Character space.
numSpaces := anInt - aCharacterCollection size.
numSpaces <= 0 "No room, just do the add"
  ifTrue:[self addAll: aCharacterCollection.]
  ifFalse:[ | leftSpaces |
    leftSpaces := numSpaces // 2.
    leftSpaces timesRepeat:[self add: aSpace].
    self addAll: aCharacterCollection.
    (numSpaces - leftSpaces) timesRepeat:[self add: aSpace].
].
^self

]

{ #category : 'Decrypting' }
CharacterCollection >> aesDecryptWith128BitKey: aKey salt: aSalt [
"Decrypts the receiver using 128 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith128BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Decrypting' }
CharacterCollection >> aesDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil [
"Decrypts the receiver using 128 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -1 into: destObjOrNil

]

{ #category : 'Decrypting' }
CharacterCollection >> aesDecryptWith192BitKey: aKey salt: aSalt [
"Decrypts the receiver using 192 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith192BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Decrypting' }
CharacterCollection >> aesDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil [
"Decrypts the receiver using 192 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -2 into: destObjOrNil

]

{ #category : 'Decrypting' }
CharacterCollection >> aesDecryptWith256BitKey: aKey salt: aSalt [
"Decrypts the receiver using 256 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith256BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Decrypting' }
CharacterCollection >> aesDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil [
"Decrypts the receiver using 256 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -3 into: destObjOrNil

]

{ #category : 'Encrypting' }
CharacterCollection >> aesEncryptWith128BitKey: aKey salt: aSalt [
"Encrypts the receiver using 128 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith128BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Encrypting' }
CharacterCollection >> aesEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil [
"Encrypts the receiver using 128 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 1  into: destObjOrNil

]

{ #category : 'Encrypting' }
CharacterCollection >> aesEncryptWith192BitKey: aKey salt: aSalt [
"Encrypts the receiver using 192 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith192BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Encrypting' }
CharacterCollection >> aesEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil [
"Encrypts the receiver using 192 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 2 into: destObjOrNil

]

{ #category : 'Encrypting' }
CharacterCollection >> aesEncryptWith256BitKey: aKey salt: aSalt [
"Encrypts the receiver using 256 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith256BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Encrypting' }
CharacterCollection >> aesEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil [
"Encrypts the receiver using 256 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 3 into: destObjOrNil

]

{ #category : 'Authenticated Decrypting' }
CharacterCollection >> aesGcmDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 128 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -7
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
CharacterCollection >> aesGcmDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 192 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -8
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
CharacterCollection >> aesGcmDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 256 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -9
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> aesGcmEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 128 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 7 into: destObjOrNil tag: tag extraData: eData

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> aesGcmEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 192 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 8
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> aesGcmEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 256 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 9
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Decrypting' }
CharacterCollection >> aesOcbDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 128 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -4
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
CharacterCollection >> aesOcbDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 192 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -5
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
CharacterCollection >> aesOcbDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 256 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -6
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> aesOcbEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 128 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 4
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> aesOcbEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 192 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 5
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> aesOcbEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 256 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 6
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Converting' }
CharacterCollection >> asArrayOf32PathTerms [
  "Returns an Array of path substrings held by the receiver.  The receiver
 is assumed to be a period-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 periods, the Array will hold a copy of the receiver.  The $\ Character
 is no longer recognized as an escape character.

 Raises an error if an element is not a valid path term."

  | nextName result |
  result := {}.
  nextName := self speciesForConversion new.
  self
    do: [ :c |
      c == $.
        ifTrue: [
          nextName _isValid32PathTermName
            ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: { nextName } ].
          result add: nextName asSymbol.
          nextName := self speciesForConversion new ]
        ifFalse: [ nextName add: c ] ].
  nextName size ~~ 0
    ifTrue: [
      nextName _isValid32PathTermName
        ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: { nextName } ].
      result add: nextName asSymbol ]
    ifFalse: [
      result size == 0
        ifTrue: [ result add: nextName asSymbol ] ].
  ^ result

]

{ #category : 'Converting' }
CharacterCollection >> asArrayOfKeywords [

"Returns an Array of keyword substrings held by the receiver.  The receiver
 is assumed to be a colon-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 colons, the Array will hold a copy of the receiver."

| c nextName result |

result := { }.
nextName := self speciesForConversion new.
1 to: self size do: [ :i |
  c := self at: i.
  nextName add: c.
  c == $: ifTrue: [
    result add: nextName.
    nextName := self speciesForConversion new.
  ].
].
nextName size ~~ 0 ifTrue: [
  result add: nextName
]
ifFalse: [
  result size == 0 ifTrue: [result add: nextName]
].
^result

]

{ #category : 'Converting' }
CharacterCollection >> asArrayOfPathTerms [
  "Returns an Array of path substrings held by the receiver.  The receiver
 is assumed to be a period-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 periods, the Array will hold a copy of the receiver.  The $\ Character
 is no longer recognized as an escape character.

 Raises an error if an element is not a valid path term."

  | nextName result |
  result := {}.
  nextName := self speciesForConversion new.
  self
    do: [ :c |
      c == $.
        ifTrue: [
          nextName _isValidPathTermName
            ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: {nextName} ].
          result add: nextName asSymbol.
          nextName := self speciesForConversion new ]
        ifFalse: [ nextName add: c ] ].
  nextName size ~~ 0
    ifTrue: [
      nextName _isValidPathTermName
        ifFalse: [ ^ self _error: #'rtErrInvalidIndexPathExpression' args: {nextName} ].
      result add: nextName asSymbol ]
    ifFalse: [
      result size == 0
        ifTrue: [ result add: nextName asSymbol ] ].
  (result at: 1) = #'*'
    ifTrue: [
      "* not allowed as first term"
      ^ self _error: #'rtErrInvalidIndexPathExpression' args: {(result at: 1)} ].
  ^ result

]

{ #category : 'Converting' }
CharacterCollection >> asArrayOfSubstrings [

"Returns an Array of substrings held by the receiver. The receiver
 is assumed to be a separator-separated list of substrings.  These substrings
 are extracted and collected in an Array.  If the receiver contains no
 separators, the Array will hold a copy of the receiver.  Separators not meant
 to separate substrings may be escaped with a $\ Character."

| nextName result esc sz |

result := { } .
(sz := self size) == 0 ifTrue: [
  ^result
].
nextName := self speciesForConversion new.
esc := false.
1 to: sz do: [ :i | | c |
  c := self at: i.
  esc ifTrue: [
    nextName add: c.
    esc := false.
  ] ifFalse: [
    c == $\ ifTrue: [ esc := true ]
    ifFalse: [
	c isSeparator ifTrue: [
          nextName size ~~ 0 ifTrue: [result add: nextName].
          nextName := self speciesForConversion new.
        ] ifFalse: [
          nextName add: c
        ].
    ].
  ].
].

esc ifTrue:[ nextName add: $\ ].

(nextName size ~~ 0 or: [result size == 0]) ifTrue:[ result add: nextName ].

^result

]

{ #category : 'Converting' }
CharacterCollection >> asBase64String [
"Return a String which represents the receiver represented in base64 format."

^ self asBase64StringOnOneLine: true

]

{ #category : 'Converting' }
CharacterCollection >> asBase64StringOnOneLine: aBoolean [
"Return a String which represents the receiver represented in base64 format.
 If aBoolean is true, the resulting is one long line which does not contain
 newline characters.  If aBoolean is false, newline characters are inserted
 such that each line does not exceed 64 characters."

<primitive: 1062>
aBoolean _validateClass: Boolean .
self _primitiveFailed: #asBase64StringOnOneLine:

]

{ #category : 'Converting' }
CharacterCollection >> asDecimalFloat [

"Returns a DecimalFloat whose value is represented by the receiver."

^ DecimalFloat fromString: self

]

{ #category : 'Converting' }
CharacterCollection >> asDoubleByteString [

"Returns a DoubleByteString representation of the receiver."

^ DoubleByteString withAll: self asString .       "fix 39372"

]

{ #category : 'Converting' }
CharacterCollection >> asFloat [

"Returns a SmallDouble or Float whose value is represented by the receiver."

^ Float fromString: self

]

{ #category : 'Converting' }
CharacterCollection >> asHexString [

"Returns a String containing a hexadecimal printed representation of the
 contents of the receiver.  For example, the message 'abc' asHexString
 returns the String '616263'.

 The receiver must be a byte format object."

<primitive: 467>
self _validateByteClass: CharacterCollection .
self _primitiveFailed: #asHexString .

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asHmacSumWithDigestKind: opCode key: keyString [

"Computes the keyed-hash message authentication code (HMAC) of the receiver using
 the message digest algorithm indicated by opCode and secret key keyString.

 opCode must be one of the following:

 opCode   Digest Algorithm
 =========================
  1          md5
  2          sha1
  3          sha2-256
  4          sha2-512
  5          sha3-224
  6          sha3-256
  7          sha3-384
  8          sha3-256
 =========================

 secretKey must be an instance or subclass of a ByteArray or String and must
 have a character size of one, i.e.: its class must answer 1 when sent the
 message #_bytesPerWord. secretKey must have a size between 1 and 64 bytes.

 Answers the HMAC of the receiver as a LargeInteger."

<primitive: 1060>
opCode _validateClass: SmallInteger .
((opCode < 1) or:[ opCode > 8]) ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ 1 . 8 } ].
keyString _validateClasses: { String . ByteArray } .
((keyString _basicSize > 64) or:[keyString _basicSize < 1])
  ifTrue:[ keyString _error: #rtErrBadSize args: { 64 . keyString _basicSize } ] .
self _primitiveFailed: #asHmacSumWithDigestKind:key: args: { opCode . keyString } .

]

{ #category : 'Converting' }
CharacterCollection >> asInteger [

"Returns an Integer whose value is represented by the receiver."

^ Integer fromString: self

]

{ #category : 'Converting' }
CharacterCollection >> asLowercase [

"Returns a new instance of the receiver's class, with all upper-case
 characters in the receiver changed to lower-case.

 (Subclass responsibility.)"

^ self subclassResponsibility: #asLowercase

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asMd5HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the MD5 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asMd5HmacWithKey: keyString) asHexStringWithLength: 32

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asMd5HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the MD5 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 1 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asMd5String [

"Compute the 128 bit MD5 message digest for the receiver and return it as
 a 32 character string of hexadecimal characters."

^ self md5sum asHexStringWithLength: 32

]

{ #category : 'Converting' }
CharacterCollection >> asMultiByteString: example [

"Returns a Double/QuadByteString representation of the receiver,
 depending on the class of the example string."

^ example class withAll: self

]

{ #category : 'Converting' }
CharacterCollection >> asNumber [

"Returns the receiver converted to a kind of number.  If the receiver contains
 all digits (with optional radix notation), returns a kind of Integer.  If the
 receiver has a slash, returns a Fraction.  Otherwise conversion to a Float is
 attempted.  An error may result if the receiver does not contain the proper
 format for a kind of Number."

 | strm |
 self size == 0 ifTrue: [ ^0 ].
 strm := ReadStreamPortable on: self .
 (self indexOf: $/ startingAt: 1) ~~ 0 ifTrue:[
    ^ Fraction fromStream: strm .
 ].
 ^ Number fromStream: strm .
]

{ #category : 'Converting' }
CharacterCollection >> asQuadByteString [

"Returns a QuadByteString representation of the receiver."

^ QuadByteString withAll: self.

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha1HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha1 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha1HmacWithKey: keyString) asHexStringWithLength: 40

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha1HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha1 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 2 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asSha1String [

"Compute the 160 bit SHA1 message digest for the receiver and return it as
 a 40 character string of hexadecimal characters."

^ self sha1Sum asHexStringWithLength: 40

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha256HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha256HmacWithKey: keyString) asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha256HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 3 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asSha256String [

"Compute the 256 bit SHA-2 message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha256Sum asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_224HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 224 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_224HmacWithKey: keyString) asHexStringWithLength: 56

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_224HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 224 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 5 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asSha3_224String [

"Compute the SHA3 224 bit message digest for the receiver and return it as
 a 56 character string of hexadecimal characters."

^ self sha3_224Sum asHexStringWithLength: 56

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_256HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_256HmacWithKey: keyString) asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_256HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 6 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asSha3_256String [

"Compute the SHA3 256 bit message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha3_256Sum asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_384HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 384 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_384HmacWithKey: keyString) asHexStringWithLength: 96

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_384HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 384 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 7 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asSha3_384String [

"Compute the SHA3 384 bit message digest for the receiver and return it as
 a 96 character string of hexadecimal characters."

^ self sha3_384Sum asHexStringWithLength: 96

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_512HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_512HmacWithKey: keyString) asHexStringWithLength: 128

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha3_512HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 8 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asSha3_512String [

"Compute the SHA3 512 bit message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha3_512Sum asHexStringWithLength: 128

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha512HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha512HmacWithKey: keyString) asHexStringWithLength: 128

]

{ #category : 'Message Authentication Codes' }
CharacterCollection >> asSha512HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 4 key: keyString

]

{ #category : 'Message Digests' }
CharacterCollection >> asSha512String [

"Compute the 512 bit SHA-2 message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha512Sum asHexStringWithLength: 128

]

{ #category : 'Deprecated' }
CharacterCollection >> asSmallFloat [

"SmallFloat is deprecated. Return a SmallDouble or Float whose value
 is represented by the receiver."

self deprecated: 'CharacterCollection>>asSmallFloat deprecated v3.0. Use asFloat or Float class>>#fromString: '.
^ (Float fromString: self) asSmallFloat

]

{ #category : 'Converting' }
CharacterCollection >> asString [

"Returns a String representation of the receiver."

^ String withAll: self

]

{ #category : 'Converting' }
CharacterCollection >> asSymbolKind [

"Returns a canonical symbol containing the same Characters as the receiver."

CharacterCollection subclassResponsibility: #asSymbolKind

]

{ #category : 'Converting' }
CharacterCollection >> asUppercase [

"Returns a new instance of the receiver's class, with all lower-case
 characters in the receiver changed to upper-case.

 (Subclass responsibility.)"

^ self subclassResponsibility: #asUppercase

]

{ #category : 'Comparing' }
CharacterCollection >> at: anIndex equals: aCharCollection [

"Returns true if aCharCollection is contained in the receiver starting at
 anIndex.  Returns false otherwise.

 Note that this method returns true only if aCharCollection begins exactly at
 the position designated by anIndex.  To locate a pattern beginning on or after
 anIndex, see the method findPattern:startingAt: in category 'Searching'.

 (Subclass responsibility.)"

^ self subclassResponsibility: #at:equals:

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> authenticatedEncryptionExample [

| key salt enc dec hamlet extraData tag|
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 12 .

hamlet :=
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .

extraData := 'Hamlet: Act V, Scene i'.

tag := ByteArray new.
enc := hamlet aesOcbEncryptWith256BitKey: key salt: salt
       into: ByteArray new tag: tag extraData: extraData .
dec := enc aesOcbDecryptWith256BitKey: key salt: salt
       into: String new tag: tag extraData: extraData .
^ dec = hamlet

]

{ #category : 'Comparing' }
CharacterCollection >> between: min and: max [
	"Answer whether the receiver is less than or equal to the argument max and
	 greater than or equal to the argument min."

	^self >= min and: [self <= max].

]

{ #category : 'Authenticated Decrypting' }
CharacterCollection >> chacha20Poly1305DecryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using CHACHA20-Poly1305 decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -10
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
CharacterCollection >> chacha20Poly1305EncryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using CHACHA20-Poly1305 encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 10
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Formatting' }
CharacterCollection >> charSize [

"Returns number of bytes that make up a character for this string class.
 (Subclass responsibility.)"

^ self subclassResponsibility: #charSize

]

{ #category : 'Accessing' }
CharacterCollection >> codePointAt: anIndex [

CharacterCollection subclassResponsibility: #'codePointAt:'

]

{ #category : 'Updating' }
CharacterCollection >> codePointAt: anIndex put: aValue [

CharacterCollection subclassResponsibility: #'codePointAt:put:'

]

{ #category : 'Comparing' }
CharacterCollection >> compareTo: aString collator: anIcuCollator [

"Returns -1, 0 or 1,  when receiver is less than,
 equal to, or greater than argString .
 argString must be a String, MultiByteString, or a Utf8 .
 anIcuCollator == nil is interpreted as   IcuCollator default ."

^ self compareTo: aString collator: anIcuCollator useMinSize: false

]

{ #category : 'Comparing' }
CharacterCollection >> compareTo: argString collator: anIcuCollator useMinSize: aMinSize [
  "Returns -1, 0 or 1,  when receiver is less than,
 equal to, or greater than argString .
 argString must be a String, MultiByteString, or a Utf8.
 anIcuCollator == nil is interpreted as   IcuCollator default .

 If aMinSize==true, compare stops at (self size min: argString size),
 which is Squeak semantics for comparison .

 If aMinSize is a SmallInteger >= 1, compare stops at
 aMinSize min: (self size min: argString size) ."

  ^ self subclassResponsibility: #'compareTo:collator:useMinSize:'

]

{ #category : 'Searching' }
CharacterCollection >> containsDigit [

"Answers true if the receiver contains at least one digit, otherwise
 answers false."

1 to: self size do:[:n| (self at: n) isDigit ifTrue:[ ^ true ] ] .
^ false

]

{ #category : 'Searching' }
CharacterCollection >> containsLowercase [

"Answers true if the receiver contains at least one lower case character,
 otherwise answers false.
 Differs from legacy method containsLowercaseOld for
 code points 170, 186, and about 1700 code points >= 256"

1 to: self size do:[:n| (self at: n) isLowercase ifTrue:[ ^ true ] ] .
^ false

]

{ #category : 'Legacy' }
CharacterCollection >> containsLowercaseOld [

"Answers true if the receiver contains at least one lower case character,
 otherwise answers false."

1 to: self size do:[:n| (self at: n) isLowercaseOld ifTrue:[ ^ true ] ] .
^ false

]

{ #category : 'Private' }
CharacterCollection >> containsPasswordSymbol [

"Answers true if the receiver contains at least one symbol character, with
 symbol in this case meaning any Character that is neither a letter nor a
 digit; otherwise answers false."

1 to: self size do:[:n| (self at: n) _isPasswordSymbol ifTrue:[ ^ true ] ] .
^ false

]

{ #category : 'Searching' }
CharacterCollection >> containsUppercase [

"Answers true if the receiver contains at least one upper case character,
 otherwise answers false."

1 to: self size do:[:n| (self at: n) isUppercase ifTrue:[ ^ true ] ] .
^ false

]

{ #category : 'Repository Conversion' }
CharacterCollection >> convert [
"If receiver is a committed object and receiver's class on disk was
 a subclass of one of
   ObsDoubleByteString, ObsDoubleByteSymbol, ObsQuadByteString
 the receiver is added to the writeSet of current transaction,
 otherwise has no effect.
 Returns true if receiver was added to the writeSet, false otherwise."

<primitive: 744>
self _primitiveFailed:#convert

]

{ #category : 'Copying' }
CharacterCollection >> copy [
  "optimization, no need for postCopy"
<primitive: 885>
self _primitiveFailed: #copy .
self _uncontinuableError

]

{ #category : 'Copying' }
CharacterCollection >> copyFrom: startIndex to: stopIndex [

"Returns a new SequenceableCollection containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver, unless the receiver is a Symbol or DoubleByteSymbol,
 in which case the result class is respectively String or DoubleByteString.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
 If startIndex > stopIndex and both are positive, an empty collection is returned.
 "

| result |
(startIndex > stopIndex) ifTrue: [
  stopIndex < 0 ifTrue:[ self _error: #rtErrBadCopyFromTo args: { stopIndex } ].
  ^ self class new
].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

result := (self class _newString: (stopIndex - startIndex + 1)).
result replaceFrom: 1 to: 1 + stopIndex - startIndex
	with: self startingAt: startIndex  .
^ result

]

{ #category : 'Copying' }
CharacterCollection >> copyReplaceAll: subString with: newSubString [

"Returns a copy of the receiver with all occurrences of the given substring
 replaced with newSubString."

| copy csize matches ssize idx |

copy := self copy.
matches := { } .
ssize := subString size.
ssize == 0 ifTrue: [^copy].
idx := 1 - ssize.
csize := copy size.
[ idx := idx + subString size.
  idx <= csize and: [
    idx := copy findString: subString startingAt: idx.
    idx ~~ 0
  ]
] whileTrue: [
  matches add: idx
].
matches reverseDo: [:p |
  copy removeFrom: p to: p+ssize-1.
  copy insertAll: newSubString at: p
].
^copy

]

{ #category : 'Copying' }
CharacterCollection >> copyReplaceChar: aCharacter with: secondCharacter [

"Returns a copy of the receiver with all occurrences of the given
 Character replaced with secondCharacter."

| copy |
secondCharacter _validateClass: Character .
copy := self speciesForCollect withAll: self .
1 to: copy size do:[:n |
 (copy at: n) == aCharacter ifTrue:[ copy at: n put: secondCharacter].
].
^ copy

]

{ #category : 'Copying' }
CharacterCollection >> copyWithout: anObject [

"Returns a copy of the receiver that does not contain the given object.
 Comparisons are by equality."

| copy element sz |

copy := self class _newString .

sz := 0.
1 to: self size do: [:i |
  element := self at: i.
  (element = anObject)
    ifFalse: [
      sz := sz + 1.
      copy at: sz put: element.
      ]
  ].

^copy

]

{ #category : 'Converting' }
CharacterCollection >> copyWrappedTo: rightMargin [

"Returns a String with the receiver's contents word-wrapped to the given
 right margin."

| res col ch lf |

self size < rightMargin ifTrue: [^self].
lf := Character lf.
res := self species new.
col := 0.
1 to: self size do: [ :i |
  ch := self at: i.
  res add: ch.
  ch == lf ifTrue: [ col := 0 ]
  ifFalse: [
    col := col + 1.
    (col > rightMargin and: [ch isSeparator]) ifTrue: [
      res add: lf.
      col := 1.
    ].
  ].
].
^ res

]

{ #category : 'Formatting' }
CharacterCollection >> describeClassName [

"Returns a copy of the receiver with the Character $a prepended to the
 receiver's contents.  This method is used for formatting class names in object
 descriptions, where the receiver is a string containing the name of a class.
 For example, the String 'UserClass', when sent the message describeClassName,
 returns 'aUserClass'."

| result |
result := self speciesForPrint new .
self size ~~ 0 ifTrue:[   "inline simplified isVowel using Unicode asUppercase"
  ( #( $A $E $I $O $U $Y ) includesIdentical: (self at: 1) asUppercase ) ifTrue:[
    result add: 'an' ; addAll: self .
    ^ result
  ]
].
result add: $a ; add: self.
^ result

]

{ #category : 'Encrypting' }
CharacterCollection >> encryptionExample [

| key salt enc dec hamlet |
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 16 .

hamlet :=
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .


enc := hamlet aesEncryptWith256BitKey: key salt: salt .
dec := enc aesDecryptWith256BitKey: key salt: salt into: String new.
^ dec = hamlet

]

{ #category : 'Testing' }
CharacterCollection >> equalsNoCase: aCharCollection [
    "Returns true if the receiver is equivalent to aCharCollection.
    The receiver is equivalent to aCharCollection if the receiver
    contains the same Characters as aCharCollection regardless of case
    or internal representation.  For example, if $a is in
    aCharCollection, it is equivalent to any representation of an 'a'
    in the receiver's character set.

    (Subclass responsibility.)"

^ self subclassResponsibility: #equalsNoCase:

]

{ #category : 'Case-Sensitive Searching' }
CharacterCollection >> findLastSubString: subString startingAt: startIndex [

"startIndex should be >= 1 and <= self size . Search is backwards
 through the receiver. Returns 0 if no match found. "
^  self _findLastString: subString startingAt: startIndex ignoreCase: false

]

{ #category : 'Searching' }
CharacterCollection >> findPattern: aPattern startingAt: anIndex [

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-sensitive search."

^ self _findPattern: aPattern startingAt: anIndex ignoreCase: false

]

{ #category : 'Searching' }
CharacterCollection >> findPatternNoCase: aPattern startingAt: anIndex [

"This method searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, this method returns the
 index of the first Character of the substring.  Otherwise, this returns 0.

 The argument aPattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 Performs a case-insensitive search."

^ self _findPattern: aPattern startingAt: anIndex ignoreCase: true

]

{ #category : 'Case-Sensitive Searching' }
CharacterCollection >> findString: subString startingAt: startIndex [

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.

 The search is case-sensitive."

^ self _findString: subString startingAt: startIndex ignoreCase: false

]

{ #category : 'Case-Insensitive Searching' }
CharacterCollection >> findStringNoCase: subString startingAt: startIndex [

"If a receiver contains subString beginning at some point at or after
 startIndex, this returns the index at which subString begins.  If the
 receiver does not contain subString, this returns 0.

 The search is case-insensitive."

^ self _findString: subString startingAt: startIndex ignoreCase: true

]

{ #category : 'Hashing' }
CharacterCollection >> hash [

"Returns a positive Integer based on a case-sensitive hash of the contents
 of the receiver.  The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680.

 This implementation inherited by JapaneseString.  String and
 MultiByteString each reimplement hash."

<primitive: 31>
self _primitiveFailed: #hash .
self _uncontinuableError

]

{ #category : 'LDAP Support' }
CharacterCollection >> hasLdapWildcardPattern [

"Answer true if the receiver contains one and only one '%' character,
 and the character after the '%' is 's'."

(self select:[:e| e == $%]) size == 1
  ifFalse:[^ false ].

^(self findPattern: #('%' 's') startingAt: 1) ~~ 0

]

{ #category : 'Case-Insensitive Searching' }
CharacterCollection >> includesString: aString [

"Returns true if aString is contained as a subString within the receiver,
 using a case-insensitive search.  Returns false otherwise."

^ (self _findString: aString startingAt: 1 ignoreCase: true) ~~ 0

]

{ #category : 'Searching' }
CharacterCollection >> indexOf: pattern matchCase: flag startingAt: startIndex [

"Searches the receiver, beginning at anIndex, for a substring that
 matches aPattern.  If a matching substring is found, returns the
 index of the first Character of the substring.  Otherwise, returns 0.

 The argument pattern is an Array containing zero or more CharacterCollections
 plus zero or more occurrences of the special Characters asterisk or
 question-mark.  See the description of the matchPattern: method for more
 information about this argument.

 If the flag argument is true, a case-sensitive search is performed.  Otherwise,
 a case-insensitive search is performed."

^ self _findPattern: pattern startingAt: startIndex ignoreCase: (flag not).

]

{ #category : 'Searching' }
CharacterCollection >> indexOfSubCollection: aSubColl startingAt: anIndex [

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver at starting at anIndex. Returns zero if no match is found."

(aSubColl size == 0) ifTrue: [ ^ 0 ].
^ self findString: aSubColl startingAt: anIndex .

]

{ #category : 'Searching' }
CharacterCollection >> indexOfSubCollection: aSubColl startingAt: anIndex ifAbsent: anExceptionBlock [

"Returns the index of the first element of the receiver where that element and
 the subsequent ones are equal to those in aSubColl. The search is begun in the
 receiver at starting at anIndex. Returns the value of evaluating
 anExceptionBlock if no match is found."

| idx |
(aSubColl size == 0) ifTrue: [ ^ anExceptionBlock value ].
idx := self findString: aSubColl startingAt: anIndex .
idx == 0 ifTrue:[ ^ anExceptionBlock value ].
^ idx

]

{ #category : 'Deprecated' }
CharacterCollection >> insert: aCharOrCharCollection at: anIndex [

self deprecated: 'CharacterCollection>>insert:at: deprecated long before v3.0. Use insertAll:at: instead.'.
^ self insertAll: aCharOrCharCollection at: anIndex.

]

{ #category : 'Adding' }
CharacterCollection >> insertAll: aCharOrCharCollection at: anIndex [

"Inserts aCharOrCharCollection into the receiver at the specified index and
 returns aCharOrCharCollection."

(aCharOrCharCollection isKindOf: CharacterCollection)
   ifTrue: [ ^ super insertAll: aCharOrCharCollection at: anIndex ].

(aCharOrCharCollection class == Character)
  ifTrue: [
    self replaceFrom: anIndex + 1 to: 1 + self size with: self startingAt: anIndex .
    self at: anIndex put: aCharOrCharCollection.
    ^aCharOrCharCollection.
    ].

^ aCharOrCharCollection _error: #rtErrInvalidArgClass
                        args: { Character . CharacterCollection }.

]

{ #category : 'Adding' }
CharacterCollection >> insertObject: anObject at: anIndex [

"Inserts anObject into the receiver at index anIndex and returns
 aCollection."

"reimplemented to use at:put: instead of _basicAt:put:"
| selfSize |

anIndex <= 0 ifTrue:[ ^ self _errorIndexOutOfRange: anIndex ].
selfSize := self size.
anIndex > selfSize ifTrue:[
  anIndex > (selfSize + 1) ifTrue:[ ^ self _errorIndexOutOfRange: anIndex].
  ^ self at: anIndex put: anObject.
  ].

"Not adding to the end of the receiver. Create a gap for anObject to
 be copied into."
self replaceFrom: anIndex + 1 to: 1 + selfSize with: self startingAt: anIndex .

^ self at: anIndex put: anObject.

]

{ #category : 'Testing' }
CharacterCollection >> isDigits [

"Returns true if the receiver contains only digits.  Returns false if the
 receiver contains non-digit Characters."

| sz |
(sz := self size) == 0 ifTrue:[ ^ false ].
1 to: sz do: [:i |
    (self at: i) isDigit ifFalse: [^false]
].
^true

]

{ #category : 'Testing' }
CharacterCollection >> isEquivalent: aCharCollection [

"Returns true if the receiver is equivalent to aCharCollection.  The receiver
 is equivalent to aCharCollection if the receiver contains the same Characters
 as aCharCollection regardless of case or internal representation.  For
 example, if $a is in aCharCollection, it is equivalent to any representation
 of an 'a' in the receiver's character set.

 (Subclass responsibility.)"

^ self subclassResponsibility: #isEquivalent:

]

{ #category : 'Testing' }
CharacterCollection >> isInfix [

"Returns true if the receiver is an infix (binary) selector.  Returns false
 otherwise."

| binaryChars mySize fch |

binaryChars := '+-\*~<>=|/&@%,?!'.   "fixed bug 14109"

mySize := self size.
mySize == 0 ifTrue: [ ^ false ].

fch := self at: 1 .
(fch == $_  or:[ (binaryChars indexOf: fch startingAt: 1) ~~ 0] ) ifFalse:[
  ^ false
].
2 to: mySize do: [ :i |
  ((binaryChars indexOf: (self at: 1) startingAt: 1) == 0) ifTrue:[
     ^ false
  ].
].
^ true.

]

{ #category : 'Testing' }
CharacterCollection >> isKeyword [

"Returns true if the receiver is a keyword; that is, a legal keyword method
 selector. Returns false otherwise."

| mySize limit ch idx |

mySize := self size.
mySize > 1  ifFalse: [ ^ false ].
idx := 1 .
[ idx <= mySize ] whileTrue:[
  ch := self at: idx .  "first char of next keyword within selector"
  ( ch == $_ or:[ ch isLetter ] ) ifFalse: [ ^ false ].
  idx := idx + 1 .
  limit := mySize .
  [ idx <= limit ] whileTrue:[
    ch := self at: idx .
    ch == $:  ifTrue:[
      idx == mySize ifTrue:[ ^ true "hit ending $: " ].
      limit := 0 . "end of a keyword, exit inner loop"
    ] ifFalse:[
      (ch == $_  or:[ ch isAlphaNumeric]) ifFalse:[ ^ false ].
    ] .
    idx := idx + 1
  ]
].
^ false

]

{ #category : 'Testing' }
CharacterCollection >> isMinusAndDigits [

"Returns true if the receiver contains a minus sign followed only by digits.
 Returns false if the receiver has any other Characters."

((self at: 1) == $- ) ifFalse:[ ^ false ].
2 to: self size do: [:i |
    (self at: i) isDigit ifFalse:[ ^ false]
].
^true

]

{ #category : 'Testing' }
CharacterCollection >> isValidIdentifier [

"Returns true if the receiver is a valid GemStone Smalltalk variable name,
 and false otherwise.
 Returns true if first character is $_  or answers true to   isLetter ,
  and subsequent characters are all $_  or answer true to   isAlphaNumeric,
  and size is > 0 and <= 1024 .  
 Otherwise returns false .
 Note 1024==GCI_MAX_SYMBOL_SIZE "

 <primitive: 37>
 ^ self _primitiveFailed: #isValidIdentifier
]

{ #category : 'Updating' }
CharacterCollection >> lf [

"Appends a line-feed to the receiver and returns the receiver."

self addCodePoint: 10

]

{ #category : 'Formatting' }
CharacterCollection >> linesIndentedBy: anInt [

"Returns a copy of the receiver in which all lines have been indented
 by anInt spaces."

| c newStr indentStr lfInd lf targEndInd selfCurrInd sz destIdx sp |

indentStr := self class new: anInt.
indentStr atAllPut: (sp := Character space).
lf := Character lf.
lfInd := self indexOf: lf.
lfInd == 0 ifTrue:[ | res |
  (res := indentStr) addAll: self .
  ^ res
].

newStr := self class new.
selfCurrInd := 1.
c := self copy.

[(lfInd := c indexOf: lf) ~~ 0 ] whileTrue: [
  targEndInd := newStr size + indentStr size.
  newStr addAll: indentStr.
 "self copyFrom: selfCurrInd to: lfInd into: newStr startingAt: targEndInd + 1. "
  destIdx := targEndInd + 1 .
  newStr replaceFrom: destIdx to: destIdx + lfInd - selfCurrInd
	  with: self startingAt: selfCurrInd .
  selfCurrInd := lfInd + 1.
  c at: lfInd put: sp .
].
selfCurrInd < (sz := self size) ifTrue: [
  newStr addAll: indentStr.
 "self copyFrom: selfCurrInd to: sz into: newStr startingAt: newStr size + 1."
  destIdx := newStr size + 1 .
  newStr replaceFrom: destIdx to: destIdx + sz - selfCurrInd
         with: self startingAt: selfCurrInd .
].
^newStr

]

{ #category : 'Comparing' }
CharacterCollection >> match: prefix [

"Returns true if the argument prefix is a prefix of the receiver, and
 false if not.  The comparison is case-sensitive."

self size == 0 ifTrue: [ ^ prefix size == 0 ].
^ self at: 1 equals: prefix

]

{ #category : 'Comparing' }
CharacterCollection >> matchesAnyOf: aCollectionOfCharacterColls [

"Returns true if the receiver returns true to the message match: with any of
 the objects in the given collection; returns false otherwise.  Examples:

   'xyz' matchesAnyOf: #('xyz' 'abc*')
     true
   'xyz' matchesAnyOf: #('ayz' 'abc')
     false
   'x#z' matchesAnyOf: #('x@z' '*')
     false

 The deprecated class JISString does not support this method."

aCollectionOfCharacterColls do: [:coll |
  (self match: coll) ifTrue: [ ^true ]
].
^false

]

{ #category : 'Comparing' }
CharacterCollection >> matchPattern: aPattern [

"Returns true if the receiver matches aPattern, false if it doesn't.  An exact
 match is required.  For partial matching, use the 'Searching' method
 findPattern:startingAt: instead.

 The argument aPattern is a kind of Array containing zero or more
 CharacterCollections, plus zero or more occurrences of the special Characters
 $* or $?.  If either $* or $? occurs in aPattern, it acts as a wild card.
 The Character $? matches any single Character in the receiver, and $* matches
 any sequence of zero or more Characters in the receiver.  For example,

 'weimaraner' matchPattern: #('w' $* 'r')

 returns true, because the Character $* is interpreted as a wild card.

 If either of these special Characters occurs in the receiver, it is
 interpreted literally.  For example,

 'w*r' matchPattern: #('weimaraner')

 returns false - because the Character $* occurs in the receiver, it is
 interpreted as a literal asterisk (not as a wild card)."

| match         "indicates if the current pattern matched"
  pattern       "the Array of pattern elements"
  selfSize      "the size of the receiver"
  patternSize   "the number of elements in the pattern Array"
  startIndexArr "an Array of indexes into the receiver; this Array is
                 parallel to the pattern Array (each element in this Array
                 corresponds to the starting index for each pattern element)"
  index         "index into the pattern Array and the startIndexArr Array"
  thisPattern   "the current element of the pattern to match"
  selfIndex     "an index into the receiver"
|

 "The pattern Array must be processed so that there are no *? pairs in it.
 They must all be converted to ?* pairs for the algorithm to work correctly."
 pattern:= Array withAll: aPattern.
 patternSize:= pattern size.
 index := 1.
 [ index < patternSize ]
 whileTrue:
 [ (((pattern at: index) isEquivalent: $* )
    and:[ (pattern at: index+1) isEquivalent: $? ])
   ifTrue:[
     pattern at: index put: $?.
     pattern at: index+1 put: $*.
     index := 1 max: index-1.
   ]
   ifFalse:
   [ index := index + 1 ].
 ].

 "initialize"
 selfSize := self size.
 startIndexArr:= Array new: (patternSize + 1).
                 "last element is set, but not used"
 index := 1.
 startIndexArr at: 1 put: 1.

 "if no patterns to match, exit early"
 (patternSize == 0)
 ifTrue:
    [^ selfSize == 0 ].

 [index <= patternSize]
 whileTrue:
 [ thisPattern := pattern at: index.
   selfIndex := startIndexArr at: index.
   match := true.

   (thisPattern isKindOf: CharacterCollection) "pattern to match is a string"
   ifTrue:
   [ (selfIndex + thisPattern size - 1) > selfSize
     ifTrue: "this pattern too big to match rest of receiver"
        [^ false ].

     index = patternSize "processing the final pattern"
     ifTrue:
     [ ((index > 1) and: [ (pattern at: index - 1) isEquivalent: $* ])
       ifTrue: "is not the first pattern _and_ previous pattern was a $*"
       [ ((selfSize == 0) and: [thisPattern size == 0])
         ifTrue: [^ true].
         ^(self findString: thisPattern
                startingAt: selfSize - thisPattern size + 1) ~~ 0
         "find the pattern far enough back in the string so that only
         the final chars match"
       ]
       ifFalse: "processing first pattern _or_ previous pattern was not $*"
       [ (match:= (selfIndex + thisPattern size - 1 = selfSize))
         ifTrue: "exactly enough chars in self to match thisPattern"
         [ (selfSize == 0)
           ifTrue: [match:= (thisPattern size == 0)]
           ifFalse: [match:= self at: selfIndex equals: thisPattern ].
         ]
       ].
     ]
     ifFalse: "not processing the final pattern"
     [ ((index > 1) and: [ (pattern at: index - 1) isEquivalent: $* ])
       ifTrue: "not first pattern _and_ previous pattern was $*"
       [ (((selfSize == 0) and: [thisPattern size == 0]) or:
         [(selfIndex:= self findString: thisPattern
                           startingAt: selfIndex) ~~ 0])
         ifTrue: "thisPattern was found"
         [ startIndexArr at: index put: selfIndex.
           startIndexArr at: index + 1 put: selfIndex + thisPattern size.
         ]
         ifFalse: "thisPattern was not found"
            [^ false ]
       ]
       ifFalse: "first pattern _or_ previous pattern was not $*"
       [ (((selfSize == 0) and: [thisPattern size == 0]) or:
         [(self at: selfIndex equals: thisPattern)])
         ifTrue:
            [startIndexArr at: index + 1 put: selfIndex+ thisPattern size]
         ifFalse:
            [match := false ].
       ].
     ]
   ]
   ifFalse: "thisPattern is not a string"
   [ (thisPattern isEquivalent: $*)
     ifTrue:
        [startIndexArr at: (index + 1) put: selfIndex]
     ifFalse:
     [ (thisPattern isEquivalent: $?)
       ifTrue:
       [ selfIndex > selfSize
         ifTrue: "no char to match; already at end of self"
            [^ false ].
         startIndexArr at: (index + 1) put: (selfIndex + 1).
         index = patternSize "processing the last pattern"
         ifTrue:
            [match := selfIndex = selfSize.].
       ]
       ifFalse: "next pattern is neither a $* or $?"
       [ ^ aPattern _error: #rtErrBadPattern].
     ].
   ].  "end ifTrue:ifFalse"

   match
   ifTrue:
      [index := index + 1 ] "advance to the next term in the pattern"
   ifFalse:
   [ "If there is a preceding $* term in the pattern, backup to the
      term following it, and advance position in the string by 1."
      [ index := index - 1.
        index < 2 ifTrue:
           [^ false ].
        (pattern at: index - 1) isEquivalent: $*
      ] untilTrue.
      startIndexArr at: index put: ((startIndexArr at: index) + 1).
   ].
].  "end whileTrue:"

^ true

]

{ #category : 'Comparing' }
CharacterCollection >> max: another [

"If the receiver is greater than the argument, return the receiver.
 Otherwise return the argument."

	^self > another
		ifTrue: [self]
		ifFalse: [another].

]

{ #category : 'Searching' }
CharacterCollection >> maxConsecutiveSubstring [

"Returns the largest substring within the receiver that contains Characters with
 consecutive ASCII values.  For example, the message

   'abxabcdxabc' maxConsecutiveSubstring

 yields the result 'abcd'.

 If there are no such substrings larger than 2 Characters, returns a String that
 contains the first Character in the receiver."

| mySize bigSize bigStart aStart aSize lastVal thisVal |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastVal := (self at: 1) codePoint  .
2 to: mySize do:[:j |
  lastVal := lastVal + 1 .
  thisVal := (self at: j) codePoint .
  thisVal == lastVal
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastVal := thisVal .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .

]

{ #category : 'Searching' }
CharacterCollection >> maxRepeatingSubstring [

"Returns the largest substring within the receiver that contains repetitions of
 a Character, using case-sensitive comparison.  For example, the message

   'aaxbbbBxccc' maxRepeatingSubstring

 yields the result 'bbb'.

 If there are no such substrings larger than 1 Character, returns a String that
 contains the first Character in the receiver."

| mySize bigSize bigStart aStart aSize lastChar |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastChar := self at: 1 .
2 to: mySize do:[:j | | thisChar |
  thisChar := self at: j .
  thisChar == lastChar
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastChar := thisChar .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .

]

{ #category : 'Searching' }
CharacterCollection >> maxSameTypeSubstring [

"Returns the largest substring within the receiver that contains either all
 digits, all alphabetic characters, or all special characters.  For example, the
 message

   'axv2435,.-' maxSameTypeSubstring

 yields the result '2435'.

 If there are no such substrings larger than 1 Character, returns a String that
 contains the first Character in the receiver.

 This method may generate an error if the receiver is a JapaneseString."

| mySize bigSize bigStart aStart aSize lastType |

mySize := self size .
mySize < 2 ifTrue:[ ^ self ].
bigSize := 1 .
bigStart := 1 .
aStart := 1 .
aSize := 1 .
lastType := (self at: 1) _type  .
2 to: mySize do:[:j | | thisType |
  thisType := (self at: j) _type .
  thisType == lastType
    ifTrue:[ aSize := aSize + 1 ]
    ifFalse:[
      aSize > bigSize ifTrue:[
	bigStart := aStart .
	bigSize := aSize .
	].
      aSize := 1 .
      lastType := thisType .
      aStart := j .
      ].
   ].
aSize > bigSize ifTrue:[
  bigStart := aStart.
  bigSize := aSize.
  ].
^ self copyFrom: bigStart to: (bigStart + bigSize - 1) .

]

{ #category : 'Message Digests' }
CharacterCollection >> md5sum [

"Return the 128 bit MD5 checksum of the receiver as a LargeInteger.

 Computation is per RFC 1321 , http://www.ietf.org/rfc/rfc1321.txt,
 using L. Peter Deutsch's C implementation from
 http://sourceforge.net/projects/libmd5-rfc/

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 1

]

{ #category : 'Message Digests' }
CharacterCollection >> md5sumBytes [

"Return the 128 bit MD5 checksum of the receiver as a ByteArray.

 Computation is per RFC 1321 , http://www.ietf.org/rfc/rfc1321.txt,
 using L. Peter Deutsch's C implementation from
 http://sourceforge.net/projects/libmd5-rfc/

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -1

]

{ #category : 'Comparing' }
CharacterCollection >> min: another [

"If the receiver is less than the argument, return the receiver.
 Otherwise return the argument."

	^self < another
		ifTrue: [self]
		ifFalse: [another].

]

{ #category : 'Accessing' }
CharacterCollection >> numArgs [

"Returns the number of arguments the receiver would take, were the receiver
 a message selector."

| idx count sz |
(sz := self size) == 0 ifTrue:[ ^ 0 ].
(self at: sz) == $: ifTrue:[
  count := 1 .
  idx := 0 .
  [ idx := self indexOf: $: startingAt: idx + 1 .
    idx ~~ sz
  ] whileTrue:[
    count := count + 1
  ].
  ^ count
] ifFalse:[
  self isInfix ifTrue: [ ^ 1 ].
  ^ 0
]

]

{ #category : 'Json' }
CharacterCollection >> printJsonOn: aStream [

	aStream nextPut: $".
	1 to: self size do: [:j |
		| codePoint char |
		codePoint := (char := self at: j) codePoint.
    (32 <= codePoint and: [codePoint <= 127]) ifTrue: [
      char == $" "34" ifTrue: [aStream nextPutAll: '\"' ] ifFalse: [
      char == $\ "92" ifTrue: [aStream nextPutAll: '\\' ] ifFalse: [
        aStream nextPut: char
      ]]
    ] ifFalse:[
      (codePoint <= 13 and:[ codePoint >= 8]) ifTrue:[
		    codePoint == 8 	ifTrue: [aStream nextPutAll: '\b'	. codePoint := nil. ] ifFalse: [
		    codePoint == 9 	ifTrue: [aStream nextPutAll: '\t'	. codePoint := nil. ] ifFalse: [
		    codePoint == 10 	ifTrue: [aStream nextPutAll: '\n'. codePoint := nil. 	] ifFalse: [
		    codePoint == 12 	ifTrue: [aStream nextPutAll: '\f'. codePoint := nil. 	] ifFalse: [
		    codePoint == 13 	ifTrue: [aStream nextPutAll: '\r'. codePoint := nil. 	]
      ]]]]].
      codePoint ifNotNil:[ | hex |
        hex := '0123456789ABCDEF'.
			  aStream nextPutAll: '\u' ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -12) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -8) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + ((codePoint bitShift: -4) bitAnd: 16rF)) ;
             nextPut: (hex at: 1 + (codePoint bitAnd: 16rF)) .
      ]
	  ].
  ].
	aStream nextPut: $".

]

{ #category : 'Formatting' }
CharacterCollection >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

"Should be reimplemented for more efficiency in subclasses."

aStream nextPutAll: self quoted .

]

{ #category : 'Formatting' }
CharacterCollection >> printOn: aStream recursionSet: anIdentitySet [
	"Put a displayable representation of the receiver on the given stream
	 since CharacterCollections cannot have recursive references."

	self printOn: aStream

]

{ #category : 'Formatting' }
CharacterCollection >> printString [

"Returns a CharacterCollection whose contents are a displayable representation of the
 receiver."

^ self printStringWithMaxSize: 100000

]

{ #category : 'Formatting' }
CharacterCollection >> printStringWithMaxSize: n [

"Returns a CharacterCollection whose contents are a displayable representation of the
 receiver, limited to a specified number of characters <n>.

 If the number of characters in the displayable representation  exceeds <n>,
 display only the first <n>-1 characters and then display '. . .'. "

| ws |

ws := PrintStream printingOn: self speciesForPrint new maxSize: n.
self printOn: ws.
^ws contents

]

{ #category : 'Formatting' }
CharacterCollection >> quoted [

"Returns a copy of the receiver enclosed in single-quote marks, with contained
 single-quote Characters doubled.  The copy is of the same class as the
 receiver."

| sz result targetIdx lastIdx idx |
sz := self size.
result := self class _newString: sz + 2.
result at: 1 put: $'.
targetIdx := 2.
lastIdx := 1.
[ (idx := self indexOf: $' startingAt: lastIdx) == 0 ] whileFalse: [
 "self copyFrom: lastIdx to: idx into: result startingAt: targetIdx . "
  result replaceFrom: targetIdx to: targetIdx + idx - lastIdx
	 with: self startingAt: lastIdx .

  targetIdx := targetIdx + (idx - lastIdx) + 2.
  result at: targetIdx - 1 put: $'.
  lastIdx := idx + 1
].
lastIdx <= sz ifTrue: [
 "self copyFrom: lastIdx to: sz into: result startingAt: targetIdx "
  result replaceFrom: targetIdx to: targetIdx + sz - lastIdx
         with: self startingAt: lastIdx .
].
result at: targetIdx + (sz - lastIdx) + 1 put: $'.
^result

]

{ #category : 'Comparing' }
CharacterCollection >> sameAs: aCharCollection [

"Returns true if the receiver is equivalent to aCharCollection.  The receiver
 is equivalent to aCharCollection if the receiver contains the same Characters
 as aCharCollection regardless of case or internal representation.  For
 example, if $a is in aCharCollection, it is equivalent to any representation
 of an 'a' in the receiver's character set.
 From ANSI.  Used in Seaside. "

(aCharCollection isKindOf: CharacterCollection) ifFalse: [ ^false ].

self size ~~ aCharCollection size ifTrue: [ ^false ].

^ self isEquivalent: aCharCollection

]

{ #category : 'Message Digests' }
CharacterCollection >> sha1Sum [

"Return the 160 bit SHA1 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 2

]


{ #category : 'Message Digests' }
CharacterCollection >> sha1SumBytes [

"Return the 160 bit SHA1 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -2

]

{ #category : 'Message Digests' }
CharacterCollection >> sha256Sum [

"Return the 256 bit SHA256 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 3

]


{ #category : 'Message Digests' }
CharacterCollection >> sha256SumBytes [

"Return the 256 bit SHA256 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -3

]

{ #category : 'Message Digests' }
CharacterCollection >> sha3_224Sum [

"Return the SHA3 224 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 5

]

{ #category : 'Message Digests' }
CharacterCollection >> sha3_224SumBytes [

"Return the SHA3 224 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -5

]

{ #category : 'Message Digests' }
CharacterCollection >> sha3_256Sum [

"Return the SHA3 256 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 6

]


{ #category : 'Message Digests' }
CharacterCollection >> sha3_256SumBytes [

"Return the SHA3 256 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -6

]

{ #category : 'Message Digests' }
CharacterCollection >> sha3_384Sum [

"Return the SHA3 384 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 7

]

{ #category : 'Message Digests' }
CharacterCollection >> sha3_384SumBytes [

"Return the SHA3 384 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -7

]

{ #category : 'Message Digests' }
CharacterCollection >> sha3_512Sum [

"Return the SHA3 512 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 8

]

{ #category : 'Message Digests' }
CharacterCollection >> sha3_512SumBytes [

"Return the SHA3 512 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -8

]

{ #category : 'Message Digests' }
CharacterCollection >> sha512Sum [

"Return the 512 bit SHA512 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 4

]

{ #category : 'Message Digests' }
CharacterCollection >> sha512SumBytes [

"Return the 512 bit SHA512 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -4

]

{ #category : 'Digital Signature Creation - EC' }
CharacterCollection >> signWithEcPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Signs the receiver with the given elliptic curve private key. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #EC ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 0 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - DSA' }
CharacterCollection >> signWithSha1AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha1AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the
 resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha1AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -2 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - DSA' }
CharacterCollection >> signWithSha256AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -3 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_224AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 5 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_224AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -5 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 6 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -6 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_384AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 7 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_384AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -7 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 8 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha3_512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -8 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - DSA' }
CharacterCollection >> signWithSha512AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
CharacterCollection >> signWithSha512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -4 signature: aByteArrayOrNil

]

{ #category : 'Updating' }
CharacterCollection >> space [

"Appends a space to the receiver and returns the receiver."

self add: $ .

]

{ #category : 'Private' }
CharacterCollection >> speciesForConversion [

"Return the class of the receiver.  Subclasses should reimplement this method."

^ self class.

]

{ #category : 'Converting' }
CharacterCollection >> subStrings [

"Returns an Array of CharacterCollections where element represents a word in
 the receiver.  A word is a group of Characters separated by one or more
 separators."

^ self asArrayOfSubstrings.

]

{ #category : 'Converting' }
CharacterCollection >> subStrings: separators [
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by any of the Characters in separators.  For compatibility with
	 existing code, separators may be a single Character or a collection of Characters.
	 The result will include empty substrings when two adjacent separators exist, as well as if
	 a separator is the first or last element."

	^separators size == 0
		ifTrue: [self subStringsDelimitedBy: separators]
		ifFalse: [self subStringsDelimitedByAny: separators].

]

{ #category : 'Converting' }
CharacterCollection >> subStringsDelimitedBy: aCharacter [
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by aCharacter.  The result will include empty substrings when
	 two adjacent separators exist, as well as if a separator is the first or last element."

	| result startIndex endIndex sz |
	result := {}.
	startIndex := 1.

	[endIndex := self indexOf: aCharacter startingAt: startIndex.
	endIndex == 0]
			whileFalse:
				[endIndex == startIndex ifTrue: [result add: self class _newString].
				endIndex > startIndex
					ifTrue: [result add: (self copyFrom: startIndex to: endIndex - 1)].
				startIndex := endIndex + 1].
	startIndex > (sz := self size)
		ifTrue: [result add: self class _newString]
		ifFalse: [result add: (self copyFrom: startIndex to: sz)].
	^result

]

{ #category : 'Converting' }
CharacterCollection >> subStringsDelimitedByAny: separators [
	"Returns an Array of CharacterCollections in which each element represents a
	 substring separated by any of the Characters in separators.
	 The result will include empty substrings when two adjacent separators exist,
	 as well as if a separator is the first or last element."

	| nextName result |
	result := {}.
	nextName := self speciesForConversion new.
	self do:
			[:c |
			(separators includes: c)
				ifTrue:
					[result add: nextName.
					nextName := self speciesForConversion new]
				ifFalse: [nextName add: c]].
	result add: nextName.
	^result

]

{ #category : 'Deprecated' }
CharacterCollection >> toServerTextFile: aFileSpec [

self deprecated: 'CharacterCollection >> toServerTextFile: deprecated long before v3.0. Use an instance of GsFile
 to access the file system.'.
self _toServerTextFile: aFileSpec

]

{ #category : 'Converting' }
CharacterCollection >> trimBlanks [

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading and trailing blanks removed."

self size == 0 ifTrue: [ ^ self ].
^ (self trimLeadingBlanks) trimTrailingBlanks.

]

{ #category : 'Converting' }
CharacterCollection >> trimLeadingBlanks [

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading blanks removed."

| idx blank sz |
(sz := self size) == 0 ifTrue: [ ^ self ].
blank := Character space.
((self at: 1) == blank) ifFalse: [ ^ self ].
idx := 2.
[ true ] whileTrue:[
  idx <= sz ifFalse:[ ^ '' ].
  (self at: idx) == blank ifFalse:[ ^ self copyFrom: idx to: sz].
  idx := idx + 1
]

]

{ #category : 'Converting' }
CharacterCollection >> trimLeadingSeparators [

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading separators removed."

| idx sz |
(sz := self size) == 0 ifTrue:[ ^ self ].
(self at: 1) isSeparator ifFalse: [ ^ self ].
idx := 2.
[ true ] whileTrue:[
  idx <= sz ifFalse:[ ^ '' ].
  (self at: idx) isSeparator ifFalse:[ ^ self copyFrom: idx to: sz].
  idx := idx + 1
]

]

{ #category : 'Converting' }
CharacterCollection >> trimSeparators [

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with leading and trailing separators removed."

self size == 0 ifTrue: [ ^ self ].
^ (self trimLeadingSeparators) trimTrailingSeparators.

]

{ #category : 'Converting' }
CharacterCollection >> trimTrailingBlanks [

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with trailing blanks removed."

| idx blank sz |

(sz := self size) == 0 ifTrue: [ ^ self ].
blank := Character space.
(self at: sz ) == blank ifFalse: [ ^ self ].
idx := sz  - 1.
[ true ] whileTrue:[
  idx < 1 ifTrue:[ ^ '' ] .
  (self at: idx) == blank ifFalse:[ ^ self copyFrom: 1 to: idx ].
  idx := idx - 1
]

]

{ #category : 'Converting' }
CharacterCollection >> trimTrailingSeparators [

"Returns a CharacterCollection containing the same Characters as the receiver,
 but with trailing separators removed."

| idx sz |

(sz := self size) == 0 ifTrue: [ ^ self ].
(self at: sz ) isSeparator ifFalse: [ ^ self ].
idx := sz  - 1.
[ true ] whileTrue:[
  idx < 1 ifTrue:[ ^ '' ] .
  (self at: idx) isSeparator ifFalse:[ ^ self copyFrom: 1 to: idx ].
  idx := idx - 1
]

]

{ #category : 'Formatting' }
CharacterCollection >> trimWhiteSpace [

"Returns a copy of the receiver with leading and trailing white space removed."

| first limit selfSize |

((selfSize := self size) == 0) ifTrue: [
   ^ self class _newString
].

limit := selfSize + 1.

first := 1 .
(self at: 1) isSeparator ifTrue: [ | j |
  first := nil .
  j := 2.
  [ j == limit ] whileFalse: [
      (self at: j) isSeparator ifTrue: [
         j := j + 1.
      ] ifFalse:[
         first := j.
         j := limit .
       ].
  ].
  first ifNil: [ ^ self class _newString ].
].

(self at: selfSize) isSeparator ifTrue: [ | k |
  k := selfSize - 1.
  [ k == 0 ] whileFalse: [
     (self at: k) isSeparator ifFalse:[
       ^ self copyFrom: first to: k
     ].
     k := k - 1.
  ].
  ^ self class _newString
].

first == 1 ifTrue:[  ^ self copy ].
^ self copyFrom: first to: selfSize .

]

{ #category : 'Digital Signature Creation - EC' }
CharacterCollection >> verifyWithEcPublicKey: aGsTlsPublicKey signature: aByteArray [
"Verifies the receiver using the given elliptic curve key and signature.  Returns true
 if the signature is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #EC ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 0 signature: aByteArray

]

{ #category : 'Digital Signature Verification - DSA' }
CharacterCollection >> verifyWithSha1AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA1 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
  # _primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha1AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -2 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha1AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray

]

{ #category : 'Digital Signature Verification - DSA' }
CharacterCollection >> verifyWithSha256AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA256 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -3 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_224AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -5 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_224AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 5 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -6 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 6 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_384AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -7 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_384AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 7 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -8 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha3_512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 8 signature: aByteArray

]

{ #category : 'Digital Signature Verification - DSA' }
CharacterCollection >> verifyWithSha512AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA512 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -4 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
CharacterCollection >> verifyWithSha512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray

]

{ #category : 'Formatting' }
CharacterCollection >> width: anInteger [

"Pads the receiver with spaces to create an object of size anInteger.
 If anInteger is positive, the spaces are added to the right of the receiver.
 If anInteger is negative, the spaces are added to the left of the receiver.
 If the size of the receiver is already greater than anInteger, the receiver
 is left unchanged."

| onRight s change |

change := anInteger abs - self size.
change > 0
ifTrue:
  [ onRight := anInteger > 0.
    s := (self speciesForConversion) new: change.
    s atAllPut: $ .
    onRight
    ifTrue:
      [ self addAll: s ]
    ifFalse:
      [ self insertAll: s at: 1 ]
  ]

]

{ #category : 'Copying' }
CharacterCollection >> withCRs [

"Supplied for Smalltalk-80 compatibility.  This is equivalent to withLFs."

^self withLFs

]

{ #category : 'Formatting' }
CharacterCollection >> wrapTo: col [

"Word-wrap the receiver to column col, treating tab Characters as modulo-8."

^ self _wrapTo: col indentingWith: ''

]


{ #category : 'Private' }
CharacterCollection >> _lineNumberFor:  offset [
  "Counts LF characters in the receiver up to offset and
   returns an Array of the form { line number,  offset in line } "
  | cnt lf eolPos |
  cnt := 1 . lf := Character lf .
  eolPos := 0 .
  1 to: (offset min: self size) do:[:n|
    (self at: n) == lf ifTrue:[ eolPos := n . cnt := cnt + 1 ].
  ].
  ^ { cnt . offset - eolPos }
]

