Extension { #name : 'FloatingPointError' }

{ #category : 'Private' }
FloatingPointError class >> _checkFpStatus [
  "If any of the currently enabled exceptions have occurred,
   clear those bits from the VM state and signal an error.
   Currently enabled exceptions are returned by
     FloatingPointError(C) >> enabledExceptions .
   This method is called from primitive failure paths in in each floating
   point primitive."

| flags |
flags := self _getExceptions: false .
flags ~~ 0 ifTrue:[
  self new actual: flags ; signal
].

]

{ #category : 'Private' }
FloatingPointError class >> _enabledBitsAsArray: bits [
  "The result is in sorted order."
  | res |
  res := { } .
  bits ifNotNil:[
    (bits bitAnd: 1) ~~ 0 ifTrue:[ res add: #divideByZero ].
    (bits bitAnd: 16r10) ~~ 0 ifTrue:[ res add: #inexactResult ].
    (bits bitAnd: 8) ~~ 0 ifTrue:[ res add: #invalidOperation ].
    (bits bitAnd: 2) ~~ 0 ifTrue:[ res add: #overflow ].
    (bits bitAnd: 4) ~~ 0 ifTrue:[ res add: #underflow ].
  ].
  ^ res

]

{ #category : 'Private' }
FloatingPointError class >> _enableExceptions: aSmallInteger [

"aSmallInteger may have one or more bits in the range 16r1F,
 per the constants in FloatingPointError(C)>>_initializeConstants .
 other bits are ignored.  The specified exceptions will be checked for
 after each arithmetic primitive in Float and SmallDouble and if
 any of those execptions occurs, a FloatingPointError will be signaled.
 Returns the value which was passed in the previous call to this primitive.

 If aSmallInteger == -1 , then returns the positive value
 which was passed in the previous call to this primitive without changing
 the state of the enabled exceptions.
"

<primitive: 122>

aSmallInteger _validateClass: SmallInteger .
self _primitiveFailed: #enableExceptions: args: { aSmallInteger }

]

{ #category : 'Private' }
FloatingPointError class >> _exceptionList [

^  #( #divideByZero #overflow #underflow #invalidOperation #inexactResult)

]

{ #category : 'Private' }
FloatingPointError class >> _getExceptions: aBoolean [

"Fetch and clear bits representing floating point exceptions that have occurred
 since last execution of this primitive.  Result is a SmallInteger.

 aBoolean == false  means get bits enabled by the last call to
 FloatingPointError(C)>>enableExceptions: , and clear all bits.
 aBoolean == true means get and clear all bits .

 The bits in the result are in the range 16r1F and
 are defined by the constants in FloatingPointError(C)>>_initializeConstants ."

<primitive: 129>

aBoolean _validateClass: Boolean .
self _primitiveFailed: #_getExceptions: args: { aBoolean }

]

{ #category : 'Private' }
FloatingPointError class >> _initializeConstants [
   "VM changes needed if you change any of these bit definitions.
    Also fix sends of _enableExceptions: in image."
   self _addInvariantClassVar: #divideByZero value: 1 ;
         _addInvariantClassVar: #overflow value: 2  ;
         _addInvariantClassVar: #underflow value: 4  ;
         _addInvariantClassVar: #invalidOperation  value: 8  ;
         _addInvariantClassVar: #inexactResult  value: 16r10 .

]

{ #category : 'Private' }
FloatingPointError class >> _symbolToBit: aSymbol [

  | bit |
  bit := classVars at: aSymbol otherwise: nil .
  bit ifNil:[ Error signal:'invalid name of a floating point exception bit, ' , aSymbol].
  ^ bit

]

{ #category : 'Private' }
FloatingPointError class >> enableAllExceptions [

^ self enableExceptions: self _exceptionList

]

{ #category : 'Private' }
FloatingPointError class >> enabledExceptions [
  "Returns an Array containing zero or more of the Symbols
    #divideByZero #overflow #underflow #invalidOperation #inexactResult
  reflecting the most recent call to either of
     FloatingPointError class>>enableAllExceptions
     FloatingPointError class>>enableExceptions:
  "
  ^ self _enabledBitsAsArray: (self _enableExceptions: -1)  .

]

{ #category : 'Accessing' }
FloatingPointError class >> enableExceptions: argument [

"Argument may be one of the symbols
   #divideByZero #overflow #underflow #invalidOperation #inexactResult
 or an Array containing zero or more of those Symbols ,
 or nil which means enable none of the exceptions.

 The specified exceptions will be checked for
 after each arithmetic primitive in Float and SmallDouble and if
 any of those execptions occurs, a FloatingPointError will be signaled.
 Overrides the settings of a previous call to enableExceptions: .
 Returns a SmallInteger, the previously enabled exception bits."

 | bits |
 argument ifNil:[ ^ self _enableExceptions: 0 ].
 argument _isSymbol ifTrue:[
   ^ self _enableExceptions: (self _symbolToBit: argument)
 ].
 argument _isArray ifTrue:[
   bits := 0 .
   argument do:[:aSym |
     bits := bits bitOr:( self _symbolToBit: aSym)
   ].
   ^ self _enableExceptions: bits
 ].
 argument _validateClasses: { Array . Symbol }.

]

{ #category : 'Private' }
FloatingPointError >> actual: aValue [
  actual := aValue

]

{ #category : 'Formatting' }
FloatingPointError >> asString [
  | str arr cls |
  str := 'a ' copy .
  str add: (cls := self class) name .
  arr := cls _enabledBitsAsArray: actual .
  arr size == 0 ifTrue:[
    gsNumber == ERR_LargeIntegerOverflow
      ifTrue:[ str add:' overflow during LargeInteger arithmetic'].
  ] ifFalse:[
    1 to: arr size do:[:n | str add: $  ; add: (arr at: n) ].
  ].
  ^ str .

]

{ #category : 'Accessing' }
FloatingPointError >> exceptionList [
 "Return an Array of Symbols , the list of exceptions
  which produced this instance.  The result will contain one or more of
     #divideByZero #overflow #underflow #invalidOperation #inexactResult  .
  and is sorted alphabetically.
 "
  ^ self class _enabledBitsAsArray: actual

]

{ #category : 'Instance initialization' }
FloatingPointError >> initialize [
  gsNumber := ERR_FloatingPointError.
  gsResumable := true .
  gsTrappable := true .
  actual := 0 .

]
