Extension { #name : 'GsProcess' }

{ #category : 'Private' }
GsProcess class >> _current [
  "Returns currently active process .
   If current execution was initiated by GCI and no GsProcess has yet been created,
   creates a GsProcess and returns that .
   The result should be used for accessing thread-local data only.
   Instance methods related to accessing the stack frames do not work
   for the active process."

<primitive: 693>
self _primitiveFailed: #_current
]

{ #category : 'Private' }
GsProcess class >> _currentOrNil [
  "Returns currently active process if one exists. Returns nil if
   current execution was initiated by GCI and no GsProcess has yet been created.
   The result should be used for accessing thread-local data only.
   Instance methods related to accessing the stack frames do not work
   for the active process."

<primitive: 1110>
self _primitiveFailed: #_currentOrNil

]

{ #category : 'Private' }
GsProcess class >> _forkBlock: aBlock with: blockArgs env: envId prio: aPriority [
  "Answer an ready instance of the receiver representing a potential
   process on the given block and with the active priority."
  | proc sched |
  (blockArgs _isArray ) ifFalse:[
    blockArgs _error: #rtErrInvalidArgClass args: { Array } .
    self _uncontinuableError .
  ].
  (proc := self _basicNew) _init: aBlock args: blockArgs env: envId
      stackSerial: self _current _stackSerialNum
      forked:  ModeInfo_forked .
  aPriority ifNil:[
    (sched := self _scheduler) _scheduleNewProcess: proc prio: sched activePriority .
  ] ifNotNil:[
    (sched := self _scheduler) _scheduleNewProcess: proc prio: aPriority  .
    sched activePriority < aPriority ifTrue:[
       sched yield .   "fix 49114"
    ].
  ].
  ^ proc

]

{ #category : 'Debugging Support' }
GsProcess class >> _frameContentsAt: aLevel [

"Returns an Array describing the specified stack level in the currently
 executing process.

 aLevel == 1 means top of stack with respect to the
               sender of _frameContentsAt: .

 If aLevel is out of range, returns nil.

 The contents of the result Array are the same as for the instance method
 GsProcess | _frameContentsAt:, except for :
    element  3  is always nil
             4  varContext is always nil .
             8 (self) is nil
             9 (argAndTempNames) is nil
 "

<primitive: 195>
self _primitiveFailed: #_frameContentsAt: args: { aLevel }

]

{ #category : 'Debugging Support' }
GsProcess class >> _fullStackReport [
  "used in ernie framework"
| curr res |
curr := GsProcess _current .
[ res := curr stackReportToLevel: 300 withArgsAndTemps: true andMethods: false ] forkAt: 39 .
^ res asString .

]

{ #category : 'Private' }
GsProcess class >> _maxProcessStacks [

  ^ MaxProcessStacks "from C constant OM_MAX_PROCESS_STACKS"

]

{ #category : 'Debugging Support' }
GsProcess class >> _methodInFrameContents: aFrameContentsArray [

^ aFrameContentsArray at: 1 "virtual machine constant"

]

{ #category : 'Private' }
GsProcess class >> _newForBlock: aBlock with: blockArgs env: envId [
  "Answer a suspended instance of the receiver representing a potential
   process on the given block and with the active priority."
  | proc |
  (blockArgs _isArray ) ifFalse:[
    blockArgs _error: #rtErrInvalidArgClass args: { Array } .
    self _uncontinuableError .
  ].
  (proc := self _basicNew) _init: aBlock args: blockArgs env: envId
       stackSerial: self _current _stackSerialNum
      forked:  ModeInfo_forked .
  self _scheduler _suspendNewProcess: proc .
  ^ proc

]

{ #category : 'Debugging Support' }
GsProcess class >> _receiverAt: aLevel [

"Returns the receiver at the given level of the currently executing
 process' stack.

 aLevel == 1 means top of stack with respect to the
               sender of _selfAt: .

 If aLevel is out of range, returns nil."

|frame|
(frame := self _frameContentsAt: aLevel) ifNil:[ ^ nil ].
^ frame at: 10 "virtual machine constant"

]

{ #category : 'Private' }
GsProcess class >> _scheduler [

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler

]

{ #category : 'Debugging Support' }
GsProcess class >> _selfAt: aLevel [

"Returns self at the given level of the currently executing process' stack.

 aLevel == 1 means top of stack with respect to the
               sender of _selfAt: .

 If aLevel is out of range, returns nil."

|frame|
(frame := self _frameContentsAt: aLevel + 1) ifNil:[ ^ nil ].
^ frame at: 8 "virtual machine constant"

]

{ #category : 'Debugging Support' }
GsProcess class >> _topazAsString: anObject [
  ^ [ anObject asString
    ] onSynchronous: AbstractException do:[:ex |
      [  | cls |
         cls := Reflection classOf: anObject .
         cls == UnauthorizedObjectStub ifTrue:[ anObject __asString]
                           ifFalse:[ 'a ', cls name ]
      ] onSynchronous: AbstractException do:[:exb |
        '<error during asString>'
      ]
    ]

]

{ #category : 'Debugging Support' }
GsProcess class >> _topazExceptionName: anException [
  ^ [ | ecls |
      (ecls := anException class) _isExceptionClass ifFalse:[ ^ nil ].
      ecls name  .
    ] onSynchronous: AbstractException do:[:ex |
      ex return: '<error during _topazExceptionName:>'
    ].

]

{ #category : 'Instance Creation' }
GsProcess class >> continuationFromLevel: numLevels [

"Returns an instance of the receiver, that is a copy of the current
 process'  stack , from the stack base to the frame that is the
 specified number of levels above the frame executing this primitive.
 anInt must be >= 1 .

 When  value:   is sent to the result,
 execution resumes in the frame at the top of the instance's saved stack,
 and with top-of-stack value replaced by the argument to value: .

 The result contains copies of all method temps and args from the
 stack it was created from ;  value: sent to the result restores
 the contents of of all method temps and args to the identical values when
 the instance was created. "

<primitive: 47>
numLevels _validateClass: SmallInteger .
numLevels < 1 ifTrue:[ self _errorIndexOutOfRange: numLevels ].
self _primitiveFailed: #continuationFromLevel: args: { numLevels } .
self _uncontinuableError

]

{ #category : 'Debugging Support' }
GsProcess class >> cpuOsKind [

"Returns an integer specifying the CPU and OS on which
 this VM is running. The result is one of
   1 - reserved for use within VM
   2 - sparc Solaris
   3 - x86_64 Linux
   4 - PowerPc AIX
   5 - x86_64 Apple Unix
   6 - x86_64 Solaris
   7 - Itanium HPUX
"
<primitive: 345>
self _primitiveFailed: #cpuOsKind

]

{ #category : 'Accessing' }
GsProcess class >> current [
  "Returns currently active process .
   If current execution was initiated by GCI and no GsProcess has yet been created,
   creates a GsProcess and returns that .
   The result should be used for accessing thread-local data only.
   Instance methods related to accessing the stack frames do not work
   for the active process."

<primitive: 693>
self _primitiveFailed: #current
]

{ #category : 'Partial Continuations' }
GsProcess class >> installPartialContinuation: aGsProcess atLevel: targetLevel value: anObject [

" aGsProcess must be a partial continuation, i.e., the result from
  partialContnuationFromLevel:to:, or an error will be generated.

  Replaces targetLevel frames of stack with partialContinuation
  and resumes execution with top-of-stack to be replaced by anObject."

<primitive: 2031>
self _uncontinuableError  "should never reach here"

]

{ #category : 'Accessing' }
GsProcess class >> methodAt: aLevel [
  "Return the GsNMethod in which the sender of methodAt: is executing."
  ^ GsProcess _methodInFrameContents:(GsProcess _frameContentsAt: aLevel + 1) 
]

{ #category : 'Instance Creation' }
GsProcess class >> new [

"Disallowed."

self shouldNotImplement: #new

]

{ #category : 'Partial Continuations' }
GsProcess class >> partialContinuationFromLevel: levelsToTrim to: targetLevel [

"Returns an instance of the receiver, that is a copy of the stack
 for the current process from frame levelsToTrim+1 to frame targetLevel-1.
 This method (partialContinuationFromLevel:to: will be at top of stack
 which is frame 1 , when the partialContinuation is created.
 levelsToTrim and targetLevel must be >= 1 .
 targetLevel must be >= levelsToTrim+2 .

 The result is expected to be used as an argument to
 #installPartialContinuation:atLevel:value: .

 The result contains copies of all method temps and args from the
 stack it was created from and when used as an argument to
 #installPartialContinuation:atLevel:value:, the contents of all
 method temps and args are restored to the identical values when
 the instance was created."

<primitive: 900>
levelsToTrim _validateClass: SmallInteger .
levelsToTrim < 1 ifTrue:[ self _errorIndexOutOfRange: levelsToTrim ].
targetLevel _validateClass: SmallInteger .
targetLevel < 1 ifTrue:[ self _errorIndexOutOfRange: targetLevel ].
targetLevel > (levelsToTrim + 1) ifFalse:[ self _errorIndexOutOfRange: targetLevel ].
self _primitiveFailed: #partialContinuationFromLevel:to:
     args: { levelsToTrim . targetLevel } .
self _uncontinuableError

]

{ #category : 'Debugging Support' }
GsProcess class >> stackReportToLevel: aLevel [

"Returns a String describing the currently active stack, starting with
 to the sender of this method (which is considered level 1).  The aLevel

 The format of the result is subject to change with each release of GemStone."

| framesArr aFrame report level lf |
framesArr := { }  .
level := 1 .
[ level <= aLevel and:[ (aFrame := self _frameContentsAt: level + 1) ~~ nil] ]
whileTrue:[
  framesArr at: level put: aFrame.
  level := level + 1.
  ].
report := String new .
lf := Character lf .
1 to: framesArr size do:[:j | | gsMethod |
  report add: j asString; add:' ' .
  aFrame := framesArr at: j .
  gsMethod := aFrame at: 1 .
  gsMethod ifNil:[
    report add:'<Reenter marker>'; add: lf .
  ] ifNotNil:[ | stepPoint aRcvr |
    aRcvr := aFrame at: 10 .
    report add:  (gsMethod _descrForStackPadTo: 0 rcvr: aRcvr ) .
    stepPoint := gsMethod _previousStepPointForIp: (aFrame at: 2) .
    report add:' @' ; add: stepPoint asString ;
      add: ' line ';
      add: (gsMethod _lineNumberForStep: stepPoint) asString ;
      add:'  [GsNMethod '; add: gsMethod asOop asString ; add:']';
      add: lf .
    ].
  ].
^ report

]

{ #category : 'Debugging Support' }
GsProcess class >> usingNativeCode [

  "returns true if the currently executing process is using native code"
<primitive: 855>
self _primitiveFailed: #usingNativeCode

]

{ #category : 'Private Debugging Support' }
GsProcess >> __returnAddrIdxAt: aLevel [

| fpIdx theMeth frDesc |
(frDesc := self _frameDescrAt: aLevel ) ifNil:[ ^ nil ].
fpIdx := frDesc at: 2 .
theMeth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS)) .
theMeth ifNil:[ ^  nil  ].
aLevel > 1 ifTrue:[ | calleeFpIdx |
  calleeFpIdx :=  frDesc at: 3 .
  ^ calleeFpIdx + FP_rtnAddr_OFS .
] ifFalse:[
  ^ 1 "savedIP at TOS"  .
]

]

{ #category : 'Private' }
GsProcess >> _addJoiner: aDelay [

| js |
(js := joiners ) ifNil:[
  joiners := { aDelay }.
  ^ self .
].
js at: js size + 1 put: aDelay

]

{ #category : 'Debugging Support' }
GsProcess >> _calleeIsAsync: aLevel [
  self _currentOrNil == self ifTrue:[ ^ false "an approximation"].
  aLevel > 1 ifTrue:[
    (self _frameDescrAt: aLevel -1) ifNotNil:[:calleeDesc |
      (self _fetchMethod:(arStack at:(calleeDesc at:2) + FP_codePtr_OFS)) ifNotNil:[:calleeMeth|
        | sel |
        sel := calleeMeth selector .
        (sel == #_signalAsync: or:[ sel == #_signalGcFinalize: or:[ sel == #_signalTimeout:]]) ifTrue:[
           ^ calleeMeth inClass == AbstractException
        ]
      ]
    ]
  ].
  ^ false
]

{ #category : 'Accessing' }
GsProcess >> _cannotReturn: aValue [

"Raises an error message in the event that the virtual machine cannot
 resume after the current method or block context."

^ self _error: #rtErrCantReturn

]

{ #category : 'Private' }
GsProcess >> _canWaitOnSocket [

  ^ true

]

{ #category : 'Private' }
GsProcess >> _checkIfDebuggable [
  "Check to make sure the receiver is debuggable. Currently this
   means that it is in the debug or ready states.
   If it is not debuggable then raise an error."

  | status |

  status := self _statusString.
  (status = 'ready' or:[status = 'debug' or:[ status = 'active' ]]) ifFalse: [
    ImproperOperation new _number: 2376 ; reason: #rtErrGsProcessNotDebuggable;
  object: self ; signal
  ].

]

{ #category : 'Private Debugging Support' }
GsProcess >> _clearAllStackBreaks [

  1 to: self stackDepth - 1 do:[:n | self _clearStackBreakAt: n ]

]

{ #category : 'Private Debugging Support' }
GsProcess >> _clearAllStepBreaksAt: aLevel [

"Legacy code, not used in the current debugging implementation.
 Clears single step breakpoints at the specified level on the stack.  If
 aLevel is 1, clears the step-into bit in the virtual machine flag word, which
 could be left over from a step-into flag argument to a GciPerform* or
 GciExecute* call."

| meth |
aLevel == 1 ifTrue:[  | flags |
  flags := self interruptFlag .
  flags := flags bitAnd:( 16r200 bitInvert) .
  self setInterruptFlag: flags  .
].
meth := (self == GsProcess _currentOrNil) ifTrue:[ GsProcess _frameContentsAt: 1 ]
                                     ifFalse:[ self localMethodAt: aLevel ].
meth ifNotNil:[ meth _clearAllStepBreaks ]

]

{ #category : 'Private Debugging Support' }
GsProcess >> _clearStackBreakAt: aLevel [

"Clear stack breakpoint at specified level.

 Debugger implementations should use GciStep rather than invoking
 this method directly."

| ipOffset dp |
self _nativeStack ifTrue:[
  ^ self _error: #rtErrNativeCodeNoBreakpts
].
(aLevel < 1 or:[ aLevel > (dp := self stackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
  ].
(self __returnAddrIdxAt: aLevel) ifNotNil:[ :rtnAddrIdx |
  ipOffset := arStack at: rtnAddrIdx .
  ipOffset < 0 ifTrue:[
    arStack at: rtnAddrIdx put: ipOffset abs
  ].
].

]

{ #category : 'Private' }
GsProcess >> _current [
  "Returns currently active process .
   The result should be used for accessing thread-local data only.
   Instance methods related to accessing the stack frames do not work
   for the active process."

<primitive: 693>
self _primitiveFailed: #_current

]

{ #category : 'Private' }
GsProcess >> _currentOrNil [
  "Returns currently active process if one exists. Returns nil if
   current execution was initiated by GCI and no GsProcess has yet been created.
   The result should be used for accessing thread-local data only.
   Instance methods related to accessing the stack frames do not work
   for the active process."

<primitive: 1110>
self _primitiveFailed: #_currentOrNil

]

{ #category : 'Process Properties' }
GsProcess >> _environment [

  ^ environment

]

{ #category : 'Changing Process State' }
GsProcess >> _executeEnsureBlocks [
  | pairs |
  pairs := self _getEnsureBlocks .
  1 to: pairs size by: 2 do: [:k |
    self _removeEnsureAtFP:( pairs at: k) .
    (pairs at: k + 1 ) value .
  ].

]

{ #category : 'Accessing' }
GsProcess >> _fetchMethod: aCodePtr [

"translate a codePtr to a GsNMethod"
| codeCls  |
codeCls := aCodePtr class .
codeCls == GsNMethod ifTrue:[
 ^ aCodePtr
] ifFalse:[
  self _halt: 'should not be here' .
  ^ nil
].

]

{ #category : 'Private' }
GsProcess >> _finishDelay [
  "Return 1 if receiver ready to run, 0 otherwise.
   For use only by the scheduler. "
  waitingOn := nil .
  self _isSuspended ifTrue:[
    ^ 0
  ].
  self _scheduler _scheduleProcess: self.
  ^ 1
]

{ #category : 'Debugging Support' }
GsProcess >> _frameAt: aLevel offsetOfTempNamed: aString [

"For the method activation at level aLevel, returns a SmallInteger which
 is the 1-based offset of the method argument or temporary with name aString.
 Returns 0 if no temp exists at aLevel with name aString.
 Generates an error if aLevel is out of range.

 Deprecated.
"
 | frameContents aSym tempNames dp |
 (aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
   aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
 ].
 aSym := Symbol _existingWithAll: aString .
 aSym ifNil:[ ^ 0 ].
 frameContents := self _frameContentsAt: aLevel .
 tempNames := frameContents at: 9 "virtual machine constant" .
 ^ tempNames indexOf: aSym
]

{ #category : 'Debugging Support' }
GsProcess >> _frameAt: aLevel tempAt: anOffset put: aValue [

"In the method activation at level aLevel, alter the method argument or
 temporary at anOffset to have value aValue.
 Generates an error if aLevel or anOffset is out of range.

 Deprecated
 "

| frDesc fpIdx codePtr theMeth nArgs argAndTempNames vcOfs
  theVc argsTmpsOfss absIp lastTmpIdx sd |
(aLevel < 1 or:[ aLevel > (sd := self localStackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . sd } .
].
frDesc := self _frameDescrAt: aLevel .
fpIdx := frDesc at: 2 .
codePtr := arStack at:( fpIdx + FP_codePtr_OFS ).
theMeth := self _fetchMethod: codePtr  .
  nArgs := theMeth numArgs .
argAndTempNames := theMeth  argsAndTemps .
vcOfs := fpIdx + FP_VC_OFS .
theVc := arStack at: vcOfs .
(theVc class == VariableContext) ifFalse:[ theVc := nil ].
argAndTempNames := theMeth  argsAndTemps .
argAndTempNames ifNil:[ argAndTempNames := { } ] .
argsTmpsOfss := theMeth _argsAndTempsOffsets .
anOffset <= nArgs ifTrue:[ | anOfs |
  anOfs := argsTmpsOfss at: anOffset  .
  arStack at:(fpIdx + (anOfs bitShift:-8) ) put: aValue .
  ^ self
].
absIp := (self _ipOffsetFromFrameDescr: frDesc) abs .
absIp <= OC_GSNMETHOD_FIRST_INSTR_OFFSET ifTrue:[
  ArgumentError signal:'method temps not allocated yet' .
  ^ nil
].
lastTmpIdx := argsTmpsOfss size - theMeth _numCopyingBlockArgs .
anOffset <= lastTmpIdx ifTrue:[ | aOfs |
  aOfs := argsTmpsOfss at: anOffset  .
  aOfs < 0 ifTrue:[
    " temp on evaluation stack"
    arStack at: (fpIdx + (aOfs bitShift:-8) ) put: aValue .
    ^ self
  ] ifFalse:[
    aOfs > 0 ifTrue:[ | lev ofs aVc |
      " temp allocated in a VariableContext,
         allow for theVc==nil to handle corrupted stacks during development. "
      ofs := aOfs bitShift: -8 .
      lev := aOfs bitAnd: 16rFF .
      aVc := theVc .
      [ lev > 0 ] whileTrue:[
	aVc := aVc ifNotNil:[ aVc parent ] .
	lev := lev - 1 .
      ].
      aVc ifNotNil:[
         aVc _primitiveAt: (ofs + 1) put: aValue . ^ self
      ] ifNil:[ ArgumentError signal:'theVc not found'. ^ nil ].
    ] ifFalse:[
      ArgumentError signal:'internal error, invalid offset for VC access'.
      ^ nil
    ].
  ].
].
anOffset <= argsTmpsOfss size ifTrue:[ "handle copying Block args"
  | aOfs rcvr |
  rcvr := arStack at:(fpIdx + FP_lastArg_OFS + nArgs ) .
  aOfs := argsTmpsOfss at: anOffset .
  rcvr at: aOfs + 1 put: aValue .
  ^ self
].
ArgumentError signal:'method temp or arg not found' .
^ nil
]

{ #category : 'Debugging Support' }
GsProcess >> _frameAt: aLevel tempNamed: aString [

"For the method activation at level aLevel, returns an Array of the form
      { offset .  (self _frameContentsAt: aLevel) at: offset . self _frameContentsAt: aLevel }
 for the receiver , method arg or temp named aString . 
 is the 1-based offset within framecontents of the method argument or temporary with name aString.
 Returns nil if no temp exists at aLevel with name aString.
 Generates an error if aLevel is out of range."

 | frameContents tempNames dp ofs |
 (aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
   aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
 ].
 frameContents := self _frameContentsAt: aLevel .
 tempNames := frameContents at: 9 "virtual machine constant" .
     "use compareTo:collator: for case-sensitive isEquivalent:"
 (aString compareTo: 'receiver' collator: nil) == 0 ifTrue:[ 
   ^ { 0 . frameContents at: 10 . frameContents }  
].
 1 to: tempNames size do:[:n | 
   ((tempNames at: n) compareTo: aString collator: nil) == 0 ifTrue:[ ofs ifNil:[ ofs := n ]].
 ].
 ofs ifNotNil:[ ^ { ofs . frameContents at: (10 + ofs) . frameContents} ].
 ^ nil
]

{ #category : 'Debugging Support' }
GsProcess >> _frameAt: aLevel tempNamed: aString put: aValue [
  | ary idx kind ofs oldVal varInfo |
  ary := self _frameAt: aLevel tempNamed: aString .
  ary ifNil:[ Error signal:'temp not found' ].
  (ofs := ary at: 1) == 0 ifTrue:[ ^ ArgumentError signal:'cannot alter receiver' ].
  varInfo := ((ary at: 3"result of _frameContents:") at: 3"varDetails") at: ofs .
  (idx := varInfo at: 1) == (ofs + 1) ifFalse:[ ^Error signal:'inconsistent var offset'].
  oldVal := ary at: 2 . 
  oldVal _isExecBlock ifTrue:[ ^ ArgumentError signal:'previous value is an ExecBlock'].
  oldVal class == VariableContext ifTrue:[ ^ ArgumentError signal:'previous value is a VariableContext'].
  kind := varInfo at: 2 .
  kind == #calleeRcvr ifTrue:[  ^ ArgumentError signal:'cannot alter callee receiver'].
  kind == #calleeArg ifTrue:[  ^ ArgumentError signal:'cannot alter callee arg'].
  (kind == #arg or:[ kind == #temp or:[ kind == #vcTemp]]) ifTrue:[ | obj k |
     obj := varInfo at: 3 . "arStack or a VC"
     obj ifNil:[ Error signal:'arStack or VC is nil' ].
     k := varInfo at: 4 .  "offset into arStack or a VC"
     obj at: k put: aValue 
  ] ifFalse:[
    ^ Error signal:'unknown var kind ', kind asString .
  ]
]

{ #category : 'Formatting' }
GsProcess >> _frameContentsAt: aLevel [

"Private.  Returns an Array describing the specified level in the receiver.
 aLevel == 1 is top of stack.  If aLevel is less than 1 or greater than
 stackDepth, returns nil.

 The result Array contains:
 offset item
 -----  -----
   1    gsMethod  (a GsNMethod)
   2    ipOffset    (absolute instruction offset in portable code
                     negative means a stack breakpoint is present)
   3    anArray ,   varDetails for use by   _frameAt:tempNamed:put:
   4    varContext
   5    saveProtectedMode (always nil in v3.0)
   6    markerOrException (always nil in v3.0)
   7    homeMethod if gsMethod is for a block, otherwise nil .
   8    self   (possibly nil in a ComplexBlock)
   9    argAndTempNames   (an Array of Symbols or Strings)
  10    receiver
  11... arguments and temp values, if any

 Notes to GemStone implementors:

 If result of this method is changed, you must change tpaux.c in the
 topaz C sources, other methods in this class, and the code for primitive 195.

 Note that if execution stops at a breakpoint at the entry to a method,
 the method temporaries may not be allocated yet and so some or all of
 the method temporaries may be missing from the result."

| absIp argsTmpsOfss calleeCodePtr calleeMeth calleeNArgs calleeRcvrIdx codePtr fpIdx 
  frDesc ofsIdx result theIp theMeth theVc varDetails vcOfs |

(aLevel < 1 or:[ aLevel > self localStackDepth ]) ifTrue:[ ^ nil ].
frDesc := self _frameDescrAt: aLevel .
frDesc ifNil:[ ^ nil ].
fpIdx := frDesc at: 2 .
result := Array new: 10 "virtual machine constant" .
arStack ifNil:[ ^ nil ].
codePtr := arStack atOrNil:( fpIdx + FP_codePtr_OFS ).
theMeth := self _fetchMethod: codePtr  .

result at: 1 put:  theMeth .
theIp := (self _ipOffsetFromFrameDescr: frDesc) .
result at: 2 put: theIp .
absIp := theIp abs .
varDetails := { } .  "each element of varDetails is an Array of form
                       { display_offset, kindSymbol, holder . offset_in_holder } 
                     holder is either arStack or a VC "
result at: 3 put: varDetails.   
vcOfs := fpIdx + FP_VC_OFS .
arStack ifNotNil:[ theVc := arStack atOrNil: vcOfs ].

(Reflection classOf: theVc) == VariableContext ifTrue:[
  result at: 4 put: theVc
 ] ifFalse: [
  theVc := nil "IP at start of method, VC not allocated yet"
 ].
"result at: 5 , leave as nil,  saveProtectedMode not available"
theMeth ifNotNil:[
  | homeMeth nArgs receiver theSelf argAndTempNames lastTmpIdx |
  homeMeth := theMeth homeMethod .
  homeMeth ~~ theMeth ifTrue:[ result at: 7 put: homeMeth ].

  nArgs := theMeth numArgs .
  receiver := arStack atOrNil:(fpIdx + FP_lastArg_OFS + nArgs ) .
  result at: 10 put: receiver .
  (receiver _isExecBlock and:[theMeth inClass ~~ ExecBlock]) ifTrue:[
    theSelf := receiver selfValue  .
  ] ifFalse:[
    theSelf := receiver
  ].
  result at: 8 "virtual machine constant" put: theSelf.

  argAndTempNames := theMeth  argsAndTemps .
  argAndTempNames size == 0 ifTrue:[ argAndTempNames := { } ] .
  result at: 9 put: argAndTempNames .

  argsTmpsOfss := theMeth _argsAndTempsOffsets .
  lastTmpIdx := argsTmpsOfss size - theMeth _numCopyingBlockArgs .
  ofsIdx := 1 .
  1 to: nArgs do:[:k | | anOfs xofs |
    anOfs := argsTmpsOfss at: ofsIdx  .
    xofs := anOfs bitShift:-8 .
    result addLast:( arStack ifNotNil:[:ars | ars at:(fpIdx + xofs )] ) .
    ofsIdx := ofsIdx + 1 .
    varDetails add: { ofsIdx . #arg . arStack . fpIdx + xofs  }.
  ].
  absIp <= OC_GSNMETHOD_FIRST_INSTR_OFFSET ifTrue:[
    "temps not allocated yet"
    argAndTempNames size > nArgs ifTrue:[
      argAndTempNames size: nArgs
    ].
  ] ifFalse:[ | aOfs tmpVal lastStkTmpIdx calleeLastArgIdx calleeFpIdx tIdx |
    [ ofsIdx <= lastTmpIdx ] whileTrue:[
      aOfs := argsTmpsOfss at: ofsIdx  .
      tmpVal := nil .
      aOfs < 0 ifTrue:[ | idx |
        " temp on evaluation stack"
        idx := fpIdx + (aOfs bitShift:-8) .
        lastStkTmpIdx ifNil:[ lastStkTmpIdx := idx ]
           ifNotNil:[ lastStkTmpIdx := lastStkTmpIdx min: idx ].
        tmpVal := arStack ifNotNil:[:ars | ars atOrNil: idx ].
        varDetails add: { ofsIdx . #temp . arStack . idx }.
      ] ifFalse:[
        aOfs > 0 ifTrue:[  | ofs lev aVc |
          ofs := aOfs bitShift: -8 .
          lev := aOfs bitAnd: 16rFF .
          aVc := theVc .
          [ lev > 0 ] whileTrue:[
            aVc := aVc ifNotNil:[ aVc parent ] .
            lev := lev - 1 .
          ].
          " temp allocated in a VariableContext,
            allow for theVc==nil to handle corrupted stacks during development. "
          tmpVal := aVc ifNotNil:[ aVc _primitiveAt: (ofs + 1)] ifNil:[ nil] .
          varDetails add: { ofsIdx . #vcTemp . aVc . ofs + 1  } .
        ] ifFalse:[
          self error:'illegal offset'.
        ].
      ].
      result addLast: tmpVal .
      ofsIdx := ofsIdx + 1 .
    ].
    calleeFpIdx := frDesc at: 3 .
    calleeRcvrIdx := 1 . "at frame 1, there is a savedIP in (arStack at:1)"
    calleeFpIdx + FP_codePtr_OFS > 1 ifTrue:[
      calleeCodePtr := arStack atOrNil:( calleeFpIdx + FP_codePtr_OFS ).
      calleeMeth := self _fetchMethod: calleeCodePtr  .
      calleeNArgs := calleeMeth numArgs .
      calleeLastArgIdx := calleeFpIdx + FP_lastArg_OFS .
      calleeRcvrIdx := calleeLastArgIdx + calleeNArgs .
    ].
		tIdx := 1 .
     "GsFile gciLogServer:' ofsIdx  ', ofsIdx asString,' lastTmpIdx ', lastTmpIdx asString.
      GsFile gciLogServer:' lastStkTmpIdx ', lastStkTmpIdx asString ."
		lastStkTmpIdx ifNil:[ lastStkTmpIdx := fpIdx + FP_codePtr_OFS ].
     "GsFile gciLogServer:' lastStkTmpIdx ', lastStkTmpIdx asString, 
       ' calleeRcvrIdx ', calleeRcvrIdx asString."
		(lastStkTmpIdx - 1) to: calleeRcvrIdx + 1  by: -1 do:[:idx |
			tmpVal := arStack ifNotNil:[:ars | ars atOrNil: idx ].
			result addLast: tmpVal .
			argAndTempNames add: '.t' , (tIdx asString ) .
			varDetails add: { ofsIdx + tIdx . #temp . arStack . idx } .
			tIdx := tIdx + 1 .
		].
    "Don't include any eval temps that are receiver or arg of the callee"
  ].
 ].
 ^ result
]

{ #category : 'Accessing' }
GsProcess >> _frameDescrAt: aLevel [

"Private.
 Returns nil if aLevel is less than zero or greater than stackDepth,
 otherwise returns an Array  { aLevel . fpIdx . calleeFpIdx } ,
 where fpIdx is one-based frameOffset at aLevel in the receiver,
 and  where aLevel == 1 is top of stack.
 If aLevel == 1, calleeFpIdx will be zero.

 (self.arStack at: fpIdx) is the savedFP for aLevel.
 if aLevel > 1, (self.arStack at: calleeIdx) is the savedFP for aLevel - 1.
 "
| fpIdx calleeFpIdx cache dp |

(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  ^ nil.
].
(cache := dbgFpsCache) ifNil:[  | ar_size idx ar_stk depth |
  (ar_stk := arStack) ifNil:[ ^ nil ].
  ar_size := ar_stk size .
  fpIdx := ar_size + (self _toWordOfs:topFpOffset) + 1.
  depth := dp .
  cache := Array new: depth .
  cache at: 1 put: fpIdx .
  idx := 1 .
  [ idx < depth ] whileTrue:[ | callerFpOfs |
    idx := idx + 1 .
    callerFpOfs := ar_stk at: fpIdx"+ FP_savedFP_OFS==0" .
    calleeFpIdx := fpIdx .
    fpIdx := ar_size + (self _toWordOfs:callerFpOfs) + 1.
    cache at: idx put: fpIdx .
  ].
  dbgFpsCache := cache .
].
fpIdx := cache at: aLevel .
calleeFpIdx := aLevel > 1 ifTrue:[ cache at: aLevel - 1 ] ifFalse:[ 0 ] .
^ { aLevel . fpIdx . calleeFpIdx }

]

{ #category : 'Accessing' }
GsProcess >> _frameOffsetAt: aLevel [

"Private.
 Returns nil if aLevel is less than zero or greater than stackDepth,
 otherwise returns  idx ,
 where idx is one-based frameOffset at aLevel in the receiver,
 and  where aLevel == 1 is top of stack.

 In self.arStack at idx is the savedFP for aLevel."

| arr |
arr := self _frameDescrAt: aLevel .
arr ifNil:[ ^ nil ].
^ arr at: 2

]

{ #category : 'Private' }
GsProcess >> _getEnsureBlocks [

"For a GsProcess returns an Array of pairs,
     FP offset,  block that was arg to an ensure:
 for the ensure blocks currently on the stack of the receiver"

<primitive: 826>  "primitive fails if receiver not in stack memory"
| ar_stk pairs fpOfs ar_size |
pairs := { }  .
(ar_stk := arStack) ifNotNil:[  "handle a suspended process not in stack memory"
  fpOfs := topFpOffset .
  ar_size := ar_stk size .
  [ fpOfs < 0 ] whileTrue:[ | fpIdx arOfs anFP |
    fpIdx := ar_size  + (anFP := self _toWordOfs: fpOfs) + 1 .
    arOfs := fpIdx + FP_markerNil_OFS .
    arOfs >= 1 ifTrue:[ | marker aBlk |
      marker := ar_stk at: arOfs .
      marker == OOP_ENSURE_Mark_NIL ifTrue:[
        aBlk := ar_stk at:(fpIdx + FP_lastArg_OFS).
        aBlk _isExecBlock ifTrue:[
          pairs add: anFP ; add: aBlk .
        ] ifFalse:[
          UncontinuableError signal:'corrupt stack, expected an ExecBlock'.
        ].
      ].
    ].
    fpOfs := ar_stk at: fpIdx"+ FP_savedFP_OFS==0" . "get callerFpOfs"
  ].
].
^ pairs

]

{ #category : 'Process Groups' }
GsProcess >> _group [
  "Return the Smalltalk process group (a SmallInteger) the receiver is in,
   assigning to a new group if needed ."
  | g |
  (g := group) ifNil:[ ^ self _newGroup].
  ^ g

]

{ #category : 'Process Groups' }
GsProcess >> _groupOrNil [
  "Return the process group the receiver is in,
   or return nil if receiver has not been assigned to a group."

  ^ group

]

{ #category : 'Process Groups' }
GsProcess >> _groupSameAs: anInteger [
  "Return true if the receiver's group is the same as anInteger.
   Otherwise return false."

  "Note we don't use 'self _group' because that would initialize
   group which we don't need done for a same as test."

  ^group = anInteger

]

{ #category : 'Debugging Support' }
GsProcess >> _gsiDebuggerDetailedReportAt: aLevel [

"If aLevel is less than 1 or greater than stackDepth of the receiver,
 returns nil.  Otherwise, returns an Array containing:

 offset item
 ------ ----
   1    gsMethod  (a GsNMethod)
   2    receiver
   3    self
   4    selector  (a Symbol)
   5    quickStepPoint (SmallInteger index into sourceOffsets, or nil)
   6    sourceOffsets  (Array of SmallIntegers)
   7    argAndTempNames  (Array of Symbols)
   8    argAndTempValues (Array) (may be smaller than argAndTempNames if
   execution halted at entry to a method, may be larger than
         argAndTempNames if compiler has allocated additional
   unnamed stack temporaries. For aLevel == 1, Last element is TOS )
   9    sourceString
  10    ipOffset   in portable code (SmallInteger)
  11    markerOrException  (SmallInteger in a reenter marker,
         or an Exception, or nil )

 The quickStepPoint is the step point in the method, if the method has
 50 or less step points.  Otherwise, the quickStepPoint will be nil
 and the expression

 gsMethod _stepPointForIp: ipOffset level: aLevel
    useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel])

 may be used to obtain search for and obtain the step point."
| frameContents |

(frameContents := self _frameContentsAt: aLevel) ifNil:[ ^ nil ].
^ self _gsiDetailedReportWithFrameContents: frameContents
             forLevel: aLevel

]

{ #category : 'Debugging Support' }
GsProcess >> _gsiDetailedReportWithFrameContents: frameContents forLevel: aLevel [

"Private."
"See _gsiDebuggerDetailedReportAt: for documentation."

| result receiver gsMethod argAndTempNames argAndTempValues aself |

result := Array new: 11 .

result
  at: 1 put: (gsMethod := frameContents at: 1  "virtual machine constant") ; "gsMethod"
  at: 2 put: (receiver := frameContents at: 10 "virtual machine constant" ); "receiver"
  at: 3 put: (aself := frameContents at: 8 "virtual machine constant" ); "self"
  at: 4 put: (gsMethod == nil
    ifTrue: [#'<UserAction>']
    ifFalse: [ gsMethod _selector]); "selector"
  at: 5 put: (gsMethod == nil
    ifTrue: [ 0 ]
    ifFalse: [ gsMethod
      _stepPointForIp: (frameContents at: 2 "virtual machine constant") abs
      level: aLevel useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel])
    ]) ;
  at: 6 put: (gsMethod == nil
    ifTrue: [ #() ]
    ifFalse: [ gsMethod _blockSourceOffsets ]). "source offsets"

argAndTempNames := frameContents at: 9 "virtual machine constant" .

argAndTempValues := (frameContents size >= 11 "virtual machine constant")
    ifTrue:[ frameContents copyFrom: 11 to: frameContents size]
    ifFalse:[ { }  ].

result  at: 7 put: argAndTempNames ;
        at: 8 put: argAndTempValues ;
        at: 9 put: (gsMethod == nil
          ifTrue: [ 'source not available' ]
          ifFalse: [ gsMethod sourceString  ]);
        at: 10 put: (frameContents at: 2 "virtual machine constant") abs ;  "ipOffset"
        at: 11 put: (frameContents at: 6 "virtual machine constant"). "markerOrException"

^ result

]

{ #category : 'Debugging Support' }
GsProcess >> _gsiStackReportFromLevel: startLevel toLevel: stopLevel [

"Returns an Array describing the receiver.  For each stack level in the
 receiver, the result contains 2 elements,  a String describing that
 level, and an Array containing the result of the _gsiDebuggerDetailedReportAt:
 method for that level.

 Level 1 is the top of the stack.  If startLevel is out of range, or if
 stopLevel is less than startLevel, returns nil.  If stopLevel is beyond the end
 of the stack, returns information up to the end of the stack."

|result actualStop frameContents dp |

stopLevel < startLevel ifTrue:[ ^ nil ].
startLevel > (dp := self localStackDepth) ifTrue:[ ^ nil ].

actualStop := (stopLevel > dp )
  ifTrue: [ dp ]
  ifFalse: [ stopLevel ].

result := { } .
startLevel to: actualStop do:[:j|
  frameContents := self _frameContentsAt: j .
  frameContents
    ifNil:[ result addLast: nil; addLast:nil ]
    ifNotNil:[ | frameStr |
      frameStr := String new .
      frameStr addAll: (self _reportWithFrameContents: frameContents level: j) .
      result addLast: frameStr ;
             addLast:(self _gsiDetailedReportWithFrameContents: frameContents
                            forLevel: j ) .
      ].
  ].
^ result

]

{ #category : 'Private' }
GsProcess >> _init: aBlock args: aArgs env: environmentId stackSerial: serialNum forked: forkedBit [
  "returns receiver.
   caller must _scheduleNewProcess or _suspendNewProcess "
  | currProc |
  stackId := -1 .
  "scheduler will set priority_use to match active process"
  modeInfo := ModeInfo_debugEnabled bitOr: forkedBit .
  block := aBlock.
  blockResult := _remoteNil .
  args := aArgs.
  msgEnvironmentId := environmentId .
  serialNum _isSmallInteger ifFalse:[
      InternalError new details:'GsProcess invalid stackSerialNum' ; signalNotTrappable.
  ].
  stackSerialNum := serialNum .
  environmentId == 0 ifTrue:[ | asz bNargs |
    (asz := args size) ~~ (bNargs := block argumentCount) ifTrue: [
      "Generate an error because we don't have the correct number of args."
      OutOfRange new _number: 2373; expectedNumArgs: bNargs actual: asz ;
     object: block ; details:'wrong number of arguments for invoking a block' ;
          signal.
      ^ nil
    ].
  ].
  debugActionBlock := (currProc := GsProcess _current) debugActionBlock . "inherit from parent"
  breakpointLevel := currProc breakpointLevel .
  startSeconds := System _timeGmtFloat - 978307200.0 . "aSmallDouble, secondsSince2001"
  parentProcess := self _current .
  "self _setClientData . debugging only"

]

{ #category : 'Accessing' }
GsProcess >> _ipOffsetAt: aLevel [

"return the portable code IP offset for the specified frame"
^ self _ipOffsetFromFrameDescr: (self _frameDescrAt: aLevel)


]

{ #category : 'Accessing' }
GsProcess >> _ipOffsetFromFrameDescr: frDescr [

"return the portable code IP offset for frame described by
 frDescr.  frDescr must be non-nil result from _frameDescrAt:

 if receiver is a stack from native code, result maybe approximate

If frame has a stack breakpoint, answer will be negative.
"

| ipOff aLevel notTopFrame |
aLevel := frDescr at: 1.
(notTopFrame := aLevel >  1) ifTrue:[ | calleeFpIdx |
  calleeFpIdx := frDescr at: 3 .
  ipOff := arStack at:( calleeFpIdx + FP_rtnAddr_OFS ) .
] ifFalse:[
  ipOff := arStack at: 1 "savedIP at TOS" .
].
ipOff == nil ifTrue:[
  "can have nil for reenter markers."
  ipOff := 0
].
self _nativeStack ifTrue:[ | fpIdx meth anIp |
  ipOff < 0 ifTrue: [
    "due to biasing of return addresses for native code, can have negative
     offset for start of method, so treat as zero for stack display."
    ipOff := 0
  ].
  fpIdx := frDescr at: 2 .
  meth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS )).
  " arg to asReturn: is false here because callers (or topaz) are using one of
      GsProcess>>_StepPointAt:
      GsNMethod>>_stepPointForIp:level:useNext:
    to convert the IP to a step point
  "
  anIp := meth _nativeIpOffsetToPortable: ipOff asReturn: false .
  anIp <= 0 ifTrue:[
    anIp < -1 ifTrue:[
      anIp := anIp negated.  "approximate result"
    ] ifFalse:[
      self error:'ipOff=', anIp asString, ' out of range' .
    ].
  ].
  ipOff := anIp .
].
^ ipOff .
]

{ #category : 'Private' }
GsProcess >> _isSuspended [
  ^ self _scheduler _isSuspended: self 
]

{ #category : 'Private' }
GsProcess >> _isTerminated [

  "returns true if the receiver has been terminated."

  ^ (modeInfo bitAnd: ModeInfo_terminated ) ~~ 0

]

{ #category : 'Private' }
GsProcess >> _setTerminated [

  self _setModeinfoBit: ModeInfo_terminated value: 1  

]

{ #category : 'Process Groups' }
GsProcess >> _joinGroup: aGroup [
  "Have the receiver join the specified processor group. Return self.
   For Smalltalk,  aGroup is an Integer,
   For Ruby aGroup is a RubyThreadGroup .
  "
  group := aGroup .

]

{ #category : 'Debugging Support' }
GsProcess >> _localStepPointAt: aLevel [

"Used by topaz debugger,
 returns an array { stepPoint . aGsNMethod }
 or nil if no step point available "

| fpIdx theMeth  ipOffset stepPt frDesc dp |
(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
  ].
(frDesc := self _frameDescrAt: aLevel) ifNil:[ ^ nil ].
fpIdx := frDesc at: 2 .
theMeth := self _fetchMethod:( arStack at:( fpIdx + FP_codePtr_OFS)) .
theMeth ifNil:[ ^  nil  ].
ipOffset := self _ipOffsetFromFrameDescr: frDesc .
stepPt := theMeth _stepPointForIp: ipOffset level: aLevel
        useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel]) .
stepPt ifNil:[ ^ nil ].
^ { stepPt . theMeth }

]

{ #category : 'Debugging Support' }
GsProcess >> _methodInFrameContents: aFrameContentsArray [

^ aFrameContentsArray at: 1 "virtual machine constant"

]

{ #category : 'Private' }
GsProcess >> _stackKind [

  "returns 0 if the receiver represents execution in interpreted code,
          1 receiver is in mixed mode (not used).
          2 receiver is pure native code

   Only usable on instances created by a VM stack save. "

  ^ ((modeInfo bitAnd: ModeInfo_stackKind_mask ) bitShift: 0 - ModeInfo_stackKind_shift) 
]

{ #category : 'Private' }
GsProcess >> _nativeStack [
  "Returns true if receiver is running native code"
  ^ self _stackKind > 0 
]

{ #category : 'Process Groups' }
GsProcess >> _newGroup [
  "Add the receiver to a new Smalltalk process group. Return the group value."
  | g |
  g := self _scheduler _newGroup.
  group := g .
  ^ g

]

{ #category : 'Private' }
GsProcess >> _nextBytecodeIsSendAtLevel: aLevel [
  " returns a Boolean "
  | frameContents method ip |
  frameContents := self _frameContentsAt: aLevel.
  method := self _methodInFrameContents: frameContents.
  ip := frameContents at: 2.
  ^ ((method _opcodeKindAt: ip) bitAnd: 4) ~= 0 
]

{ #category : 'Private' }
GsProcess >> _onQueue: anObj [
  " for use by ProcessorScheduler only"
  onQueue := anObj

]

{ #category : 'Private' }
GsProcess >> _primContinue: argUnused [

<primitive: 2012>
"no image code here, compiler emits Bc_RETURN_TOS here"

]

{ #category : 'Private' }
GsProcess >> _primStep: breakpointsToIgnore from: argUnused with: saveArray mode: raiseExcept [

<primitive: 2011>
^ self  "normal execution path"

]

{ #category : 'Updating' }
GsProcess >> _raisePriority [
  "Raises priority of receiver to  ((priority +20) max: 40) .
   Returns the previous priority of receiver.  "
  | old p won |
  old := self priority .
  old <= 20 ifTrue:[   "inline highestPriority logic"
    p := old + 20
  ] ifFalse:[
    p := 40
  ].
  p ~~ old ifTrue:[
    self _setPriority: p .
    self _scheduler _changePriority: self from: p .
    (won := waitingOn) ifNotNil:[
      won _changePriority: self from: old .
    ]
 ].
 ^ old

]

{ #category : 'Private' }
GsProcess >> _reapSignal: signalSource [
  "Make the receiver ready to run."
  "caller enters critical region"

  waitingOn := nil .
  self _isSuspended ifTrue:[
    self _scheduler _suspendProcess: self . 
  ] ifFalse:[
    self _scheduler _scheduleProcess: self.
  ].
]

{ #category : 'Debugging Support' }
GsProcess >> _receiverInFrameContents: aFrameContentsArray [

^ aFrameContentsArray at: 10 "virtual machine constant"

]

{ #category : 'Private' }
GsProcess >> _removeEnsureAtFP: anFP [

 "Dereference the ensure block in specified frame,
  so it won't be executed more than once.
  Frame changed  from  ensure:[] to  ensure: nil ."

 <primitive: 958>
 "handle receiver not in stack memory "
 | ar_stk ar_size fpIdx arOfs |
 anFP _isSmallInteger ifTrue:[
   ar_stk := arStack .
   ar_size := ar_stk size .
   fpIdx := ar_size  + anFP + 1.
   arOfs := fpIdx + FP_markerNil_OFS .
   arOfs >= 1 ifTrue:[ | marker |
     marker := ar_stk atOrNil: arOfs .
     marker == OOP_ENSURE_Mark_NIL ifTrue:[
      ar_stk at:(fpIdx + FP_lastArg_OFS) put: nil .
      ^ self
     ]
   ]
 ].
 self _primitiveFailed: #_removeEnsureAtFP: args: { anFP }

]

{ #category : 'Private' }
GsProcess >> _removeJoiner: aDelay [

| js ofs |
(js := joiners ) ifNotNil:[
  ofs := (js := joiners ) indexOfIdentical: aDelay .
  ofs ~~ 0 ifTrue:[ js at: ofs put: nil ].
].

]

{ #category : 'Debugging Support' }
GsProcess >> _reportAt: aLevel [

"Return a one line String describing the specified level in the receiver.
 aLevel == 1 means top of stack with respect

 If aLevel is out of range,  nil is returned. "

| fc isCurr |
(isCurr :=  self == GsProcess _currentOrNil) ifTrue:[
   (fc := GsProcess _frameContentsAt: aLevel ) ifNil:[ ^ nil ].
   fc at: 8 put: (GsProcess _selfAt: aLevel) . "get self"
] ifFalse:[
   (fc := self _frameContentsAt: aLevel) ifNil:[ ^ nil ].
].
^ self _reportWithFrameContents: fc level: aLevel  .

]

{ #category : 'Debugging Support' }
GsProcess >> _reportOfSize: aSize [

"Returns an Array describing the receiver up to aSize entries in length.  Each
 element in the result is a String produced by GsProcess | _reportAt:."

| result dp |
result := { }  .
dp := self == GsProcess _currentOrNil ifTrue:[ System stackDepth ]
                                ifFalse:[ self stackDepth ].
1 to: (dp min: aSize) do:[:j |
   (self _reportAt: j) ifNotNil:[:r | result add: r ]
].
^ result

]

{ #category : 'Debugging Support' }
GsProcess >> _reportString [

"Returns a String with each Activation object in the receiver, up to 2000
 levels, described on separate lines."

| s arr |
s := String new.
s add: self class name.
s add: '(oop=' ; add: self asOop printString ; add: ', ' .

s add: 'status=' ; add: (self _statusString) ; add: ', '.

clientData ifNotNil:[
  s add: 'clientData=' ; add: (clientData printString) ; add: ', '.
].

s add: 'priority=' ; add: (self priority printString) ; add: ', '.

group ifNotNil:[
  s add: 'group=' ; add: (group printString) ; add: ', '.
].

block ifNotNil:[
  s add: 'block=' ; add: (block printString) ; add: ', '.
  s add: 'args=' ; add: (args printString) ; add: ', '.
].
s lf .

arr := self _reportOfSize: 2000.
arr size > 0 ifTrue:[
  1 to: arr size do:[:j | s add: '   ' ; add: (arr at: j) ; lf ].
  s add: $) ; lf .
] ifFalse:[
  s add: $)
].
^ s

]

{ #category : 'Debugging Support' }
GsProcess >> _reportWithFrameContents: frameContents level: aLevel [
"Return a one line String describing the argument.
 Example result:
   SmallInteger (Object) >> doesNotUnderstand:  @9 line 10 "

| gsMethod receiver aself result ip stepPt |

gsMethod := self _methodInFrameContents: frameContents .
receiver := self _receiverInFrameContents: frameContents .

aself := self _selfInFrameContents: frameContents .

gsMethod ifNil:[ ^ self == GsProcess _currentOrNil ifTrue:['GCI'] ifFalse:['User Action' ]].

result := String new .
result add: (gsMethod _descrForStackPadTo: 0 rcvr: receiver) .
ip := (frameContents at: 2) abs .
stepPt := gsMethod _stepPointForIp: ip level: aLevel
        useNext: (self _nativeStack or:[ self _calleeIsAsync: aLevel]) .
result add: ' @' ; add: stepPt asString ;
       add: ' line ' ;
       add: (gsMethod _lineNumberForStep:  stepPt) asString .
^ result

]

{ #category : 'Private' }
GsProcess >> _scheduler [

"Returns   ProcessorScheduler scheduler "
<primitive: 457>

self _primitiveFailed: #_scheduler

]

{ #category : 'Debugging Support' }
GsProcess >> _selfInFrameContents: aFrameContentsArray [

^ aFrameContentsArray at: 8 "virtual machine constant"

]

{ #category : 'Private Debugging Support' }
GsProcess >> _setBreaksForStepInto [

"Set breakpoints so that a subsequent IntContinue on the receiver will
 execute a step-into.  step-into across a fork of a new GsProcess
 is not supported.  (The step breakpoint in the forked block will
 be ignored since it not in the current GsProcess).

 For use only from within implementation of GciStep.

 Return a 3 element Array,
     { <the GsNMethod in which single-step breaks were set> .
       <the level at which stack break was set, or zero> .
       <a SmallInteger, number of breakpoints to ignore> }

Algorithm
Step Into (is always from TOS == level 1)
  1. set all step breakpoints in TOS.codeOop
  2. if next instruction to execute is BC_CALL_PRIMITIVE,
      set all step breakpoints in level 2 and set stack breakpoint in level 3
    else
      set stack breakpoint at level 2
"

| tosFrame aMethod stackBreakLevel result breakpointsToIgnore ipOfs |

self convertToPortableStack .

result := Array new: 3 .
tosFrame := self _frameContentsAt: 1 .
aMethod := tosFrame at: 1 "virtual machine constant" .
ipOfs := tosFrame at: 2 "virtual machine constant" "ipOffset" .
(aMethod _opcodeAtIsCallPrimitive: ipOfs) ifTrue:[
  "next instruction to be executed is a call-primitive"
  aMethod := self localMethodAt: 2 .
  aMethod _setAllStepBreaks: true frame: nil process: self .
  result at: 1 put: aMethod.
  breakpointsToIgnore := 0.
  stackBreakLevel := 3  .
] ifFalse:[
  "next instruction is not a primitive call"
  aMethod _setAllStepBreaks: false frame: nil process: self .
  result at: 1 put: aMethod .
  breakpointsToIgnore := (aMethod _opcodeAtIsSend: ipOfs) ifTrue:[ 0 ] ifFalse:[ 1]. 
  stackBreakLevel := 2 .
].
  " use <  in following compare to not set stack break in _gsReturnToC"
stackBreakLevel < self stackDepth
  ifTrue:[ self _setStackBreakAt: stackBreakLevel ]
  ifFalse:[ stackBreakLevel := 0 "no stack breaks to set" ].
result at: 2 put: stackBreakLevel .
result at: 3 put: breakpointsToIgnore.
^ result

]

{ #category : 'Private Debugging Support' }
GsProcess >> _setBreaksForStepOverFromLevel: flags [
	"Set breakpoints so that a subsequent IntContinue on the receiver will
 execute a step-over.
 For use only from within implementation of GciStep.
 Return an Array, { <the GsNMethod in which single-step breaks were set> .
        <the level at which stack break was set, or zero> .
                    <number of breakpoints to ignore> }."

	| aFrame aMethod stackBreakLevel result breakpointsToIgnore aLevel thruBool fpOffset trc doTrace |
	doTrace ifNotNil: [ trc := String new ].
	self convertToPortableStack.
	result := Array new: 3.
	aLevel := flags bitAnd: 16rFFFFFFFF.
	thruBool := (flags bitAnd: 16r100000000) ~~ 0.
	aFrame := self _frameContentsAt: aLevel.
	aMethod := aFrame at: 1.	"virtual machine constant"
	fpOffset := aLevel == 1
		ifTrue: [ self _toWordOfs: topFpOffset ]
		ifFalse: [ ((self _frameDescrAt: aLevel) at: 2) - (arStack size + 1) ].
	aLevel >= 2
		ifTrue: [ 
			stackBreakLevel := aLevel + 1.
			breakpointsToIgnore := 0.
			aMethod
				_setStepOverBreaks: false
				frame: fpOffset
				process: self
				stepThrough: thruBool.
			doTrace ifNotNil: [ trc add: $A ] ]
		ifFalse: [ 
			| hmm "aLevel is 1" |
			"set breaks in TOS method, excluding first step point"
			aMethod
				_setStepOverBreaks: false
				frame: fpOffset
				process: self
				stepThrough: thruBool.
			result at: 1 put: aMethod.
			breakpointsToIgnore := 0.	"no special logic after lastBreakpt instVar added"
			(thruBool and: [ (hmm := aMethod homeMethod) ~~ aMethod ])
				ifTrue: [ 
					| depth j |
					depth := self localStackDepth.
					j := 2.
					stackBreakLevel := 0.
					[ j < depth ]
						whileTrue: [ 
							((self _frameContentsAt: j) at: 1) == hmm
								ifTrue: [ 
									stackBreakLevel := j.	"stack break for return to/across home method"
									j := depth.	"break out of whileTrue"
									doTrace ifNotNil: [ trc add: $B ] ].
							j := j + 1 ] ]
				ifFalse: [ 
					stackBreakLevel := aLevel + 1.
					doTrace ifNotNil: [ trc add: $C ] ] ].	" use <  in following compare to not set stack break in _gsReturnToC"
	doTrace
		ifNotNil: [ 
			GsFile
				gciLogServer:
					'dp ' , self localStackDepth asString , ' stackBreakLevel '
						, stackBreakLevel asString ].
	(stackBreakLevel > 0 and: [ stackBreakLevel < self localStackDepth ])
		ifTrue: [ 
			self _setStackBreakAt: stackBreakLevel.
			doTrace ifNotNil: [ trc add: $D ] ]
		ifFalse: [ 
			stackBreakLevel := 0.	"no stack breaks to set"
			doTrace ifNotNil: [ trc add: $E ] ].
	doTrace
		ifNotNil: [ 
			GsFile
				gciLogServer:
					'level ' , aLevel asString , ' aMethod ' , aMethod asOop asString , ' hmm '
						, aMethod homeMethod asOop asString , ' breakpointsToIgnore '
						, breakpointsToIgnore asString , ' stackBreakLevel '
						, stackBreakLevel asString , ' trc ' , trc ].
	result at: 3 put: breakpointsToIgnore.
	result at: 2 put: stackBreakLevel.
	^ result
]

{ #category : 'Private' }
GsProcess >> _setInterpretedStack [

  | mask val |
  mask := ModeInfo_stackKind_mask bitInvert .
  val :=  0 .
  modeInfo := (modeInfo bitAnd: mask) bitOr: val  .

]

{ #category : 'Private' }
GsProcess >> _setModeinfoBit: aMask value: aValue [

  "Sets/clears bit specified by aSmallInteger in  the modeInfo instVar.
   aMask must be a SmallInteger, one of
     ModeInfo_terminated - set terminated and terminationStarted bits ;
       if the process has an in-memory stack, unlocks it and clears its use count;
       stores nil in instVars arStack,clientData.
     ModeInfo_terminationStarted - sets specified bit
     ModeInfo_breakptsIgnore_mask - sets specified bit field.
     ModeInfo_debugEnable - sets specified bit field

   aValue must be a SmallInteger >= 0 , and not larger than the specified bit field.
  "
  <primitive: 708>
  aMask _validateClass: SmallInteger .
  aValue _validateClass: SmallInteger .
  self _primitiveFailed: #_setModeinfoBit:value: args: { aMask . aValue }

]

{ #category : 'Private' }
GsProcess >> _setPriority: anInt [
  "Update priority of the receiver, and of receiver's in-memory
   stack if the receiver  has one. 
   Public priority range is 1 .. 40 .
   Priority 41 is private, used when terminating a process at priority 40.
  "
  <primitive: 706>
  anInt _validateClass: SmallInteger .
  (anInt < 1 or:[ anInt > 41]) ifTrue:[
     anInt _error: #rtErrArgOutOfRange args:{ 1 . 40 } ].
  self _primitiveFailed: #_setPriority: args: { anInt }
]

{ #category : 'Private Debugging Support' }
GsProcess >> _setStackBreakAt: aLevel [

"Sets a stack breakpoint that is hit when a return from a
 block would return into or across the specified level on the stack.
 When a stack breakpoint is hit, the breakpoint is automatically cleared,
 the return from block is executed, and execution stops with a breakpoint
 error.  If both an unwind block (as installed by ensure:)
 and a stack breakpoint are present at the same level in the stack
 the stack breakpoint will be cleared and execution will stop with a
 breakpoint error before executing the unwind block."

| ipOffset dp |
self convertToPortableStack .

" disallow stack break at aLevel == self stackDepth
  to prevent setting set stack break in _gsReturnToC"

(aLevel < 1 or:[ aLevel >= (dp := self stackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp - 1 } .
  ].
(self __returnAddrIdxAt: aLevel) ifNotNil:[ :rtnAddrIdx |
  ipOffset := arStack at: rtnAddrIdx .
  arStack at: rtnAddrIdx put: ipOffset abs negated
].

]

{ #category : 'Private' }
GsProcess >> _signalAll [
  "Wake up the receiver"  "caller is in critical region"

  ^ self _reapSignal: nil .
]

{ #category : 'Private' }
GsProcess >> _signalJoiners [
| js sched wasInCrit |
js := joiners .
js ifNotNil:[
  joiners := nil .
  sched := self _scheduler.
  wasInCrit := sched _enterCritical .
  1 to: js size do:[:n | | aDelay |
    aDelay := js at: n .
	"Removed joiners are replaced with nil."
    aDelay ifNotNil:[aDelay signal].
  ].
  wasInCrit == 0 ifTrue:[ sched _exitCritical ].
].
]

{ #category : 'Private' }
GsProcess >> _signalTime [
  ^ signalTime

]

{ #category : 'Private' }
GsProcess >> _signalTime: time [

  signalTime := time.

]

{ #category : 'Debugging Support' }
GsProcess >> _stackReport [
  "For use with topaz"
  ^ self stackReportToLevel: 1000 withArgsAndTemps: false andMethods: true
		includeSource: false

]

{ #category : 'Debugging Support' }
GsProcess >> _stackReportIncludeSourceFor: gsMethod inclSrc: inclSrc [

  "inclSrc is a SmallInteger produced by
     stackReportToLevel:withArgsAndTemps:andMethods:includeSource:lineLimit:  .
   Returns true if source should be displayed."

  | cls sel |
  inclSrc <= 0 ifTrue:[ ^ false "none" ].
  inclSrc >= 3 ifTrue:[ ^ true  "all"  ].
  inclSrc == 1 ifTrue:[ ^  gsMethod inClass == nil  "anonymous methods"].

  "inclSrc == 2 "
  (cls := gsMethod inClass ) ifNil:[ ^ true "an anonymous method"].
  sel := gsMethod isMethodForBlock ifTrue:[ gsMethod homeMethod selector ]
                                ifFalse:[ gsMethod selector ].
  sel ifNil:[ ^ true "failsafe" ].
  (cls compiledMethodAt: sel environmentId: gsMethod environmentId otherwise: nil usePackages: true)
      ifNotNil:[ ^ false "exists in a method dict, don't report sources"].
  ^ true  "block from block execution, etc"

]

{ #category : 'Private' }
GsProcess >> _stackSerialNum [
  ^ stackSerialNum

]

{ #category : 'Private' }
GsProcess >> _start [
  "Called from C to start a new process.
   NOTE, exception handling differs in extent0.dbf and extent0.ruby.dbf .
   This version of _start from extent0.dbf .

   This method will never return. Once the receiver completes it should
   find another process to run.
   This method is preloaded during VM startup.  You must logout/login
   to have any changes to this method take effect."

  | res curr |
  (modeInfo bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    self error:'cannot start an instance that is a continuation'
  ].
  block ifNotNil:[
    res := block valueWithArguments: args
  ].
  "if this process resumed a continuation, then current process may
   no longer be self at this point."
  curr := self _terminateCurrentWithResult: res . "terminate currently running process"
  joiners ifNotNil:[
    curr == self ifTrue:[ self _signalJoiners ]
               ifFalse:[ self error:'cannot _signalJoiners after resuming continuation'].
  ].
  self _scheduler _runNextProcess .

]

{ #category : 'Private' }
GsProcess >> _statusString [
  "Returns a string that describes the receiver's status as a thread."

  ^ self _scheduler _statusString: self.

]

{ #category : 'Private' }
GsProcess >> _stepCleanup: saveArray [
  | newStackDepth flags |

  self _storeBit: 1 value: false . "debuggingInProgress:=0"

  newStackDepth := self stackDepth.

  self _clearAllStackBreaks .
  GsNMethod _clearAllStepBreaks .

  "clear INT_STEP_INTO_FROM_TOS_MASK "
  flags := self interruptFlag .
  flags := flags bitAnd: (16r200 bitInvert).
  self setInterruptFlag: flags .

]

{ #category : 'Debugging Support' }
GsProcess >> _stepOverFromLevel: flags breakpointLevel: brkLevel [
  "For use by an in-process debugger .
     (flags bitAnd:  16rFFFFFFFF) is the stack level to step from , 0 == step into .
     (flags bitAnd: 16r100000000) is the step through boolean .

     brkLevel, if not nil, is a SmallInteger >=0 which sets breakpointLevel of
     the receiver , 0 means signal single step errors to GCI , 1 to Smalltalk ."
  | actualLevel dp |
  self == GsProcess _currentOrNil ifTrue:[
    ^ Error signal:'cannot single-step the currently running GsProcess'
  ].
  dp := self localStackDepth . "force to object memory"
  actualLevel := flags bitAnd:  16rFFFFFFFF .
  (actualLevel > dp) ifTrue:[
    actualLevel _error: #rtErrArgOutOfRange args:{ 0 . dp } .
  ].
  self convertToPortableStack .

  brkLevel ifNotNil:[ self breakpointLevel: brkLevel ].

  "VM clears stack and step BPs before signalling a single step Breakpoint."

  "clear any leftover INT_STEP_INTO_FROM_TOS_MASK"
  self setInterruptFlag: (self interruptFlag bitAnd:(16r200 bitInvert)) .
  
  self _setBreaksForStepOverFromLevel: flags .
]

{ #category : 'Private Debugging Support' }
GsProcess >> _stepOverInFrame: flags [
  "Used by GBS , which expects breakpoint and step point errors
    to be returned to the GCI. "
  ^ self _stepOverInFrame: flags breakpointLevel: 0 
]

{ #category : 'Private Debugging Support' }
GsProcess >> _stepOverInFrame: flags breakpointLevel: brkLevel [
    "For use by a debugger controlled from GCI.
     Step execution so that it stops after the next message send
     in the specified stack level.  If a return causes the
     context at the specified level to be removed from the
     stack, execution will stop immediately after that return.
     Exceptions for step-point and breakpoint errors are suppressed.
     Returns the receiver unless the receiver completes in which
     case the result of the completion is returned.

     (flags bitAnd:  16rFFFFFFFF) is the stack level to step from (0 = step into)
     (flags bitAnd: 16r100000000) is the step through boolean .

     brkLevel, if not nil, is a SmallInteger >=0 which sets breakpointLevel of
     the receiver , 0 means signal single step errors to GCI , 1 to Smalltalk .
    "
  ^ self _stepOverInFrame: flags mode: false replace: false tos: nil breakpointLevel: brkLevel .

]

{ #category : 'Private' }
GsProcess >> _stepOverInFrame: flags mode: raiseException replace: replaceTosBool tos: newTosValue breakpointLevel: brkLevel [
    "Used in the implementation of GciStep and within the server
     image. topaz uses this via GsProcess >> _topazStepOverInFrame .
     Do not use when implementing in-process debugging.
     Step execution so that it stops after the next message send
     in the specified stack level.  If a return causes the
     context at the specified level to be removed from the
     stack, execution will stop immediately after that return.
     If 'raiseException' then debugger exceptions are raised
     so that a GCI app can catch them.
     Returns the receiver unless the receiver completes in which
     case the result of the completion is returned.
     Execution of this method must be in interpreted mode.

     flags == 0 is stepInto semantics.
     flags == 1 is typical step over in top frame .
     flags == 16r100000001 is typical step through (stopping in blocks of the method)
     (flags bitAnd:  16rFFFFFFFF) is the stack level to step from .
     (flags bitAnd: 16r100000000) is the step through boolean .

     brkLevel, if not nil, is a SmallInteger >=0 which sets breakpointLevel of
     the receiver , 0 means signal single step errors to GCI , 1 to Smalltalk .
    "
  | actualLevel result saveArray breakpointsToIgnore
    mySaveArray saveStackDepth status sched dp doTrace wasInCrit |
  self isContinuation ifTrue:[
    Error signal:'cannot use stepping to resume a continuation' .
  ].
  "self _checkIfDebuggable ."
  self convertToPortableStack .
  self _nativeStack ifTrue:[
    Error signal:'unable to convert GsProcess to be stepped to portable stack'
  ].
  GsProcess _currentOrNil ifNotNil:[:currProc| 
    currProc _nativeStack ifTrue:[ | cvtProc |
      cvtProc := [ currProc convertToPortableStack ] fork .
      [ cvtProc _isTerminated ] whileFalse:[
        self _scheduler yield .
      ].
      currProc _nativeStack ifTrue:[
        Error signal:'unable to convert current GsProcess to portable for stepping another GsProcess'.
      ].
    ].
  ].
  brkLevel ifNotNil:[ self breakpointLevel: brkLevel ].
  sched := self _scheduler .
  wasInCrit := sched _enterCritical .
  status := self _statusString.
  (status = 'ready') ifTrue: [
    self _unscheduleProcess .
  ].
  dp := self localStackDepth .
  flags < 0 ifTrue:[
    flags _error: #rtErrArgOutOfRange args:{ 0 . dp } .
  ].
  actualLevel := flags bitAnd:  16rFFFFFFFF .
  (actualLevel > dp) ifTrue:[
    actualLevel _error: #rtErrArgOutOfRange args:{ 0 . dp } .
  ].
  saveStackDepth := dp .
  (actualLevel == 0) ifTrue: [  | interrFlags |
    saveArray := self _setBreaksForStepInto.
    breakpointsToIgnore := (saveArray at: 3).
    "set INT_STEP_INTO_FROM_TOS_MASK bit"
    interrFlags := self interruptFlag .
    interrFlags := flags bitOr: 16r200.
    self setInterruptFlag: interrFlags .
  ] ifFalse: [
    saveArray := self _setBreaksForStepOverFromLevel: flags .
    breakpointsToIgnore := (saveArray at: 3).
        	" deleted INT_STEP_OVER_FROM_TOS_MASK logic "
  ].

  (replaceTosBool) ifTrue: [
    mySaveArray := { (saveArray at: 1) . (saveArray at: 2) . saveStackDepth .
                       newTosValue }.
  ] ifFalse: [
    mySaveArray := { (saveArray at: 1) . (saveArray at: 2) . saveStackDepth }.
  ].

  self _storeBit: 1 value: true . "debuggingInProgress:=1"
  wasInCrit == 0 ifTrue:[ sched _exitCritical ].

  doTrace ifNotNil:[
    GsFile gciLogServer:'breakpointsToIgnore ' , breakpointsToIgnore asString ].
  result := self _primStep: breakpointsToIgnore
                 from: nil with: mySaveArray mode: raiseException.

  wasInCrit := sched _enterCritical .
  (status = 'ready') ifTrue: [
    (result == self) ifTrue: [
      (self _isTerminated) ifFalse: [
        sched _scheduleProcess: self .
      ].
    ].
  ].
  wasInCrit == 0 ifTrue:[ sched _exitCritical ].
  "_stepCleanup may run a method that was stepped into (Object>>at:), so run with
   #_performNoDebug:with: to ensure that the cleanup is performed"
  self _performNoDebug: #_stepCleanup:  env: 0  with: mySaveArray.

  ^result

]

{ #category : 'Private Debugging Support' }
GsProcess >> _stepOverInFrame: flags return: anObject [
    "Used by GBS , which expects breakpoint and step point errors
     to be returned to the GCI.
     Do not use when implementing in-process debugging.
     Step execution so that it stops after the next message send
     in the specified stack frame.  If a return causes the
     context at the specified frame to be removed from the
     stack, execution will stop immediately after that return.
     The argument anObject will be the return value of the
     current top of stack - useful for passing back the return
     value of a client forwarder send.
     Returns the receiver unless the receiver completes in which
     case the result of the completion is returned.
     Execution of this method must be in interpreted mode.

     (flags bitAnd:  16rFFFFFFFF) is the stack level to step from (0 means step into).
     (flags bitAnd: 16r100000000) is the step through boolean .
    "
  ^ self _stepOverInFrame: flags mode: false replace: true tos: anObject breakpointLevel: nil 

]

{ #category : 'Debugging Support' }
GsProcess >> _stepPointAt: aLevel [

"Used by topaz debugger,
 returns a stepPoint , or nil if no step point available "

^ (self _localStepPointAt: aLevel) at: 1

]

{ #category : 'Debugging Support' }
GsProcess >> _stepPointStringAt: aLevel [

"Used by topaz debugger."

| stepPtArr result stepPoint theMeth |
stepPtArr := self _localStepPointAt: aLevel .
stepPtArr ifNil:[ ^ '' ].

stepPoint := stepPtArr at: 1 .
theMeth := stepPtArr at: 2 .
result := String withAll:'@' .
result addAll: stepPoint asString ;
    addAll: ' line ' ;
    addAll: (theMeth _lineNumberForStep: stepPoint) asString .
^ result


]

{ #category : 'Private' }
GsProcess >> _storeBit: bitNum value: aValue [
  "  aValue must be true,false, or nil(bitNum 0 only)
   bitNum 0 ==  set/clear  lockedInMemory returns previous value ;
               aValue == nil means no change and return current value.
   bitNum 1 ==  store  debuggingInProgress, per aValue
                returns receiver ;
   bitNum 2 == set  THREAD_SIGNAL_EXC interrupt bit, aValue must be true.
        returns previous value, for non-current process only.
   The primitive accesses/updates either C memory state or object memory
   as required by whether process is in stack memory or not.
 "
  <primitive: 707>

  self _primitiveFailed: #_storeBit:value: args: { bitNum . aValue }

]

{ #category : 'Private' }
GsProcess >> _targetProcess [
  "Returns the GsProcess that is waiting for the receiver."

  ^ self

]

{ #category : 'Private Debugging Support' }
GsProcess >> _topazDescrForFrame: frameId padTo: minSize [
  "Used by topaz stack display code .
   Returns nil if frameId is out of range, or an Array
    { aGsNMethod . aString } .
   aGsNMethod in the result is nil if frame is a reenter marker.
  "
  | frame aMethod |
  frame := self _frameContentsAt: frameId .
  frame ifNil:[ ^ nil ].
  aMethod := frame at: 1 .
  aMethod ifNil:[ ^ { nil . 'reenter marker' } ].
  ^ { aMethod .
      aMethod _descrForStackPadTo: minSize rcvr: (frame at: 10 "rcvr")
    }

]

{ #category : 'Debugging Support' }
GsProcess >> _topazMethodAt: aLevel [

  "Returns nil if aLevel is out of range"

| fpIdx dp |
(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  ^ nil
  ].
fpIdx := self _frameOffsetAt: aLevel .
fpIdx ifNil:[ ^ nil].
^ self _fetchMethod:( arStack at: (fpIdx + FP_codePtr_OFS )) .

]

{ #category : 'Private' }
GsProcess >> _topazPrintName [
  "Used by topaz ALLSTACKS command"
  | res  knd |
  res := String new .
  name ifNotNil:[ res add: ' name:'; add: name asString ].
  res add: ' priority:' ; add: self priority asString ;
     add: ' breakpointLevel:'; add: breakpointLevel asString . 
  msgEnvironmentId ~~ 0 ifTrue:[ res add:' env:', msgEnvironmentId asString ]. 
   knd := self _stackKind .
  res add:(  knd == 0 ifTrue:[ ' interpreted' ] ifFalse:[
        knd == 1 ifTrue:[ ' mixedMode' ] ifFalse:[ ' native']]).
  ^ res
]

{ #category : 'Private Debugging Support' }
GsProcess >> _topazStepOverInFrame: flags [
  "Used by topaz and GciStep which
   expect breakpoint and step point errors to be returned to the GCI .
   Do not use when implementing in-process debugging.

   Step execution so that it stops after the next message send
   in the specified stack level.  If a return causes the
   context at the specified level to be removed from the
   stack, execution will stop immediately after that return.
   Exceptions for step-point and breakpoint errors are NOT suppressed.
   If the receiver completes result of the completion is returned.

   (flags bitAnd:  16rFFFFFFFF) is the stack level to step from (0 = step into).
    (flags bitAnd: 16r100000000) is the step through boolean."

  ^ self _stepOverInFrame: flags mode: true replace: false tos: nil breakpointLevel: 0 

]

{ #category : 'Private' }
GsProcess >> _toWordOfs: aByteOffset [

^ aByteOffset bitShift: -3

]

{ #category : 'Private' }
GsProcess >> _unscheduleProcess [

  "remove the given process from the queues it is in.
   caller puts scheduler in critical region."
  | sched won st |
  sched := self _scheduler .
  (st := signalTime) ifNotNil: [
     sched _delayUnschedule: self.
  ].
  (won := waitingOn) ifNotNil:[
    won _unscheduleProcess: self .
  ].
  sched _unschedule: self.
  st ifNotNil: [
    signalTime := nil
  ].
  won ifNotNil:[
    waitingOn := nil
  ].

]

{ #category : 'Private' }
GsProcess >> _wait [
  "Suspend the receiver who should be the active process"
  self _scheduler _reschedule .
  ^ self
]

{ #category : 'Private' }
GsProcess >> _waitingOn: anObj [
  " for use by ProcessorScheduler only"
  waitingOn := anObj

]

{ #category : 'Private' }
GsProcess >> beForked [
  "Private.
   Prevents termination of the receiver from causing a return to the GCI.
   if the scheduler has something else to run."

  modeInfo := modeInfo bitOr: ModeInfo_forked

]

{ #category : 'Accessing' }
GsProcess >> breakpointLevel [
  "values 0  process debugged by GCI. 
   value  >= 1  process debugged by Smalltalk"
  ^ breakpointLevel
]

{ #category : 'Accessing' }
GsProcess >> breakpointLevel: aSmallInteger [
  "values 0  process debugged by GCI. 
   value  >=1  process debugged by Smalltalk"

  aSmallInteger _validateClass: SmallInteger .
  aSmallInteger < 0 ifTrue:[ Error signal:'breakpointLevel must be >= 0 '].
  ^ breakpointLevel := aSmallInteger
]

{ #category : 'In-Process Debugger Support' }
GsProcess >> clearLastBreakpointHistory [
  "If a Breakpoint is handled by handler block installed with on:do: or
   onException:do:, and neither #return nor #resume is sent to the instance
   of Breakpoint, then
      GsProcess _current clearLastBreakpointHistory
   is needed in order for the next Breakpoint to be signalled properly."

  lastBreakpt := nil

]

{ #category : 'Process Properties' }
GsProcess >> clientData [
  "answer the client data associated with the receiver.
   See also environmentAt:  "

  ^clientData

]

{ #category : 'Process Properties' }
GsProcess >> clientData: anobject [
  "set the client data object associated with the receiver.
   See also environmentAt:put:  "

  clientData := anobject

]

{ #category : 'Debugging Support' }
GsProcess >> convertToPortableStack [
  "Ensure that the receiver is executing in interpreted mode,
   used so that subsequent breakpoint operations will work."

  self == GsProcess _currentOrNil ifTrue:[
    GsProcess usingNativeCode ifTrue:[
      ^ Error signal:'cannot alter stack of the currently running GsProcess'
    ]
  ] ifFalse:[ | d |
    d := self localStackDepth . "ensure stack pushed to object memory"
    self _nativeStack ifTrue:[
      d == 0 ifTrue:[ Error signal:'process has not started or has been terminated'].
      self _convertToPortableStack: -1 .
    ].
  ]

]

{ #category : 'Debugging Support' }
GsProcess >> convertToPortableStack: forTrimToLevel [

  "Ensure that the receiver is executing in interpreted mode,
   used so that subsequent breakpoint operations and execution
   restart will work.
   forTrimToLevel should be -1 if not being called for stack trim,
   otherwise a positive stack level. "

  self == GsProcess _currentOrNil ifTrue:[
    GsProcess usingNativeCode ifTrue:[
      ^ Error signal:'cannot alter stack of the currently running GsProcess'
    ]
  ] ifFalse:[
    self localStackDepth "ensure stack pushed to object memory".
    self _nativeStack ifTrue:[
      self _convertToPortableStack: forTrimToLevel .
    ].
  ]

]

{ #category : 'Copying' }
GsProcess >> copy [

"Disallowed."

self shouldNotImplement: #copy

]

{ #category : 'Accessing' }
GsProcess >> createdByApplication [
  "Returns true if the receiver was created by the application.
   Returns false if the receiver was created by a Smalltalk method."

  ^(args == nil)

]

{ #category : 'Accessing' }
GsProcess >> debugActionBlock [
   ^ debugActionBlock
   
]

{ #category : 'Updating' }
GsProcess >> debugActionBlock: aBlock [
  aBlock ifNotNil:[ 
    aBlock _validateClass: ExecBlock.
    aBlock argumentCount <= 1 ifFalse:[
      ArgumentError signal:'block must take 0 or 1 arguments'.
    ].
  ].
  debugActionBlock := aBlock

]

{ #category : 'Debugging Support' }
GsProcess >> disableDebugging [
  "causes the receiver to ignore any breakpoints which may be set."
  ^ self _setModeinfoBit: ModeInfo_debugEnabled value: 0

]

{ #category : 'Debugging Support' }
GsProcess >> enableDebugging [
  "causes the receiver to ignore any breakpoints which may be set."
  ^ self _setModeinfoBit: ModeInfo_debugEnabled value: 1 .
]

{ #category : 'Accessing' }
GsProcess >> envId [
 "Returns a SmallInteger in the range 0..255 "
 ^ modeInfo bitAnd: ModeInfo_threadEnvMask
]

{ #category : 'Process Properties' }
GsProcess >> environment [
  | env |
  (env := environment) ifNil:[
     env := Dictionary new .
     environment := env
  ].
  ^ env

]

{ #category : 'Process Properties' }
GsProcess >> environment: aDictionary [

  environment := aDictionary

]

{ #category : 'Process Properties' }
GsProcess >> environmentAt: key [

  | env |
 (env := environment) ifNil:[ ^ nil ].
 ^ env at: key otherwise:  nil

]

{ #category : 'Process Properties' }
GsProcess >> environmentAt: key ifAbsent: aBlock [
  | env |
  (env := environment) ifNil:[ ^ aBlock value ].
  ^ env at: key ifAbsent:[ aBlock value ]

]

{ #category : 'Process Properties' }
GsProcess >> environmentAt: key put: value [

  ^ (environment ifNil:[ self environment]) at: key put: value

]

{ #category : 'Debugging Support' }
GsProcess >> gciStepIntoFromLevel: aLevel [
  "For use by a Gci application.
   Breakpoint and step point errors will be returned as errors to the GCI.
   Do not use when implementing in-process debugging.

  Step execution so that it stops at the beginning of the next method
  invoked.  If a return causes the
  context at the specified level to be removed from the
  stack, execution will stop immediately after that return.
  If the receiver completes result of the completion is returned."
  aLevel < 1 ifTrue:[ ArgumentError signal:'level argument must be >= 1' ].
  ^ self _stepOverInFrame: aLevel - 1 mode: true replace: false tos: nil breakpointLevel: 0 .

]

{ #category : 'Debugging Support' }
GsProcess >> gciStepOverFromLevel: aLevel [
  "For use by a Gci application.
   Breakpoint and step point errors will be returned as errors to the GCI.
   Do not use when implementing in-process debugging.

  Step execution so that it stops after the next message send
  in the specified stack level.  If a return causes the
  context at the specified level to be removed from the
  stack, execution will stop immediately after that return.
  If the receiver completes result of the completion is returned."
  aLevel < 1 ifTrue:[ ArgumentError signal:'level argument must be >= 1' ].
  ^ self _stepOverInFrame: aLevel  mode: true replace: false tos: nil breakpointLevel: 0 .

]

{ #category : 'Debugging Support' }
GsProcess >> gciStepThruFromLevel: aLevel [
  "For use by a Gci application.
   Breakpoint and step point errors will be returned as errors to the GCI.
   Do not use when implementing in-process debugging.

  Step execution so that it stops after the next message send
  in the specified stack level.  If a return causes the
  context at the specified level to be removed from the
  stack, execution will stop immediately after that return.
  If the receiver completes result of the completion is returned."

  | flags |
  aLevel < 1 ifTrue:[ ArgumentError signal:'level argument must be >= 1' ].
  flags := (aLevel bitAnd: 16rFFFFFFFF) bitOr: 16r100000000"thru flag" .
  ^ self _stepOverInFrame: flags mode: true replace: false tos: nil breakpointLevel: 0 .

]

{ #category : 'Debugging Support' }
GsProcess >> ignoreNextBreakpoint [
  "Use this prior to resuming from a breakpoint if your
   debugger is not using GciContinue to resume execution."

  ^ self _setModeinfoBit: ModeInfo_breakptsIgnore_mask value: 1 .

]

{ #category : 'Updating' }
GsProcess >> instVarAt: anIndex put: aValue [

"Disallowed."

self shouldNotImplement: #instVarAt:put:

]

{ #category : 'Accessing' }
GsProcess >> interruptFlag [

"Return the interruptFlag value"
^ interruptFlag_stEntryCount bitShift: -32

]

{ #category : 'Continuations' }
GsProcess >> isContinuation [

"Return true if the receiver is a continuation."

^ (modeInfo bitAnd: ModeInfo_continuationMask) == ModeInfo_isContinuation

]

{ #category : 'Debugging Support' }
GsProcess >> isForked [
  ^ (modeInfo bitAnd: ModeInfo_forked) ~~ 0

]

{ #category : 'Continuations' }
GsProcess >> isPartialContinuation [

"Return true if the receiver is a continuation."

^ (modeInfo bitAnd: ModeInfo_continuationMask) == ModeInfo_isPartialContinuation

]

{ #category : 'Changing Process State' }
GsProcess >> join: limitSeconds [

  "Waits up to limitSeconds for receiver to terminate.
   Returns self if thread already terminated, nil
   if the time limit expires before receiver terminates"

  | curr delay mi |
  limitSeconds _isSmallInteger ifFalse:[
    ArgumentError signal:'expected a SmallInteger' .
  ].
  limitSeconds < 0 ifTrue:[
    ArgumentError signal:'argument must be >= zero ' .
  ].
  self == (curr := self _currentOrNil) ifTrue:[
    ThreadError signal:'tried to join itself' .
    ^ nil
  ].
  mi := modeInfo .
  (mi bitAnd: ModeInfo_terminated ) ~~ 0 ifTrue:[ ^ self ].
  limitSeconds ~~ 0 ifTrue:[
    (mi bitAnd: ModeInfo_terminationStarted) ~~ 0 ifTrue:[ | t limMs |
      limMs := 1000 * limitSeconds .
      t := 0 .
      [ t < limMs ] whileTrue:[
        delay := Delay forMilliseconds: 20.
        delay highPriorityWait  .
        self _isTerminated ifTrue:[ ^ self ].
        t := t + 20 .
      ]
    ] ifFalse:[
      delay := Delay forMilliseconds: 1000 * limitSeconds .
      self _addJoiner: delay .
      delay highPriorityWait  .
      self _removeJoiner: delay .
      self _isTerminated ifTrue:[ ^ self ].
    ]
  ].
  ^ nil

]

{ #category : 'Accessing' }
GsProcess >> localMethodAt: aLevel [

"Returns the GsNMethod that is active at aLevel in the receiver, where
 aLevel == 1 is the top of the stack.  Generates an error if aLevel less than
 zero or greater than stackDepth.  Returns nil if there is a reenter marker at
 the specified level."

| fpIdx dp |

(aLevel < 1 or:[ aLevel > (dp := self localStackDepth) ]) ifTrue:[
  aLevel _error: #rtErrArgOutOfRange args:{ 1 . dp } .
  ].
fpIdx := self _frameOffsetAt: aLevel .
fpIdx ifNil:[ ^ nil].
^ self _fetchMethod:( arStack at: (fpIdx + FP_codePtr_OFS )) .

]

{ #category : 'Accessing' }
GsProcess >> localStackDepth [

"Returns the local stack depth of the receiver, or 0 if receiver is running.
 If receiver not running but has an in-memory stack, forces the receiver out
 to object memory.  If receiver not running but is locked in memory,
 this method will unlock it.
 Returns 0 if receiver was found in memory but has been terminated, or
 has never been run.
"

<primitive: 719>  "process push to OM, primitive always fails"
| uacd |
arStack ifNil:[ ^ 0 ].   "v3.7: allow examination of a terminated GsProcess"
(uacd := uaCount_depth) ifNil:[
  ^ 0
].
^ uacd bitAnd: 16rFFFFFFFF

]

{ #category : 'Accessing' }
GsProcess >> lockedInMemory [

  ^ self _storeBit: 0 value: nil .

]

{ #category : 'Updating' }
GsProcess >> lockInMemory [

 "Returns true if successful.
  Returns false if too many processes would be locked in stack memory,
  or if the receiver is not currently in stack memory."

 ^ self _storeBit: 0 value: true

]

{ #category : 'Accessing' }
GsProcess >> methodAt: aLevel [

"Returns the GsNMethod that is active at aLevel in the receiver, where
 aLevel == 1 is the top of the stack.  Generates an error if aLevel less than
 zero or greater than stackDepth.  Returns nil if there is a reenter marker at
 the specified level."

^ self localMethodAt: aLevel

]

{ #category : 'Accessing' }
GsProcess >> name [
  ^ name ifNil:[ '' ]

]

{ #category : 'Updating' }
GsProcess >> name: aString [
  name := aString 

]

{ #category : 'Accessing' }
GsProcess >> onQueue [
  ^ onQueue

]

{ #category : 'Accessing' }
GsProcess >> parentProcess [
  ^ parentProcess "a GsProcess"

]

{ #category : 'Formatting' }
GsProcess >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

aStream nextPutAll: self _reportString

]

{ #category : 'Formatting' }
GsProcess >> printString [

"Returns a String whose contents are a displayable representation of the
 receiver."

^ self _reportString

]

{ #category : 'Accessing' }
GsProcess >> priority [
  "Answer the scheduling priority of the receiver"

  | p_u |
  (p_u := priority_use) ifNil:[ ^ 15 "inline userSchedulingPriority"].
  ^ p_u bitShift: -40

]

{ #category : 'Updating' }
GsProcess >> priority: anInt [
  | oldPriority p_u |
      "inline _priorityRangeCheck"
  (anInt < 1 or:[ anInt > 40])  "inline lowestPriority, highestPriority" ifTrue:[
    anInt _error: #rtErrArgOutOfRange args:{ 1 . 40 } .
  ].
  (p_u := priority_use) ifNil:[
    oldPriority := 15 "inline userSchedulingPriority" .
  ] ifNotNil:[
    oldPriority :=  p_u bitShift: -40 .
  ].
  anInt ~~ oldPriority ifTrue:[ 
    self _setPriority: anInt.
    waitingOn ifNotNil:[
      waitingOn _changePriority: self from: oldPriority .
    ].
    self _scheduler _changePriority: self from: oldPriority  .
  ].
]

{ #category : 'Accessing' }
GsProcess >> result [
  "Retrieve the result from a completed thread. Returns nil
  if receiver still running, or did not complete normally."
  | res |
  self _isTerminated ifFalse:[ ^ nil ].
  (res := blockResult) == _remoteNil ifTrue:[ ^ nil ].
  ^ res

]

{ #category : 'Changing Process State' }
GsProcess >> resume [
  "If receiver is suspended, puts the receiver in the queue of ready processes at 
   its priority, otherwise has no effect"
  | sched mI |
  ((mI:= modeInfo) bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    self error:'cannot resume an instance that is a continuation'
  ].
  ((sched := self _scheduler) _isSuspended: self) ifTrue:[ "fix 49802"
    sched  _resumeProcess: self .
    sched activePriority < self priority ifTrue:[ sched yield  "fix 49114"].
  ].
]

{ #category : 'Updating' }
GsProcess >> setInterruptFlag: anInt [

" set the interruptFlag value. Receiver is expected to have been suspended."
| val |
stackId >= 0 ifTrue:[ self error:'cannot update C stack state correctly' ] .
val := interruptFlag_stEntryCount bitAnd:(16rFFFFFFFF). "keep stEntryCount"
val := val bitOr:(( anInt bitAnd:16rFFFFFFF) bitShift: 32) .
interruptFlag_stEntryCount := val

]

{ #category : 'In-Process Debugger Support' }
GsProcess >> setStepIntoBreaksAtLevel: aLevel [
  ^ self setStepIntoBreaksAtLevel: aLevel breakpointLevel: nil .
]

{ #category : 'In-Process Debugger Support' }
GsProcess >> setStepIntoBreaksAtLevel: aLevel breakpointLevel: brkLevel [
	  "Step execution into the next message send in stack frame specified by aLevel.
     To be used within implementation of an in-process debugger,
     after the receiver has signaled a Breakpoint from a previous single step
     or from a method breakpoint.
     After executing this method,
     the sender must yield  so the receiver can run.
     Receiver needs to have an on:do: handler for Breakpoint which
     yields to the debuggger process and then resumes from the Breakpoint.

     The step_into interrupt bit in the VM will be set by the returnNothing
     bytecode when the receiver returns from AbstractException>>_signalAsync:.

   brkLevel, if not nil, is a SmallInteger >=0 which sets breakpointLevel of
   the receiver , 0 means signal single step errors to GCI , 1 to Smalltalk ."

  | frameContents method ip fpOffset |
	self _stepOverFromLevel: aLevel breakpointLevel: brkLevel.
  frameContents := self _frameContentsAt: aLevel.
  method := self _methodInFrameContents: frameContents.
  ip := frameContents at: 2.
  ((method _opcodeKindAt: ip) bitAnd: 4) ~= 0 ifTrue:[ "_nextBytecodeIsSendAtLevel"
    "next opcode is a send"
    fpOffset := aLevel == 1
      ifTrue: [ self _toWordOfs: topFpOffset ]
      ifFalse: [ ((self _frameDescrAt: aLevel) at: 2) - (arStack size + 1) ].
    method _setBreakAtIp: ip operation: 5 frame: fpOffset process: self 
        breakpointLevel: self breakpointLevel .
  ].
]

{ #category : 'In-Process Debugger Support' }
GsProcess >> setStepOverBreaksAtLevel: aLevel [
  ^ self setStepOverBreaksAtLevel: aLevel breakpointLevel: nil .
]

{ #category : 'In-Process Debugger Support' }
GsProcess >> setStepOverBreaksAtLevel: aLevel breakpointLevel: brkLevel [
    "Step execution over next message in stack frame specified by aLevel.
     To be used within implementation of an in-process debugger.
     After executing this method,
     the sender must yield  so the receiver can run.
     Receiver needs to have an on:do: handler for Breakpoint which
     yields to the debuggger process and then resumes from the Breakpoint.

   brkLevel, if not nil, is a SmallInteger >=0 which sets breakpointLevel of
   the receiver , 0 means signal single step errors to GCI , 1 to Smalltalk ."

  | dp |
  self == GsProcess _currentOrNil ifTrue:[
    ^ Error signal:'cannot single-step the currently running GsProcess'
  ].
  dp := self localStackDepth . "force to object memory"
  aLevel _validateClass: SmallInteger .
  (aLevel < 1 or:[ aLevel > dp]) ifTrue:[
     ^ OutOfRange new name:'alevel' min: 1 max: dp actual: aLevel ; signal.
  ].
  self _stepOverFromLevel: aLevel breakpointLevel: brkLevel 

]

{ #category : 'In-Process Debugger Support' }
GsProcess >> setStepThroughBreaksAtLevel: aLevel [
  ^ self setStepThroughBreaksAtLevel: aLevel breakpointLevel: nil .
]

{ #category : 'In-Process Debugger Support' }
GsProcess >> setStepThroughBreaksAtLevel: aLevel breakpointLevel: brkLevel [
  "Step execution over the next message send in stack frame specified
   by aLevel, stopping in blocks which share a home method with the
   specified frame.
   To be used within implementation of an in-process debugger.
   After executing this method,
   the sender must yield  so the receiver can run.
   Receiver needs to have an on:do: handler for Breakpoint which
   yields to the debuggger process and then resumes from the Breakpoint.

   brkLevel, if not nil, is a SmallInteger >=0 which sets breakpointLevel of
   the receiver , 0 means signal single step errors to GCI , 1 to Smalltalk ."

  | dp |
  self == GsProcess _currentOrNil ifTrue:[
    ^ Error signal:'cannot single-step the currently running GsProcess'
  ].
  dp := self localStackDepth . "force to object memory"
  aLevel _validateClass: SmallInteger .
  (aLevel < 1 or:[ aLevel > dp]) ifTrue:[
     ^ OutOfRange new name:'alevel' min: 1 max: dp actual: aLevel ; signal.
  ].
  self _stepOverFromLevel: (16r100000000 bitOr: aLevel) breakpointLevel: brkLevel 

]

{ #category : 'Changing Process State' }
GsProcess >> signalException: anException [

  | prevBit |
  "signal an exception to the receiver."

  self == self _currentOrNil ifTrue:[
    ^ anException signal
  ].
  signalledException := anException .
  prevBit := self _storeBit: 2  value: true . "set INT_THREAD_SIGNAL_EXC_MASK bit"
  self resume .

]

{ #category : 'Accessing' }
GsProcess >> stackDepth [

"Returns the stack depth of the receiver."

| result |
result := self localStackDepth .
^ result

]

{ #category : 'Accessing' }
GsProcess >> stackId [

  "returns -1 if not in a stack area for execution.
   Only the VM can update stackId ."

  ^ stackId

]

{ #category : 'Debugging Support' }
GsProcess >> stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps andMethods: includeMethods [
  "Returns a String describing the stack of the receiver.
   The report includes frames from level 1 (top of stack) through the lesser of aLevel and
   the stack depth.

   The format of the result is subject to change with each release of GemStone.
   aLevel should be a SmallInteger >= 1 .
   includeArgsAndTemps and includeMethods should be Booleans."

^ self stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps
 	  andMethods: includeMethods includeSource: false

]

{ #category : 'Debugging Support' }
GsProcess >> stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps andMethods: includeMethods includeSource: includeSource [

^ self stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps
      andMethods: includeMethods includeSource: includeSource
      lineLimit: 100"characters per line"

]

{ #category : 'Debugging Support' }
GsProcess >> stackReportToLevel: aLevel withArgsAndTemps: includeArgsAndTemps andMethods: includeMethods includeSource: includeSource lineLimit: lineLim [
  "Returns a String describing the stack of the receiver.
   The report includes frames from level 1 (top of stack) through the lesser of aLevel and
   the stack depth.

   The format of the result is subject to change with each release of GemStone.
   aLevel should be a SmallInteger >= 1 .
   includeArgsAndTemps and includeMethods should be Booleans.
   includeSource may be: false or #none   no source ,
                         true or #execution  Execution sources,
                         #blocks    Sources for blocks and methods not in method dictionary
			                   2  or #all  method sources.
   lineLim is a SmallInteger specifing the maximum length in Characters 
       of each line of the report."

     | framesArr aFrame report gsMethod level inclSrc rptBlk tmpBlk |
     rptBlk := [:str | str size > lineLim
                   ifTrue:[ report add: (str copyFrom: 1 to: lineLim) ]
                   ifFalse:[ report add: str ]] .
     tmpBlk := [:anObj | | s | s := String new .
          anObj isSpecial ifFalse:[
            anObj isCommitted ifTrue:[ s add: 'oop:'; add: anObj asOop asString; add:' ' ].
            anObj class isIndexable ifTrue:[ s add: 'size:'; add: anObj size asString; add:' '].
          ].
          s size > 0 ifTrue:[ s add:' '].
          s add: anObj describe.
          s  ].
     inclSrc := includeSource .
     (inclSrc == true or:[ inclSrc == #execution]) ifTrue:[ inclSrc := 1 ] ifFalse:[
     (inclSrc == false or:[ inclSrc == #none]) ifTrue:[ inclSrc := 0 ] ifFalse:[
     (inclSrc == #blocks)                ifTrue:[ inclSrc := 2 ] ifFalse:[
     (inclSrc == #all or:[ inclSrc == 2 ]) ifTrue:[ inclSrc := 3 ] ifFalse:[
         ArgumentError signal: 'unrecognized includeSource arg ', includeSource asString]]]].
     framesArr := { } .
     level := 1.
     [ level <= aLevel and: [(aFrame := self _frameContentsAt: level) ~~ nil]] whileTrue:[
       framesArr at: level put: aFrame.
       level := level + 1
     ].
     report := String new.
     1 to: framesArr size do: [:j |
       report add: j asString; add: ' '.
       aFrame := framesArr at: j.
       (gsMethod := aFrame at: 1) ifNil: [report add: '<Reenter marker>'; lf
       ] ifNotNil: [
         | stepPoint |
         report add: gsMethod _descrForStack.
         stepPoint := gsMethod _stepPointForIp: (aFrame at: 2) level: j
             useNext: (self _nativeStack or:[ self _calleeIsAsync: j]) .
         report
           add: ' @'; add: stepPoint asString; add: ' line ';
           add: (gsMethod _lineNumberForStep: stepPoint) asString.
         includeMethods ifTrue:[
          report add: '  [GsNMethod '; add: gsMethod asOop asString; add: $]
         ].
         report lf.
         includeArgsAndTemps ifTrue:[ | argsAndTempsNames argsAndTempsValues |
          report add: '    receiver '.
             rptBlk value: (tmpBlk value: (aFrame at: 10)) .  report lf .
           argsAndTempsNames := aFrame at: 9.
           argsAndTempsValues := aFrame copyFrom: 11 to: aFrame size.
           [ argsAndTempsNames with: argsAndTempsValues do:[:nam :value |
                report add: '    '; add: nam; add: ' '.
                rptBlk value: ( tmpBlk value: value ) .
                report  lf
             ]
           ] on: ArgumentError do: [:ex |
             report
               add: '    <<<args and temps mismatch (';
               add: argsAndTempsNames size printString;
               add: ' versus ';
               add: argsAndTempsValues size printString;
               add: ', frame size = ';
               add: aFrame size printString;
               add: ') - ';
               add: '    names: ';
               add: argsAndTempsNames printString;
               add: '    values: '.
             rptBlk value: argsAndTempsValues describe .
             report add: '>>>'; lf .
            ]
          ].
          (self _stackReportIncludeSourceFor: gsMethod inclSrc: inclSrc) ifTrue:[
            gsMethod isMethodForBlock ifTrue:[
              report add: '  Block source:'; add:  (aFrame at: 10) _sourceString; lf
            ] ifFalse:[
              report add: (gsMethod inClass ifNil:[ '  Executed'] ifNotNil:['  Method']);
                     add: ' source: ';
                     add: gsMethod sourceString ; lf .
            ]
          ]
       ].
     ].
     ^report

]

{ #category : 'Accessing' }
GsProcess >> startSeconds [
  "Returns start time of the reciever, a SmallDouble, number of seconds since Jan 1, 2001
   with resolution of microseconds."
  ^ startSeconds

]

{ #category : 'Deprecated' }
GsProcess >> stepIntoFromLevel: aLevel [
  self deprecated:  'GsProcess >> stepIntoFromLevel: Deprecated as of GS/64 3.7, use setStepIntoBreaksAtLevel:breakpointLevel:'.
  ^ self setStepIntoBreaksAtLevel: aLevel breakpointLevel: nil .
]

{ #category : 'Deprecated' }
GsProcess >> stepOverFromLevel: aLevel [
  self deprecated:  'GsProcess >> stepOverFromLevel: Deprecated as of GS/64 3.7, use setStepOverBreaksAtLevel:breakpointLevel:'.
  ^ self setStepOverBreaksAtLevel: aLevel breakpointLevel: nil .
]

{ #category : 'Deprecated' }
GsProcess >> stepThroughFromLevel: aLevel [
  self deprecated:  'GsProcess >> stepThroughFromLevel: Deprecated as of GS/64 3.7, use setStepThroughBreaksAtLevel:breakpointLevel:'.
  ^ self setStepThroughBreaksAtLevel: aLevel breakpointLevel: nil .
]

{ #category : 'Changing Process State' }
GsProcess >> suspend [
  "Suspends the receiver from processing and does not
   schedule it for further processing."

  (modeInfo bitAnd: ModeInfo_continuationMask) ~~ 0 ifTrue:[
    self error:'cannot suspend an instance that is a continuation'
  ].
  self _scheduler _suspendProcess: self .

]

{ #category : 'Deprecated' }
GsProcess >> threadRubyEnvId [
 self deprecated:  'GsProcess >> threadRubyEnvId Deprecated as of GS/64 3.7.'.
 ^ modeInfo bitAnd: ModeInfo_threadEnvMask
]

{ #category : 'Updating' }
GsProcess >> unlockInMemory [

 "If receiver is locked in stack memory, allow it to be preempted.
  Returns true if receiver was locked in stack memory."

  ^ self _storeBit:0 value: false

]

{ #category : 'Continuations' }
GsProcess >> value [

"See value: for documentation "

^ self value: nil


]

{ #category : 'Continuations' }
GsProcess >> value: anArg [

"The receiver must be a continuation, i.e. a result
 from continuationFromLevel: , or an error will be generated.

 Resumes execution of the continuation with top-of-stack
 replaced by anArg . "

<primitive: 2016>
self _uncontinuableError  "should never reach here"

]

{ #category : 'Accessing' }
GsProcess >> waitingOn [
  ^ waitingOn
]

{ #category : 'Private' }
GsProcess >> _arStack [
  ^ arStack
]

