Extension { #name : 'GsSession' }

{ #category : 'Private Symbol Access' }
GsSession class >> _newSymbolsAsArray [

"Returns an Array containing the enumeration of all Symbols
 in   GsSession _newSymbolsDictionary ."

<primitive: 2001>  "enter protected mode"
| res prot |
prot := System _protectedMode .
[ | dict |
  dict := self _newSymbolsDictionary .
  res := { }  .
  dict keysDo:[:aSym | res add: aSym ].
] ensure:[
  prot _leaveProtectedMode
].
^ res

]

{ #category : 'Private Symbol Access' }
GsSession class >> _newSymbolsDictionary [

"Returns the session-private dictionary that is a cache of Symbols
 created during the session's current transaction
 on behalf of this session by the Symbol Creation Session."

<protected primitive: 152>
self _primitiveFailed: #_newSymbolsDictionary

]

{ #category : 'Private Symbol Access' }
GsSession class >> _newSymbolsSize [

"Returns the number of Symbols in
    GsSession _newSymbolsDictionary ."

<primitive: 2001>  "enter protected mode"
| res prot |
prot := System _protectedMode .
[ | dict |
  dict := self _newSymbolsDictionary .
  res := dict size .
] ensure:[
  prot _leaveProtectedMode
].
^ res

]

{ #category : 'Instance Creation' }
GsSession class >> currentSession [

"Returns the sole instance of GsCurrentSession that represents this login
 session."

<primitive: 311>
self _primitiveFailed: #currentSession

]

{ #category : 'Testing' }
GsSession class >> isSolo [

  ^ self currentSession isSolo

]

{ #category : 'Private' }
GsSession class >> serialOfSession: aSessionId [

"Returns a SmallInteger, the unique serial number of the session that is
 currently logged in as aSessionId, or -1 if that session is not logged in.

 Requires SessionAccess privilege if the session described by aSessionId is
 not the current session."

| descr |
descr := System _descriptionOfSessionSerialNum: 0 sessionId: aSessionId .
^ descr at: 9

]

{ #category : 'Private' }
GsSession class >> sessionIdOfSerial: aSerialNumber [

"Returns the current session ID of the session uniquely identified by
 aSerialNumber, or -1 if that session is not logged in.

 Requires SessionAccess privilege if the session described by aSerialNumber is
 not the current session."

| descr |
descr := System _descriptionOfSessionSerialNum: aSerialNumber sessionId: -1 .
^ descr at: 10

]

{ #category : 'Instance Creation' }
GsSession class >> sessionWithSerialNumber: anInteger [

"Returns an instance of a kind of GsSession that represents the GemStone session
 on the same repository, whose serial number is anInteger.  Returns nil if no
 logged-in session has that serial number.

 Requires SessionAccess privilege if the session described by anInteger is
 not the current session."

| dict aSess currSess descr systm |
currSess := GsSession currentSession.
anInteger == currSess sessionSerialNum ifTrue:[ ^ currSess ].
systm := System .
dict := systm __sessionStateAt: 9 .
dict == nil ifTrue:[
  dict := IdentityKeyValueDictionary new .
  dict at: (currSess sessionSerialNum) put: currSess .
  systm __sessionStateAt: 9 put: dict .
  ].
descr := systm _descriptionOfSessionSerialNum: anInteger sessionId: -1 .
(descr at: 1 "UserProfile" ) == nil ifTrue:[
  dict removeKey: anInteger otherwise: nil .
  ^ nil .
  ].
aSess := dict at: anInteger otherwise: nil .
aSess ~~ nil ifTrue:[ ^ aSess ].
aSess := super new _initFromSessionDescription: descr .
dict at: anInteger put: aSess.
^ aSess

]

{ #category : 'Private' }
GsSession >> _initFromSessionDescription: anArray [

"Initializes the receiver using  anArray, which must be a result from
 System (C) | _descriptionOfSessionSerialNum:sessionId:."

sessionSerialNum := anArray at: 9 .
userProfile := anArray at: 1

]

{ #category : 'Signalling' }
GsSession >> enableInterSessionSignalling: aBoolean [

"If aBoolean is true, enables receipt of intersession signals in the session
 represented by the receiver.  Otherwise, disables receipt of such signals."

System enableSignaledGemStoneSessionError

]

{ #category : 'Testing' }
GsSession >> hasRemoteSessions [

"Returns true if the receiver has remote sessions, false otherwise.

 If the receiver is itself a remote session, returns nil."

| rsess |

rsess := self remoteSessions.
rsess ifNil:[ ^ nil "don't know the answer" ].
^ rsess size ~~ 0

]

{ #category : 'Testing' }
GsSession >> isCurrent [

"Returns true if the receiver represents the current login session of this Gem;
 returns false otherwise."

^ self == GsSession currentSession

]

{ #category : 'Testing' }
GsSession >> isRemote [

"Returns true if the receiver represents a remote federated session."

^ self ~~ GsSession currentSession

]

{ #category : 'Testing' }
GsSession >> isSolo [

"Returns true if receiver represents a login that used the GCI_LOGIN_SOLO flag.

 A SOLO session is created by a linked login in a topaz -l process
 using the GEM_SOLO_EXTENT from the topaz's config file.
 In a remote login (gem forked by netldi), the gem must have an
 appropriate GEM_SOLO_EXTENT value in its config.
 GEM_SOLO_EXTENT can be in the gem config file, or passed on the topaz -l
 or gemnetobject command line with  -C 'GEM_SOLO_EXTENT=anExtent0.dbf' .

 The repository specified by GEM_SOLO_EXTENT must be a single extent
 repository, last written by a clean shutdown of a stone.
 If the extent on the filesystem has either write or execute permission
 for any of user,group,world  then session will open the extent with an
 exclusive file lock, to prevent opening an extent that is also open
 by a stone.  Set the extent to read-only permission on the filesystem
 to allow sharing by multiple Solo sessions.  The amount of internal
 free space in the extent does not affect the Solo sessions.

 Each Solo gem or topaz -l process also opens a read-write 10MB
 /tmp/gemRO_<pid>_extent1.dbf which is deleted on logout
 or process exit.

 Object creation in a Solo session is all in temporary object memory.
 Use -T on the topaz -l command line or in the gemnet NRS to specify
 appropriate temp obj memory config, or use appropriate config file
 options in the -C argument or in a gem config file.

 In a Solo session,
   A large number of classmethods in System
   will signal an ImproperOperation (number 2050)
   if the operation would require connection to a stone.

   Many methods in Repository signal an ImproperOperation including
      markForCollection, reclaimAll, fullBackup*, restoreFromBackup*

   Nested transactions work normally, but outer level commit
   signals an ImproperOperation unless the commit is readonly.
"

^ sessionSerialNum == 0

]

{ #category : 'Accessing' }
GsSession >> remoteSessions [

"Returns a collection of GsSession instances representing all the remote
 sessions spawned by the session represented by the receiver.  If the receiver
 has no remote sessions, the resulting collection is empty.  This list is
 maintained in the session transient state."

| sessionDict |

self == GsSession currentSession ifFalse:[ ^ nil "result is unknown" ].
" sessionDict maps parameters to sessions "
sessionDict := System __sessionStateAt: 8. "xxx not yet updated"
sessionDict == nil ifTrue:[ ^ #() ].
^ sessionDict values

]

{ #category : 'Signalling' }
GsSession >> sendSignalObject: aGsInterSessionSignal [

"Sends the signal and message of aGsInterSessionSignal to the session
 represented by the receiver."

System _sendSignal: aGsInterSessionSignal signal
       toSess: sessionSerialNum
       withMessage: aGsInterSessionSignal message

]

{ #category : 'Accessing' }
GsSession >> serialNumber [

"Returns the serial number issued to the session represented by the receiver
 when that session logged in."

^ sessionSerialNum

]

{ #category : 'Accessing' }
GsSession >> sessionSerialNum [

"Returns the value of the sessionSerialNum instance variable of the receiver."

^ sessionSerialNum

]

{ #category : 'Session Control' }
GsSession >> stop [

"Aborts the current transaction and terminates the session.  If  the receiver is
 the current session, no operation is performed.  If the UserProfile of the
 current session lacks SystemControl privilege, an exception occurs."

self == GsSession currentSession ifTrue:[ ^ self "do nothing" ].

System _stopSession: sessionSerialNum kind: 0 timeout: 0

]

{ #category : 'Accessing' }
GsSession >> userProfile [

"Returns the UserProfile attached to the session."

^ userProfile

]
