Extension { #name : 'Symbol' }

{ #category : 'Instance Creation' }
Symbol class >> _basicNew [
^ self shouldNotImplement: #_basicNew

]

{ #category : 'Instance Creation' }
Symbol class >> _basicNew: aSize [
^ self shouldNotImplement: #_basicNew:

]

{ #category : 'Class Membership' }
Symbol class >> _classHistoryIncludesIdentical: aClass [
  ^ aClass == Symbol or:[ aClass == DoubleByteSymbol or:[ aClass == QuadByteSymbol]]

]

{ #category : 'Instance Creation' }
Symbol class >> _existingWithAll: aString [

"Returns an existing canonical symbol that has the same value as 'aString'.
 If no such Symbol, DoubleByteSymbol or QuadByteSymbol already exists, returns nil."

"This method is useful in processing Strings that are supposed to be names
 of already existing instance variables, such as in indexing and sorting."

<primitive: 310>
aString _validateClass: CharacterCollection .
^ self _existingWithAll: aString asString .

]

{ #category : 'Converting' }
Symbol class >> _keywords: aSymbol [
| result str |
result := { } .
str := String new .
1 to: aSymbol size do:[:n | | ch |
  ch := aSymbol at: n .
  ch == $:  ifTrue:[
    str add: $: .  result add: str . str := String new .
  ] ifFalse:[
    str add: ch .
  ].
].
str size ~~ 0 ifTrue:[ result add: str ].
^ result

]

{ #category : 'Instance Creation' }
Symbol class >> _newString [

"Returns a new instance of String."

^ String new

]

{ #category : 'Instance Creation' }
Symbol class >> _newString: aSize [

"Returns a new instance of String of the specified size."

^ String new: aSize

]

{ #category : 'Private' }
Symbol class >> _validateNewSymbols: newSymbols  allSymbols: allSymbols [
newSymbols keysDo:[ :aSym | | val |
  val := allSymbols at: aSym ifAbsent:[
    (Globals at: #GsFile) gciLogServer:'symbol ', aSym ,
          ', oop ' , aSym asOop asString , ' not in AllSymbols' .
    nil pause .
  ].
  val == aSym ifFalse:[
    (Globals at: #GsFile) gciLogServer:'bad symbol identity', aSym ,
		', oop ' , aSym asOop  asString.
    nil pause .
  ].
].
^ true

]

{ #category : 'Class Membership' }
Symbol class >> isSubclassOf: aClassHistoryOrClass [

  (self _classHistoryIncludesIdentical: aClassHistoryOrClass) ifTrue:[ ^ true].
  ^ super isSubclassOf: aClassHistoryOrClass

]

{ #category : 'Storing and Loading' }
Symbol class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| result inst theSize marker |

"since Symbols can't refer to other objects, the 'hasRead:' message
 may be sent after values have been filled in."

theSize := passiveObj readSize .
inst := String new .
marker := passiveObj objectPositionMarker .
passiveObj next: theSize bytesTo: inst .
passiveObj next .
result := inst asSymbol .
passiveObj hasRead: result marker: marker .
^ result

]

{ #category : 'Instance Creation' }
Symbol class >> migrateNew [

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew

]

{ #category : 'Instance Creation' }
Symbol class >> new [

"Disallowed.  To create a new Symbol, use the class method withAll: instead."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
Symbol class >> new: size [

"Disallowed.  To create a new Symbol, use the class method withAll: instead."

self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
Symbol class >> withAll: aString [

"Returns a canonical symbol that has the same Characters as aString.  Returns
 an existing canonical symbol if it is already in AllSymbols, or if it was
 created earlier in the current session.  Otherwise, creates and returns a new
 canonical symbol.

 The result is a Symbol, DoubleByteSymbol, or QuadByteSymbol as required
 to hold the largest code point in aString ."

<primitive: 300>
aString _isOneByteString ifTrue:[  "charSize 1 and not a ByteArray"
  ^ self _primitiveFailed: #withAll: args: { aString } .
].
aString stringCharSize == 0 ifTrue:[ "not charSize 2 or 4"
  aString _validateClass: CharacterCollection .
  ^ self withAll: aString asString
].
aString _error: #rtErrBadArgKind2 args: { String . DoubleByteString . QuadByteString }.

]

{ #category : 'Decompiling without Sources' }
Symbol >> _asSource [

| result |
result := String new .
result add: $#  .
result addAll: super _asSource .
^ result

]

{ #category : 'New Indexing Comparison' }
Symbol >> _idxForCompareEqualTo: arg [

  ^arg _idxForCompareEqualToSymbol: self

]

{ #category : 'New Indexing Comparison' }
Symbol >> _idxForCompareEqualTo: aCharacterCollection collator: anIcuCollator [
  ""

  ^ aCharacterCollection _idxForCompareEqualToSymbol: self

]

{ #category : 'New Indexing Comparison - for Compare' }
Symbol >> _idxForCompareEqualToCharacterCollection: aCharacterCollection [
  "second half of a double dispatch call from CharacterCollection>>_idxForCompareEqualTo:."

  ^ false

]

{ #category : 'New Indexing Comparison - for Compare' }
Symbol >> _idxForCompareEqualToUnicode: aUnicodeString collator: aCollator [
"second half of a double dispatch call from CharacterCollection>>_idxForCompareEqualTo:collator:."

  ^ false

]

{ #category : 'New Indexing Comparison' }
Symbol >> _idxForSortEqualTo: arg [
  ""

  ^ arg _idxForSortEqualToSymbol: self

]

{ #category : 'New Indexing Comparison' }
Symbol >> _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator [
  ""

  ^ self asString
    _idxForSortEqualTo: aCharacterCollection
    collator: anIcuCollator

]

{ #category : 'New Indexing Comparison' }
Symbol >> _idxForSortEqualToSymbol: aSymbol [
  "second half of a double dispatch call from Symbol>>_idxForSortEqualTo:."

  ^ self asString _idxPrimCompareEqualTo: aSymbol asString

]

{ #category : 'New Indexing Comparison' }
Symbol >> _idxForSortNotEqualTo: aCharacterCollection collator: anIcuCollator [
  ""

  ^ (self _idxForSortEqualTo: aCharacterCollection collator: anIcuCollator) not

]

{ #category : 'Converting' }
Symbol >> -> anObject [

"Returns a SymbolAssociation with the receiver as the key and the given object
 as the value."

^ SymbolAssociation newWithKey: self value: anObject

]

{ #category : 'Concatenating' }
Symbol >> , aCharOrCharCollection [

"Returns a new instance of String that contains the elements of the receiver
 followed by the elements of aCharOrCharCollection.  A String is returned rather
 than a Symbol to avoid the expense of unnecessary creation and canonicalization
 of Symbols."

^ ((self _unicodeCompareEnabled ifTrue:[ Unicode7 ] ifFalse:[ String])
    withAll: self) addAll: aCharOrCharCollection ; yourself

]

{ #category : 'Comparing' }
Symbol >> = aCharCollection [

"Returns true if and only if the receiver and aCharCollection are identical."

^ self == aCharCollection

]

{ #category : 'Comparing' }
Symbol >> ~= aCharCollection [

"This method can be optimized for Symbols since they are canonical."

^ self ~~ aCharCollection

]

{ #category : 'Compatibility' }
Symbol >> argumentCount [

 ^ self numArgs

]

{ #category : 'Formatting' }
Symbol >> asString [

"Returns a copy of the receiver as an instance of class String."

^ String withAll: self

]

{ #category : 'Converting' }
Symbol >> asSymbol [

"Returns the receiver."

^ self

]

{ #category : 'Converting' }
Symbol >> asSymbolKind [

"Equivalent to asSymbol."

 ^self

]

{ #category : 'Clustering' }
Symbol >> cluster [

"Has no effect.  Clustering of Symbols is only performed by
 the clusterAllSymbols method in class System . "

^ true

]

{ #category : 'Clustering' }
Symbol >> clusterInBucket: aClusterBucketOrId [

"Has no effect.  Clustering of Symbols is only performed by
 the clusterAllSymbols method in class System . "

^ true

]

{ #category : 'Copying' }
Symbol >> copy [

"Returns self, copies of canonical symbols are not allowed."

^ self

]

{ #category : 'Copying' }
Symbol >> copyReplacing: oldObject withObject: newObject [
	"Returns a String comprising a copy of the receiver in which all occurrences
	 of objects equal to oldObject have been replaced by newObject."

	^self asString copyReplacing: oldObject withObject: newObject

]

{ #category : 'Comparing' }
Symbol >> hash [

"Returns a numeric hash key for the receiver.
 This implementation for new databases only and is
 replaced in the upgrade scripts."

^ self identityHash

]

{ #category : 'Converting' }
Symbol >> keywords [

"Returns an Array of the keywords in the receiver, treating
 any colon-delimited segment as if it was a legal keyword."

^ Symbol _keywords: self

]

{ #category : 'Testing' }
Symbol >> precedence [

"Returns the precedence of the receiver, were it a message selector, with
 1=unary, 2=binary and 3=keyword."

^self isInfix
    ifTrue: [ 2 ]
    ifFalse: [ self isKeyword ifTrue: [ 3 ]
                              ifFalse: [ 1 ] ]

]

{ #category : 'Formatting' }
Symbol >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream.
 That representation conforms to GemStone Smalltalk parsing rules."

aStream nextPut: $# .
super printOn: aStream

]

{ #category : 'Copying' }
Symbol >> shallowCopy [

"Returns self, copies of canonical symbols are not allowed."

^ self

]

{ #category : 'Class Membership' }
Symbol >> species [

"Returns a class similar to, or the same as, the receiver's class which
 can be used for containing derived copies of the receiver."

^ String

]

{ #category : 'Private' }
Symbol >> speciesForConversion [

"Returns String."

^ String.

]

{ #category : 'Formatting' }
Symbol >> withNoColons [

"Returns a String containing the value of the receiver with all colons removed.

 A String is returned rather than a Symbol to avoid the expense of unnecessary
 creation and canonicalization of Symbols."

^ self copyWithout: $:   .

]
