Extension { #name : 'UserProfileSet' }

{ #category : 'Creation' }
UserProfileSet class >> new [

"Disallowed"

self shouldNotImplement: #new

]

{ #category : 'Private' }
UserProfileSet >> __add: aUserProfile [

""
"Reimplementation from IdentityBag.
 Adds anObject to the receiver only if it is not already an element of the
 receiver.  Returns anObject.  Has no effect if anObject is nil."

<protected primitive: 208>
self _primitiveFailed: #add: args: { aUserProfile } .
self _uncontinuableError

]

{ #category : 'Private' }
UserProfileSet >> _add: aUserProfile [

"Adds a new UserProfile to the receiver.  Generates an error if aUserProfile
 has a userId that duplicates an existing element of the receiver."

<protected>

| userId |

aUserProfile _validateClass: UserProfile.
userId := aUserProfile userId.
self userWithId: userId ifAbsent:[
  userIdDictionary ~~ nil ifTrue:[
     userIdDictionary at: userId put: aUserProfile .
     ].
  aUserProfile _newSecurityData .
  self __add: aUserProfile .
  ^ aUserProfile
  ].
^ self _error: #rtErrUserIdAlreadyExists args: { userId }

]

{ #category : 'Illegal Operations' }
UserProfileSet >> _addIfAbsent: aUserProfile [

"Disallowed."
self shouldNotImplement: #_addIfAbsent:

]

{ #category : 'Private' }
UserProfileSet >> _addToDeletedUsers: aUserProfile [

| cls |
cls := Globals at: #DeletedUserProfile .
(Globals at: #AllDeletedUsers) add: (cls createFrom: aUserProfile)

]

{ #category : 'Private' }
UserProfileSet >> _initialize [

"Private.  Only SystemUser has permission to execute this successfully."

| saveArr |

System myUserProfile userId = 'SystemUser' ifFalse:[
  self _halt:'Only SystemUser may execute this method.'
  ].

userSecurityData == nil ifTrue:[
  GsObjectSecurityPolicy setCurrent: self objectSecurityPolicy while:[
    self size > 0 ifTrue:[
      saveArr := { }  .
      saveArr addAll: self .
      saveArr do: [ :aProfile | super remove: aProfile ].
    ] .

    userSecurityData := StringKeyValueDictionary new .

    userIdDictionary == nil
       ifTrue:[ userIdDictionary := StringKeyValueDictionary new .  ]
      ifFalse:[  self _halt:'inconsistent state' ].

    maxPasswordSize := 0 .
    minPasswordSize := 1 .
    maxRepeatingChars := 0 .
    maxConsecutiveChars := 0 .
    maxCharsOfSameType := 0 .

    disallowedPasswords := Set new .
    securityBits := 0 .

    passwordAgeLimit := nil .
    passwordAgeWarning := nil .
    staleAccountAgeLimit := nil .

    saveArr ~~ nil ifTrue:[
      saveArr accompaniedBy: self do:[ :me :aProfile | me add: aProfile ].
    ].
  ].							"fix 12466"
].

]

{ #category : 'Private' }
UserProfileSet >> _migrateSecurityPoliciesFrom: aUserProfile to: anotherUserProfile [

| sr |
anotherUserProfile _validateClass: UserProfile .
anotherUserProfile isCommitted
  ifFalse:[ anotherUserProfile _error: #rtErrObjMustBeCommitted ] .
aUserProfile == anotherUserProfile
  ifTrue:[ aUserProfile halt: 'Cannot migrate GsSecurityPolicyObjects to this UserProfile' ] .
sr := SystemRepository .
1 to: sr size do:[:n|
  (sr at: n ) ifNotNil:[ :policy |
    policy owner == aUserProfile
      ifTrue:[ policy owner: anotherUserProfile ] .
  ].
].

]

{ #category : 'Updating' }
UserProfileSet >> _oldUserId: oldUserId newUserId: newUserId for: aUserProfile [

"Private."

<protected>

| oldUserPro duplicateUser |

self _validatePrivilege ifFalse:[ ^ nil ].
System myUserProfile _validatePrivilegeName: #ChangeUserId .

duplicateUser := self userWithId: newUserId ifAbsent:[ nil ].
duplicateUser ifNotNil:[
  ^ self _error: #rtErrUserIdAlreadyExists args: { newUserId }
  ].

oldUserPro := self userWithId: oldUserId .
oldUserPro == aUserProfile ifFalse:[
  ^ ArgumentError signal: 'oldUserId does not match the given UserProfile' .
  ].

userIdDictionary removeKey: oldUserId .
userIdDictionary at: newUserId put: aUserProfile .
self _userSecurityDataOldKey: oldUserId newKey: newUserId .

]

{ #category : 'Illegal Operations' }
UserProfileSet >> _remove: aUserProfile [

"Disallowed."
self shouldNotImplement: #_remove:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> _removeAll: aCollection errIfAbsent: aBlock [

"Disallowed."
self shouldNotImplement: #_removeAll:errIfAbsent:

]

{ #category : 'Private' }
UserProfileSet >> _removeSecurityDataFor: aUserProfile [

""
<protected>

self _validatePrivilege ifTrue:[
  "use _removeKey: to avoid accessing value in userSecurityData, fix 36312"
  userSecurityData _removeKey: aUserProfile userId .
  super _remove: aUserProfile
]

]

{ #category : 'Private' }
UserProfileSet >> _updateSecurityBitsForMask: bitMask newState: aBoolean [

self _validatePrivilege ifTrue:[
  aBoolean
    ifTrue:[ securityBits := securityBits bitOr: bitMask ] "set"
    ifFalse:[ securityBits := securityBits bitAnd: bitMask bitInvert ]. "clear"
]

]

{ #category : 'Private' }
UserProfileSet >> _userSecurityDataOldKey: oldUserId newKey: newUserId [
  <protected primitive: 952>

  oldUserId _validateClass: String  .
  newUserId _validateClass: String  .
  self _primitiveFailed: #_userSecurityDataOldKey:newKey: args: { oldUserId . newUserId }

]

{ #category : 'Private' }
UserProfileSet >> _validateNewPassword: aString [

"Generates an error if aString is not a valid new password for the receiver;
 otherwise, returns true.
 If execution continued from the generated error, returns false."

<protected>
| subStr maxS |

maxS := maxPasswordSize > 0 ifTrue:[ maxPasswordSize ] ifFalse:[ 1024 ] .
aString size > maxS ifTrue:[
  aString _error: #rtMaxPasswordSize args:{ maxS } .
  ^ false .
].
(minPasswordSize > 0 and:[ aString size < minPasswordSize]) ifTrue:[
  aString _error: #rtMinPasswordSize args:{ minPasswordSize } .
  ^ false
].
(maxConsecutiveChars > 0) ifTrue:[
  subStr := aString maxConsecutiveSubstring .
  subStr size > maxConsecutiveChars ifTrue:[
    aString _error: #rtMaxConsecutiveChars args:{ maxConsecutiveChars . subStr }.
    ^ false
  ].
].
(maxRepeatingChars > 0) ifTrue:[
  subStr := aString maxRepeatingSubstring .
  subStr size > maxRepeatingChars ifTrue:[
    aString _error: #rtMaxRepeatingChars args:{ maxRepeatingChars . subStr }.
    ^ false
  ].
].
(maxCharsOfSameType > 0) ifTrue:[
  subStr := aString maxSameTypeSubstring .
  subStr size > maxCharsOfSameType ifTrue:[
    aString _error: #rtMaxCharsOfSameType
	    args:{ maxCharsOfSameType . (subStr at: 1) _typeAsSymbol asString .
                    subStr } .
    ^ false
  ].
].
(disallowedPasswords includes: aString) ifTrue:[
  aString _error: #rtDisallowedPassword  .
  ^ false
].

(self passwordRequiresUppercase and:[ aString containsUppercase not ]) ifTrue:[
  aString _error: #rtErrPasswordMissingChar args: { 'Uppercase' }  .
  ^ false
] .

(self passwordRequiresLowercase and:[ aString containsLowercase not ]) ifTrue:[
  aString _error: #rtErrPasswordMissingChar args: { 'Lowercase' }.
  ^ false
] .

(self passwordRequiresDigit and:[ aString containsDigit not ]) ifTrue:[
  aString _error: #rtErrPasswordMissingChar args: { 'Digit' } .
  ^ false
].

(self passwordRequiresSymbol and:[ aString containsPasswordSymbol not ]) ifTrue:[
  aString _error: #rtErrPasswordMissingChar args: { 'Symbol' }  .
  ^ false
] .
"disallowUsedPasswords tested by instance method in UserProfile."

^ true

]

{ #category : 'Private' }
UserProfileSet >> _validatePrivilege [

" You must have #OtherPassword privilege to modify UserProfileSets "

  ^ System myUserProfile _validatePrivilegeName: #OtherPassword

]

{ #category : 'Adding' }
UserProfileSet >> add: aUserProfile [

"Adds a new UserProfile to the receiver.  Generates an error if aUserProfile
 has a userId that duplicates an existing element of the receiver.
 Requires OtherPassword privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[
  self _validatePrivilege ifTrue:[
    result := self _add: aUserProfile .
  ]
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Group Membership' }
UserProfileSet >> addGroup: aGroupString [

"Adds all the users in the receiver to the group represented by aGroupString.
 If the current session does not have the authorizations required for this
 operation, raises an error.
 Requires OtherPassword privilege."

self _validatePrivilege ifTrue:[
  self accompaniedBy: aGroupString do:[ :grpStr :aUserProfile | aUserProfile addGroup: grpStr ]
]

]

{ #category : 'Logging' }
UserProfileSet >> addMsgToSecurityLog: aString [

"This method adds the date, time and userId prefix to the specified message
 string and includes the resulting string in the security log.  The Stone
log file is the security log.

 Requires OtherPassword privilege. "

| message |

self _validatePrivilege ifFalse:[ ^ nil ].
message := String withAll:'    SecurityLog: ' .
message addAll:  DateTime now asString ; addAll: ', ';
        addAll: System myUserProfile userId ;  lf ;
        addAll: aString .
System addAllToStoneLog: message .

]

{ #category : 'Adding' }
UserProfileSet >> addNewUserWithId: userIdString
password: passwordString [

"Creates a new UserProfile and adds it to the receiver.  The new UserProfile
 has no privileges and belongs to no groups and has the default
 GsObjectSecurityPolicy, which is nil.

 This default method can be used by the data curator in batch user installations.
 Returns the new UserProfile.

 The current session must be in a transaction with no uncommitted changes
 or an error will be generated.
 This method will writeLock AllUsers and SystemRepository for
 the duration of the method for the creation of the new UserProfile .
 If a writeLock cannot be obtained on AllUsers or SystemRepository ,
 an error will be generated.

 This method will commit its changes.

 Requires OtherPassword privilege and write access to DataCuratorObjectSecurityPolicy."

^ self addNewUserWithId: userIdString
       password: passwordString
       createNewSecurityPolicy: false

]

{ #category : 'Adding' }
UserProfileSet >> addNewUserWithId: userIdString
password: passwordString
createNewSecurityPolicy: aBoolean [

"Creates a new UserProfile and adds it to the receiver.  The new UserProfile
 has no privileges and belongs to no groups.  If aBoolean is true, a new
 GsObjectSecurityPolicy is created for the new UserProfile.  Otherwise the
 new UserProfile has the default GsObjectSecurityPolicy (nil).
 Returns the new UserProfile.

 The current session must be in a transaction with no uncommitted changes
 or an error will be generated.
 This method will writeLock AllUsers and SystemRepository for
 the duration of the method for the creation of the new UserProfile .
 If a writeLock cannot be obtained on AllUsers or SystemRepository ,
 an error will be generated.

 This method will commit its changes.

 Requires OtherPassword privilege and write access to DataCuratorObjectSecurityPolicy.
 If aBoolean is true, requires ObjectSecurityPolicyCreation."

| commitAttempted |
self _validatePrivilege ifFalse:[ ^ nil ].
^ [ | newSeg newUp oldUp |
  oldUp := self userWithId: userIdString ifAbsent:[ nil ].
  oldUp ifNotNil:[ self _error: #rtErrUserIdAlreadyExists args: { userIdString } ].
  System writeLock: self
    ifDenied:[ self error:'unable to writeLock AllUsers' ]
    ifChanged:[
       System needsCommit ifTrue:[
	 System removeLock: self .
	 self _error: #rtErrAbortWouldLoseData
       ].
       System beginTransaction
    ].
  System writeLock: SystemRepository
    ifDenied:[ System removeLocksForSession.
	       self error:'unable to writeLock SystemRepository' ]
    ifChanged:[
      System needsCommit ifTrue:[
	System removeLocksForSession .
	self _error: #rtErrAbortWouldLoseData
      ].
      System beginTransaction
    ].
  "New GsObjectSecurityPolicy must be committed first!"
  aBoolean ifTrue:[
    newSeg := GsObjectSecurityPolicy newInRepository: SystemRepository .
    newSeg name: userIdString, 'ObjectSecurityPolicy' .
    newSeg worldAuthorization: #read .
    System commitTransaction ifFalse:[
      System removeLocksForSession .
      System abortTransaction.
      self error:'commit of new GsObjectSecurityPolicy failed'
    ].
  ].
  newUp := self addNewUserWithId: userIdString password: passwordString
	    defaultObjectSecurityPolicy: newSeg privileges:  #() inGroups:  #() .
  newSeg ifNotNil:[ newSeg owner: newUp] .
  commitAttempted := true .
  System commitAndReleaseLocks ifFalse:[
    System removeLocksForSession .
    System abortTransaction.
    self error:'commit of new UserProfile failed'
  ].
  newUp
] ensure:[
  commitAttempted ifNil:[ System removeLocksForSession ].
].

]

{ #category : 'Adding' }
UserProfileSet >> addNewUserWithId: userIdString
password: passwordString
defaultObjectSecurityPolicy: anObjectSecurityPolicy [

"Creates a new UserProfile and adds it to AllUsers.  The new UserProfile
 has no privileges, and belongs to no groups.

 anObjectSecurityPolicy is used as the defaultObjectSecurityPolicy of the new UserProfile and
 must be a committed GsObjectSecurityPolicy or nil.
 Pass nil as the value of anObjectSecurityPolicy to have the default GsObjectSecurityPolicy be
 the world-write security policy for applications not using object-level security.
"

^ self addNewUserWithId: userIdString
          password: passwordString
          defaultObjectSecurityPolicy: anObjectSecurityPolicy
          privileges:  #()
          inGroups:  #()

]

{ #category : 'Adding' }
UserProfileSet >> addNewUserWithId: userIdString
password: passwordString
defaultObjectSecurityPolicy: anObjectSecurityPolicy
privileges: anArrayOfStrings
inGroups: aCollectionOfGroupStrings [

"Creates a new UserProfile with the associated characteristics and adds it to
 AllUsers.  Generates an error if the userIdString duplicates the userId
 of any existing element of the receiver. UserIdString must be a single-byte
 string with only alphanumeric characters or underscore, period, dash or space. 
 Returns the new UserProfile.

 In order to log in to GemStone, the user must be authorized to read and write 
 in the specified default GsObjectSecurityPolicy.
 anObjectSecurityPolicy must be a committed GsObjectSecurityPolicy or nil.
 Pass nil as the value of anObjectSecurityPolicy to have the default GsObjectSecurityPolicy be
 the world-write security policy for applications not using object-level security.

 Requires OtherPassword privilege and write access to the DataCurator objectSecurityPolicy.
"

self _validatePrivilege ifTrue:[
  ^ UserProfile newWithUserId: userIdString
                       password: passwordString
                       defaultObjectSecurityPolicy: anObjectSecurityPolicy
                       privileges: anArrayOfStrings
                       inGroups: aCollectionOfGroupStrings
].
^ nil

]

{ #category : 'Deprecated' }
UserProfileSet >> addNewUserWithId: userIdString
password: passwordString
defaultSegment: anObjectSecurityPolicy [
"Deprecated - use method with #defaultObjectSecurityPolicy: instead of #defaultSegment: keyword"

self deprecated: 'UserProfileSet>>addNewUserWithId:password:defaultSegment: deprecated v3.0.
Use #addNewUserWithId:password:defaultObjectSecurityPolicy: instead.'.

^ self addNewUserWithId: userIdString
          password: passwordString
          defaultObjectSecurityPolicy: anObjectSecurityPolicy
          privileges:  #()
          inGroups:  #()

]

{ #category : 'Deprecated' }
UserProfileSet >> addNewUserWithId: userIdString
password: passwordString
defaultSegment: anObjectSecurityPolicy
privileges: anArrayOfStrings
inGroups: aCollectionOfGroupStrings [
"Deprecated - use method with #defaultObjectSecurityPolicy: instead of #defaultSegment: keyword"

self deprecated: 'UserProfileSet>>addNewUserWithId:password:defaultSegment:privileges:inGroups: deprecated v3.0.
Use #addNewUserWithId:password:defaultObjectSecurityPolicy:privileges:inGroups: instead.'.

^self addNewUserWithId: userIdString
	password: passwordString
	defaultObjectSecurityPolicy: anObjectSecurityPolicy
	privileges: anArrayOfStrings
	inGroups: aCollectionOfGroupStrings

]

{ #category : 'Group Membership' }
UserProfileSet >> auditGroups [
"Audit each user and every group, to make sure each
 user's groups and each group's members match.
 Raise an error on the first mismatch and return false."

self do: [:aUser | (aUser auditGroups) ifFalse: [^false] ].
^UserProfileGroup auditGroups.

]

{ #category : 'Enumerating' }
UserProfileSet >> classesDo: aBlock [

"Execute 3-argument block over all classes (plus those referenced by
 its class history) that are contained by all SymbolDictionaries held
 by all users held in this UserProfileSet.  Checks are made to avoid
 redundant processing of classes/symbol dictionaries already scanned.

 Arguments to block are:
    [:aUserProfile :aSymbolDictionary :aClass | ... ]

"

  | coveredSymDics coveredClasses |

  coveredSymDics := IdentitySet new.
  coveredClasses := IdentitySet new.

  self do: [ :aUserProfile |
    aUserProfile symbolList do: [ :aSymbolDictionary |
      (coveredSymDics includes: aSymbolDictionary) ifFalse: [
        coveredSymDics add: aSymbolDictionary.
        aSymbolDictionary valuesDo: [ :namedClass |
          (namedClass isKindOf: Class) ifTrue: [
            namedClass classHistory do: [ :aClass |
              (coveredClasses includes: aClass) ifFalse: [
                coveredClasses add: aClass.
                aBlock value: aUserProfile
                       value: aSymbolDictionary
                       value: aClass ]]]]]]]

]

{ #category : 'Formatting' }
UserProfileSet >> dictionaryNames [

"Returns a formatted String describing each user's symbol list.  For each user,
 the String contains the userId, and the position and name of each Dictionary
 in that user's symbol list.

 This method assumes that each Dictionary in the symbol list contains an
 Association whose value is that Dictionary.  If any Dictionary does not
 contain such an Association, it is represented in the result String as
 '(unnamed Dictionary)'."

"Example use: 'AllUsers dictionaryNames' "

| tempStr aUserPro |

tempStr := String new.

1 to: self size do:[:j|
  aUserPro := self _at: j .
  tempStr add: (Character lf).
  tempStr add: (aUserPro userId asString).
  tempStr add: (Character lf).
  tempStr add: (aUserPro dictionaryNames).
  ].

^tempStr

]

{ #category : 'Querying' }
UserProfileSet >> disableUsersWithOldPasswordEncryption [

"Disables logins for all the elements of the receiver that
 are using the old password encryption scheme.  The old encryption
 scheme was used in versions prior to 3.1.0.

 Generates an error if you do not have the #OtherPassword privilege.

 Answers the number of elements in the receiver for which logins were
 disabled."

| result |
result := 0 .
self do: [:each |
              each usesOldPasswordEncryption
                 ifTrue:[
                   result := result + 1 .
                   each disableWithReason: 'No logins since upgrade' ]
].
^ result

]

{ #category : 'Accessing' }
UserProfileSet >> disallowedPasswords [

"Returns the set of disallowed passwords defined for the receiver."

 ^ disallowedPasswords

]

{ #category : 'Accessing' }
UserProfileSet >> disallowUsedPasswords [

"Returns the value of the disallowUsedPasswords instance variable."

^ (securityBits bitAnd: 1) ~~ 0

]

{ #category : 'Updating' }
UserProfileSet >> disallowUsedPasswords: aBoolean [

"Determines whether the system prevents users from changing their password to
 a password previously used by the same user.  The number of previous passwords
 disallowed may be set by the #numberOfUsedPasswordsDisallowed: method.

 Requires the OtherPassword privilege. "

self _validatePrivilege ifFalse: [ ^ nil ].
aBoolean _validateClass: Boolean .
self disallowUsedPasswords == aBoolean
  ifFalse:[ self _updateSecurityBitsForMask: 1 newState: aBoolean ]

]

{ #category : 'Querying' }
UserProfileSet >> findDisabledUsers [

"Returns a SortedCollection of UserProfiles that are disabled.  The result
 includes users whose accounts have been disabled because their passwords have
 expired, or whose accounts were not used within the interval defined by the
 staleAccountAgeLimit, or who failed to login within the number of tries
 specified by the Stone configuration parameter.

 Generates an error if you do not have OtherPassword privilege."

| result |

result := SortedCollection sortBlock:[:a :b | a userId < b userId ].
self accompaniedBy: result do:[ :res :aUserProfile |
  aUserProfile isDisabled ifTrue:[ res add: aUserProfile ].
].
^ result .

]

{ #category : 'Disk Space Management' }
UserProfileSet >> findObjectsLargerThan: aSize limit: aLimit [

"Searches the symbol list of each user in the receiver for named objects larger
 than aSize.  Returns an Array of the form { { aUserId . aKey . anObject} }
 where aKey is the symbolic representation of anObject such that:

 ((AllUsers userWithId: aUserId)
   resolveSymbol: aKey ) value == anObject

 is true.  For each user in the receiver, the search continues until there are
 no more named objects in the user's symbol list, or until the size of the
 result reaches the specified maximum aLimit.

 Generates an error if an authorization violation occurs."

| test result optimizeSet theUserId aUser aSymList aDict |

aSize _validateClass: SmallInteger.
aLimit _validateClass: SmallInteger.

test := { } .
result := { } .
optimizeSet := IdentitySet new.

1 to: self size do:[:j|
    aUser := self _at: j .
    theUserId := aUser userId.
    aSymList := aUser symbolList .
    1 to: aSymList size do:[:k|
          aDict := aSymList at: k .
          (optimizeSet includesIdentical: aDict) "avoid repeated scans of
                                                  shared Dictionaries"
          ifFalse:
            [ optimizeSet add: aDict.
              aDict associationsDo: [:anAssoc | | theVal |
                theVal := anAssoc value.
                  (theVal size > aSize)
                  ifTrue:
                    [ (result size >= aLimit)
                      ifTrue:
                        [ ^result ].
                      (test includesIdentical: theVal)
                      ifFalse:
                        [ test add: theVal.
                          result add: { theUserId . anAssoc key . theVal }.
                        ].
                   ].  "ifTrue: "
                ].  "aDict do: "
            ].  "ifFalse: "
      ].  "aUser symbolList do:"
  ].  "AllUsers do: "

^result

]

{ #category : 'Querying' }
UserProfileSet >> findProfilesWithAgingPassword [

"Returns a SortedCollection of UserProfiles whose passwords will expire sooner
 than passwordAgeWarning hours from now.

 Generates an error if you do not have OtherPassword privilege."

| result nowDtime endWarningDtime |

self _validatePrivilege ifFalse:[ ^ nil ].
result := SortedCollection sortBlock:[:a :b | a userId < b userId ] .

(((passwordAgeWarning ~~ nil and:[ passwordAgeWarning > 0])
  and:[ passwordAgeLimit ~~ nil]) and:[ passwordAgeLimit > 0])
ifTrue:[
  nowDtime := DateTime now.
  endWarningDtime := nowDtime addHours: passwordAgeWarning .
  self do:[ :aUserProfile | | expireDtime |
    (aUserProfile passwordNeverExpires) ifFalse: [
      (aUserProfile lastPasswordChange ~~ nil) ifTrue:[
        expireDtime := aUserProfile lastPasswordChange addHours: passwordAgeLimit.
        expireDtime <= endWarningDtime ifTrue:[ result add: aUserProfile ].
      ].
    ].
  ].
].
^ result

]

{ #category : 'Accessing' }
UserProfileSet >> maxCharsOfSameType [

"Returns the value of the maxCharsOfSameType instance variable."

^ maxCharsOfSameType

]

{ #category : 'Updating' }
UserProfileSet >> maxCharsOfSameType: aPositiveInteger [

"Sets the value of the maxCharsOfSameType instance variable.
 Requires OtherPassword privilege."

self _validatePrivilege ifTrue:[
  (aPositiveInteger _validateClass: SmallInteger ) ifTrue:[
    aPositiveInteger < 0 ifTrue:[
      ^ aPositiveInteger _error: #rtErrArgNotPositive
    ].
    maxCharsOfSameType := aPositiveInteger
  ]
]

]

{ #category : 'Accessing' }
UserProfileSet >> maxConsecutiveChars [

"Returns the value of the maxConsecutiveChars instance variable."

^ maxConsecutiveChars

]

{ #category : 'Updating' }
UserProfileSet >> maxConsecutiveChars: aPositiveInteger [

"Sets the value of the maxConsecutiveChars instance variable.
 Requires OtherPassword privilege."

self _validatePrivilege ifTrue:[
  (aPositiveInteger _validateClass: SmallInteger ) ifTrue:[
    aPositiveInteger < 0 ifTrue:[
     ^  aPositiveInteger _error: #rtErrArgNotPositive
    ].
    maxConsecutiveChars := aPositiveInteger
  ]
]

]

{ #category : 'Accessing' }
UserProfileSet >> maxPasswordSize [

"Returns the value of the maxPasswordSize instance variable."

^ maxPasswordSize

]

{ #category : 'Updating' }
UserProfileSet >> maxPasswordSize: aPositiveInteger [

"Sets the value of the maxPasswordSize instance variable.
 Requires OtherPassword privilege."

self _validatePrivilege ifTrue:[
  (aPositiveInteger _validateClass: SmallInteger) ifTrue:[
    aPositiveInteger < 0 ifTrue:[
      ^ aPositiveInteger _error: #rtErrArgNotPositive
    ].
    aPositiveInteger > 1024 ifTrue:[
       aPositiveInteger _error: #errArgTooLarge args:{ 1024 }
    ].
    maxPasswordSize := aPositiveInteger
  ]
]

]

{ #category : 'Accessing' }
UserProfileSet >> maxRepeatingChars [

"Returns the value of the maxRepeatingChars instance variable."

^ maxRepeatingChars

]

{ #category : 'Updating' }
UserProfileSet >> maxRepeatingChars: aPositiveInteger [

"Sets the value of the maxRepeatingChars instance variable.
 Requires OtherPassword privilege."

self _validatePrivilege ifTrue:[
  (aPositiveInteger _validateClass: SmallInteger ) ifTrue:[
    aPositiveInteger < 0 ifTrue:[
      ^ aPositiveInteger _error: #rtErrArgNotPositive
    ].
    maxRepeatingChars := aPositiveInteger
  ]
]

]

{ #category : 'Deprecated' }
UserProfileSet >> membersOfGroup: aString [

self deprecated: 'UserProfileSet class >> membersOfGroup: deprecated v3.4.  Replace with
   (UserProfileGroup groupWithName aString) users, or userIdsInGroup:'.
^self userIdsInGroup: aString

]

{ #category : 'Accessing' }
UserProfileSet >> minPasswordSize [

"Returns the value of the minPasswordSize instance variable."

^ minPasswordSize

]

{ #category : 'Updating' }
UserProfileSet >> minPasswordSize: aPositiveInteger [

"Sets the value of the minPasswordSize instance variable.
 Requires OtherPassword privilege."

self _validatePrivilege ifTrue:[
  (aPositiveInteger _validateClass: SmallInteger) ifTrue:[
    aPositiveInteger < 0 ifTrue:[
      ^ aPositiveInteger _error: #rtErrArgNotPositive
    ].
    aPositiveInteger == 0 ifTrue:[
      ^ aPositiveInteger _error: #rtErrArgOutOfRange args:{ 1 . 1024 }
    ].
    minPasswordSize := aPositiveInteger
  ]
]

]

{ #category : 'Accessing' }
UserProfileSet >> numberOfUsedPasswordsDisallowed [

"Answers a SmallInteger between 0 and 65535 which is the number of previous passwords
 which may not be reused by users when changing their password.  A value of zero means
 users may not reuse any previous password.

 To disallow previous passwords, the following code must also be executed:
   AllUsers disallowUsedPasswords: true
"

^ (securityBits bitShift: -5) bitAnd: 65535

]

{ #category : 'Updating' }
UserProfileSet >> numberOfUsedPasswordsDisallowed: aSmallInt [

"Sets the number of previous passwords which may not be reused by users when
 changing their password.  aSmallInt must be a SmallInteger in the range of
 0 to 65535.  A value of zero means users may not reuse any previous password.

 Requires the OtherPassword privilege.

 To disallow previous passwords, the following code must also be executed:
   AllUsers disallowUsedPasswords: true
"

self _validatePrivilege ifFalse: [ ^ nil ].
(aSmallInt _validateClass: SmallInteger) ifFalse: [ ^ nil].
((aSmallInt < 0) or:[ aSmallInt > 65535 ])
     ifTrue:[ ^ aSmallInt  _error: #rtErrArgOutOfRange args:{ 0 . 65535 } ] .
self numberOfUsedPasswordsDisallowed == aSmallInt
  ifFalse:[securityBits := (securityBits bitAnd: 31) bitOr: (aSmallInt bitShift: 5) ]

]

{ #category : 'Accessing' }
UserProfileSet >> passwordAgeLimit [

"Returns the value of the passwordAgeLimit instance variable. Units is hours ."

^ passwordAgeLimit

]

{ #category : 'Updating' }
UserProfileSet >> passwordAgeLimit: numberOfHours [

"Sets the global, default password age limit for UserProfiles.

 If numberOfHours is greater than zero, the password of a UserProfile
 will expire after the specified number of hours since the password was
 last changed if the following conditions are all true:

 1. The UserProfile is not a reserved, system UserProfile.  The special system
    UserProfiles are: SystemUser, SymbolUser, DataCurator, Nameless, and GcUser.

 2. The passwordAgeLimit has not been overridden or disabled in the UserProfile.
    See the method UserProfile>>passwordAgeLimit: for more information.

 The argument numberOfHours must be a SmallInteger or a Float and must be at
 least zero and at most 536870911.

 Applies only to UserProfiles which use GemStone authentication.

 Requires OtherPassword privilege."

 | message |

  self _validatePrivilege ifFalse:[ ^ nil ].
  (numberOfHours _validateClasses:{ SmallInteger . Float }) ifFalse:[ ^ nil ] .
  numberOfHours < 0 ifTrue:[
    ^ numberOfHours _error: #rtErrArgNotPositive
  ].
  numberOfHours > 536870911 ifTrue:[
    ^ numberOfHours _error: #errArgTooLarge args:{ 536870911 }
  ].

  passwordAgeLimit := numberOfHours .

  message :=
'      passwordAgeLimit for AllUsers changed to '
   , numberOfHours asString , ' hours.' .
  self addMsgToSecurityLog: message .

]

{ #category : 'Accessing' }
UserProfileSet >> passwordAgeWarning [

"Returns the value of the passwordAgeWarning instance variable."

^ passwordAgeWarning

]

{ #category : 'Updating' }
UserProfileSet >> passwordAgeWarning: numberOfHours [

"Sets the global, default password age warning for UserProfiles.

 If numberOfHours is greater than zero, a warning indicating the password for
 the UserProfile will be issued at login time if all of the following conditions
 are true:

 1. The UserProfile is not a reserved, system UserProfile.  The special system
    UserProfiles are: SystemUser, SymbolUser, DataCurator, Nameless, and GcUser.

 2. A non-zero passwordAgeLimit has been set which applies to the UserProfile.

 3. The remaining number of hours before the password for the UserProfile expirees is
    less than or equal to numberOfHours.

 The warning issued at login time is in the form of a non-fatal error raised immediately
 after the login succeeds.

 The argument numberOfHours must be a SmallInteger or a Float and must be at
 least zero and at most 536870911.

 Requires OtherPassword privilege."

self _validatePrivilege ifFalse:[ ^ nil ].
(numberOfHours _validateClasses:{ SmallInteger . Float }) ifFalse:[ ^ nil ].
numberOfHours < 0 ifTrue:[
    ^ numberOfHours _error: #rtErrArgNotPositive
] .
numberOfHours > 536870911 ifTrue:[
  ^ numberOfHours _error: #errArgTooLarge args:{ 536870911 }
] .

passwordAgeWarning := numberOfHours

]

{ #category : 'Accessing' }
UserProfileSet >> passwordRequiresDigit [

"Answers true if passwords changed by users are required to contain
 at least one digit"

^ (securityBits bitAnd: 8) ~~ 0

]

{ #category : 'Updating' }
UserProfileSet >> passwordRequiresDigit: aBoolean [

"Enables or disables the requirement that new passwords contain at least one
 digit.

 Requires the OtherPassword privilege."

self _validatePrivilege ifFalse: [ ^ nil ].
aBoolean _validateClass: Boolean .
self passwordRequiresDigit == aBoolean
  ifFalse:[ self _updateSecurityBitsForMask: 8 newState: aBoolean ].

]

{ #category : 'Accessing' }
UserProfileSet >> passwordRequiresLowercase [

"Answers true if passwords changed by users are required to contain
 at least one lower case character."

^ (securityBits bitAnd: 4) ~~ 0

]

{ #category : 'Updating' }
UserProfileSet >> passwordRequiresLowercase: aBoolean [

"Enables or disables the requirement that new passwords contain at least one
 lower case character.

 Requires the OtherPassword privilege."

self _validatePrivilege ifFalse: [ ^ nil ].
aBoolean _validateClass: Boolean .
self passwordRequiresLowercase == aBoolean
  ifFalse:[ self _updateSecurityBitsForMask: 4 newState: aBoolean ].

]

{ #category : 'Accessing' }
UserProfileSet >> passwordRequiresSymbol [

"Answers true if passwords changed by users are required to contain
 at least one Symbol character.  In the scope of passwords, a Symbol
 character is any character that answers false to the message
 #isAlphaNumeric ."

^ (securityBits bitAnd: 16) ~~ 0

]

{ #category : 'Updating' }
UserProfileSet >> passwordRequiresSymbol: aBoolean [

"Enables or disables the requirement that new passwords contain at least one
 symbol character.  In the scope of passwords, a Symbol character is any
 character that answers false to the message #isAlphaNumeric .

 Requires the OtherPassword privilege."

self _validatePrivilege ifFalse: [ ^ nil ].
aBoolean _validateClass: Boolean .
self passwordRequiresSymbol == aBoolean
  ifFalse:[ self _updateSecurityBitsForMask: 16 newState: aBoolean ].

]

{ #category : 'Accessing' }
UserProfileSet >> passwordRequiresUppercase [

"Answers true if passwords changed by users are required to contain
 at least one upper case character."

^ (securityBits bitAnd: 2) ~~ 0

]

{ #category : 'Updating' }
UserProfileSet >> passwordRequiresUppercase: aBoolean [

"Enables or disables the requirement that new passwords contain at least one
 upper case character.

 Requires the OtherPassword privilege."

self _validatePrivilege ifFalse: [ ^ nil ].
aBoolean _validateClass: Boolean .
self passwordRequiresUppercase == aBoolean
  ifFalse:[ self _updateSecurityBitsForMask: 2 newState: aBoolean ]

]

{ #category : 'Illegal Operations' }
UserProfileSet >> remove: aUserProfile [

"Disallowed."
self shouldNotImplement: #remove:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> remove: aUserProfile ifAbsent: aBlock [

"Disallowed."
self shouldNotImplement: #remove:ifAbsent:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> remove: aUserProfile otherwise: something [

"Disallowed."
self shouldNotImplement: #remove:otherwise:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> removeAll: aCollection [

"Disallowed."
self shouldNotImplement: #removeAll:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> removeAllPresent: aCollection [

"Disallowed."
self shouldNotImplement: #removeAllPresent:

]

{ #category : 'Removing' }
UserProfileSet >> removeAndCleanup: aUserProfile [

"Remove a UserProfile from the receiver.
 Also transfers ownership of any GsObjectSecurityPolicy objects owned
 by the UserProfile being removed to the UserProfile for the current session.
 Also adds a new DeletedUserProfile to the AllDeletedUsers collection
 in Globals.

 Requires write access to the GsObjectSecurityPolicy objects for both DataCurator
 and the UserProfile object to be removed.  Also requires the #OtherPassword
 privilege."

^ self removeAndCleanup: aUserProfile migrateSecurityPoliciesTo: System myUserProfile

]

{ #category : 'Removing' }
UserProfileSet >> removeAndCleanup: aUserProfile migrateSecurityPoliciesTo: anotherUserProfile [


"Remove a UserProfile from the receiver.
 Also transfers ownership of any GsObjectSecurityPolicy objects owned
 by the UserProfile being removed to anotherUserProfile.  Also adds a new
 DeletedUserProfile to the AllDeletedUsers collection in Globals.

 Requires write access to the GsObjectSecurityPolicy objects for both DataCurator
 and the UserProfile object to be removed.  Also requires the #OtherPassword
 privilege."

<primitive: 2001>
| result prot |
prot := System _protectedMode .
[ | userId oldval |
  self _validatePrivilege ifFalse:[ ^ nil ].
  "protect against removal of special users"
  userId := aUserProfile userId.
  (UserProfile isSpecialUserId: userId) ifTrue:[ userId _error: #rtErrRemoveSpecialUser ].

  oldval := userIdDictionary at: userId otherwise: nil .
  (oldval ~~ aUserProfile) ifTrue:[
    ArgumentError signal:'(AllUsers at: aUserProfile userId) not identical to aUserProfile'
  ].
  oldval := userIdDictionary removeKey: userId otherwise: nil .
  (oldval ~~ aUserProfile) ifTrue:[
    ArgumentError signal:'(AllUsers at: aUserProfile userId) not identical to aUserProfile (2)'
  ].
  oldval ifNotNil:[
      aUserProfile removeFromAllGroups .
      self _removeSecurityDataFor: aUserProfile .
      self _migrateSecurityPoliciesFrom: aUserProfile to: anotherUserProfile .
      self _addToDeletedUsers: aUserProfile .
      ^ aUserProfile
  ].
  result := super remove: aUserProfile ifAbsent:[ nil ] .
] ensure:[
  prot _leaveProtectedMode
].
^ result

]

{ #category : 'Removing' }
UserProfileSet >> removeAndCleanupUserWithId: aUserId ifAbsent: aBlock [

"Remove a UserProfile whose userId is equal to aUserId from the receiver.
 Also transfers ownership of any GsObjectSecurityPolicy objects owned
 by the UserProfile being removed to the UserProfile for the current session.
 Also adds a new DeletedUserProfile to the AllDeletedUsers collection
 in Globals.

 Executes aBlock if the UserProfile could not be found.

 Requires write access to the GsObjectSecurityPolicy objects for both DataCurator
 and the UserProfile object to be removed.  Also requires the #OtherPassword
 privilege."

| aUserProfile |
aUserProfile := self userWithId: aUserId ifAbsent:[ nil ] .
aUserProfile ifNil:[ ^ aBlock value ] .

^ self removeAndCleanup: aUserProfile migrateSecurityPoliciesTo: System myUserProfile

]

{ #category : 'Removing' }
UserProfileSet >> removeAndCleanupUserWithId: aUserId migrateSecurityPoliciesToUserWithId: anotherId ifAnyAbsent: aBlock [

"Remove a UserProfile whose userId is equal to aUserId from the receiver.
 Also transfers ownership of any GsObjectSecurityPolicy objects owned
 by the UserProfile being removed to another UserProfile whose userId is equal
 to anotherId.  Also adds a new DeletedUserProfile to the AllDeletedUsers
 collection in Globals.

 Executes aBlock and returns if either UserProfile could not be found.

 Requires write access to the GsObjectSecurityPolicy objects for both DataCurator
 and the UserProfile object to be removed.  Also requires the #OtherPassword
 privilege."

| aUserProfile anotherUserProfile |
aUserProfile := self userWithId: aUserId ifAbsent:[ nil ] .
aUserProfile ifNil:[ ^ aBlock value ] .

anotherUserProfile := self userWithId: anotherId ifAbsent:[ nil ] .
anotherUserProfile ifNil:[ ^ aBlock value ] .

^ self removeAndCleanup: aUserProfile migrateSecurityPoliciesTo: anotherUserProfile

]

{ #category : 'Group Membership' }
UserProfileSet >> removeGroup: aGroupString [

"Removes all the users in the receiver from the group represented by
 aGroupString.  If the current session does not have the authorizations required
 for this operation, raises an error.
 Requires OtherPassword privilege."

self _validatePrivilege ifTrue:[
  self accompaniedBy: aGroupString do:[ :grpStr :aUserProfile | aUserProfile removeGroup: grpStr ].
]

]

{ #category : 'Illegal Operations' }
UserProfileSet >> removeIdentical: aUserProfile [

"Disallowed."
self shouldNotImplement: #removeIdentical:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> removeIdentical: aUserProfile ifAbsent: someBlock [

"Disallowed."
self shouldNotImplement: #removeIdentical:ifAbsent:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> removeIdentical: aUserProfile otherwise: something [

"Disallowed."
self shouldNotImplement: #removeIdentical:otherwise:

]

{ #category : 'Illegal Operations' }
UserProfileSet >> removeIfPresent: aUserProfile [

"Disallowed."
self shouldNotImplement: #removeIfPresent:

]

{ #category : 'Class Membership' }
UserProfileSet >> species [

"Returns a Class, an instance of which should be used as the result of
 collect: or other projections applied to the receiver."

^ IdentitySet

]

{ #category : 'Class Membership' }
UserProfileSet >> speciesForCollect [

"Returns a Class, an instance of which should be used as the result of
 collect: or other projections applied to the receiver."

^ Array

]

{ #category : 'Accessing' }
UserProfileSet >> staleAccountAgeLimit [

"Returns the value of the staleAccountAgeLimit instance variable."

^ staleAccountAgeLimit

]

{ #category : 'Updating' }
UserProfileSet >> staleAccountAgeLimit: numberOfHours [

"Sets the global, default stale account age limit for UserProfiles.

 If numberOfHours is greater than zero, the UserProfile will be disabled
 if all of the following conditions are true:

 1. The UserProfile is not a reserved, system UserProfile.  The special system
    UserProfiles are: SystemUser, SymbolUser, DataCurator, Nameless, and GcUser.

 2. The staleAccountAgeLimit has not been overridden or disabled in the UserProfile.
    See the method UserProfile>>staleAccountAgeLimit: for more information.

 3. The last successful login for the UserProfile occurred more than numberOfHours in the
    past.  Note that the staleAccountAgeLimit will never prevent the first login to a
    newly-created UserProfile.

 The argument numberOfHours must be a SmallInteger or a Float and must be at
 least zero and at most 536870911.

 Applies only to UserProfiles which use GemStone authentication.

 Requires OtherPassword privilege."

 | message |
  self _validatePrivilege ifFalse:[ ^ nil ].
  (numberOfHours _validateClasses:{ SmallInteger . Float })  ifFalse:[ ^ nil ].
  numberOfHours < 0 ifTrue:[
    ^ numberOfHours _error: #rtErrArgNotPositive
    ].
  numberOfHours > 536870911 ifTrue:[
    ^ numberOfHours _error: #errArgTooLarge args:{ 536870911 }
    ].

  staleAccountAgeLimit := numberOfHours .

  message :=
'      staleAccountAgeLimit for AllUsers changed to '
   , numberOfHours asString , ' hours.' .
  self addMsgToSecurityLog: message .

]

{ #category : 'Group Membership' }
UserProfileSet >> userIdsInGroup: aString [

"Returns an IdentitySet containing the userId for each member of the
 group with the sepcified name.
 If the group contains no members, returns an empty IdentitySet.
 Generates an error if aString is not a kind of String, or if aString is not
 defined in the global object AllGroups."

|  theGroup |

theGroup := UserProfileGroup _validateGroupString: aString .
^ theGroup userIds

]

{ #category : 'Group Membership' }
UserProfileSet >> usersInGroup: aGroupString [

"Returns all the elements of the receiver that are in the group represented by
 aGroupString.  If the current session does not have the authorizations required
 for this operation, raises an error."

| theGroup |
theGroup := UserProfileGroup _validateGroupString: aGroupString .
^ self * theGroup

]

{ #category : 'Querying' }
UserProfileSet >> usersWithOldPasswordEncryption [

"Returns all the elements of the receiver that are using the old
 password encryption scheme.  The old encryption scheme was
 used in versions prior to 3.1.0.

 Generates an error if you do not have the #OtherPassword privilege."


^ self select: [:each | each usesOldPasswordEncryption]

]

{ #category : 'Querying' }
UserProfileSet >> userWithId: aString ifAbsent: aBlock [

"Searches the receiver for a UserProfile whose userId is equal to aString, and
 returns that UserProfile.  Evaluates the argument aBlock if no userId
 is equal to aString."

"Example: (AllUsers userWithId: 'DataCurator') "

| aUserProfile aStr |
aStr := aString asString .
userIdDictionary ifNotNil:[
  ^ userIdDictionary at: aStr ifAbsent: aBlock
  ].
aUserProfile:=
   self detect: [:aUserPro | aUserPro.userId = aStr] ifNone: aBlock .

^aUserProfile "Should only be one as UserId are unique"

]
{ #category : 'Adding' }
UserProfileSet >> addAll: aCollection [
    ^ self shouldNotImplement: #addAll:
]
