Extension { #name : 'DateAndTime' }

{ #category : 'Private' }
DateAndTime class >> _scaledDecimal6mantissa: aNumber [
  "Return a mantissa of a ScaledDecimal with scale 6 and equal to aNumber 
   if aNumber has precision no greater than 0.000001, otherwise return nil"
  | f |
  aNumber _isSmallInteger ifTrue:[ ^ aNumber * 1000000 ].
  aNumber _isScaledDecimal ifTrue:[ | sc |
     (sc := aNumber scale) == 6 ifTrue:[ ^ aNumber mantissa ].
     sc < 6 ifTrue:[ ^ (aNumber asScaledDecimal: 6) mantissa  ]
            ifFalse:[ ^ nil ]
  ].
  f := aNumber asFloat .
  ((f truncateTo: 0.000001) - f) abs < 0.000001 ifTrue:[
     ^ (aNumber asScaledDecimal: 6) mantissa 
  ].
  ^ nil
]

{ #category : 'Private' }
DateAndTime class >> _smallFromMicrosecs: aSmallInteger offset: offsetSmallInt [
  "If receiver is DateAndTime or SmallDateAndTime ,
   returns a SmallDateAndTime if args within range. 
   Otherwise returns nil.
   aSmallInteger is positive offset from 2001 in microseconds.
   offsetSmallInt is in seconds ."
  <primitive: 1112>
   ^ self _primitiveFailed: #_smallFromMicrosecs:offset:

]

{ #category : 'other' }
DateAndTime class >> clockPrecision [
"  Answer a <Duration> such that after that period of time passes,
   #now is guaranteed to give a different result.
"
	^ Duration seconds: 1.0e-6  

]

{ #category : 'Migration' }
DateAndTime class >> migrateNew [
  "Use instance creation protocol so SmallDateAndTime creation works"
  self shouldNotImplement: #migrateNew

]

{ #category : 'Instance Creation' }
DateAndTime class >> now [ 
  " Answer a SmallDateAndTime if possible, or a DateAndTime, representing the current 
   date and time in the local time specified by the implementation."
  | posixMicrosec us res ofs |
  posixMicrosec := System _timeGmtMicroSeconds .
  us := posixMicrosec - 978307200000000 . "microsecs from 2001"
	res := self _smallFromMicrosecs: us offset: 0 .
	res ifNil:[  | sd |
    sd := ScaledDecimal numerator: us denominator: 1000000 scale: 6 .
    ^ self basicNew _secondsUTC: sd offsetSeconds: nil 
  ].
	ofs := self _zoneOffsetAtPosix: posixMicrosec // 1000000 .
	res := self _smallFromMicrosecs: us offset: ofs . 
	res ifNotNil:[ ^ res ] .
  (res := self basicNew) _secondsUTC: us asFloat / 1000000.0 offsetSeconds: nil .
  ^ res.
]

{ #category : 'Instance Creation' }
DateAndTime class >> nowWithScale: subsecondResolution [
   self deprecated: 'DateAndTime class>>nowWithScale: deprecated v3.6. use #now '.
   ^ self now
]

{ #category : 'Instance Creation' }
DateAndTime class >> posixSeconds: aNumber offset: aDuration [
  "Return a SmallDateAndTime if possible without loosing precision.
   aNumber is UTC time in seconds since 1970, 
   aDuration is offset in seconds from UTC specifying a timezone. "
  (self _scaledDecimal6mantissa: aNumber) ifNotNil:[:m | | res us ofs |
    us := m - 978307200000000 .
    ofs := aDuration ifNotNil:[ aDuration asSeconds ]
          ifNil:[ res := self _smallFromMicrosecs: us offset: 0 .
                  res ifNil:[ ^ self basicNew _posixSeconds: aNumber offset: nil ].
                  self _zoneOffsetAtUTC: res ].
    res := self _smallFromMicrosecs: us offset: ofs .
    res ifNil:[ ^ self basicNew _posixSeconds: aNumber offset: aDuration].
    ^ res
  ].
  ^ self basicNew _posixSeconds: aNumber offset: aDuration 

]

{ #category : 'Instance Creation' }
DateAndTime class >> secondsLocal: aNumber offset: aDuration [
  "Return a SmallDateAndTime if possible without loosing precision.
   aNumber is local time in seconds since 2001, 
   aDuration is offset in seconds from UTC specifying a timezone. "
  (self _scaledDecimal6mantissa: aNumber) ifNotNil:[:us | | res ofs |
    ofs := aDuration ifNotNil:[ aDuration asSeconds ]
            ifNil:[ res := self _smallFromMicrosecs: us offset: 0 .
                    res ifNil:[ ^ self basicNew _secondsLocal: aNumber offset: nil ].
                    self _zoneOffsetAtLocal: res ].
    res := self _smallFromMicrosecs: us - (ofs * 1000000) offset: ofs .
    res ifNil:[ ^ self basicNew _secondsLocal: aNumber offset: aDuration ].
    ^ res
  ].
  ^ self basicNew _secondsLocal: aNumber offset: aDuration 

]

{ #category : 'other' }
DateAndTime class >> secondsSince2001 [

	"^DateTime now asSecondsGmt - 3155760000."
	^System _timeGmtFloat - 978307200.0  .

]

{ #category : 'Instance Creation' }
DateAndTime class >> secondsUTC: aNumber offsetSeconds: anOffset [
  "Return a SmallDateAndTime if possible without loosing precision.
   aNumber is UTC time in seconds since 2001, 
   aDuration is offset in seconds from UTC specifying a timezone. "
  (self _scaledDecimal6mantissa: aNumber) ifNotNil:[:us | | res ofs |
    ofs := anOffset ifNil:[
              res := self _smallFromMicrosecs: us offset: 0 .
              res ifNil:[ ^ self basicNew _secondsUTC: aNumber offsetSeconds: nil ].
              self _zoneOffsetAtUTC: res .
    ].
    res := self _smallFromMicrosecs: us offset: ofs .
    res ifNil:[ ^ self basicNew _secondsUTC: aNumber offsetSeconds: anOffset ].
    ^ res
  ].
  ^ self basicNew _secondsUTC: aNumber offsetSeconds: anOffset 
]

{ #category : 'Deprecated' }
DateAndTime class >> setDefaultScale: anIntOrNil [
   "anIntOrNil should be nil or a SmallInteger in the range 0 .. 10 .
    It specifies the scale of a ScaledDecimal. "

   self deprecated: 'DateAndTime class>>setDefaultScale: deprecated v3.6. All DateAndTimes/SmallDateAndTimes have 6 digits of scale; this method has no effect'.
   DefaultScale := anIntOrNil ifNotNil: [anIntOrNil asInteger].

]

{ #category : 'New Indexing Comparison' }
DateAndTime >> _classSortOrdinal [

^ 70

]

{ #category : 'private' }
DateAndTime >> _topazAsString [
 ^ self asStringMs
]

{ #category : 'Converting' }
DateAndTime >> asCanonicalForm [
	"Answer self, or, if I am a DateAndTime with an equivalent
	SmallDateAndTime, answer that SmallDateAndTime."

	| cls res |
	(cls := self class) ~~ SmallDateAndTime
		ifTrue: [ 
			cls == DateAndTime
				ifTrue: [ 
					res := cls secondsLocal: self asSeconds offset: self offset.
					res class == SmallDateAndTime
						ifTrue: [ ^ res ] ] ].
	^ self
]

{ #category : 'private' }
DateAndTime >> asStringMs [
  "prints with millisecond resolution"
| strm |
strm := AppendStream on: String new .
self printLocalWithSubsecondDigits: 3 on: strm .
^ strm contents

]

{ #category : 'private' }
DateAndTime >> asStringUs [
  "prints with microsecond resolution"
| strm |
strm := AppendStream on: String new .
self printLocalWithSubsecondDigits: 6 on: strm .
^ strm contents

]

{ #category : 'initialize-release' }
DateAndTime >> currentTimeZone [

	^TimeZone current.

]

{ #category : 'private' }
DateAndTime >> printLocalMsOn: aStream [
    "print the receiver on aStream, omitting the timezone offset and using space instead of T.
     printed time is rounded to 0.001 second."
  self printLocalWithSubsecondDigits: 3 on: aStream.

]

{ #category : 'private' }
DateAndTime >> printLocalOn: aStream [
  "print the receiver on aStream, omitting the timezone offset and using space instead of T"
  | anArray |
  anArray := self asFloatParts.
  self printDateFrom: anArray on: aStream.
  aStream nextPut: $ .
  self printTimeFrom: anArray on: aStream.
]

{ #category : 'private' }
DateAndTime >> printLocalWithSubsecondDigits: anInteger on: aStream [
    "print the receiver on aStream, omitting the timezone offset and using space instead of T.
     Print the given number of subsecond digits, rounding or printing additional trailing 0s."
 | arr |
  anInteger == 0 ifTrue:[ arr := self asIntegerParts ] 
    ifFalse:[ arr := self asFloatParts .
      anInteger = 6 ifFalse:[
      arr at: 7 put:(ScaledDecimal for: (arr at: 7) scale: anInteger)] ].
  self printDateFrom: arr on: aStream.
  aStream nextPut: $ .
  self printTimeFrom: arr on: aStream.
]
