"
  A Duration represents a length of time, in integer days, hours, minutes, 
  with seconds that can be integer or fractional of artibrary presicion. 

  (from ANSI).
"
Class {
	#name : 'Duration',
	#superclass : 'Magnitude',
	#instVars : [
		'seconds'
	],
	#category : nil
}

{ #category : 'Instance Creation' }
Duration class >> days: days hours: hours minutes: minutes seconds: seconds [
"  Answer a <Duration> of the number of days, hours, minutes, and seconds. If any of the operands
   are negative, the result is smaller by that number of days, hours, minutes, or seconds as
   appropriate.
   Parameters
      days <integer> unspecified
      hours <integer> unspecified
      minutes <integer> unspecified
      seconds <number> unspecified"

	^self seconds: days * 24 + hours * 60 + minutes * 60 + seconds.

]

{ #category : 'other' }
Duration class >> seconds: seconds [
" Answer a <Duration> which is <seconds> in length. If seconds is negative,
  answer a <Duration> that is abs (seconds) less than Duration zero."

  | res |
  (res := self basicNew) _seconds: seconds.
  ^ res

]

{ #category : 'Instance Creation' }
Duration class >> zero [
" Answer a <Duration> representing a length of no time."

	^self seconds: 0.

]

{ #category : 'Instance Creation' }
Duration class >> new [
	^self seconds: 0.

]
 

{ #category : 'Private' }
Duration >> _fractionFormat [

^ #( 5 3 false )

]

{ #category : 'Private' }
Duration >> _seconds: aNumber [

	seconds := aNumber.

]

{ #category : 'Arithmetic' }
Duration >> - operand [
"  Answer a <Duration> whose value is the result of subtracting operand from the receiver.
   Parameters
     operand <Duration> unspecified"

	^self class seconds: seconds - operand asSeconds.

]

{ #category : 'Arithmetic' }
Duration >> * operand [
"  Answer a <Duration> that is the result of multiplying the receiver by operand.
   Parameters
      operand <number> unspecified"

	^self class seconds: seconds * operand.

]

{ #category : 'Arithmetic' }
Duration >> / operand [
"  If operand is a <number> answer a new <Duration> whose value is the result of dividing the
   receiver by operand. If operand equals zero the ZeroDivide exception is signaled.
   If operand is a <Duration> answer a <number> whose value is the result of dividing the receiver
   by operand. If operand is <Duration factory> #zero the ZeroDivide exception is signaled.
   Parameters
     operand<number> | <Duration> unspecified
   Return Values
     <Duration> unspecified | <number> unspecified"

	^(operand isKindOf: Duration)
		ifTrue: [seconds / operand asSeconds]
		ifFalse: [Duration seconds: seconds / operand].

]

{ #category : 'Arithmetic' }
Duration >> + operand [
"  Answer a <Duration> whose value is the result of adding the receiver and operand.
   Parameters
     operand <Duration> unspecified"

	^self class seconds: seconds + operand asSeconds.

]

{ #category : 'Comparing' }
Duration >> < operand [
"  Answer true if operand represents a <Duration> that is larger than the receiver. Answer false
   otherwise.
   Parameters
     operand <Duration> unspecified"

	^seconds < operand asSeconds.

]

{ #category : 'Comparing' }
Duration >> = comparand [
"  Answer true if the comparand is a <Duration> representing the same length of time as the
   receiver. Answer false otherwise.
   Parameters
     comparand <Object> uncaptured"

	^(comparand isKindOf: Duration) and: [seconds = comparand asSeconds].

]

{ #category : 'Arithmetic' }
Duration >> abs [
"  If the receiver is greater than or equal to Duration zero answer a <Duration> which is
   equal to the receiver. Otherwise answer a <Duration> which has the same magnitude as the
   receiver but the opposite sign."

	^0 <= seconds
		ifTrue: [self]
		ifFalse: [self class seconds: seconds negated].

]

{ #category : 'Converting' }
Duration >> asSeconds [
"  Answer the total number of seconds in the length of time represented by the receiver including any
   fractional part of a second. If the receiver is less than <Duration factory> #zero then the result will
   be less than 0."

	^seconds.

]

{ #category : 'Converting' }
Duration >> asMilliseconds [
  "Return the number of milliseconds in the length of time represented by the receiver"
  ^ seconds * 1000 
] 

{ #category : 'Formatting' }
Duration >> asString [
  ^ self printString

]

{ #category : 'Accessing' }
Duration >> days [
"  Answer the number of complete days in the receiver. If the receiver is less than Duration zero 
   then the result will be less than or equal to 0."

	^ (seconds abs // 60) // 60 // 24 * seconds sign.

]

{ #category : 'Comparing' }
Duration >> hash [

	^seconds hash.

]

{ #category : 'Accessing' }
Duration >> hours [
"  Answer an <integer> between -23 and 23 inclusive that represents the number of complete hours
   in the1G receiver, after the number of complete days has been removed. If the receiver is less than
   Duration zero then the result will be less than or equal to 0."

	^ (seconds abs // 60) // 60 \\ 24 * seconds sign.

]

{ #category : 'Accessing' }
Duration >> minutes [
"  Answer an <integer> between -59 and 59 inclusive that represents the number of complete
   minutes in the receiver, after the number of complete days and hours have been removed. If the
   receiver is less than <Duration factory> #zero then the result will be less than or equal to 0."

	^ ( seconds abs // 60) \\ 60 * seconds sign.

]

{ #category : 'Arithmetic' }
Duration >> negated [
"  Answer a <Duration> which is of the same magnitude but opposite sign as the receiver."

	^self class seconds: seconds negated.

]

{ #category : 'Testing' }
Duration >> negative [
"  Answer true if the receiver is less than <Duration factory> #zero, false otherwise."

	^seconds < 0.

]

{ #category : 'Testing' }
Duration >> positive [
"  Answer true if the receiver is greater than or equal to the <Duration factory> #zero, false
   otherwise."

	^0 <= seconds.

]

{ #category : 'Formatting' }
Duration >> printOn: aStream [
"  Write a description of the receiver on aStream that is formatted as
     [-]D:HH:MM:SS[.SSS] 
   where
     -   is a minus sign if the receiver represents a length of time going from the future into
         the past,
     D   is the number of complete days with leading zeros to fill one place,
     HH  is the number of complete hours with leading zeros to fill two places,
     MM  is the number of complete minutes with leading zeros to fill two places,
     SS  is the number of complete seconds with leading zeros to fill two places, and
    .SSS is the fractional part of the number of seconds, if any, rounded and 0-padded to three places.
"

  | secs mins hours absSecs |

  seconds < 0 ifTrue: [aStream nextPut: $-].
  self days abs printOn: aStream.
  aStream nextPut: $:.
  (hours := self hours abs) < 10 ifTrue: [aStream nextPut: $0].
  hours printOn: aStream.
  aStream nextPut: $:.
  (mins := self minutes abs) < 10 ifTrue: [aStream nextPut: $0].
  mins printOn: aStream.
  aStream nextPut: $:.
  (absSecs := (secs := self seconds) asInteger abs) < 10 ifTrue: [aStream nextPut: $0].
  absSecs printOn: aStream.
  (secs _isInteger) ifFalse: [
    | float |
    0 < (float := (secs abs - absSecs) asFloat) ifTrue: [ | string |
      aStream nextPut: $. .
      string := ((float roundTo: 0.001) min: 0.999) asStringUsingFormat: self _fractionFormat.
      aStream nextPutAll: (string copyFrom: 3 to: string size).
    ].
  ].

]

{ #category : 'Formatting' }
Duration >> printOnHoursMinutes: aStream [

	aStream nextPut: (seconds negative ifTrue: [$-] ifFalse: [$+]).
	self hours abs < 10 ifTrue: [aStream nextPut: $0].
	self hours abs printOn: aStream.
	aStream nextPut: $:.
	self minutes abs < 10 ifTrue: [aStream nextPut: $0].
	self minutes abs printOn: aStream.

]

{ #category : 'Accessing' }
Duration >> seconds [
"  Answer a <number> strictly greater than -60 and strictly less than 60
   that represents the number of seconds in the receiver, after the complete days,
   hours, and minutes have been removed.
   If the receiver is less than Duration zero then the result
   will be less than or equal to 0."

  | s |
  ^(s := seconds) abs \\ 60 * s sign.

]
