"
This provides GemStone/S overrides and extensions to the
ANSI TimeZone behavior.
"
Class {
	#name : 'TimeZone',
	#superclass : 'TimeZoneInfo',
	#instVars : [
		'standardPrintString',
		'dstPrintString',
		'dstStartTimeList',
		'dstEndTimeList',
		'secondsForDst',
		'secondsFromGmt',
		'yearStartDst'
	],
	#category : nil
}

{ #category : 'instance creation' }
TimeZone class >> _fromPath: aString onClient: onClient [
  | f |
  f := GsFile open: aString mode: 'rb' onClient: onClient .
  f ifNil:[
     | errStr |
     errStr := aString asString,' ' , 'does not specify a TimeZone file; '.
     errStr add: (onClient 
        ifTrue: [GsFile lastErrorString] 
        ifFalse: [GsFile serverErrorString]).
     Error signal: errStr
     ].
  ^self fromGsFile: f .
]

{ #category : 'Initialization' }
TimeZone class >> _initialize3 [
  | hist |
  default ifNil:[ |zone osZone |
    "executed during slowfilein only"
    osZone := self fromOS .
    zone := self fromGemPath: '$GEMSTONE/pub/timezone/usr/share/zoneinfo/America/Los_Angeles' .
    osZone ifNotNil:[ 
      zone asString = osZone asString ifFalse:[  "bad timezone on build host"
        Error signal:'inconsistent timezones zone ', zone asString ,' osZone ', osZone asString
      ].
    ] ifNil:[  "AIX slowfilein takes this path"].
    "init classVars in superclass"
    default := zone  .
    cache ifNil:[ cache := SymbolDictionary new ].
    self cacheAt: #'America/Los_Angeles' put: zone.
    GsFile gciLogServer:'Set TimeZone default to ', zone asString .
  ] ifNotNil:[:z |
    GsFile gciLogServer:'TimeZone default is already ', z asString .
  ].
  (Globals at: #TimeZone) == self ifFalse:[ Error signal:'bad TimeZone binding'].
  hist := (ObsoleteClasses at: #ObsoleteTimeZone) classHistory.
  (hist size == 1 and:[ self classHistory size == 1]) ifTrue:[
    hist add: (ObsoleteClasses at: #ObsoleteTimeZone2) .
    hist add: self .
    (ObsoleteClasses at: #ObsoleteTimeZone2) classHistory: hist .
    self classHistory: hist .
  ].
]

{ #category : 'private' }
TimeZone class >> _isTimeZoneFile: aPath [

(GsFile openReadOnServer: aPath) ifNotNil:[:f | | str |
  str := String new .
  f read: 4 into: str .
  f close .
  str = 'TZif' ifTrue:[ ^ true ].
].
^ false

]

{ #category : 'private' }
TimeZone class >> _olsonPath [
  | path list suffix sufSize |
  path := (GsFile _expandEnvVariable:'GEMSTONE' isClient: false) , '/pub/timezone/usr/share'.
  " use contentsAndTypesOfDirectory on parent of target directory so as to expand symlinks"
  list := GsFile contentsAndTypesOfDirectory: path onClient: false .
  suffix := '/zoneinfo' .
  sufSize := suffix size .
  1 to: list size by: 2 do:[:n | | kind |
    kind := list at: n + 1 .
    kind ifFalse:[ "a directory" | elem |
       elem := list at: n .
       (elem at:(elem size - sufSize + 1) equals: suffix) ifTrue:[ ^ elem].
    ]
  ].
  Error signal:'could not find zoneinfo subdirectory'

]

{ #category : 'querying' }
TimeZone class >> availableZones [
 "Returns a sorted Array of Strings which are time zones supported by
  the Olson database shipped in $GEMSTONE/pub/timezone/usr/share/zoneinfo/.
  These Strings may be used as arguments to TimeZone(C)>>named: "
  | res dirs ofs basePath basePathSize |
  res := { } .
  dirs := { (basePath := self _olsonPath , $/)  } .
  basePathSize := basePath size .
  ofs := 1 .
  [ ofs <= dirs size ] whileTrue:[ | list aDir |
    aDir := dirs at: ofs .
    list := GsFile contentsAndTypesOfDirectory: aDir onClient: false .
    1 to: list size by: 2 do:[:n | | elem isFile elemSiz |
      elem := list at: n .  elemSiz := elem size .
      isFile := list at: n + 1 .
      ((elem at: elemSiz) == $. or:[ elem at: elemSiz - 3 equals: '.tab']) ifFalse:[
        isFile ifTrue:[
          (self _isTimeZoneFile: elem) ifTrue:[ | relPath |
            relPath := elem copyFrom: basePathSize + 1 to: elemSiz .
            res add:  relPath
          ].
        ] ifFalse:[ dirs add: elem , $/ ].
      ].
    ].
    ofs := ofs + 1.
  ].
  ^ Array withAll: (SortedCollection withAll: res)

]

{ #category : 'singleton' }
TimeZone class >> current [

"Returns the current session's current TimeZoneInfo. E.g. TimeZoneInfo current."

^ System __sessionStateAt: 17.

]

{ #category : 'singleton' }
TimeZone class >> default [

	^default.

]

{ #category : 'singleton' }
TimeZone class >> default: aTimeZone [

"Makes the specified time zone the default time zone. Returns aTimeZone.
 Must be SystemUser to do so."

aTimeZone _validateClass: TimeZone.
System myUserProfile userId = 'SystemUser' ifFalse:[
  self error:'instance only modifiable by SystemUser'.
  self _uncontinuableError .
].
super default: aTimeZone.
(ObsoleteClasses at:#ObsoleteTimeZone) default: aTimeZone.  "work-around for #36178"
^aTimeZone.

]

{ #category : 'cache' }
TimeZone class >> for: aPlace [

"Returns a TimeZone object for the specified place if it has been defined
 and stored in the receiver's cache . Returns nil if it is not cached.

 E.g. TimeZone for: #'America/Los_Angeles'.

 Example to create a TimeZone for one of the availableZones  
     TimeZone named: 'Etc/UTC' 

 Example To create aned cache a TimeZone for one of the availableZones  use   
    | name | 
    name := 'Etc/UTC' .
    TimeZone for: name put:( TimeZone named: name) . 
    System commit .
"

^self cacheAt: aPlace.

]

{ #category : 'cache' }
TimeZone class >> for: aPlace put: aTimeZone [

"Stores aTimeZone as the TimeZoneInfo object identified with a particular
 place. A single TimeZoneInfo can be associated with any number of places.
 E.g. TimeZoneInfo for: #'America/Los_Angeles' put: aTimeZone;
 TimeZoneInfo for: #'Europe/Berlin' put: aTimeZone.
 Returns aTimeZone."

self cacheAt: aPlace put: aTimeZone.
^aTimeZone.

]

{ #category : 'instance creation' }
TimeZone class >> fromGciPath: pathString [
  ^ self _fromPath: pathString onClient: true

]

{ #category : 'instance creation' }
TimeZone class >> fromGemPath: pathString [
  ^ self _fromPath: pathString onClient: false

]

{ #category : 'instance creation' }
TimeZone class >> fromGsFile: aGsFile [

	| instance |
	[
		instance := self fromStream: aGsFile.
	] ensure: [
		aGsFile close.
	].
	^instance.

]

{ #category : 'instance creation' }
TimeZone class >> fromLinux [

	^TimeZone fromOS

]

{ #category : 'instance creation' }
TimeZone class >> fromObsolete: anObsoleteTimeZone [
"
TimeZone fromObsolete: ObsoleteTimeZone default.
"
  | res |
  (res := self basicNew) initializeFromObsolete: anObsoleteTimeZone.
  ^ res

]

{ #category : 'instance creation' }
TimeZone class >> fromOS [
  "This is suitable for Linux and Mac"

  ^TimeZone fromGemPath: '/etc/localtime'

]

{ #category : 'instance creation' }
TimeZone class >> fromPath: aString [

	| block |
	block := [:prefix |
		| path |
		path := prefix , aString.
		(GsFile existsOnServer: path) == true ifTrue: [
			^self fromGemPath: path.
		].
	].
	block
		value: '';	"full path"
		value: '/usr/share/lib/zoneinfo/';	"AIX and Solaris"
		value: '/usr/share/zoneinfo/';		"Linux"
		value: '$GEMSTONE/pub/timezone/usr/share/zoneinfo/'. "GemStone distribution"
	^nil.

]

{ #category : 'other' }
TimeZone class >> migrateNew [

"Override default migrateNew behavior with #_basicNew because
we disallow #new (which is called by Behavior>>migrateNew)."

^ self _basicNew

]

{ #category : 'instance creation' }
TimeZone class >> named: aString [
  "Return an instance of TimeZone using the specified zone
   from the Olson database shipped in $GEMSTONE/pub/timezone/usr/share/zoneinfo/.
  See TimeZone(C)>>availableZones for legal arguments .
  Does not store the result in the receiver's cache .
  "

  | f path |
  path := self _olsonPath , $/ , aString .
  f := GsFile openReadOnServer: path .
  f ifNil:[
    Error signal: aString asString,' ' , 'does not specify a TimeZone; ' ,
      GsFile lastErrorString
  ].
  ^ self fromGsFile: f .

]

{ #category : 'instance creation' }
TimeZone class >> timeDifferenceHrs: hours dstHrs: dstHrs atTimeHrs: startTimeHrs
fromDayNum: startDay toDayNum: endDay on: nameOfDay beginning: startYear
stdPrintString: stdString dstPrintString: dstString [

| oldTZ |
oldTZ := (ObsoleteClasses at:#ObsoleteTimeZone)
	timeDifferenceHrs: hours
	dstHrs: dstHrs
	atTimeHrs: startTimeHrs
	fromDayNum: startDay
	toDayNum: endDay
	on: nameOfDay
	beginning: startYear
	stdPrintString: stdString
	dstPrintString: dstString.
^self fromObsolete: oldTZ.

]

{ #category : 'instance creation' }
TimeZone class >> timeDifferenceMin: minutes dstMin: dstMins atTimeMin: startTimeMins
fromDayNum: startDay toDayNum: endDay on: nameOfDay beginning: startYear
stdPrintString: stdString dstPrintString: dstString [

| oldTZ |
oldTZ := (ObsoleteClasses at:#ObsoleteTimeZone)
	timeDifferenceMin: minutes
	dstMin: dstMins
	atTimeMin: startTimeMins
	fromDayNum: startDay
	toDayNum: endDay
	on: nameOfDay
	beginning: startYear
	stdPrintString: stdString
	dstPrintString: dstString.
^self fromObsolete: oldTZ.

]

{ #category : 'Initialization' }
TimeZone class >> installOsTimeZone [
  "Install a TimeZone based on the OS time zone settings for the host on which
  the Gem executing this command is running. This uses /etc/localtime which is 
  correct for Linux and Mac. The new TimeZone is made the default/current.  
  The existing default TimeZone is sent become: with the new TimeZone, so existing
  references to the old TimeZone automatically now refer to the new TimeZone."

  | zone |
  zone := TimeZone fromOS.
  self _installDefaultZone: zone
]

{ #category : 'private' }
TimeZone class >> _installDefaultZone: zone [
  | def |
  zone = (def := TimeZone default) ifFalse:[
    def ifNotNil:[ def become: zone ]
        ifNil:[ TimeZone default: zone ]. 
    TimeZone default installAsCurrentTimeZone.
    System commit.
  ].
]

{ #category : 'Initialization' }
TimeZone class >> installNamedZone: zonenameString [
  "Install the named timezone, from the GemStone distribution Olson zoneinfo 
  database in $GEMSTONE/pub/timezone. The new TimeZone is made the default/current.
  The existing default TimeZone is sent become: with the new TimeZone, so existing 
  references to the old TimeZone automatically now refer to the new TimeZone."

  | zone path |
  path := '$GEMSTONE/pub/timezone/usr/share/zoneinfo/'.
  zone := TimeZone fromGemPath: path , zonenameString.
  self _installDefaultZone: zone
]

{ #category : 'Legacy Accessors' }
TimeZone >> _secondsForDst [
	"calculate and save in cache"

	| isDST isNotDST |
	isDST := self detectLastTransition: [:each | each isDST].
	isDST == nil ifTrue: [^secondsForDst := 0].
	isNotDST := self detectLastTransition: [:each | each isDST not].
	isNotDST == nil ifTrue: [^secondsForDst := 0].
	^secondsForDst := isDST offsetFromUTC - isNotDST offsetFromUTC.

]

{ #category : 'Legacy Accessors' }
TimeZone >> _secondsFromGmt [
	"calculate and save in cache"

	| transition |
	transition := self detectLastTransition: [:each | each isDST not].
	^secondsFromGmt := transition == nil
		ifTrue: [0]
		ifFalse: [transition offsetFromUTC].

]

{ #category : 'Legacy Accessors' }
TimeZone >> _yearStartDst [
	"calculate and save in cache"

	| transition |
	transition := transitions
		detect: [:each | each isDST]
		ifNone: [nil].
	^yearStartDst := transition
		ifNil: [SmallInteger maximumValue]
		ifNotNil: [transition asDateAndTimeUTC year].

]

{ #category : 'accessors' }
TimeZone >> = aTimeZone [

	^(aTimeZone isKindOf: TimeZone)
		and: [self transitions = aTimeZone transitions
		and: [self leapSeconds = aTimeZone leapSeconds
		and: [self standardPrintString = aTimeZone standardPrintString
		and: [self dstPrintString = aTimeZone dstPrintString]]]].

]

{ #category : 'Printing' }
TimeZone >> asString [
  | str |
  str := self standardPrintString .
  (str size == 0 and:[ self secondsFromGmt == 0]) ifTrue:[ str := 'UTC'] .
  ^ str

]

{ #category : 'Updating' }
TimeZone >> become: anObject [

	super become: anObject.
	self initializeCache.
	anObject initializeCache.

]

{ #category : 'legacy protocol' }
TimeZone >> dateTimeClass [

"Returns the class of DateTime objects that are to be created by the
 various methods in this class."

^ DateTime

]

{ #category : 'Legacy Accessors' }
TimeZone >> dayEndDst [

	| transition dt |
	transition := self detectLastTransition: [:each | each isDST not].
	dt := transition asDateAndTimeUTC asLocal.
	^dt dayOfYear.

]

{ #category : 'Legacy Accessors' }
TimeZone >> dayStartDst [

	| transition dt |
	transition := self detectLastTransition: [:each | each isDST].
	dt := transition asDateAndTimeUTC asLocal.
	dt := dt + (Duration seconds: self secondsForDst negated).
	^dt dayOfYear.

]

{ #category : 'Legacy Accessors' }
TimeZone >> dstEndTimeList [

"Returns the dstEndTimeList instance variable."

^ dstEndTimeList

]

{ #category : 'legacy protocol' }
TimeZone >> dstPrintString [

	^dstPrintString.

]

{ #category : 'legacy protocol' }
TimeZone >> dstPrintString: aString [

"Sets the dstPrintString instance variable. Returns the receiver."

dstPrintString := aString.
^ self

]

{ #category : 'Legacy Accessors' }
TimeZone >> dstStartTimeList [

"Returns the dstStartTimeList instance variable."

^ dstStartTimeList

]

{ #category : 'queries' }
TimeZone >> endOfDstFor: aYear [

	^ (dstEndTimeList at: aYear otherwise: nil)
		ifNil:[ self endOfDstForA: aYear].

]

{ #category : 'internal' }
TimeZone >> endOfDstForA: aYear [

	| dt transition next |
	dt := DateAndTime year: aYear + 1 day: 1 hour: 0 minute: 0 second: 0.
	[
		(next := self transitionAtUTC: dt) == nil ifTrue: [^nil].
		next = transition ifTrue: [^nil].
		transition := next.
		dt := transition asDateAndTimeUTC.
		dt year ~~ aYear ifTrue: [^nil].
		dt := dt - (Duration seconds: 1).
		transition isDST.
	] whileTrue: [].
	dt := DateTime
		newGmtWithYear: dt year
		month: dt month
		day: dt dayOfMonth
		hours: dt hour
		minutes: dt minute
		seconds: dt second asInteger
		timeZone: self.
	^dt addSeconds: 1.

]

{ #category : 'accessors' }
TimeZone >> hash [

	^transitions first hash + standardPrintString hash.

]

{ #category : 'internal' }
TimeZone >> initialize: aStream [

	| transition |
	super initialize: aStream.
	transition := self detectLastTransition: [:each | each isDST].
	dstPrintString := transition == nil
		ifTrue:  ['']
		ifFalse: [transition abbreviation].
	transition := self detectLastTransition: [:each | each isDST not].
	standardPrintString := transition == nil
		ifTrue:  ['']
		ifFalse: [transition abbreviation].
	self initializeCache.

]

{ #category : 'internal' }
TimeZone >> initializeCache [

	dstStartTimeList := IntegerKeyValueDictionary new.
	dstEndTimeList := IntegerKeyValueDictionary new.
	self
		populateCacheFor: (1950 to: 2050);
		_yearStartDst;
		_secondsForDst;
		_secondsFromGmt;
		yourself.

]

{ #category : 'internal' }
TimeZone >> initializeFromObsolete: anObsoleteTimeZone [
	| base startArray endArray old new currentZ |
  currentZ := TimeZone current .
  base :=	DateTime newWithYear: 1970 dayOfYear: 1 seconds: 0 .
	transitions := OrderedCollection new.
	standardPrintString := anObsoleteTimeZone standardPrintString.
	dstPrintString := anObsoleteTimeZone dstPrintString.
	startArray := { {
		anObsoleteTimeZone secondsFromGmt + anObsoleteTimeZone secondsForDst .
		true .
		dstPrintString } } .
	endArray := { {
		anObsoleteTimeZone secondsFromGmt .
		false .
		standardPrintString } } .
	currentZ ifNotNil:[
		anObsoleteTimeZone secondsForDst = 0 ifTrue: [
			| year trTim transition |
			year := (anObsoleteTimeZone yearStartDst max: 1900) printString.
			trTim := (DateTime fromStringGmt: '01/01/' ,  year , ' 00:00:00' ) asSecondsGmt - base asSeconds.
			(transition := TimeZoneTransition new)
				localTimeTypeID: 1;
				transitionTime: trTim ;
				typeList: endArray.
			transitions add: transition.
			self initializeCache.
			^self.
		].
		anObsoleteTimeZone yearStartDst to: 2030 do: [:year |
			| endDateTime startSec endSec start end |
			startSec := (anObsoleteTimeZone startOfDstFor: year) asSecondsGmt - base asSeconds.
			(start := TimeZoneTransition new)
				localTimeTypeID: 1;
				transitionTime: startSec;
				typeList: startArray.
			(endDateTime := anObsoleteTimeZone endOfDstFor: year) ifNil: [
				transitions add: start.
			] ifNotNil: [
				endSec := endDateTime asSecondsGmt - base asSeconds.
				(end := TimeZoneTransition new)
					localTimeTypeID: 1;
					transitionTime: endSec;
					typeList: endArray.
				start transitionTimeUTC < end transitionTimeUTC ifTrue: [
					transitions add: start; add: end.
				] ifFalse: [
					transitions add: end; add: start.
				].
			].
		].
	].
	transitions := transitions asArray.
	self initializeCache.
	old := anObsoleteTimeZone.
	new := self.
  false ifTrue:[
    "Gs64 v3.6, disable this check. seeing failures in testBug43664, but not worth
     fixing ObsoleteTimeZone issues."
	  anObsoleteTimeZone yearStartDst to: 2030 do: [:year |
		  | oldStart oldEnd newStart newEnd |
		  oldStart := old startOfDstFor: year.
		  newStart := new startOfDstFor: year.
		  oldEnd   := old endOfDstFor:   year.
		  newEnd   := new endOfDstFor:   year.
		  oldStart = newStart ifFalse: [Warning signal: 'start date calculation error',
           oldStart asString , ', ' , newStart asString ].
		  oldEnd   = newEnd   ifFalse: [Warning signal: 'end date calculation error ',
		     oldEnd asString , ', ' , newEnd asString ].
	  ].
  ].

]

{ #category : 'singleton' }
TimeZone >> installAsCurrentTimeZone [

"Sets the receiver as the current session's current Timezone. Returns the
 receiver."

System __sessionStateAt: 17 put: self.
^ self.

]

{ #category : 'accessors' }
TimeZone >> leapSeconds [

	^leapSeconds.

]

{ #category : 'Instance Migration' }
TimeZone >> migrateFrom: anotherObject instVarMap: otherivi [

	super migrateFrom: anotherObject instVarMap: otherivi.
	self
		_yearStartDst;
		_secondsForDst;
		_secondsFromGmt;
		yourself.

]

{ #category : 'internal' }
TimeZone >> populateCacheFor: anInterval [

	anInterval do: [:year |
		dstStartTimeList
			at: year
			put: (self startOfDstForA: year).
		dstEndTimeList
			at: year
			put: (self endOfDstForA: year).
	].

]

{ #category : 'Legacy Accessors' }
TimeZone >> secondsForDst [

	^secondsForDst

]

{ #category : 'Legacy Accessors' }
TimeZone >> secondsFromGmt [

	^secondsFromGmt

]

{ #category : 'legacy protocol' }
TimeZone >> shouldWriteInstVar: instVarName [

"Returns whether the given instance variable should be written out."

"exclude the ditionaries"

instVarName == #dstStartTimeList ifTrue:[ ^ false ].
instVarName == #dstEndTimeList ifTrue:[ ^ false ].
^ true

]

{ #category : 'legacy protocol' }
TimeZone >> standardPrintString [

	standardPrintString == nil ifTrue: [
		standardPrintString := (self detectLastTransition: [:each | each isDST not]) abbreviation.
	].
	^standardPrintString.

]

{ #category : 'legacy protocol' }
TimeZone >> standardPrintString: aString [

"Sets the standardPrintString instance variable. Returns the receiver."

standardPrintString := aString.
^ self

]

{ #category : 'queries' }
TimeZone >> startOfDstFor: aYear [

	^ ( dstStartTimeList at: aYear otherwise: nil)
		ifNil:[ self startOfDstForA: aYear ].

]

{ #category : 'internal' }
TimeZone >> startOfDstForA: aYear [

	| dt transition next |
	dt := DateAndTime year: aYear + 1 day: 1 hour: 0 minute: 0 second: 0.
	[
		(next := self transitionAtUTC: dt) == nil ifTrue: [^nil].
		next = transition ifTrue: [^nil].
		transition := next.
		dt := transition asDateAndTimeUTC.
		dt year ~~ aYear ifTrue: [^nil].
		dt := dt - (Duration seconds: 1).
		transition isDST not.
	] whileTrue: [
	].
	dt := DateTime
		newGmtWithYear: dt year
		month: dt month
		day: dt dayOfMonth
		hours: dt hour
		minutes: dt minute
		seconds: dt second asInteger
		timeZone: self.
	^dt addSeconds: 1.

]

{ #category : 'Legacy Accessors' }
TimeZone >> timeStartDst [

	| transition dt |
	transition := self detectLastTransition: [:each | each isDST].
	dt := transition asDateAndTimeUTC - (Duration seconds: 1).
	dt := DateAndTime
		secondsUTC: dt asSeconds
		offset: (Duration seconds: (self offsetAtUTC: dt)).
	^dt hour * 60 + dt minute * 60 + dt second + 1.

]

{ #category : 'accessors' }
TimeZone >> transitions [

	^transitions.

]

{ #category : 'Legacy Accessors' }
TimeZone >> weekDayStartDst [

	| transition year dateTime date |
	transition := self detectLastTransition: [:each | each isDST].
	year := transition asDateAndTimeUTC year.
	dateTime := self startOfDstFor: year.
	dateTime := dateTime addSeconds: self secondsForDst negated.
	date := dateTime asDateIn: self.
	^date weekDayName asSymbol.

]

{ #category : 'Legacy Accessors' }
TimeZone >> yearStartDst [
	^yearStartDst
]
