"
A Bag is an UnorderedCollection in which any distinct object can occur any
 number of times.  Adding the same (identical) object to a Bag multiple times
 simply causes it to occur multiple times in the Bag.

 Since a Bag is an equality-based collection, different (non-identical) but
 equivalent (equal) objects are not treated as distinct from each other.  In
 IdentityBags, they are distinct.  Adding multiple equivalent objects to a Bag
 yields a Bag with multiple occurrences of the object that was added last.

Constraints:
	_varyingSize: Object
	_numEntries: Object
	_indexedPaths: Object
	_levels: Object
	dict: KeyValueDictionary
	size: Integer
--- instVar dict
A KeyValueDictionary that organizes the elements and element counts for the
 Bag.
--- instVar size
For GemStone internal use.

"
Class {
	#name : 'Bag',
	#superclass : 'UnorderedCollection',
	#instVars : [
		'dict',
		'size'
	],
	#gs_options : [
		'disallowGciStore'
	],
	#gs_reservedoop : '102657',
	#category : nil
}

{ #category : 'Instance Creation' }
Bag class >> new [

"Returns an instance of the receiver whose contents are empty."

^ (self basicNew) initialize: 0

]

{ #category : 'Instance Creation' }
Bag class >> new: initialSize [

"Returns an instance of the receiver whose contents are empty."

^ (self basicNew) initialize: initialSize.

]

{ #category : 'Private' }
Bag >> __dict [

^ dict

]

{ #category : 'Converting' }
Bag >> _asIdentityBag [

"Returns an IdentityBag that contains all of the elements of the receiver."

"Used by index creation."

| result tmp |

"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

result := IdentityBag new .
dict keysAndValuesDo:[:aKey :numOcc|
  numOcc timesRepeat:[ result add: aKey ].
  ].
^ result

]

{ #category : 'Private' }
Bag >> _deepCopyWith: copiedObjDict [

| copy |
copy := copiedObjDict at: self otherwise: nil.
copy ~~ nil ifTrue: [ ^ copy ].

copy := self class  new: (self size).
copiedObjDict at: self put: copy.

self _deepCopyNamedIvsWith: copiedObjDict to: copy .

dict keysAndValuesDo: [ :aKey :aValue |
  aValue timesRepeat: [copy add: (aKey _deepCopyWith: copiedObjDict)]
  ].

^ copy.

]

{ #category : 'Private' }
Bag >> _deferredGciUpdateWith: valueArray [

"Private."

1 to: valueArray size do:[:j |
  self add: (valueArray at: j)
  ].

]

{ #category : 'Searching' }
Bag >> _detect: aBlock [

dict keysDo: [:each |
    (aBlock value: each)
        ifTrue:[ ^ each ].
].
^ self _error: #assocErrNoElementsDetected args: { aBlock } .

]

{ #category : 'Searching' }
Bag >> _detect: aBlock ifNone: exceptionBlock [

dict keysDo: [:each |
    (aBlock value: each)
        ifTrue:[ ^ each ].
].
^ exceptionBlock value

]

{ #category : 'Instance Initialization' }
Bag >> _gciInitialize [

"Private."

self initialize: 0 .
^ true

]

{ #category : 'Testing' }
Bag >> _isLarge [

"Returns true if the object is implemented as a tree of private smaller objects"

^ true

]

{ #category : 'Private' }
Bag >> _keysAndValuesDo: aBlock [
  "aBlock should be a 2 argument block with arguments  element, count "

  dict keysAndValuesDo:[ :each :count | aBlock value: each value: count ]

]

{ #category : 'Searching' }
Bag >> _reject: aBlock [

| result |
result:= self species new.
dict keysAndValuesDo: [:each :count |
    (aBlock value: each)
        ifFalse: [ result add: each withOccurrences: count ]
].
^ result

]

{ #category : 'Searching' }
Bag >> _select: aBlock [

| result |
result:= self species new.
dict keysAndValuesDo: [:each :count |
    (aBlock value: each)
        ifTrue: [ result add: each withOccurrences: count ]
].
^ result

]

{ #category : 'Set Arithmetic' }
Bag >> _union: aBagOrSet [

"Union with minimal result.
 An object must be present in either the receiver or the argument in order
 to appear in the result.
 For each object in the result the number of occurrances in the result
 is the maximum of the number of occurrances in the receiver or argument. "

| res |
(aBagOrSet isKindOf: IdentityBag ) ifTrue:[
  res := IdentityBag new .
  res addAll: aBagOrSet .
  dict keysAndValuesDo:[ :each :count | | occurrences resOccurrences |
    resOccurrences := res occurrencesOf: each.
    occurrences := (count max: resOccurrences) - resOccurrences.
    res add: each withOccurrences: occurrences ]  .
  ^ res
].
((aBagOrSet isKindOf: Bag) or:[ aBagOrSet isKindOf: Set ]) ifFalse:[
  aBagOrSet _validateKindOfClasses: { Bag . Set } .
].
res := self copy .
aBagOrSet _keysAndValuesDo:[ :each :count | | occurrences resOccurrences |
    resOccurrences := res occurrencesOf: each.
    occurrences := (count max: resOccurrences) - resOccurrences.
    res add: each withOccurrences: occurrences ].
^ res

]

{ #category : 'Set Arithmetic' }
Bag >> - aBagOrSet [

"Difference. The result containing exactly those elements of
 the receiver that have a greater number of occurrences in the receiver than in
 the argument.
 If argument is a kind of IdentityBag, result will be an IdentityBag ,
 otherwise result is an instance of the class of the receiver."

| res |
(aBagOrSet isKindOf: IdentityBag ) ifTrue:[ | s |
  res := IdentityBag new .
  res addAll: aBagOrSet .
  s := IdentityBag new .
  dict keysAndValuesDo:[ :each :count | s add: each withOccurrences: count ]  .
  ^ res - s .
].
((aBagOrSet isKindOf: Bag) or:[ aBagOrSet isKindOf: Set ]) ifFalse:[
  aBagOrSet _validateKindOfClasses: { Bag . Set } .
].
res := self copy .
aBagOrSet _keysAndValuesDo:[ :each :count | | oldCnt |
  oldCnt := dict at: each otherwise: 0 .
  oldCnt > count ifTrue:[ res remove: each withOccurrences: count ]
                ifFalse:[ oldCnt ~~ 0 ifTrue:[ res remove: each withOccurrences: oldCnt ]].
  ] .
^ res

]

{ #category : 'Set Arithmetic' }
Bag >> * aBagOrSet [

"Intersection.  The result containing only the elements that
 are present in both the receiver and the argument aBagOrSet.

 If aBagOrSet is a kind of IdentityBag, the result is an IdentityBag,
 otherwise the result is a Bag ."

| res |
(aBagOrSet isKindOf: IdentityBag ) ifTrue:[ | s |
  s := IdentityBag new .
  dict keysAndValuesDo:[ :each :count | s add: each withOccurrences: count ]  .
  ^ s * aBagOrSet
].
res := self class new .
aBagOrSet _keysAndValuesDo:[ :each :count | | oldCnt newCnt |
  oldCnt := dict at: each otherwise: 0 .
  newCnt := oldCnt min: count .
  newCnt > 0 ifTrue:[ res add: each withOccurrences: newCnt ].
].
^ res

]

{ #category : 'Set Arithmetic' }
Bag >> + aBagOrSet [

"Union.  The result contains exactly the elements that are
 present in either the receiver or the argument aBagOrSet.
 If aBagOrSet is a kind of IdentityBag, the result is an IdentityBag,
 otherwise result is an instance of the class of the receiver."

| res |
(aBagOrSet isKindOf: IdentityBag ) ifTrue:[
  res := IdentityBag new .
  res addAll: aBagOrSet .
  dict keysAndValuesDo:[ :each :count | res add: each withOccurrences: count ]  .
  ^ res
].
((aBagOrSet isKindOf: Bag) or:[ aBagOrSet isKindOf: Set ]) ifFalse:[
  aBagOrSet _validateKindOfClasses: { Bag . Set } .
].
res := self copy .
aBagOrSet _keysAndValuesDo:[ :each :count | res add: each withOccurrences: count ].
^ res

]

{ #category : 'Comparing' }
Bag >> = aCollection [

"Returns true if all of the following conditions are true:

 1.  The receiver and aCollection are of the same class.
 2.  The two collections are of the same size.
 3.  The corresponding elements of the receiver and aCollection are equal
     and have the same count in both."

(self == aCollection)
  ifTrue: [ ^ true ].

(self class == aCollection class)
  ifFalse: [ ^ false ].

(self size == aCollection size)
  ifFalse: [ ^ false ].

dict keysAndValuesDo: [:key :value |
  (aCollection occurrencesOf: key) = value
     ifFalse: [ ^ false ].
  ].

^ true.

]

{ #category : 'Adding' }
Bag >> add: newObject [
	"Makes newObject one of the receiver's elements and returns newObject."

	^self add: newObject withOccurrences: 1

]

{ #category : 'Adding' }
Bag >> add: anObject withOccurrences: anInteger [
	"Adds anObject anInteger number of times to the receiver and returns anObject."
	"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."

	| originalObject numOccurrences |
	anObject == nil ifTrue: [^nil].
	(anInteger _isInteger and: [anInteger >= 0])
		ifFalse:
			[^anInteger _error: #rtErrInvalidArgument
				args: {'must be a non-negative Integer'}].

	"v3.0.1 varyingConstraints no longer enforced."
	_levels := _levels.

	"Because a Bag is equality based but the indexing objects are
    identity based, ensure the original object is used for index
    maintenance when an equal object is added more than once."

	originalObject := dict keyAt: anObject otherwise: anObject.
	_indexedPaths ~~ nil
		ifTrue: [
			anInteger timesRepeat: [
         self _updateIndexesForAdditionOf: originalObject logging: true.
      ]].
	numOccurrences := dict at: originalObject otherwise: 0.
	dict at: originalObject put: numOccurrences + anInteger.
	size := size + anInteger.
	^anObject

]

{ #category : 'Adding' }
Bag >> addAll: aCollection [

"assign _levels so that authorization and concurrency
 conflicts on the root object can be detected."
_levels := _levels .

(aCollection isKindOf: Bag)
ifFalse: [ super addAll: aCollection ]
ifTrue: [
    aCollection __dict keysAndValuesDo: [ :each :count |
        self add: each withOccurrences: count
    ]
].
^ aCollection

]

{ #category : 'Converting' }
Bag >> asBag [

"Returns a Bag with the contents of the receiver."

^ Bag withAll: self

]

{ #category : 'Converting' }
Bag >> asIdentitySet [

"Returns a Set"

^ IdentitySet withAll: dict keys

]

{ #category : 'Converting' }
Bag >> asSet [

"Returns a Set"

^ Set withAll: dict keys

]

{ #category : 'Accessing' }
Bag >> at: anIndex [

"Disallowed."

^ self shouldNotImplement: #at:

]

{ #category : 'Updating' }
Bag >> at: anIndex put: anObject [

"Disallowed."

^ self shouldNotImplement: #at:put:

]

{ #category : 'Enumerating' }
Bag >> do: aBlock [

"Evaluates the one-argument block aBlock using each element of the
 receiver. Returns the receiver."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

dict keysAndValuesDo: [ :aKey :aValue |
  aValue timesRepeat: [ aBlock value: aKey ]
  ].

^ self.

]

{ #category : 'Searching' }
Bag >> includes: anObject [

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

^ dict includesKey: anObject

]

{ #category : 'Searching' }
Bag >> includesIdentical: anObject [

"Returns true if anObject is identical to one of the elements of the receiver.
 Returns false otherwise."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

anObject == nil ifTrue:[ ^ false ].
^ anObject == (dict keyAt: anObject otherwise: nil)

]

{ #category : 'Searching' }
Bag >> includesValue: anObject [

"Returns true if anObject is equal to one of the elements of the receiver.
 Returns false otherwise."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

^ dict includesKey: anObject

]

{ #category : 'Instance Initialization' }
Bag >> initialize: initialSize [

"Initializes the receiver immediately after creation. Returns the receiver."

_indexedPaths ~~ nil ifTrue:[
   dict keysAndValuesDo:[:anObj :numOcc |
      numOcc timesRepeat: [
        self _updateIndexesForRemovalOf: anObj.
        ]
      ]
   ].
dict := KeyValueDictionary new: initialSize.
size := 0.

]

{ #category : 'Updating' }
Bag >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assigns the receiver and its private objects to the given security policy."

super objectSecurityPolicy: anObjectSecurityPolicy .
dict objectSecurityPolicy: anObjectSecurityPolicy  .

]

{ #category : 'Searching' }
Bag >> occurrencesOf: anObject [

"Returns the number of the receiver's elements that are equal to anObject."

| tmp |
"read _levels so that authorization and concurrency conflicts on the
 root object can be detected."
tmp := _levels.

^ dict at: anObject otherwise: 0 .

]

{ #category : 'Copying' }
Bag >> postCopy [

 super postCopy .
 dict := dict copy .

]

{ #category : 'Hashing' }
Bag >> rehash [

"Rebuilds the receiver to ensure its consistency. Returns the receiver."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."
_levels := _levels .

dict _rebuild.
^self.

]

{ #category : 'Removing' }
Bag >> remove: anObject ifAbsent: anExceptionBlock [

"Removes an object that is equivalent to anObject from the receiver and
 returns anObject.  If several elements of the receiver are equivalent to
 anObject, only one instance is removed.  If anObject has no equivalent
 elements in the receiver, this method evaluates anExceptionBlock and returns
 the result."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."

| count |
count := dict at: anObject otherwise: 0.
count == 0 ifTrue: [^anExceptionBlock value].
_levels := _levels .
_indexedPaths == nil ifFalse:[
   | originalObject |
     "Since a Bag is equality based, use the originally added object
     for index maintenance. Index objects have identity based
     methods."

     originalObject := dict keyAt: anObject otherwise: nil.
     originalObject == nil ifTrue:
         [self error: 'Internal Gemstone error, a Bag''s contents are suspect.'].
      self _updateIndexesForRemovalOf: originalObject.
    ].
count > 1
    ifTrue: [dict at: anObject put: count - 1]
    ifFalse: [dict removeKey: anObject].
size := size - 1.
^anObject

]

{ #category : 'Removing' }
Bag >> remove: anObject withOccurrences: anInteger [
	"Remove anObject anInteger number of times to the receiver and returns anObject."
	"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."

	| originalObject numOccurrences finalCount |
	anObject == nil ifTrue: [^nil].
	(anInteger _isInteger and: [anInteger >= 0])
		ifFalse:
			[^anInteger _error: #rtErrInvalidArgument
				args: {'must be a non-negative Integer'}].

	"Because a Bag is equality based but the indexing objects are
    identity based, ensure the original object is used for index
    maintenance when an equal object is removed more than once."

	originalObject := dict keyAt: anObject otherwise: nil.
	originalObject == nil ifTrue: [^self _errorNotFound: anObject].
	numOccurrences := dict at: originalObject otherwise: 0.
	anInteger > numOccurrences ifTrue: [^self _errorNotFound: anObject].
	_levels := _levels.
	_indexedPaths ~~ nil
		ifTrue: [
			anInteger timesRepeat:[
         self _updateIndexesForRemovalOf: originalObject.
    ]].
	(finalCount := numOccurrences - anInteger) > 0
		ifTrue: [dict at: originalObject put: finalCount]
		ifFalse: [dict removeKey: originalObject].
	size := size - anInteger.
	^anObject

]

{ #category : 'Removing' }
Bag >> removeAll: aCollection [

"Removes one occurrence of each element of aCollection from the
 receiver and returns the argument.  Generates an error if any
 element of aCollection is not present in the receiver."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."
_levels := _levels.

aCollection == self
ifTrue:[
  self initialize: 0
] ifFalse:[
  (aCollection isKindOf: Bag)
  ifFalse: [ aCollection accompaniedBy: self do: [:me :anObject | me remove: anObject ] ]
  ifTrue: [
      aCollection __dict accompaniedBy: self keysAndValuesDo: [:me :each :count |
	  me remove: each withOccurrences: count
      ]
  ]
].
^aCollection

]

{ #category : 'Removing' }
Bag >> removeAllPresent: aCollection [

"Removes one occurrence of each element of aCollection from the
 receiver and returns the receiver.  Does not generate an error
 if any element of aCollection is not present in the receiver.
 Returns aCollection."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."
_levels := _levels.

aCollection == self
ifTrue:[
    self initialize: 0
] ifFalse:[
    (aCollection isKindOf: Bag)
    ifFalse: [ super removeAllPresent: aCollection ]
    ifTrue: [
        aCollection __dict accompaniedBy: self keysAndValuesDo: [:me :each :count |
            me removeIfPresent: each withOccurrences: count
            ]
    ]
].
^aCollection

]

{ #category : 'Removing' }
Bag >> removeIdentical: anObject ifAbsent: anExceptionBlock [

"Removes an object that is Identical to anObject from the receiver and
 returns anObject.  If several elements of the receiver are identical to
 anObject, only one instance is removed.  If anObject has no equivalent
 elements in the receiver, this method evaluates anExceptionBlock and returns
 the result."

"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."

(self includesIdentical: anObject) ifFalse: [^anExceptionBlock value].
_levels := _levels .
^self remove: anObject

]

{ #category : 'Removing' }
Bag >> removeIfPresent: anObject withOccurrences: anInteger [
	"Remove anObject anInteger number of times to the receiver and returns anObject.
 Does not generate an error if anObject is not present (or there are not
 enough occurrences)."
	"assign _levels  so that authorization and concurrency conflicts on the
 root object can be detected."

	| originalObject numOccurrences countToRemove finalCount |
	anObject == nil ifTrue: [^nil].
	(anInteger _isInteger and: [anInteger >= 0])
		ifFalse:
			[^anInteger _error: #rtErrInvalidArgument
				args: {'must be a non-negative Integer'}].

	"v3.0.1 varyingConstraints no longer enforced."

	"Because a Bag is equality based but the indexing objects are
    identity based, ensure the original object is used for index
    maintenance when an equal object is removed more than once."

	originalObject := dict keyAt: anObject otherwise: nil.
	originalObject == nil ifTrue: [^nil].
	numOccurrences := dict at: originalObject otherwise: 0.
	anInteger > numOccurrences
		ifTrue: [countToRemove := numOccurrences]
		ifFalse: [countToRemove := anInteger].
	countToRemove > 0
		ifTrue:
			[_levels := _levels.
			_indexedPaths ~~ nil
				ifTrue: [
					countToRemove timesRepeat:[
            self _updateIndexesForRemovalOf: originalObject.
        ]].
			(finalCount := numOccurrences - countToRemove) > 0
				ifTrue: [dict at: originalObject put: finalCount]
				ifFalse: [dict removeKey: originalObject].
			size := size - countToRemove].
	^anObject

]

{ #category : 'Accessing' }
Bag >> size [

"Returns the number of elements contained in the receiver."

^ size .

]
