Extension { #name : 'BinaryFloat' }

{ #category : 'Exception Handling' }
BinaryFloat class >> _exceptionList [

"Returns the list of available exceptions. See FloatingPointError."

^ (Globals at: #FloatingPointError) _exceptionList

]

{ #category : 'Private' }
BinaryFloat class >> _finishFromStream: aStream sign: signChar integerPart: integerPart [
  "parse this subset of the BNF
   Number = [ '-' ] NumericLiteral                      (excluding RadixLiteral* )
   BinaryExponent = ( 'e' | 'E' | 'd' | 'D' | 'q' ) ['-' | '+'] Digits
   DecimalExponent = ( 'f' | 'F' ) ['-' | '+'] Digits
   Digit = '0' | '1' | '2' | ... | '9'
   Digits = Digit {Digit}
   Exponent = BinaryExponent | DecimalExponent | ScaledDecimalExponent | FixedPointExponent
   FractionalPart = '.' Digits [Exponent]
   FixedPointExponent = 'p' [ ['-' | '+'] Digits ]
   Numeric = Digit | 'A' | 'B' | ... | 'Z'
   NumericLiteral = Digits ( [FractionalPart] | [Exponent] )
   Numerics = Numeric { Numeric }
   ScaledDecimalExponent = 's' [ ['-' | '+'] Digits ]
"
  | fractionalPart ch  expon exponChar |
  fractionalPart := String new.
  [ (ch := aStream peek) ~~ nil and: [ ch isDigit ] ] whileTrue: [
    aStream next.
    fractionalPart add: ch.
  ].
  expon := String new .
  exponChar := $e .
  (#( $e $E $f $F $s $q $p $d $D ) includesIdentical: ch) ifTrue:[
     "parse Digits for Exponent"
     exponChar := ch .
     aStream next.
     (aStream peek == $- ) 
       ifTrue:[ aStream next . expon add:  $-  ]
       ifFalse:[ (aStream peek == $+ ) ifTrue: [ aStream next ] ].
     [ (ch := aStream peek) ~~ nil and: [ ch isDigit ] ] whileTrue: [
        aStream next.
        expon add: ch .
     ].
  ].
  ^ Number _finishFromStream: aStream sign: signChar integerPart: integerPart
        fractionalPart: fractionalPart expChar: exponChar expPart: expon 

]


{ #category : 'Exception Handling' }
BinaryFloat class >> _raiseInvalidOperationException [

"This method sets the invalid operation exception of the floating point
 processor.  If FloatingPointError is enabled for #invalidOperation,
 signals a FloatingPointError."

MinusSignalingNaN + 3.0E0 .
^self

]

{ #category : 'Exception Handling' }
BinaryFloat class >> clearAllExceptions [

  "Deprecated, has no effect"

  self deprecated: 'BinaryFloat class>>clearAllExceptions obsolete in v3.4; see FloatingPointError class >> enableExceptions:'.

]

{ #category : 'Exception Handling' }
BinaryFloat class >> clearException: aString [

  "Deprecated, has no effect"

  self deprecated: 'BinaryFloat class>>clearException obsolete in v3.4; see FloatingPointError class >> enableExceptions:'.

]

{ #category : 'Exception Handling' }
BinaryFloat class >> enabledExceptions [

"Deprecated, returns an empty Array."

  self deprecated: 'BinaryFloat class>>enabledExceptions obsolete in v3.4; see FloatingPointError class >> enabledExceptions'.
 ^ #()

]

{ #category : 'Instance Creation' }
BinaryFloat class >> fromStream: aStream [

"Generates a BinaryFloat from aStream.  Generates an error if an attempt is
 made to read beyond the end of the stream.

 The Stream must contain a legal BinaryFloat, as defined by the following BNF
 construction:

 BinaryFloat = ( Integer '.' Digit {Digit} [ E Integer ] ) |
         ( Integer E Integer )
 Integer = [ ('+' | '-') ] Digit {Digit}
 Point = ( '.' | ',' ) depending on Locale
 E = ( 'E' | 'e' | 'q' | 'd')

 Note that the syntax does not allow certain valid BinaryFloats (such as
 PlusInfinity and MinusInfinity) to be read."

| ch s getDigits getChar getSign |

self _checkReadStream: aStream forClass: CharacterCollection.

ch := aStream next.
[ ch == $  ] whileTrue: [ ch := aStream next ].
aStream skip: -1.
s := aStream contents class new.

getDigits := [ | c cnt | 
               cnt := 0 .
               [ (c := aStream peek) ~~ nil and: [ c isDigit ] ]
               whileTrue:
                 [ s add: aStream next . cnt := cnt + 1 ].
               cnt == 0 ifTrue:
                 [ self _errIncorrectFormat: aStream ].
             ].

getChar := [ :c |
             (aStream peek isEquivalent: c)
             ifTrue:
               [ s add: aStream next ]
           ].

getSign := [ (getChar value: $-) == nil
              ifTrue:
                [ getChar value: $+ ].
           ].

getSign value.
getDigits value.

(getChar value: (Locale decimalPoint at: 1))
ifNotNil:
  [ getDigits value ].

((getChar value: $e)
	ifNil: [(getChar value: $d)
		ifNil: [(getChar value: $q)]]
) ifNotNil:[
		getSign value.
		getDigits value.
].

^ self fromString: s

]

{ #category : 'Instance Creation' }
BinaryFloat class >> fromString: aString [

"Returns an instance of Float, constructed from aString.  The String must
 contain only Characters representing the object to be created, although
 leading and trailing blanks are permitted."

^ Float fromString: aString

]

{ #category : 'Storing and Loading' }
BinaryFloat class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

BinaryFloat subclassResponsibility: #loadFrom:

]

{ #category : 'Exception Handling' }
BinaryFloat class >> on: aSymbol do: aBlock [

  "Deprecated, has no effect"

  self deprecated: 'BinaryFloat class>>on:do: obsolete in v3.4; see FloatingPointError'.

]

{ #category : 'Exception Handling' }
BinaryFloat class >> operationException: aSymbol [

  "Deprecated, always returns false"

  self deprecated: 'BinaryFloat class>>operationExceptions: obsolete in v3.4; see FloatingPointError'.
 ^ false

]

{ #category : 'Exception Handling' }
BinaryFloat class >> operationExceptions [

  "Deprecated, returns an empty Array."

  self deprecated: 'BinaryFloat class>>operationExceptions obsolete in v3.4; see FloatingPointError class >> enabledExceptions'.
 ^ #()

]

{ #category : 'Arithmetic' }
BinaryFloat class >> pi [

"Returns the value of pi, accurate to twenty decimal places."

^ 3.14159265358979323846

]

{ #category : 'Exception Handling' }
BinaryFloat class >> raisedException: aString [

  "Deprecated, always returns false."

  self deprecated: 'BinaryFloat class>>raisedException: obsolete in v3.4; see FloatingPointError'.
 ^ false

]

{ #category : 'Exception Handling' }
BinaryFloat class >> raisedExceptions [

  "Deprecated, returns an empty Array."

  self deprecated: 'BinaryFloat class>>raisedExecptions obsolete in v3.4; see FloatingPointError class >> enabledExceptions'.
  ^ #()

]

{ #category : 'Truncation and Rounding' }
BinaryFloat class >> roundingMode [

"Returns the current rounding mode , one of
  #nearestEven #towardMinusInfinity, #towardPlusInfinity  #towardZero.

 Returns  #unknown  if access to rounding mode is not implemented,
 in which case the operating system or hardware default is always used."

 self deprecated:'BinaryFloat class>>roundingMode has no effect in 3.4 and previous releases'.
^ #unknown

]

{ #category : 'Truncation and Rounding' }
BinaryFloat class >> roundingMode: aSymbol [

 "Not implemented, has no effect."

 "Future support for rounding will include
  #nearestEven #towardMinusInfinity #towardPlusInfinity #towardZero"

 self deprecated:'BinaryFloat class>>roundingMode: has had no effect in 3.4 and previous releases.'.
 ^ self

]

{ #category : 'Exception Handling' }
BinaryFloat class >> status [

 "Deprecated, Returns an empty Array"

  self deprecated: 'BinaryFloat class>>status obsolete in v3.4.  See FloatingPointError'.
 ^ #()

]

{ #category : 'Exception Handling' }
BinaryFloat class >> status: aString [

 "Deprecated, Has no effect in this release."

  self deprecated: 'BinaryFloat class>>status: obsolete in v3.4; see FloatingPointError'.

]

{ #category : 'Exception Handling' }
BinaryFloat class >> trapEnabled: aString [

 "Deprecated, returns false."

  self deprecated: 'BinaryFloat class>>trapEnabled: obsolete in v3.4; see FloatingPointError'.
 ^ false

]

{ #category : 'Indexing Support' }
BinaryFloat >> _isNaN [

"(Subclass responsibility.)  Returns whether the receiver is quiet NaN or
 a signaling NaN.  This method is only to be used by the indexing subsystem."

BinaryFloat subclassResponsibility: #_isNaN

]

{ #category : 'Comparing' }
BinaryFloat >> >= aMagnitude [

"Returns true if the receiver is greater than or equal to aMagnitude;
 returns false otherwise."

"Reimplemented from Magnitude to handle NaNs correctly."

^ aMagnitude <= self

]

{ #category : 'Arithmetic' }
BinaryFloat >> abs [

"Returns a Number that is the absolute value of the receiver."

"reimplemented for efficiency"

(self < 0.0) ifTrue: [ ^ 0.0 - self ].
^ self

]

{ #category : 'Converting' }
BinaryFloat >> asFixedPoint: scale [

"Returns a ScaledDecimal that represents the receiver.  If the receiver is
 a NaN or Infinity, returns the receiver.  The argument scale should be a
 non-negative SmallInteger."

| myKind |

myKind := self kind.
myKind == #normal ifFalse:[
  ((myKind == #infinity or:[ myKind == #signalingNaN ])
    or:[ myKind == #quietNaN])  ifTrue:[
      BinaryFloat _raiseInvalidOperationException .
      ^ self.
    ].
  ].
^ self asFraction asFixedPoint: scale.

]

{ #category : 'Converting' }
BinaryFloat >> asFraction [

"Returns a Fraction that represents the receiver.  If the receiver is a NaN,
 or Infinity, returns the receiver."

| knd |

"If an infinite or quiet NaN, returns self"
knd := self _getKind .
knd > 2 ifTrue:[ | myKind |
  myKind := self kind .
  myKind == #zero ifTrue:[ ^ 0 ].
  ((myKind == #infinity) or: [ myKind == #quietNaN ]) ifTrue: [ ^self ].

     "If a signaling NaN, raises a floating point exception & returns self"
  (myKind == #signalingNaN) ifTrue: [
    BinaryFloat _raiseInvalidOperationException.
    ^self _makeQuietNaN
  ].
  self error:'logic error'. "should not reach here"
].
" we have either normal or subnormal "
^ self _mathPrim: 12  "_asFraction" .

]

{ #category : 'Converting' }
BinaryFloat >> asScaledDecimal: scale [

"Returns a ScaledDecimal that represents the receiver.  If the receiver is
 a NaN or Infinity, returns the receiver.  The argument scale should be a
 non-negative SmallInteger."

| myKind |

myKind := self kind.
myKind == #normal ifFalse:[
  ((myKind == #infinity or:[ myKind == #signalingNaN ])
    or:[ myKind == #quietNaN])  ifTrue:[
      BinaryFloat _raiseInvalidOperationException .
      ^ self.
    ].
  ].
^ self asFraction asScaledDecimal: scale.

]

{ #category : 'Formatting' }
BinaryFloat >> asString [

"(Subclass responsibility.)  Returns a String corresponding to the value of the
 receiver.  Where applicable, returns one of the following Strings:
 'PlusInfinity', 'MinusInfinity', 'PlusQuietNaN', 'MinusQuietNaN',
 'PlusSignalingNaN', or 'MinusSignalingNaN'."

BinaryFloat subclassResponsibility: #asString

]

{ #category : 'Storing and Loading' }
BinaryFloat >> asStringLocaleC [

"Return the receiver as a String, but using the . decimal point,
 rather than any localized decimal point."

^self asFloat asStringLocaleC

]

{ #category : 'Formatting' }
BinaryFloat >> asStringUsingFormat: anArray [

"(Subclass responsibility.)  Returns a String corresponding to the receiver,
 using the format specified by anArray."

BinaryFloat subclassResponsibility: #asStringUsingFormat:

]

{ #category : 'Accessing' }
BinaryFloat >> at: anIndex put: aValue [

"Disallowed.  You may not change the value of a Float."

self shouldNotImplement: #at:put:

]

{ #category : 'Accessing' }
BinaryFloat >> denominator [

"Returns the denominator of a Fraction representing the receiver."

| myKind |
myKind := self kind .
myKind == #normal ifFalse:[
     "If an infinite or quiet NaN, returns self"
  ((myKind == #infinity) or: [ myKind == #quietNaN ])
     ifTrue: [ ^self ].

     "If a signaling NaN, raise a floating point exception & returns self"
  (myKind == #signalingNaN)
     ifTrue: [ self class _raiseInvalidOperationException.
               ^self _makeQuietNaN ].
  ].

^ (self asFraction) denominator

]

{ #category : 'Testing' }
BinaryFloat >> even [

"Returns true if the receiver is an even integer, false otherwise."
 ^ Float noInexactResultDo:[ (self \\ 2.0) == 0.0 ]

]

{ #category : 'Arithmetic' }
BinaryFloat >> factorial [

"Returns the factorial of the integer part of the receiver.  Returns 1 if the
 receiver is less than or equal to 1."

| x result |
result := 1.0E0  .
x := result .
self asInteger timesRepeat:[ result := result * x.  x := x + 1.0E0 ] .
^ result .

]

{ #category : 'Truncation and Rounding' }
BinaryFloat >> integerPart [

"Returns an integer representing the receiver truncated toward zero."

^ self truncated asFloat

]

{ #category : 'Arithmetic' }
BinaryFloat >> negated [

"Returns a Number that is the negation of the receiver."

"reimplemented for efficiency"

^ (-0.0 - self)

]

{ #category : 'Testing' }
BinaryFloat >> negative [

"Returns true if the receiver is less than zero, false if the receiver is zero
 or greater."

^ self < 0.0

]

{ #category : 'Accessing' }
BinaryFloat >> numerator [

"Returns the numerator of a Fraction representing the receiver."

| myKind |

   "If an infinite or quiet NaN, returns self"
myKind := self kind.
myKind == #normal ifFalse:[
  ((myKind == #infinity) or: [ myKind == #quietNaN ])
     ifTrue: [ ^self ].

     "If a signaling NaN, raise a floating point exception & returns self"
  (myKind == #signalingNaN)
     ifTrue: [ self class _raiseInvalidOperationException.
               ^self _makeQuietNaN ].
  ].

^ (self asFraction) numerator

]

{ #category : 'Testing' }
BinaryFloat >> odd [

"Returns true if the receiver is an odd integer, false otherwise.
 Compare to self truncated to reject values that are not quite integers,
 such as -9.9999999999999989E-01, which due to rounding would answer true below."

 self = (self _truncated: false) ifFalse: [ ^ false ].
 ^ Float noInexactResultDo:[ (self \\ 2.0) == 1.0 ]

]

{ #category : 'Testing' }
BinaryFloat >> positive [

"Returns true if the receiver is greater than or equal to zero, false if the
 receiver is less than zero."

^ self >= 0.0

]

{ #category : 'Arithmetic' }
BinaryFloat >> rem: aNumber [

"Returns the integer remainder defined in terms of quo: (division of the
 receiver by aNumber, with truncation toward zero)."

  "x rem: y | x=infinity or y=0 are invalid floating point
   operations and returns quiet NaNs"

(aNumber = 0.0) "0.0/0.0 is also invalid"
   ifTrue: [ ^ (aNumber asFloat) / (aNumber asFloat)].

(self _getKind == 3 "infinity" ) "infinity/infinity is also invalid"
   ifTrue: [ ^ self / self ].
^ super rem: aNumber

]

{ #category : 'Accessing' }
BinaryFloat >> size: anInteger [

"Disallowed.  You may not change the size of a Float."

self shouldNotImplement: #size:

]

{ #category : 'Testing' }
BinaryFloat >> strictlyPositive [

"Returns true if the receiver is greater than zero and false if it is less than
 or equal to zero."

^ self > 0.0

]

{ #category : 'Storing and Loading' }
BinaryFloat >> writeTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj writeClass: self class;
      nextPutAll: self asStringLocaleC ;
      space

]

{ #category : 'Testing' }
BinaryFloat >> isZero [
  ^ self = 0.0
]

