Extension { #name : 'ByteArray' }

{ #category : 'Encoded OOPs' }
ByteArray class >> bytesPerOop [
"Number of bytes consumed by OopType in C."
^ 8

]

{ #category : 'instance creation' }
ByteArray class >> fromBase64String: aString [
"Creates an instance of the receiver by decoding aString, which must be an
 instance of String or another single-byte character collection class. The
 argument must be in valid base64 format. The argument may contain newline
 and other whitespace charcaters, which are ignored.

 Raises an exception if aString is not a valid base64 string."

<primitive: 1063>
aString _validateClass: String .
self _primitiveFailed: #fromBase64String: args: { aString }

]

{ #category : 'instance creation' }
ByteArray class >> fromHexString: aHexString [
"Create a new instance of the receiver containing the byte values contained
 in aHexString. aHexString must be single-byte character collection (i.e,
 it must answer 1 to the message #charSize) and contain characters in the
 range of 0 - 9 and/or of a-f/A-F. Alphabetic hex characters may be in upper
 or lower case. A C hex prefix of 0x or 0X present in the first two characters
 of aHexString is permitted. A Smalltalk hex prefix of 16r in the first three
 characters is also permitted.

 If aHexString contains an odd number of hex digits, then the result object
 is left-padded, meaning the first four bits of the first byte in the result
 object will be zero.

 Example:
   (ByteArray fromHexString: 'abc') => 0a,bc

 Returns a new instance of the receiver on success or raises an exception
 on error."

^ self fromHexString: aHexString leftPadded: true

]

{ #category : 'instance creation' }
ByteArray class >> fromHexString: aHexString leftPadded: padLeft [

"Create a new instance of the receiver containing the byte values contained
 in aHexString. aHexString must be single-byte character collection (i.e,
 it must answer 1 to the message #charSize) and contain characters in the
 range of 0 - 9 and/or of a-f/A-F. Alphabetic hex characters may be in upper
 or lower case. A C hex prefix of 0x or 0X present in the first two characters
 of aHexString is permitted. A Smalltalk hex prefix of 16r in the first three
 characters is also permitted.

 If aHexString contains an odd number of hex digits, then the result object
 must be padded. The Boolean argument padLeft determines if left or right
 padding is used. If padLeft is true, the result is left-padded, meaning
 the first four bits of the first byte in the result object will be zero. If
 padLeft is false, the result is right-padded, meaning the last four bits of
 the last byte in the result object will be zero.

 Examples:
   (ByteArray fromHexString: 'abc' leftPadded: true)  => 0a,bc
   (ByteArray fromHexString: 'abc' leftPadded: false) => ab,c0

 Returns a new instance of the receiver on success or raises an exception
 on error."

<primitive: 1061>
aHexString _validateClasses: { String }.
padLeft   _validateClass: Boolean .
self _primitiveFailed: #fromHexString:leftPadded: args: { aHexString . padLeft }

]

{ #category : 'instance creation' }
ByteArray class >> fromString: aString [

"aString must be a kind of String or MultiByteString.  For MultiByteString
 arguments, the result will be in big-endian byte order."

<primitive: 653>
aString _validateClasses:{ String }.
self _primitiveFailed: #fromString: args: { aString }

]

{ #category : 'Encoded OOPs' }
ByteArray class >> oldBytesPerOop [
"Number of bytes consumed by Gs64 v1.1 OopType in C."
^ 4

]

{ #category : 'instance creation' }
ByteArray class >> withRandomBytes: howMany [

"Returns a new instance of the receiver of size howMany containing
 randomly generated data."

^ self new addRandomBytes: howMany startingAt: 1

]

{ #category : 'Message Digests' }
ByteArray >> _asMessageDigestKind: opCode [

"
opCode   Digest Algorithm   Digest Bits
=======================================
  1          md5              128
  2          sha1             160
  3          sha2-256         256
  4          sha2-512         512
  5          sha3-224         224
  6          sha3-256         256
  7          sha3-384         384
  8          sha3-256         512
=======================================

Postive opCode means return result as a LargeInteger.
Negative opCode means return result as a ByteArray.

"
<primitive: 666>
opCode _validateClass: SmallInteger .
(opCode == 0 or:[(opCode < -8) or:[ opCode > 8]])
  ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ -8 . 8 } ].
^ self _primitiveFailed: #_asMessageDigestKind:

]

{ #category : 'Converting' }
ByteArray >> _asUnicode16 [

<primitive: 926>

"Return an instance of Unicode7 or Unicode16 using the
minimum bytes per character required to represent the receiver.
Return nil if the receiver is not representable as Unicode7 nor Unicode16."

self _primitiveFailed:#_asUnicode16

]

{ #category : 'Converting' }
ByteArray >> _asUnicode7 [

<primitive: 925>
"Return an instance of Unicode7 if receiver can be represented
 as a Unicode7 string, else return nil."

self _primitiveFailed:#_asUnicode7

]

{ #category : 'Updating' }
ByteArray >> _basicAt: anIndex put: aValue [

"Store aValue in the receiver at anIndex .
 aValue should be a SmallInteger >= 0 and <= 255 . "
<primitive: 1002>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(aValue _isSmallInteger not or: [ (aValue < 0) | (aValue > 255) ])
  ifTrue: [^ aValue _error: #rtErrExpectedByteValue].

self _primitiveFailed: #at:put: args: { anIndex . aValue } .
self _uncontinuableError

]

{ #category : 'Compression' }
ByteArray >> _compressBytes [

^ self _compress

]

{ #category : 'Compression' }
ByteArray >> _computeCRC32: crc [
"Compute the crc32 value for the receiver, given the starting value.
 The crc32 value for a given receiver should be the same both before
 and after compression.
 The argument and result are 32bit unsigned values."

<primitive: 649>
crc _validateInstanceOf: SmallInteger .
(crc < 0 or:[ crc > 16rFFFFFFFF ]) ifTrue:[
  OutOfRange new name: 'initialCrc32' min: 0 max: 16rFFFFFFFF actual: crc; signal
].
self _primitiveFailed: #_computeCRC32: args: { crc }

]

{ #category : 'Private' }
ByteArray >> _decodeFromUtf8: unicodeResultBool [

"Decode UTF8 contents of the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString."

^ self _decodeFromUtf8: unicodeResultBool maxSize: nil

]

{ #category : 'Private' }
ByteArray >> _decodeFromUtf8: unicodeResultBool maxSize: aSize [
"Decode UTF8 contents of the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString.

 aSize should be nil for no limit on result size, or
 a SmallInteger specifying approximate maximum size of the result."

^ self _decodeUtf8StartingAt: 1 unicodeResult: unicodeResultBool maxSize: aSize bytesConsumed: nil

]

{ #category : 'Converting' }
ByteArray >> _decodeUtf8At: anOffset bytesConsumed: anArray [

"Attempt to decode one Utf8 codePoint starting at anOffset in the receiver.
 anArray should be a temporary Array of size 1 .
 If successful, returns a Character and (anArray at: 1) is a SmallInteger,
 the number of bytes consumed by the decode.
 If  (self size - anOffset ) are not enough bytes for a decode returns nil.
 Signals an Error if the bytes starting at anOffset are not legal UTF8 data .
 Receiver must be a small ByteArray ( self size <= 16272), otherwise an Error is signalled."

<primitive: 1086>
anOffset _validateClass: SmallInteger .
anArray _validateClass: Array .
^ self _primitiveFailed: #_decodeUtf8At:bytesConsumed: args: { anOffset . anArray }

]

{ #category : 'Converting' }
ByteArray >> _decodeUtf8StartingAt: anOffset unicodeResult: unicodeResultBool
                 maxSize: aSize bytesConsumed: anArray [

"Attempt to decode Utf8 data starting at anOffset in the receiver.
 If unicodeResultBool == true, result is a Unicode7 , Unicode16 or Unicode32.
 If unicodeResultBool == false, result is a String, DoubleByteString or
  QuadByteString.
 If there are insufficient bytes in the receiver and anArray ~~ nil,
 returns nil .
  If there are insufficient bytes in the receiver and anArray == nil, signals an Error.
 Signals an Error if the bytes starting at anOffset are not legal UTF8 data .
 If anArray ~~ nil, returns in (anArray at: 1) the number of bytes consumed by the decode."

<primitive: 493>
unicodeResultBool _validateClass: Boolean .
aSize _validateClass: SmallInteger .
anArray ifNotNil:[ anArray _validateClass: Array  ].
anOffset _validateClass: SmallInteger .

^ self _primitiveFailed: #_decodeUtf8StartingAt:unicodeResult:maxSize:bytesConsumed:
          args: { anOffset . unicodeResultBool . aSize . anArray }

]

{ #category : 'Compression' }
ByteArray >> _decompress [

"Decompress receiver using zlib.so .
 If receiver contains a dynamic instVar #sourceClass ,
 then the value of that instVar is used as the class of the result.
 otherwise the result is an instance of receiver's class."

  ^self _decompress: 0

]

{ #category : 'Compression' }
ByteArray >> _decompressBytes [

^ self _decompress

]

{ #category : 'Private-Updating' }
ByteArray >> _deleteNoShrinkFrom: startIndex to: endIndex anchorTailSize: aSize [

"Deletes bytes from startIndex to endIndex by sliding bytes
 from endIndex+1 to aSize-1 to the left and filling with zeros.
 The bytes from  (self size - aSize) to (self size) are not altered.
 The size of the receiver is not changed.
 The size of the receiver must be <= 65536 ."

<primitive: 626>

(startIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: startIndex].
(endIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: endIndex].
(aSize _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: aSize].
self size > 65536 ifTrue:[ self error:'receiver size > 65536' ].
startIndex < 1 ifTrue:[ self _errorIndexOutOfRange:startIndex  ] .
(endIndex < startIndex or:[ endIndex > self size] ) ifTrue:[ self _errorIndexOutOfRange: endIndex].
(aSize < 0 or:[ endIndex > (self size - aSize)]) ifTrue:[ aSize _error: #rtErrArgOutOfRange ].
self _primitiveFailed: #_deleteNoShrinkFrom:to:anchorTailSize:
     args: { startIndex . endIndex . aSize }

]

{ #category : 'Private-Updating' }
ByteArray >> _int32LittleEndianAt: startIndex put: anInteger [

"Receiver can be any byte format object.
Does not update any indexed dependent on receiver.
Overwrite the four bytes of the receiver from offset to offset+3 with the low-order four bytes of the given SmallInteger value, in little-endian signed format.
Offset is 1-based and need not be aligned.
Note that if the given value is a SmallInteger outside the range that can be represented in four bytes, the low-order four bytes will silently be stored.
  Returns anInteger
Errors:
  * The given offset is not a SmallInteger (ArgumentTypeError)
  * Some index from offset to offset+3 is outside the size of the receiver (OutOfRange)
  * The given value is not a SmallInteger (ArgumentTypeError)"


<primitive: 946>
startIndex _validateClass: SmallInteger .
anInteger _validateClass: SmallInteger .
(startIndex < 1 or:[ startIndex + 3 > self size]) ifTrue:[
  OutOfRange new
   name:'startIndex' min: 1 max: self size - 3 actual: startIndex  ;
   signal
].
self _primitiveFailed: #int32LittleEndianAt:put: args: { startIndex . anInteger}

]

{ #category : 'Private' }
ByteArray >> _keySizeInBytesForOpCode: opCode [

| code codes128Bit codes192Bit codes256Bit |
code := opCode abs .
codes128Bit := { 1 . 4 . 7 } .
codes192Bit := { 2 . 5 . 8 } .
codes256Bit := { 3 . 6 . 9 . 10 } .

(codes128Bit includesIdentical: code)
  ifTrue:[ ^ 16 ].
(codes192Bit includesIdentical: code)
  ifTrue:[ ^ 24 ].
(codes256Bit includesIdentical: code)
  ifTrue:[ ^ 32 ].

]

{ #category : 'Private' }
ByteArray >> _primAddRandomBytes: anIntHowMany startingAt: anIntOffset [

<primitive: 954>
anIntHowMany _validateClass: SmallInteger .
anIntOffset  _validateClass: SmallInteger .
anIntHowMany < 1
  ifTrue:[ anIntHowMany _error: #rtErrArgOutOfRange ] .
anIntOffset < 1
  ifTrue:[ anIntOffset _error: #rtErrArgOutOfRange ] .
self _primitiveFailed: #_primAddRandomBytes:startingAt: .

]

{ #category : 'Private' }
ByteArray >> _primAsymSignVerifyWithKey: aKey digestKind: opCode signature: aByteArray [

"Signs or verifies the message contained in the receiver using public key
 encryption. aKey must be an instance of GsTlsPrivateKey (which indicates a
 signing operation) or GsTlsPublicKey (which indicates a verify operation).
 For signing keys that require a message digest, anInt indicates one of the
 supported message digests for the specified key type. For signing keys
 that do not require a message digest, anInt must be zero.  See the tables
 below. The same message digest must be used for both signing and
 verifying.

 For signing operations, aByteArray must be a variant instance of ByteArray
 or nil, in which case a new instance of ByteArray will be created to
 contain the signature. For verifying operations, aByteArray must be a
 non-empty instance of ByteArray containing the signature from the signing.

 For secure signing and verifying with RSA keys only, a padding scheme must
 be used. The default RSA padding scheme is RSA_PKCS1_PADDING
 (PKCS #1 v1.5 padding), which is the most common type. However the newer
 PSS (Probabilistic Signature Scheme) is more secure and is recommended
 whenever possible. See RFC 3447 for additional information on PSS padding.
 Other RSA padding schemes, including no padding, are not supported due
 to known security vulnerabilities. The padding scheme selected must be the
 same for the signature and verification else the verification will fail.

 To sign or verify with RSA_PKCS1_PSS_PADDING, negate the digest opCode.
 For example, a message signed with an RSA private key using opCode 2 uses
 SHA1 message digest and (the default) RSA_PKCS1_PADDING. Signing a message
 with an RSA private key using opCode -2 uses SHA1 message digest and
 RSA_PKCS1_PSS_PADDING padding.

 RSA keys of type RSA-PSS may only use RSA_PKCS1_PSS_PADDING. PSS padding
 will be used for such keys in all cases, even if RSA_PKCS1_PADDING is
 requested. RSA-PSS private keys will answer #EVP_PKEY_RSA_PSS when sent
 the #sslAlgorithm message.

 Signing operations return aByteArray containing the signature on success
 and raise a CryptoError exception on error. Verify operations return true
 if the verification succeeds and raise a CryptoError exception if
 verification fails or an error occurs.

 Note that not all private/public key pairs support digital signatures.

 Key Type  Supported Digest Types
 ================================
 DSA       SHA1,SHA2
 ECDSA	   SHA1,SHA2
 RSA	   SHA1,SHA2,SHA3
 EC        None
 Ed25519   None
 Ed448     None
 ==============================


 Digest Type	OpCode
 =====================
 None           0
 SHA1		2
 SHA2-256	3
 SHA2-512	4
 SHA3-224	5
 SHA3-256	6
 SHA3-384	7
 SHA3-512	8
 =====================
"

<primitive: 1089>
opCode _validateClass: SmallInteger .
aKey _validateClasses: { GsTlsPrivateKey . GsTlsPublicKey } .
(aKey isKindOf: GsTlsPrivateKey)
  ifTrue:[ "signing, arg may be nil"
    aByteArray ifNotNil:[ aByteArray _validateClass: ByteArray ].
  ]
  ifFalse:[
     "verifying, arg must be a ByteArray"
    aByteArray _validateClass: ByteArray
].
self _validateKey: aKey withDigestKind: opCode .
^ self _primitiveFailed: #_primAsymSignVerifyWithKey:digestKind:signature:

]

{ #category : 'Private' }
ByteArray >> _primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode into: destObjOrNil [
"Private method for invoking non-AEAD encrypt/decrypt modes. See the method:
    #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
    into: destObjOrNil tag: tag extraData: eData
 for more information."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: opCode
       into: destObjOrNil
       tag: nil
       extraData: nil

]

{ #category : 'Private' }
ByteArray >> _primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
into: destObjOrNil tag: tag extraData: eData [

"Private method for encrypting or decrypting the receiver.
 The following encryption schemes are currently supported:

 ==================================================
                          Key     Salt    Tag
 opCode Cipher   Mode  bits/Bytes Size   Bytes
 ==================================================
   1     AES     CBC     128/16    16     N/A
   2     AES     CBC     192/24    16     N/A
   3     AES     CBC     256/32    16     N/A
   4     AES     OCB     128/16    12     16
   5     AES     OCB     192/24    12     16
   6     AES     OCB     256/32    12     16
   7     AES     GCM     128/16    12     16
   8     AES     GCM     192/24    12     16
   9     AES     GCM     256/32    12     16
  10   CHACHA20 Poly1305 256/32    12     16
 ==================================================

 AES encryption/decryption (Advanced Encryption Standard) is performed
 using the OpenSSL open source package and the AES specification,
 available at:
   http://csrc.nist.gov/publications/fips/fips197/fips-197.pdf

 CBC is an acronym for cipher block chaining. See the referenced AES
 specification document for further details.

 OCB is an acronym for Offset Cookbook Mode. See RFC 7253 for further
 details.

 GCM is an acronym for Galois Counter Mode. See RFC 5288 for further
 details.

 OCB, GCM and Poly1305 are AEAD modes. AEAD is an acronym for
 Authenticated Encryption with Associated Data. AEAD provides data
 authenticity, confidentiality, and integrity. It also supports
 Additional Authenticated Data (AAD). AAD is not encrypted and therefore
 not kept confidential, however AAD authenticity and integrity are
 guaranteed. AAD is not included in the encrypted payload but must be
 provided in order to decrypt the data. AAD is optional and the eData
 argument may be nil if AAD is not required.

 AEAD encryption/decryption is performed using the OpenSSL open source
 package and is implemented to conform to ISO/IEC standard 19772:2009.
 See https://www.iso.org/standard/46345.html for further information.

 opCode must be an instance of SmallInteger and be one of the values from
 the first column of the above table. A positive value indicates the data
 is to be encrypted and a negative value indicates decryption.

 aKey and aSalt must be instances of ByteArray of the correct size
 per the above table, otherwise an error is raised.

 destObjOrNil must be nil or an instance of a non-invariant byte object.
 If destObjOrNil is nil, the result of the operation will be placed into a
 new instance of ByteArray (encryption) or String (decryption). Otherwise
 the result will be placed into the given byte object starting at offset 1.
 The size of destObjOrNil will be modified to correctly contain all
 encrypted or decrypted data, and may differ from the size of the receiver
 due to the automatic addition or removal of padding by the cipher
 algorithm.

 When encrypting a receiver that has a character size greater than one, data
 is placed into big-endian byte order before encryption.

 When decrypting into a destObjOrNil object that a character size greater
 than one, data is converted to big-endian byte order after decryption.

 During AEAD encryption, a tag is generated which is used during decryption
 to ensure data integrity. The tag data will be stored into the tag
 argument, which must an instance of a variant byte object. During AEAD
 decryption, tag must be a byte object containing the tag bytes returned
 during encryption. For non-AEAD modes, tag must be nil.

 During AEAD encryption, eData must be nil or a byte object with a character
 size of one containing additional data to be used in generating the tag
 value. eData is NOT added to the encrypted payload. During decryption,
 eData must be a byte object with a character size of one containing the
 same bytes provided during encryption or nil if no byte object was
 provided. For non-AEAD modes, eData must be nil.

 Successful encryption or decryption returns encrypted or decrypted data,
 which is stored into destObjOrNil if it was provided or a new byte object
 if it was not. An exception is raised if encryption or decryption fails.
"

<primitive: 953>
self _validateEncryptionOpCode: opCode ;
     _validateEncryptionKey: aKey forOpCode: opCode ;
     _validateEncryptionSalt: aSalt forOpCode: opCode ;
     _validateEncryptionExtraData: eData forOpCode: opCode .
destObjOrNil ifNotNil:[ destObjOrNil _validateIsBytes ; validateIsVariant ] .
self _validateEncryptionTag: tag forOpCode: opCode .
^ self _primitiveFailed: #_primEncryptDecryptWithKey:salt:opCode:into:tag:extraData:

]

{ #category : 'Private-Updating' }
ByteArray >> _reverseDeleteNoShrinkFrom: startIndex to: endIndex anchorHeadSize: aSize [

"Deletes bytes from startIndex to endIndex.  startIndex must be > aSize,
 and the bytes from 1 to aSize  are not altered.
 The bytes from aSize + 1 to startIndex - are shifted to the right by
 the distance (endIndex - startIndex + 1), and the bytes from (aSize + 1)
 to (endIndex - (startIndex - aSize) +1) are zeroed.
 The size of the receiver is not changed.
 The size of the receiver must be <= 65536 ."

<primitive: 627>

(startIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: startIndex].
(endIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: endIndex].
(aSize _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: aSize].
self size > 65536 ifTrue:[ self error:'receiver size > 65536' ].
startIndex < 1 ifTrue:[ self _errorIndexOutOfRange:startIndex  ] .
(endIndex < 1 or:[ endIndex > self size]) ifTrue:[ self _errorIndexOutOfRange:endIndex ].
startIndex < aSize ifTrue:[ self error:'startIndex < aSize' ].
(self size - aSize) < endIndex ifTrue:[ aSize _error: #rtErrArgOutOfRange ].
(endIndex - startIndex + 1) < 0 ifTrue:[ self error: 'numToDelete less than 0' ].
((endIndex - startIndex + 1) > ((self size) - startIndex)) ifTrue:[ self error: 'numToDelete > numDeletable' ].
self _primitiveFailed: #_reverseDeleteNoShrinkFrom:to:anchorHeadSize:
     args: { startIndex . endIndex . aSize }

]

{ #category : 'Private' }
ByteArray >> _saltSizeForOpCode: opCode [

^ opCode abs >= 4
    ifTrue:[ 12 ]
   ifFalse:[ 16 ]

]

{ #category : 'Private' }
ByteArray >> _validate: anObj isSize: expectedSize [

anObj _basicSize == expectedSize
  ifFalse:[ anObj _error: #rtErrBadSize args: { expectedSize . anObj _basicSize } ] .

]

{ #category : 'Private' }
ByteArray >> _validateEncryptionExtraData: eData forOpCode: opCode [
opCode abs > 3  "AEAD encrypt/decrypt"
  ifTrue:[ eData ifNotNil:[ eData _validateIsBytes ]]
 ifFalse:[ eData _validateClass: UndefinedObject ]

]

{ #category : 'Private' }
ByteArray >> _validateEncryptionKey: aKey forOpCode: opCode [
aKey _validateClass: ByteArray .
self _validate: aKey isSize: (self _keySizeInBytesForOpCode: opCode)

]

{ #category : 'Private' }
ByteArray >> _validateEncryptionOpCode: opCode [
opCode _validateClass: SmallInteger .
(((opCode < -10) or:[opCode > 10]) or:[opCode == 0])
  ifTrue:[opCode _error: #rtErrArgOutOfRange args:{ -10 . 10 } ] .
^ true

]

{ #category : 'Private' }
ByteArray >> _validateEncryptionSalt: salt forOpCode: opCode [
salt _validateClass: ByteArray .
self _validate: salt isSize: (self _saltSizeForOpCode: opCode)

]

{ #category : 'Private' }
ByteArray >> _validateEncryptionTag: tag forOpCode: opCode [
| absCode |
absCode := opCode abs .
absCode > 3 "AEAD encrypt/decrypt"
  ifTrue:[
    tag _validateClass: ByteArray .
    opCode < 3 ifTrue:[ tag validateIsVariant ]			"AEAD Encrypting"
              ifFalse:[ self _validate: tag isSize: 16 ] . 	"AEAD Decrypting"
  ] ifFalse:[ tag _validateClass: UndefinedObject ] "Not AEAD"

]

{ #category : 'Private' }
ByteArray >> _validateInteger: anInt inRangeFrom: lowest to: highest [
anInt _validateClass: SmallInteger .
((anInt < lowest) or:[anInt > highest])
  ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ lowest . highest } ] .
^ true

]

{ #category : 'Private' }
ByteArray >> _validateKey: aKey withDigestKind: opCode [
|alg|
alg := aKey algorithm . "#RSA, #DSA, #DH, #EC, etc"
alg == #RSA "SHA1,2,3 allowed"
  ifTrue:[ ^ self _validateInteger: opCode abs inRangeFrom: 2 to: 8 ].
alg == #DSA "SHA1 and SHA2 only"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 2 to: 4 ].
alg == #EC "No digest required/allowed"
  ifTrue:[ ^ self _validateInteger: opCode inRangeFrom: 0 to: 0 ].

^ CryptoError signal: 'Invalid key kind for signing or verifying'

]

{ #category : 'Comparing' }
ByteArray >> = aByteArray [

"Returns true if all of the following conditions are true, otherwise
  returns false.

 1.  The receiver and aByteArray are of the same class.
 2.  The two Arrays are the same size.
 3.  The corresponding elements of the receiver and aByteArray
     are equal."

<primitive: 613>

^ super = aByteArray.

]

{ #category : 'Adding' }
ByteArray >> addAll: aCollection [

"Adds all of the elements of aCollection to the receiver and returns
 aCollection."

| collectionSize |

(self == aCollection) ifTrue: [ ^ self addAll: (aCollection copy) ].

(aCollection isKindOf: ByteArray) ifTrue:[
  collectionSize := aCollection size.
  (collectionSize ~~ 0) ifTrue:[
    self insertAll: aCollection at: (self size + 1)
    ].
  ^ aCollection.
  ].
aCollection accompaniedBy: self do: [ :me :each | me add: each ].
^ aCollection

]

{ #category : 'Random Value Generation' }
ByteArray >> addRandomBytes: howMany startingAt: offset [

"Adds howMany random bytes to the receiver starting at offset.
 This method grows the receiver to accomodate the random data
 if necessary.   Any existing data in the range of offset to
 (offset + howMany - 1) is overwritten by this method.

 Both arguments are expected to be positive SmallIntegers.

 Returns the receiver."

^ self _primAddRandomBytes: howMany startingAt: offset

]

{ #category : 'Decrypting' }
ByteArray >> aesDecryptWith128BitKey: aKey salt: aSalt [
"Decrypts the receiver using 128 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith128BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Decrypting' }
ByteArray >> aesDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil [
"Decrypts the receiver using 128 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -1 into: destObjOrNil

]

{ #category : 'Decrypting' }
ByteArray >> aesDecryptWith192BitKey: aKey salt: aSalt [
"Decrypts the receiver using 192 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith192BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Decrypting' }
ByteArray >> aesDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil [
"Decrypts the receiver using 192 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -2 into: destObjOrNil

]

{ #category : 'Decrypting' }
ByteArray >> aesDecryptWith256BitKey: aKey salt: aSalt [
"Decrypts the receiver using 256 bit AES decryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesDecryptWith256BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Decrypting' }
ByteArray >> aesDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil [
"Decrypts the receiver using 256 bit AES decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: -3 into: destObjOrNil

]

{ #category : 'Encrypting' }
ByteArray >> aesEncryptWith128BitKey: aKey salt: aSalt [
"Encrypts the receiver using 128 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith128BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Encrypting' }
ByteArray >> aesEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil [
"Encrypts the receiver using 128 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 1  into: destObjOrNil

]

{ #category : 'Encrypting' }
ByteArray >> aesEncryptWith192BitKey: aKey salt: aSalt [
"Encrypts the receiver using 192 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith192BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Encrypting' }
ByteArray >> aesEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil [
"Encrypts the receiver using 192 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 2 into: destObjOrNil

]

{ #category : 'Encrypting' }
ByteArray >> aesEncryptWith256BitKey: aKey salt: aSalt [
"Encrypts the receiver using 256 bit AES encryption and places the result into
 a new instance of the class of the receiver.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self aesEncryptWith256BitKey: aKey salt: aSalt into: nil

]

{ #category : 'Encrypting' }
ByteArray >> aesEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil [
"Encrypts the receiver using 256 bit AES encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 3 into: destObjOrNil

]

{ #category : 'Authenticated Decrypting' }
ByteArray >> aesGcmDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 128 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -7
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
ByteArray >> aesGcmDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 192 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -8
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
ByteArray >> aesGcmDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 256 bit AES-GCM decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -9
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> aesGcmEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 128 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey salt: aSalt opCode: 7 into: destObjOrNil tag: tag extraData: eData

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> aesGcmEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 192 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 8
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> aesGcmEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 256 bit AES-GCM encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 9
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Decrypting' }
ByteArray >> aesOcbDecryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 128 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -4
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
ByteArray >> aesOcbDecryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 192 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -5
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Decrypting' }
ByteArray >> aesOcbDecryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using 256 bit AES-OCB decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -6
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> aesOcbEncryptWith128BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 128 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 4
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> aesOcbEncryptWith192BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 192 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 5
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> aesOcbEncryptWith256BitKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using 256 bit AES-OCB encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 6
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Converting' }
ByteArray >> asBase64String [
"Return a String which represents the receiver represented in base64 format."

^ self asBase64StringOnOneLine: true

]

{ #category : 'Converting' }
ByteArray >> asBase64StringOnOneLine: aBoolean [
"Return a String which represents the receiver represented in base64 format.
 If aBoolean is true, the resulting is one long line which does not contain
 newline characters.  If aBoolean is false, newline characters are inserted
 such that each line does not exceed 64 characters."

<primitive: 1062>
aBoolean _validateClass: Boolean .
self _primitiveFailed: #asBase64StringOnOneLine:

]

{ #category : 'Converting' }
ByteArray >> asHexString [

"Returns a String containing a hexadecimal printed representation of the
 contents of the receiver.  For example, the message 'abc' asHexString
 returns the String '616263'.

 The receiver must be a byte format object."

<primitive: 467>
self _validateByteClass: ByteArray .
self _primitiveFailed: #asHexString .

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asHmacSumWithDigestKind: opCode key: keyString [

"Computes the keyed-hash message authentication code (HMAC) of the receiver using
 the message digest algorithm indicated by opCode and secret key keyString.

 opCode must be one of the following:

 opCode   Digest Algorithm
 =========================
  1          md5
  2          sha1
  3          sha2-256
  4          sha2-512
  5          sha3-224
  6          sha3-256
  7          sha3-384
  8          sha3-256
 =========================

 secretKey must be an instance or subclass of a ByteArray or String and must
 have a character size of one, i.e.: its class must answer 1 when sent the
 message #_bytesPerWord. secretKey must have a size between 1 and 64 bytes.

 Answers the HMAC of the receiver as a LargeInteger."

<primitive: 1060>
opCode _validateClass: SmallInteger .
((opCode < 1) or:[ opCode > 8]) ifTrue:[ opCode _error: #rtErrArgOutOfRange args:{ 1 . 8 } ].
keyString _validateClasses: { String . ByteArray } .
((keyString _basicSize > 64) or:[keyString _basicSize < 1])
  ifTrue:[ keyString _error: #rtErrBadSize args: { 64 . keyString _basicSize } ] .
self _primitiveFailed: #asHmacSumWithDigestKind:key: args: { opCode . keyString } .

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asMd5HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the MD5 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asMd5HmacWithKey: keyString) asHexStringWithLength: 32

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asMd5HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the MD5 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 1 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asMd5String [

"Compute the 128 bit MD5 message digest for the receiver and return it as
 a 32 character string of hexadecimal characters."

^ self md5sum asHexStringWithLength: 32

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha1HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha1 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha1HmacWithKey: keyString) asHexStringWithLength: 40

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha1HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha1 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 2 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asSha1String [

"Compute the 160 bit SHA1 message digest for the receiver and return it as
 a 40 character string of hexadecimal characters."

^ self sha1Sum asHexStringWithLength: 40

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha256HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha256HmacWithKey: keyString) asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha256HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 3 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asSha256String [

"Compute the 256 bit SHA-2 message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha256Sum asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_224HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 224 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_224HmacWithKey: keyString) asHexStringWithLength: 56

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_224HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 224 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 5 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asSha3_224String [

"Compute the SHA3 224 bit message digest for the receiver and return it as
 a 56 character string of hexadecimal characters."

^ self sha3_224Sum asHexStringWithLength: 56

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_256HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_256HmacWithKey: keyString) asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_256HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 256 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 6 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asSha3_256String [

"Compute the SHA3 256 bit message digest for the receiver and return it as
 a 64 character string of hexadecimal characters."

^ self sha3_256Sum asHexStringWithLength: 64

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_384HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 384 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_384HmacWithKey: keyString) asHexStringWithLength: 96

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_384HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 384 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 7 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asSha3_384String [

"Compute the SHA3 384 bit message digest for the receiver and return it as
 a 96 character string of hexadecimal characters."

^ self sha3_384Sum asHexStringWithLength: 96

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_512HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha3_512HmacWithKey: keyString) asHexStringWithLength: 128

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha3_512HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the SHA3 512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 8 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asSha3_512String [

"Compute the SHA3 512 bit message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha3_512Sum asHexStringWithLength: 128

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha512HmacStringWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a String of hexadecimal characters."

^ (self asSha512HmacWithKey: keyString) asHexStringWithLength: 128

]

{ #category : 'Message Authentication Codes' }
ByteArray >> asSha512HmacWithKey: keyString [
"Computes the keyed-hash message authentication code (HMAC) of the receiver
 using the sha512 message digest algorithm and secret key keyString. Answers
 the resulting HMAC of the receiver as a LargeInteger."

^ self asHmacSumWithDigestKind: 4 key: keyString

]

{ #category : 'Message Digests' }
ByteArray >> asSha512String [

"Compute the 512 bit SHA-2 message digest for the receiver and return it as
 a 128 character string of hexadecimal characters."

^ self sha512Sum asHexStringWithLength: 128

]

{ #category : 'Converting' }
ByteArray >> asUnicodeString [

"This will eventually be Deprecated.
 New code should use bytesIntoUnicode.
 Return an instance of Unicode7 or Unicode16
 using the class with smallest character size needed to
 represent a copy of the receiver.
 The receiver is interpreted as an array of 8 bit codePoints."

<primitive: 927>

self _primitiveFailed:#asUnicodeString

]

{ #category : 'Accessing' }
ByteArray >> at: anIndex [

"Returns the value of an indexed variable in the receiver.
 The argument anIndex must not be larger than the size of the
 receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, or if the receiver is not indexable."

<primitive: 974>

(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError

]

{ #category : 'Accessing' }
ByteArray >> atOrNil: anIndex [

"Returns the value of an indexed variable in the receiver,
 or nil if anIndex is out of range .

 Generates an error if anIndex is not a SmallInteger,
 or if the receiver is not indexable."

<primitive: 974>

(anIndex _isInteger)
  ifTrue: [ ^ nil ]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError

]

{ #category : 'Private-Comparing' }
ByteArray >> at: anIndex equals: aString [

^self compareStringAt: anIndex to: aString startingAt: 1 sizeBytes: 0 useCase: true
   "size is not encoded, compare entire <aString>"

]

{ #category : 'Private-Comparing' }
ByteArray >> at: anIndex equals: aString useCase: aBool [

^self compareStringAt: anIndex to: aString startingAt: 1 sizeBytes: 0 useCase: aBool
   "size is not encoded, compare entire <aString>"

]

{ #category : 'Updating' }
ByteArray >> at: anIndex put: aValue [

"Store aValue in the receiver at anIndex .
 aValue should be a SmallInteger >= 0 and <= 255 . "
<primitive: 1002>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(aValue _isSmallInteger not or: [ (aValue < 0) | (aValue > 255) ])
  ifTrue: [^ aValue _error: #rtErrExpectedByteValue].

self _primitiveFailed: #at:put: args: { anIndex . aValue } .
self _uncontinuableError

]

{ #category : 'Private' }
ByteArray >> at: index put: aString fromOffset: stringOffset sizeBytes: numBytes [
 "Write a string into the receiver, which must be a byte object (usually
  a ByteArray).  The offset to start writing the receiver is given by the
  at: argument, the object to write (must also be a byte object, usually
  a string) is given by the put: argument.  The fromOffset: argument
  specifies the offset from within the given string to start copying.
  The sizeBytes argument specifies the width it takes to store the size
  of the string.  The size of the string is stored in the first offset
  of the byte array.  The sizeBytes must one of the following values:

  sizeBytes   max string size      Notes
     0        255                  Do not store the size in the byte array
     1        255
     2        65535
     4        4 GB - 1
     8        max legal object size - 8.
  Returns the receiver."

<primitive: 622>
self _primitiveFailed: #at:put:fromOffset:sizeBytes:
     args: { index . aString . stringOffset . numBytes }

]

{ #category : 'Private' }
ByteArray >> at: index put: aNumber signed: aBool width: aWidthInBytes [

"Store the big-endian or the native represention of an aNumber into
 the specified position in the receiver.
 aWidthInBytes is allowed to be 1,2,3,4,8, 512, 1024, or 2048 if
 aNumber is an Integer . Values larger than 8 indicate the value should
 be stored in native integer format rather than big-endian format.
 512 means store 2 bytes in native format, 1024 means 4 bytes and 
 2048 means 8 bytes.

 aWidthInBytes is allowed to be 4, 8, 1024 or 2048 if aNumber is
 a BinaryFloat. 1024 and 2048 indicate the 4 or 8 bytes of the 
 float should be stored in native format respectively.
 If representation of an Integer requires more than aWidthInBytes ,
 the primitive will fail.
 If coercion of a BinaryFloat to a 4 byte C float would produce
 loss of exponent bits, the primitive will fail.
 If coercion of a BinaryFloat to a 4 byte C float would cause loss
 of precision that would convert a non-zero value to zero ,
 the primitive will fail .
 "

<primitive: 618>
index _validateClass: SmallInteger.
aNumber _validateClass: Number.
aBool _validateClass: Boolean.
aWidthInBytes _validateClass: SmallInteger.
(aWidthInBytes < 1) ifTrue: [ self error: 'aWidthInBytes negative or zero'].
( #( 1 2 3 4 8 512 1024 2048 ) includesIdentical: aWidthInBytes)
  ifFalse: [ self error: 'illegal value for aWidthInBytes'].
self _primitiveFailed: #at:put:signed:width:
     args: { index . aNumber . aBool . aWidthInBytes }

]

{ #category : 'Encoded OOPs' }
ByteArray >> at: startIndex putAllOldOopsOfObjects: anArray [
"Store the OOP of each object contained in anArray into 4 bytes of the receiver
 starting at startIndex.  The receiver is grown as necessary."
| idx |
idx := startIndex .
1 to: anArray size do:[:j |
  self at: idx putOldOopValueOfObject: (anArray at: j ).
  idx := idx + 4 .
  ].
^ self

]

{ #category : 'Encoded OOPs' }
ByteArray >> at: startIndex putAllOopsOfObjects: anArray [
"Store the OOP of each object contained in anArray into 8 bytes of the receiver
 starting at startIndex.  The receiver is grown as necessary."
| idx |
idx := startIndex .
1 to: anArray size do:[:j |
  self at: idx putOopValueOfObject: (anArray at: j ).
  idx := idx + 8 .
  ].
^ self

]

{ #category : 'Updating' }
ByteArray >> at: index putChar: char [

"Stores the codepoint of char in the receiver at index. Stores one byte if
 char is a single byte character, two bytes if it is a double byte
 character. , 4 bytes if it is a quad byte character. Returns char."

<primitive: 502>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index > (self size + 1)) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

char _validateClass: Character .
self _validateByteClass: ByteArray .
self _primitiveFailed: #at:putChar: args: { index . char }

]

{ #category : 'Encoded OOPs' }
ByteArray >> at: aSmallInt putOldOopValueOfObject: anObject [
"Store the 4 byte Gs64 v1.1 OOP of anObject in the receiver.
 aSmallInt is an offset in the receiver where the bytes are to be stored.
 The bytes are stored in big-endian order.
 The receiver is grown if needed (use aSmallInt one past end to append)
 anObject cannot be a special object, and its oopNumber must be < 2 billion."

<primitive: 634>
aSmallInt _validateClass: SmallInteger .
aSmallInt < 1 ifTrue:[self _errorIndexOutOfRange: aSmallInt].
aSmallInt > (self size + 1) ifTrue:[self _errorIndexOutOfRange: aSmallInt].
self _primitiveFailed: #at:putOldOopValueOfObject:
     args: { aSmallInt . anObject }

]

{ #category : 'Encoded OOPs' }
ByteArray >> at: aSmallInt putOopValueOfObject: anObject [
"Store the 8 byte OOP of anObject in the receiver.
 aSmallInt is an offset in the receiver where the bytes are to be stored.
 The bytes are stored in big-endian order.
 The receiver is grown if needed (use aSmallInt one past end to append)"
<primitive: 633>
aSmallInt _validateClass: SmallInteger .
aSmallInt < 1 ifTrue:[self _errorIndexOutOfRange: aSmallInt].
aSmallInt > (self size + 1) ifTrue:[self _errorIndexOutOfRange: aSmallInt].
self _primitiveFailed: #at:putOopValueOfObject: args: { aSmallInt . anObject }

]

{ #category : 'Private' }
ByteArray >> at: index signed: aBool width: aWidthInBytes [

"Retrieve a Number stored in big-endian or native represention,
 from the specified position and width in the receiver.
 aWidthInBytes of 1,2,3,4 or 8 retrieves an Integer in big-endian
 format.
 aWidthInBytes of 512, 1024 or 2048 retrieves an Integer in native
 format.
 aWidthInBytes of -4 or -8 retrieves a SmallDouble or Float,
 in big-endian format and aBool is ignored.
 aWidthInBytes of 4096 or 8192 retrieves a SmallDouble or Float in
 native format and aBool is ignored.  " 

<primitive: 619>
self _primitiveFailed: #at:signed:width:
     args: { index . aBool . aWidthInBytes }

]

{ #category : 'Private' }
ByteArray >> at: index sizeBytes: anInt stringSize: anIntOrNil [
 "Read a string from the receiver into a new string and return the new
  string.  The offset to start writing the receiver is given by the
  at: argument. The sizeBytes argument specifies the width (in bytes) it takes to
  store the size of the string.  In this case, it is assumed the size
  of the string is stored in the first sizeBytes of the receiver. The
  stringSize: argument should be nil in this case, which means read the
  string size from the byte array.  If the stringSize: argument is not nil,
  it means the size is not encoded in the string and the caller is
  passing in the string size.
  Returns the new string object created from reading"

<primitive: 623>
self _primitiveFailed: #at:sizeBytes:stringSize:
     args: { index . anInt . anIntOrNil }

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> authenticatedEncryptionExample [

| key salt enc dec hamlet extraData tag|
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 12 .

hamlet :=
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .

extraData := 'Hamlet: Act V, Scene i'.

tag := ByteArray new.
enc := hamlet aesOcbEncryptWith256BitKey: key salt: salt
       into: ByteArray new tag: tag extraData: extraData .
dec := enc aesOcbDecryptWith256BitKey: key salt: salt
       into: String new tag: tag extraData: extraData .
^ dec = hamlet

]

{ #category : 'Accessing' }
ByteArray >> bitAtZ: zeroBasedOffset [
  "Returns a bit of the receiver, as if receiver is an Array of bytes .
   ((zeroBasedOffset bitShift: -3)+1)  defines the byte addressed in the receiver
    Within that byte,  bitAt:((zeroBasedOffset bitAnd: 7) + 1) is modified.
    Returns 0 or 1 ."
  | byte ofs  mask |
  ofs  := (zeroBasedOffset bitShift: -3) + 1 .
  byte := self at: ofs  .
  mask := 1 bitShift:( zeroBasedOffset bitAnd: 7 ).
  ^ (byte bitAnd: mask) == 0 ifTrue:[ 0 ] ifFalse:[ 1]

]

{ #category : 'Updating' }
ByteArray >> bitAtZ: zeroBasedOffset put: anInteger [
  "Stores a bit into the receiver, as if receiver is an Array of bytes .
   ((zeroBasedOffset bitShift: -3)+1)  defines the byte addressed in the receiver.
    Within that byte,  bitAt:((zeroBasedOffset bitAnd: 7) + 1) is modified.
    anInteger must be one of  0, 1, true, false ."
  | byte ofs  mask |
  ofs  := (zeroBasedOffset bitShift: -3) + 1 .
  byte := self at: ofs  .
  mask := 1 bitShift:( zeroBasedOffset bitAnd: 7 ).
  (anInteger == 0 or:[ anInteger == false]) ifTrue:[
    mask := 16rFF bitXor: mask .
    byte := byte bitAnd: mask .
    self at: ofs put: byte .
    ^ self .
  ].
  (anInteger == 1 or:[ anInteger == true]) ifTrue:[
    byte := byte bitOr: mask .
    self at: ofs put: byte .
    ^ self .
  ].
  ArgumentError signal: anInteger asString , '  is not one of  0, 1, true, false'

]

{ #category : 'Accessing' }
ByteArray >> byteAt: anIndex [

"Returns the value of an indexed variable in the receiver.
 The argument anIndex must not be larger than the size of the
 receiver, and must not be less than 1.

 Generates an error if anIndex is not a SmallInteger or is out of
 bounds, or if the receiver is not indexable."

<primitive: 974>

(anIndex _isInteger)
  ifTrue: [^ self _errorIndexOutOfRange: anIndex]
  ifFalse: [^ self _errorNonIntegerIndex: anIndex].
self _primitiveFailed: #at: args: { anIndex } .
self _uncontinuableError

]

{ #category : 'Updating' }
ByteArray >> byteAt: anIndex put: aValue [

"Store aValue in the receiver at anIndex .
 aValue should be a SmallInteger >= 0 and <= 255 . "
<primitive: 1002>

(anIndex _isInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
((anIndex < 1) | (anIndex > (self size + 1))) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].
(aValue _isSmallInteger not or: [ (aValue < 0) | (aValue > 255) ])
  ifTrue: [^ aValue _error: #rtErrExpectedByteValue].

self _primitiveFailed: #at:put: args: { anIndex . aValue } .
self _uncontinuableError

]

{ #category : 'Converting' }
ByteArray >> bytesIntoString [

"Returns an instance of String containing the bytes of the receiver
 without doing any decoding."

 ^ String withBytes: self

]

{ #category : 'Converting' }
ByteArray >> bytesIntoUnicode [
"Return an instance of Unicode7 or Unicode16
 using the class with smallest character size needed to
 represent a copy of the receiver.
 The receiver is interpreted as an array of 8 bit codePoints."

<primitive: 927>

self _primitiveFailed: #bytesIntoUnicode

]

{ #category : 'Authenticated Decrypting' }
ByteArray >> chacha20Poly1305DecryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Decrypts the receiver using CHACHA20-Poly1305 decryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: -10
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Authenticated Encrypting' }
ByteArray >> chacha20Poly1305EncryptWithKey: aKey salt: aSalt into: destObjOrNil tag: tag extraData: eData [
"Encrypts the receiver using CHACHA20-Poly1305 encryption and places the result into
 destObjOrNil.

 See the comments in method:
   #_primEncryptDecryptWithKey: aKey salt: aSalt opCode: opCode
         into: destObjOrNil tag: nil extraData: nil
 for more information on encryption and decryption."

^ self _primEncryptDecryptWithKey: aKey
       salt: aSalt
       opCode: 10
       into: destObjOrNil
       tag: tag
       extraData: eData

]

{ #category : 'Accessing' }
ByteArray >> charAt: index [

"Returns a (single byte) Character located at index in the receiver."

<primitive: 500>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index > self size) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

self _validateByteClass: ByteArray .
self _primitiveFailed: #charAt: args: { index }

]

{ #category : 'Private-Comparing' }
ByteArray >> compareCaseInsensitiveShortStringAt: startIndex to: aString startingAt: stringIndex [
"  Compare a string encoded in the receiver (aByteArray) to the given string
  starting at the offset specified in the given string.  The byte array is assumed
  to have the size of the encoded string encoded in the first 1 byte.
  Returns true if the strings are case insensitive match, false otherwise."

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 1 useCase: false
    "size is encoded in first byte of receiver"

]

{ #category : 'Private-Comparing' }
ByteArray >> compareShortStringAt: startIndex to: aString startingAt: stringIndex [

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 1 useCase: true
   "size is encoded in first byte of receiver"

]

{ #category : 'Private-Comparing' }
ByteArray >> compareShortStringAt: startIndex to: aString startingAt: stringIndex useCase: aBool [

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 1 useCase: aBool
   "size is encoded in first byte of receiver"

]

{ #category : 'Private-Comparing' }
ByteArray >> compareString4gAt: startIndex to: aString startingAt: stringIndex [

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 4
  "size is encoded in first byte of receiver"

]

{ #category : 'Private-Comparing' }
ByteArray >> compareString64kAt: startIndex to: aString startingAt: stringIndex [

^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: 2
   "size is encoded in first byte of receiver"

]

{ #category : 'Private-Comparing' }
ByteArray >> compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: numSizeBytes [

"  Compare a string encoded in the receiver (aByteArray) to the given string
  starting at the offset specified in the given string.  If sizeBytes not
  zero, the byte array is assumed to have the size of the encoded string
  encoded in the first numSizeBytes bytes at the startIndex.  If sizeBytes
  is zero, the size   has not been encoded in the byteArray and the entire
  length of the second string is compared.  numSizeBytes must be 0, 1, 2, 4,
  or 8.
  Returns true if the strings match, false otherwise."
        ^self compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: numSizeBytes useCase: true

]

{ #category : 'Private-Comparing' }
ByteArray >> compareStringAt: startIndex to: aString startingAt: stringIndex sizeBytes: numSizeBytes useCase: aBool [

"  Compare a string encoded in the receiver (aByteArray) to the given string
  starting at the offset specified in the given string.  If sizeBytes not
  zero, the byte array is assumed to have the size of the encoded string
  encoded in the first numSizeBytes bytes at the startIndex.  If sizeBytes
  is zero, the size has not been encoded in the byteArray and the entire
  length of the second string is compared.  numSizeBytes must be 0, 1, 2, 4,
  or 8.  <aBool> determines if the comparision is case sensitive.
  Returns true if the strings match, false otherwise."
<primitive: 624>
self _primitiveFailed: #compareStringAt:to:startingAt:sizeBytes:useCase:
     args: { startIndex . aString . stringIndex . numSizeBytes . aBool }

]

{ #category : 'Copying' }
ByteArray >> copyFrom: startIndex to: stopIndex [

"Returns a new Array containing the elements of the receiver
 between startIndex and stopIndex, inclusive.  The result is of the same class
 as the receiver.

 If startIndex > stopIndex then an empty collection is returned.
 Otherwise both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex.
"

<primitive: 818>
(startIndex < 1) ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].

((stopIndex > self size) or: [(stopIndex < 0)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].

self _primitiveFailed: #copyFrom:to: args: { startIndex . stopIndex }

]

{ #category : 'Copying' }
ByteArray >> copyReplaceAll: oldSubCollection with: newSubCollection [

"See SequenceableCollection>>copyReplaceAll:with: for details"

| old new |
old := oldSubCollection.
new := newSubCollection.
(old isKindOf: ByteArray) ifFalse: [ old := old asByteArray ].
(new isKindOf: ByteArray) ifFalse: [ new := new asByteArray ].
^ super copyReplaceAll: old with: new

]

{ #category : 'Copying' }
ByteArray >> copyReplaceFrom: startIndex to: stopIndex with: aSequenceableCollection [

"See SequenceableCollection>>copyReplaceFrom:to:with: for details"

| col |
col := aSequenceableCollection.
(col isKindOf: ByteArray) ifFalse: [ col := col asByteArray ].
^ super copyReplaceFrom: startIndex to: stopIndex with: col

]

{ #category : 'Copying' }
ByteArray >> copyReplacing: oldObject withObject: newObject [

"Returns a String comprising a copy of the receiver in which all occurrences
 of objects equal to oldObject have been replaced by newObject."

^ super copyReplacing: oldObject asInteger withObject: newObject asInteger

]

{ #category : 'Copying' }
ByteArray >> copyWith: anObject [

"See SequenceableCollection>>copyWith: for details"

^ super copyWith: anObject asInteger

]

{ #category : 'Copying' }
ByteArray >> copyWithout: anObject [

"See SequenceableCollection>>copyWithout: for details"

^ super copyWithout: anObject asInteger

]

{ #category : 'Private-Accessing' }
ByteArray >> dateTime32At: startIndex [

"retrieves a DateTime that was stored with seconds resolution
 in big-endian byte order."

^ self dateTimeAt: startIndex width: 4

]

{ #category : 'Private-Updating' }
ByteArray >> dateTime32At: startIndex put: aDateTime [

"Stores the DateTime with seconds resolution in big-endian byte order
 WARNING, this truncates the DateTime to seconds resolution,
 throwing away the milliseconds.
 "

^ self dateTimeAt: startIndex put: aDateTime width: 4

]

{ #category : 'Private-Accessing' }
ByteArray >> dateTime32NativeAt: startIndex [

"retrieves a DateTime that was stored with seconds resolution
 in the gem process's native byte order.
 Provided for compatiblity and renamed from dateTimeAsUnsigned32At: "

^ self dateTimeAt: startIndex width: -4

]

{ #category : 'Private-Updating' }
ByteArray >> dateTime32NativeAt: startIndex put: aDateTime [

"Stores the DateTime with seconds resolution and gem process native
 byte order.
 WARNING, this truncates the DateTime to seconds resolution,
 throwing away the milliseconds.  Provided for compatibility and
 renamed from dateTimeAsUnsigned32At:put:  "

^ self dateTimeAt: startIndex put: aDateTime width: -4

]

{ #category : 'Private-Accessing' }
ByteArray >> dateTime64At: startIndex [

"retrieves a DateTime that was stored with millisecond resolution.
 in big-endian byte order"

^ self dateTimeAt: startIndex width: 8

]

{ #category : 'Private-Updating' }
ByteArray >> dateTime64At: startIndex put: aDateTime [

"Stores the DateTime with full millisecond resolution in big-endian
 byte order."

^ self dateTimeAt: startIndex put: aDateTime width: 8

]

{ #category : 'Private-Updating' }
ByteArray >> dateTimeAt: startIndex put: aDateTime width: anInt [

"values for anInt:
   -4   second resolution (4 bytes) in gem process native byte order
    4   second resolution (4 bytes) in big-endian byte order
    8   millisecond resolution (8 bytes) in big-endian byte order"
<primitive: 620>
| yr |
aDateTime _validateClass: DateTime .
(yr := aDateTime yearGmt) < 1970 ifTrue:[
  OutOfRange new name: 'yearGmt' min: 1970 actual: yr ;
      details: 'year is less than 1970' ; signal
].
self _primitiveFailed: #dateTimeAt:put:width:
     args: { startIndex . aDateTime . anInt }

]

{ #category : 'Private-Accessing' }
ByteArray >> dateTimeAt: startIndex width: anInt [

"values for anInt:
   -4   second resolution (4 bytes) in gem process native byte order
    4   second resolution (4 bytes) in big-endian byte order
    8   millisecond resolution (8 bytes) in big-endian byte order"

<primitive: 621>
self _primitiveFailed: #dateTimeAt:width: args: { startIndex . anInt }

]

{ #category : 'Encoding' }
ByteArray >> decodeFromUTF8 [
"Decode receiver from UTF8 format.
 Returns either a Unicode7 , Unicode16 or Unicode32 ,
 using the minimum character size needed to represent decoded result."

^ self _decodeUtf8StartingAt: 1 unicodeResult: true maxSize: nil bytesConsumed: nil

]

{ #category : 'Encoding' }
ByteArray >> decodeFromUTF8ToString [
"Decode UTF8 contents of the receiver returning a String, DoubleByteString or
  QuadByteString."

^ self _decodeUtf8StartingAt: 1 unicodeResult: false maxSize: nil bytesConsumed: nil

]

{ #category : 'Encoding' }
ByteArray >> decodeFromUTF8ToUnicode [
"Decode receiver from UTF8 format.
 Returns either a Unicode7 , Unicode16 or Unicode32 ,
 using the minimum character size needed to represent decoded result."

^ self _decodeUtf8StartingAt: 1 unicodeResult: true maxSize: nil bytesConsumed: nil

]

{ #category : 'Private-Updating' }
ByteArray >> deleteIndexKeyAt: anIndex [

<primitive: 630>
self _primitiveFailed: #deleteIndexKeyAt: args: { anIndex }


]

{ #category : 'Private-Accessing' }
ByteArray >> doubleAt: startIndex [

"extract an 8 byte float from the receiver, the result will
 be either a SmallDouble or a Float. "

^self at: startIndex signed: false width: -8

]

{ #category : 'Private-Updating' }
ByteArray >> doubleAt: startIndex put: aBinaryFloat [

"aBinaryFloat must be a kind of BinaryFloat."

^self at: startIndex put: aBinaryFloat signed: false width: 8

]

{ #category : 'Accessing' }
ByteArray >> doubleByteCharAt: index [

"Returns a double-byte Character located at index in the receiver. The most
 significant byte would be the one located at index, the least significant
 one located at index + 1."

<primitive: 501>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index >= self size) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

self _validateByteClass: ByteArray .
self _primitiveFailed: #doubleByteCharAt: args: { index }

]

{ #category : 'Encrypting' }
ByteArray >> encryptionExample [

| key salt enc dec hamlet |
key  := ByteArray withRandomBytes: 32 .
salt := ByteArray withRandomBytes: 16 .

hamlet :=
'Alas, poor Yorick! I knew him, Horatio: a fellow
of infinite jest, of most excellent fancy: he hath
borne me on his back a thousand times; and now, how
abhorred in my imagination it is! my gorge rims at
it. Here hung those lips that I have kissed I know
not how oft. Where be your gibes now? your
gambols? your songs? your flashes of merriment,
that were wont to set the table on a roar? Not one
now, to mock your own grinning? quite chap-fallen?
Now get you to my ladys chamber, and tell her, let
her paint an inch thick, to this favour she must
come; make her laugh at that. Prithee, Horatio, tell
me one thing.' .


enc := hamlet aesEncryptWith256BitKey: key salt: salt .
dec := enc aesDecryptWith256BitKey: key salt: salt into: String new.
^ dec = hamlet

]

{ #category : 'Private-Accessing' }
ByteArray >> floatAt: startIndex [

"extract a 4 byte float from the receiver, the result will
 be a SmallDouble."

^self at: startIndex signed: false width: -4

]

{ #category : 'Private-Updating' }
ByteArray >> floatAt: startIndex put: aBinaryFloat [

"aBinaryFloat must be a kind of BinaryFloat, representable as
 a 4 byte IEEE float without loss of exponent bits.
 If coercion of a BinaryFloat to a 4 byte float would produce
 loss of exponent bits, the primitive will fail.
 If coercion of a BinaryFloat to a 4 byte float would cause loss
 of precision that would convert a non-zero value to zero ,
 the primitive will fail ."

^self at: startIndex put: aBinaryFloat signed: false width: 4

]

{ #category : 'Encoded OOPs' }
ByteArray >> getObjectWithOldOopValueAt: anOffset [
"Answer the object with the OOP stored in 4 bytes of the receiver
 at the given offset.
 The bytes are expected to have been stored in big-endian order.
 Returns nil if the object does not exist or is an invalid object identifier."
<primitive: 636>
anOffset _validateClass: SmallInteger.
(anOffset < 1) ifTrue:[self _errorIndexOutOfRange: anOffset].
(anOffset > (self _basicSize - (self class bytesPerOop - 1)))
  ifTrue:[self _errorIndexOutOfRange: anOffset].
self _primitiveFailed: #getObjectWithOldOopValueAt: args: { anOffset }

]

{ #category : 'Encoded OOPs' }
ByteArray >> getObjectWithOopValueAt: anOffset [
"Answer the object with the OOP stored in 8 bytes of the receiver
 at the given offset.
 The bytes are expected to have been stored in big-endian order.
 Returns nil if the object identifier
 encoded at anOffset does not exist or is an invalid object identifier."

<primitive: 635>
anOffset _validateClass: SmallInteger.
(anOffset < 1) ifTrue:[self _errorIndexOutOfRange: anOffset].
(anOffset > (self _basicSize - (self class bytesPerOop - 1)))
  ifTrue:[self _errorIndexOutOfRange: anOffset].
self _primitiveFailed: #getObjectWithOopValueAt: args: { anOffset }

]

{ #category : 'Comparing' }
ByteArray >> hash [

"Returns a positive SmallInteger based on the contents of the receiver."

"Uses a case-sensitive string hash algorithm.
 The algorithm implemented is described in:

 [Pearson 90]
 Pearson, Peter K., Fast Hashing of Variable-Length Text Strings,
 Communications of the ACM 33, 6, (June 1990), 677-680."

<primitive: 31>
self _primitiveFailed: #hash .
self _uncontinuableError

]

{ #category : 'Searching' }
ByteArray >> indexOf: searchByte [
  ^ self indexOf: searchByte startingAt: 1 stoppingAt: self size
]

{ #category : 'Searching' }
ByteArray >> indexOf: searchByte startingAt: startIndex [
  ^ self indexOf: searchByte startingAt: startIndex stoppingAt: self size
]

{ #category : 'Searching' }
ByteArray >> indexOf: searchByte startingAt: startIndex stoppingAt: stopIndex [

"Receiver is searched for a byte that has the same value
as the low-order byte (searchByte bitAnd: 16rFF) of SmallInteger
searchByte. The search starts at 1-based startIndex within the receiver
and stops when a matching byte is found or 1-based stopIndex is reached.
The index of the first matching byte is answered, or 0 is answered if no
matching byte is found.

Errors:
  * ArgumentTypeError if any of the arguments is not a SmallInteger.
  * OutOfRange if startIndex or stopIndex is outside the bounds of
    of the receiver."

<primitive: 947>
searchByte _validateClass: SmallInteger .
startIndex _validateClass: SmallInteger .
stopIndex _validateClass: SmallInteger .
(startIndex < 1 or:[ startIndex > self size]) ifTrue:[
  OutOfRange new
   name:'startIndex' min: 1 max: self size actual: startIndex  ;
   signal
].
(stopIndex < 1 or:[ stopIndex > self size]) ifTrue:[
  OutOfRange new
   name:'stopIndex' min: 1 max: self size actual: stopIndex  ;
   signal
].
self _primitiveFailed: #indexOf:startingAt:stoppingAt:
	args: { searchByte . startIndex . stopIndex}

]

{ #category : 'Copying' }
ByteArray >> insertAll: aByteArray at: anIndex [

"Inserts aByteArray into the receiver beginning at anIndex.  The receiver's
 size is modified to become its old size plus the size of aByteArray.

 The argument anIndex must be greater than or equal to 1.  If anIndex is 1
 greater than the size of the receiver, appends aByteArray to the receiver.  If
 anIndex is more than 1 greater than the size of the receiver, generates an
 error."

<primitive: 230>

aByteArray _validateClass: ByteArray.
(anIndex _isSmallInteger)
  ifFalse: [ ^ self _errorNonIntegerIndex: anIndex].
(anIndex < 1 or: [ anIndex > (self size + 1)]) "out of bounds"
  ifTrue: [ ^ self _errorIndexOutOfRange: anIndex].

^ super insertAll: aByteArray at: anIndex

]

{ #category : 'Private-Accessing' }
ByteArray >> int32LittleEndianAt: startIndex [
  "Receiver can be any byte format object.
   Answer a SmallInteger resulting from interpreting the bytes in the
   receiver from offset to offset+3 as a little-endian signed integer.
  Offset is 1-based and need not be aligned.
  Variant of signed32At:, for little endian data.
  * The given offset is not a SmallInteger (ArgumentTypeError)
  * offset+3 is outside the size of the receiver (OutOfRange)"
<primitive: 945>
startIndex _validateClass: SmallInteger .
(startIndex < 1 or:[ startIndex + 3 > self size]) ifTrue:[
  OutOfRange new
    name:'startIndex' min: 1 max: self size - 3 actual: startIndex ;
    signal
].
self _primitiveFailed: #int32LittleEndianAt: args: { startIndex }

]

{ #category : 'Message Digests' }
ByteArray >> md5sum [

"Return the 128 bit MD5 checksum of the receiver as a LargeInteger.

 Computation is per RFC 1321 , http://www.ietf.org/rfc/rfc1321.txt,
 using L. Peter Deutsch's C implementation from
 http://sourceforge.net/projects/libmd5-rfc/

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 1

]

{ #category : 'Message Digests' }
ByteArray >> md5sumBytes [

"Return the 128 bit MD5 checksum of the receiver as a ByteArray.

 Computation is per RFC 1321 , http://www.ietf.org/rfc/rfc1321.txt,
 using L. Peter Deutsch's C implementation from
 http://sourceforge.net/projects/libmd5-rfc/

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -1

]

{ #category : 'Encoded OOPs' }
ByteArray >> nextPutAllOldOopsOfObjects: anArray [
"For each object in anArray, append the 4 byte OOP to the receiver.
 The receiver is grown by 4 bytes for every object contained in
 anArray."

^ self at: self size + 1 putAllOldOopsOfObjects: anArray

]

{ #category : 'Encoded OOPs' }
ByteArray >> nextPutAllOopsOfObjects: anArray [
"For each object in anArray, append the 8 byte OOP to the receiver.
 The receiver is grown by 8 bytes for every object contained in
 anArray."

^ self at: self size + 1 putAllOopsOfObjects: anArray

]

{ #category : 'Encoded OOPs' }
ByteArray >> nextPutOldOopOfObject: anObject [
"Extend the receiver by 4 bytes and add the 4 byte OOP of anObject.
 anObject cannot be a special object, and its oopNumber must be < 2 billion."

^self at: self size + 1 putOldOopValueOfObject: anObject

]

{ #category : 'Encoded OOPs' }
ByteArray >> nextPutOopOfObject: anObject [
"Extend the receiver by 8 bytes and add the 8 byte OOP of anObject."

^self at: self size + 1 putOopValueOfObject: anObject

]

{ #category : 'Encoded OOPs' }
ByteArray >> put: anInt objectsWithOldOopValueAt: anOffset into: anArray [
"Extract a given number of objects from the OOPs stored in the receiver
 starting at the given offset. Each OOP extracted occupies 4 bytes
 in the receiver. Store the objects into the specified Array object.
 A nil is stored in the array for each object which does not exist.
 The specified Array is set to size 0 at the start of the operation."

1 to: anInt do:[:j | | obj |
  obj := self getObjectWithOldOopValueAt: j .
  anArray at: j put: obj .
  ].
anArray size: anInt .
^ self

]

{ #category : 'Encoded OOPs' }
ByteArray >> put: anInt objectsWithOopValueAt: anOffset into: anArray [
"Extract a given number of objects from the OOPs stored in the receiver
 starting at the given offset. Each OOP extracted occupies 8 bytes
 in the receiver. Store the objects into the specified Array object.
 A nil is stored in the array for each object which does not exist.
 The specified Array is set to size 0 at the start of the operation."

1 to: anInt do:[:j | | obj |
  obj := self getObjectWithOopValueAt: j .
  anArray at: j put: obj .
  ].
anArray size: anInt .
^ self

]

{ #category : 'Encoded OOPs' }
ByteArray >> putObjectsWithOldOopValuesAtIndices: anArrayOfOffsets into: anArray [
"For each offset stored in anArrayOfOffsets, extract the OOP from the
 receiver stored at that offset.  Then store the object with that OOP in
 anArray.  A nil is stored in anArray for each object which does not exist.
 Each OOP extracted occupies 4 bytes in the receiver.
 The specified Array is set to size 0 at the start of the operation."

| limit |
limit := anArrayOfOffsets size .
1 to: limit do:[:j | | obj |
   obj := self getObjectWithOldOopValueAt: (anArrayOfOffsets at: j) .
   anArray at: j put: obj .
  ].
anArray size: limit .
^ self

]

{ #category : 'Encoded OOPs' }
ByteArray >> putObjectsWithOopValuesAtIndices: anArrayOfOffsets into: anArray [
"For each offset stored in anArrayOfOffsets, extract the OOP from the
 receiver stored at that offset.  Then store the object with that OOP in
 anArray.  A nil is stored in anArray for each object which does not exist.
 Each OOP extracted occupies 8 bytes in the receiver.
 The specified Array is set to size 0 at the start of the operation."

| limit |
limit := anArrayOfOffsets size .
1 to: limit do:[:j | | obj |
   obj := self getObjectWithOopValueAt: (anArrayOfOffsets at: j) .
   anArray at: j put: obj .
  ].
anArray size: limit .
^ self

]

{ #category : 'Accessing' }
ByteArray >> quadByteCharAt: index [

"Returns a quad-byte Character located at index in the receiver. The most
 significant byte would be the one located at index, the least significant
 one located at index + 3."

<primitive: 703>

(index _isInteger)
  ifFalse: [^ self _errorNonIntegerIndex: index].

((index <= 0) or: [ (index >= self size) ])
  ifTrue: [^ self _errorIndexOutOfRange: index].

self _validateByteClass: ByteArray .
self _primitiveFailed: #quadByteCharAt: args: { index }

]

{ #category : 'Removing' }
ByteArray >> removeFrom: startIndex to: stopIndex [

"Deletes the elements of the receiver from startIndex to stopIndex.  The
 size of the receiver is decreased by stopIndex - startIndex + 1.

 Both startIndex and stopIndex must be positive integers not larger than the
 size of the receiver, with startIndex <= stopIndex."

<primitive: 386>

(stopIndex < startIndex)
ifTrue:
   [ ^ startIndex _error: #rtErrBadCopyFromTo args: { stopIndex }].
((stopIndex > self size) or: [(stopIndex < 1)])
   ifTrue: [ ^ self _errorIndexOutOfRange: stopIndex].
(startIndex < 1)
   ifTrue: [ ^ self _errorIndexOutOfRange: startIndex].
^ self _primitiveFailed: #removeFrom:to: args: { startIndex . stopIndex }

]

{ #category : 'Copying' }
ByteArray >> replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex [

"Replaces the elements of the receiver between the indexes startIndex and
 stopIndex inclusive with the elements of aSeqCollection starting at repIndex.
 If aSeqCollection is identical to the receiver, the source and
 destination blocks may overlap.

 The primitive supports directly the case where
    (aSeqCollection isKindOfClass: CByteArray) == true ,
 with repIndex being one-based .

 Returns the receiver."

<primitive: 297>
startIndex _isSmallInteger ifFalse:[ startIndex _validateClass: SmallInteger ].
stopIndex _isSmallInteger ifFalse:[ stopIndex _validateClass: SmallInteger ].
repIndex _isSmallInteger ifFalse:[ repIndex _validateClass: SmallInteger ].
(aSeqCollection stringCharSize >= 2) ifTrue:[
  "aSeqCollection is a DoubleByteString or QuadByteString"
  aSeqCollection _error: #rtErrInvalidArgClass args: { String . ByteArray }
].
^ super replaceFrom: startIndex to: stopIndex with: aSeqCollection startingAt: repIndex

]

{ #category : 'Message Digests' }
ByteArray >> sha1Sum [

"Return the 160 bit SHA1 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 2

]

{ #category : 'Message Digests' }
ByteArray >> sha1SumBytes [

"Return the 160 bit SHA1 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -2

]

{ #category : 'Message Digests' }
ByteArray >> sha256Sum [

"Return the 256 bit SHA256 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 3

]

{ #category : 'Message Digests' }
ByteArray >> sha256SumBytes [

"Return the 256 bit SHA256 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -3

]

{ #category : 'Message Digests' }
ByteArray >> sha3_224Sum [

"Return the SHA3 224 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 5

]

{ #category : 'Message Digests' }
ByteArray >> sha3_224SumBytes [

"Return the SHA3 224 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -5

]

{ #category : 'Message Digests' }
ByteArray >> sha3_256Sum [

"Return the SHA3 256 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 6

]

{ #category : 'Message Digests' }
ByteArray >> sha3_256SumBytes [

"Return the SHA3 256 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -6

]

{ #category : 'Message Digests' }
ByteArray >> sha3_384Sum [

"Return the SHA3 384 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 7

]

{ #category : 'Message Digests' }
ByteArray >> sha3_384SumBytes [

"Return the SHA3 384 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -7

]

{ #category : 'Message Digests' }
ByteArray >> sha3_512Sum [

"Return the SHA3 512 bit checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 8

]

{ #category : 'Message Digests' }
ByteArray >> sha3_512SumBytes [

"Return the SHA3 512 bit checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 202
   https://nvlpubs.nist.gov/nistpubs/fips/nist.fips.202.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -8

]

{ #category : 'Message Digests' }
ByteArray >> sha512Sum [

"Return the 512 bit SHA512 checksum of the receiver as a LargeInteger.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: 4

]


{ #category : 'Message Digests' }
ByteArray >> sha512SumBytes [

"Return the 512 bit SHA512 checksum of the receiver as a ByteArray.

 Computation is per FIPS PUB 180-3:
   http://csrc.nist.gov/publications/fips/fips180-3/fips180-3_final.pdf

 For DoubleByteString and QuadByteString, the computation is based
 on viewing the string as a ByteArray holding big-endian characters.
"

^ self _asMessageDigestKind: -4

]

{ #category : 'Private-Accessing' }
ByteArray >> shortStringAt: startIndex [

^self at: startIndex sizeBytes:  1 stringSize: nil

]

{ #category : 'Private-Comparing' }
ByteArray >> shortStringAt: anIndex compareWith: aByteObject startingAt: stringOffset opCode: anOpCode [
 "Compares, in order starting at anIndex, if the contents of the receiver
  are less than/greater than the contents of aByteObject, starting at stringOffset.
  The first byte of the receiver is assumed to be the size of the string
  held in the receiver.  GemStone character precedence is adhered to.
  The argument string must be a small object.
op code == 0 means less than
op code == 1 means greater than"

<primitive: 625>
self _primitiveFailed: #shortStringAt:compareWith:startingAt:opCode:
     args: { anIndex . aByteObject . stringOffset . anOpCode }

]

{ #category : 'Private-Comparing' }
ByteArray >> shortStringAt: anIndex greaterThan: aByteObject startingAt: stringOffset [

 " Checks, in order starting at anIndex, if the contents of the receiver
  are greater than the contents of aByteObject, starting at stringOffset.
  The first byte of the receiver is assumed to be the size of the string
  held in the receiver.
 GemStone character precedence is adhered to.
  The argument string must be a small object."

^self shortStringAt: anIndex compareWith: aByteObject startingAt: stringOffset opCode: 1

]

{ #category : 'Private-Comparing' }
ByteArray >> shortStringAt: anIndex lessThan: aByteObject startingAt: stringOffset [

 " Checks, in order starting at anIndex, if the contents of the receiver
  are less than the contents of aByteObject, starting at stringOffset.
  The first byte of the receiver is assumed to be the size of the string
  held in the receiver.
 GemStone character precedence is adhered to.
  The argument string must be a small object."

^self shortStringAt: anIndex compareWith: aByteObject startingAt: stringOffset opCode: 0

]

{ #category : 'Private-Updating' }
ByteArray >> shortStringAt: startIndex put: aString fromOffset: stringOffset [
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 1

]

{ #category : 'Private-Accessing' }
ByteArray >> signed16At: startIndex [

^self at: startIndex signed: true width: 2

]

{ #category : 'Private-Updating' }
ByteArray >> signed16At: startIndex put: anInteger [

^self at: startIndex put: anInteger signed: true width: 2

]

{ #category : 'Private-Accessing' }
ByteArray >> signed24At: startIndex [

^self at: startIndex signed: true width: 3

]

{ #category : 'Private-Updating' }
ByteArray >> signed24At: startIndex put: anInteger [

^self at: startIndex put: anInteger signed: true width: 3

]

{ #category : 'Private-Accessing' }
ByteArray >> signed32At: startIndex [

^self at: startIndex signed: true width: 4

]

{ #category : 'Private-Updating' }
ByteArray >> signed32At: startIndex put: anInteger [

^self at: startIndex put: anInteger signed: true width: 4

]

{ #category : 'Private-Accessing' }
ByteArray >> signed64At: startIndex [

^self at: startIndex signed: true width: 8

]

{ #category : 'Private-Updating' }
ByteArray >> signed64At: startIndex put: anInteger [

^self at: startIndex put: anInteger signed: true width: 8

]

{ #category : 'Private-Accessing' }
ByteArray >> signed8At: startIndex [

^self at: startIndex signed: true width: 1

]

{ #category : 'Private-Updating' }
ByteArray >> signed8At: startIndex put: anInteger [

^self at: startIndex put: anInteger signed: true width: 1

]

{ #category : 'Digital Signature Creation - EC' }
ByteArray >> signWithEcPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Signs the receiver with the given elliptic curve private key. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #EC ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 0 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - DSA' }
ByteArray >> signWithSha1AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha1AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the
 resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 2 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha1AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA1 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -2 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - DSA' }
ByteArray >> signWithSha256AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 3 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -3 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_224AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 5 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_224AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 224 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -5 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_256AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 6 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_256AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 256 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -6 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_384AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 7 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_384AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 384 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -7 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 8 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha3_512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA3 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -8 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - DSA' }
ByteArray >> signWithSha512AndDsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 DSA private key. Returns a ByteArray containing the resulting signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #DSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha512AndRsaPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key using #PKCS1 padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: 4 signature: aByteArrayOrNil

]

{ #category : 'Digital Signature Creation - RSA' }
ByteArray >> signWithSha512AndRsaPssPrivateKey: aGsTlsPrivateKey into: aByteArrayOrNil [
"Hashes the receiver using SHA2 512 and signs the resulting hash with the given
 RSA private key and PSS padding. Returns a ByteArray containing the resulting
 signature.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPrivateKey _validateAlgorithm: #RSA ; _validateIsPrivateKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPrivateKey digestKind: -4 signature: aByteArrayOrNil

]

{ #category : 'Private-Accessing' }
ByteArray >> string4gAt: startIndex [

^self at: startIndex sizeBytes: 4 stringSize: nil

]

{ #category : 'Private-Updating' }
ByteArray >> string4gAt: startIndex put: aString fromOffset: stringOffset [
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 4

]

{ #category : 'Private-Accessing' }
ByteArray >> string64kAt: startIndex [

^self at: startIndex sizeBytes: 2 stringSize: nil

]

{ #category : 'Private-Updating' }
ByteArray >> string64kAt: startIndex put: aString fromOffset: stringOffset [
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 2

]

{ #category : 'Private-Updating' }
ByteArray >> stringAt: startIndex put: aString fromOffset: stringOffset [
	^self at: startIndex put: aString fromOffset: stringOffset sizeBytes: 0

]

{ #category : 'Private-Accessing' }
ByteArray >> stringOfSize: anInt at: startIndex [

^self at: startIndex sizeBytes:  0 stringSize: anInt

]

{ #category : 'Private-Accessing' }
ByteArray >> unsigned16At: startIndex [

^self at: startIndex signed: false width: 2

]

{ #category : 'Private-Updating' }
ByteArray >> unsigned16At: startIndex put: anUnsignedInt [

^self at: startIndex put: anUnsignedInt signed: false width: 2

]

{ #category : 'Private-Accessing' }
ByteArray >> unsigned24At: startIndex [

^self at: startIndex signed: false width: 3

]

{ #category : 'Private-Updating' }
ByteArray >> unsigned24At: startIndex put: anUnsignedInt [

^self at: startIndex put: anUnsignedInt signed: false width: 3

]

{ #category : 'Private-Accessing' }
ByteArray >> unsigned32At: startIndex [

^self at: startIndex signed: false width: 4

]

{ #category : 'Private-Updating' }
ByteArray >> unsigned32At: startIndex put: anUnsignedInt [

^self at: startIndex put: anUnsignedInt signed: false width: 4

]

{ #category : 'Private-Accessing' }
ByteArray >> unsigned64At: startIndex [

^self at: startIndex signed: false width: 8

]

{ #category : 'Private-Updating' }
ByteArray >> unsigned64At: startIndex put: anUnsignedInt [

^self at: startIndex put: anUnsignedInt signed: false width: 8

]

{ #category : 'Private-Accessing' }
ByteArray >> unsigned8At: startIndex [

^self at: startIndex signed: false width: 1

]

{ #category : 'Private-Updating' }
ByteArray >> unsigned8At: startIndex put: anUnsignedInt [

^self at: startIndex put: anUnsignedInt signed: false width: 1

]

{ #category : 'Digital Signature Creation - EC' }
ByteArray >> verifyWithEcPublicKey: aGsTlsPublicKey signature: aByteArray [
"Verifies the receiver using the given elliptic curve key and signature.  Returns true
 if the signature is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #EC ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 0 signature: aByteArray

]

{ #category : 'Digital Signature Verification - DSA' }
ByteArray >> verifyWithSha1AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA1 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
  # _primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha1AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -2 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha1AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA1 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 2 signature: aByteArray

]

{ #category : 'Digital Signature Verification - DSA' }
ByteArray >> verifyWithSha256AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA256 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -3 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 3 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_224AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -5 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_224AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 224 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 5 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_256AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -6 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_256AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 256 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 6 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_384AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -7 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_384AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 384 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 7 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -8 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha3_512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA3 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 8 signature: aByteArray

]

{ #category : 'Digital Signature Verification - DSA' }
ByteArray >> verifyWithSha512AndDsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA512 and verifies the resulting hash using the
 given DSA public key and signature. Returns true if the signature is correct.
 Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #DSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha512AndRsaPssPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with PSS padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: -4 signature: aByteArray

]

{ #category : 'Digital Signature Verification - RSA' }
ByteArray >> verifyWithSha512AndRsaPublicKey: aGsTlsPublicKey signature: aByteArray [
"Hashes the receiver using SHA2 512 and verifies the resulting hash using the given
 RSA public key and signature with #PKCS1 padding. Returns true if the signature
 is correct. Otherwise raises an exception.

 See the method:
   #_primAsymSignVerifyWithKey:digestKind:signature:
 for more information."

aGsTlsPublicKey _validateAlgorithm: #RSA ; _validateIsPublicKey ; _validateIsNotRsaPss .
^ self _primAsymSignVerifyWithKey: aGsTlsPublicKey digestKind: 4 signature: aByteArray

]

{ #category : 'Encoding' }
ByteArray >> _decodeFromUTF16toUTF8: bigEndianBoolean [
  "Decode UTF16 contents of the receiver.
   bigEndianBoolean==true specifies decode from UTF16-BE , false from UTF16-LE .
   If the first code point is 16rFEFF in the byte order specified, it is passed
   through and the first 3 bytes of the result will be 16rEF 16rBB 16rBF ,
   i.e. the ZWNBSP  code point.
   If the first code point is 16rFFFE in the byte order specified, an error is
   signalled, since 16rFFFE is not a legal Unicode code point, and the byte order
   of the data is probably not as specified.
   Returns an instance of Utf8 .
   "

  <primitive: 212>
  bigEndianBoolean _validateInstanceOf: Boolean .
  self _primitiveFailed: #_decodeFromUTF16toUTF8: args: { bigEndianBoolean } .
]

{ #category : 'Encoding' }
ByteArray >> decodeFromUTF16BeToUTF8 [
  "Decode UTF16-BE contents of the receiver.
   Returns an instance of Utf8 ."

 ^ self _decodeFromUTF16toUTF8: true
]

{ #category : 'Encoding' }
ByteArray >> decodeFromUTF16LeToUTF8 [
  "Decode UTF16-LE contents of the receiver.
   Returns an instance of Utf8 ."

 ^ self _decodeFromUTF16toUTF8: false
]

{ #category : 'Compression' }
ByteArray >> compressWithLz4: opCode [

"Compress the receiver with lz4 compression and answer a ByteArray.
 There are 2 lz4 compression modes available:
   opCode 0 - frame mode
   opCode 1 - block mode

 Frame mode should almost always be used. Block mode requires the
 receiver to be completely faulted into memory which can cause out of 
 memory errors for large objects. Frame mode has no such limitation
 as the complete contents of the receiver need not be in memory
 during compression. Also the lz4 command line program compresses
 data using frame mode.

 Symbols, DoubleByteSymbols and QuadByteSymbols may not be compressed.
 Attempting to do so raises an exception.

 Returns a ByteArray on success or raises an exception on error."

<primitive: 241>
opCode _validateClass: SmallInteger .
((opCode < 0) or:[ opCode > 1])
  ifTrue:[ opCode _error: #rtErrArgOutOfRange ] .
^ self _primitiveFailed: #compressWithLz4: .

]

{ #category : 'Compression' }
ByteArray >> decompressWithLz4IntoNewInstanceOf: aClass decompressedSize: dSize [

"Decompresses the receiver using lz4 and stores the resulting  bytes into a new
 instance of aClass. aClass must be a byte format Class. The lz4 compression 
 mode (block or frame) used to compress the data is automatically detected by
 this method.

 dSize is a SmallInteger which indicates the size (in characters, not bytes) 
 of the data when decompressed. If the contents were compressed using lz4 frame
 mode, dSize may be set to 0 indicating the decompressed size is not known.
 However the decompressed size should always be provided whenever possible 
 to enable faster decompression performance.

 If the contents were compressed using lz4 block mode then dSize must be set
 correctly to the decompressed size otherwise an error may be raised.

 aClass may not be Symbol, DoubleByteSymbol nor QuadByteSymbol otherwise the
 operation fails and an error is raised.

 On success, returns a new instance of aClass containing the decompressed data.
 Raises an exception on error."

<primitive: 242>
dSize _validateClass: SmallInteger .
dSize < 0 ifTrue:[ dSize _error: #rtErrArgOutOfRange ] .
aClass validateIsClass .
aClass isBytes ifFalse:[ aClass _error: #objErrNotByteKind ] .  
^ self _primitiveFailed: #decompressWithLz4IntoNewInstanceOf:decompressedSize: .

]

{ #category : 'Accessing (Native)' }
ByteArray >> unsigned16AsNativeAt: startIndex [

"Answer the unsigned 2 byte integer at offset startIndex.
 Integer is assumed to be in native format."
 
^ self at: startIndex signed: false width: 512
]

{ #category : 'Accessing (Native)' }
ByteArray >> unsigned32AsNativeAt: startIndex [

"Answer the unsigned 4 byte integer at offset startIndex.
 Integer is assumed to be in native format."
 
^ self at: startIndex signed: false width: 1024
]

{ #category : 'Accessing (Native)' }
ByteArray >> unsigned64AsNativeAt: startIndex [

"Answer the unsigned 8 byte integer at offset startIndex.
 Integer is assumed to be in native format."
 
^ self at: startIndex signed: false width: 2048
]

{ #category : 'Accessing (Native)' }
ByteArray >> signed16AsNativeAt: startIndex [

"Answer the signed 2 byte integer at offset startIndex.
 Integer is assumed to be in native format."
 
^ self at: startIndex signed: true width: 512
]

{ #category : 'Accessing (Native)' }
ByteArray >> signed32AsNativeAt: startIndex [

"Answer the signed 4 byte integer at offset startIndex.
 Integer is assumed to be in native format."
 
^ self at: startIndex signed: true width: 1024
]

{ #category : 'Accessing (Native)' }
ByteArray >> signed64AsNativeAt: startIndex [

"Answer the signed 8 byte integer at offset startIndex.
 Integer is assumed to be in native format."
 
^ self at: startIndex signed: true width: 2048
]

{ #category : 'Accessing (Native)' }
ByteArray >> doubleAsNativeAt: startIndex [

"Answer the 8 byte double at offset startIndex.
 Double is assumed to be in native format."
 
^ self at: startIndex signed: true width: 8192
]

{ #category : 'Accessing (Native)' }
ByteArray >> floatAsNativeAt: startIndex [

"Answer the 4 byte float at offset startIndex.
 Float is assumed to be in native format."
 
^ self at: startIndex signed: true width: 4096
]

{ #category : 'Updating (Native)' }
ByteArray >> signed16At: startIndex putAsNative: anInteger [

"Store anInteger at startIndex as a native 2 byte signed integer."

^self at: startIndex put: anInteger signed: true width: 512

]

{ #category : 'Updating (Native)' }
ByteArray >> signed32At: startIndex putAsNative: anInteger [

"Store anInteger at startIndex as a native 4 byte signed integer."

^self at: startIndex put: anInteger signed: true width: 1024

]

{ #category : 'Updating (Native)' }
ByteArray >> signed64At: startIndex putAsNative: anInteger [

"Store anInteger at startIndex as a native 8 byte signed integer."

^self at: startIndex put: anInteger signed: true width: 2048

]

{ #category : 'Updating (Native)' }
ByteArray >> unsigned16At: startIndex putAsNative: anInteger [

"Store anInteger at startIndex as a native 2 byte unsigned integer."

^self at: startIndex put: anInteger signed: false width: 512

]

{ #category : 'Updating (Native)' }
ByteArray >> unsigned32At: startIndex putAsNative: anInteger [

"Store anInteger at startIndex as a native 4 byte unsigned integer."

^self at: startIndex put: anInteger signed: false width: 1024

]

{ #category : 'Updating (Native)' }
ByteArray >> unsigned64At: startIndex putAsNative: anInteger [

"Store anInteger at startIndex as a native 8 byte unsigned integer."

^self at: startIndex put: anInteger signed: false width: 2048

]

{ #category : 'Updating (Native)' }
ByteArray >> doubleAt: startIndex putAsNative: aDouble [

"Store aDouble at startIndex as a native 8 byte double."

^self at: startIndex put: aDouble signed: false width: 2048

]

{ #category : 'Updating (Native)' }
ByteArray >> floatAt: startIndex putAsNative: aFloat [

"Store aFloat at startIndex as a native 4 byte float."

^self at: startIndex put: aFloat signed: false width: 1024

]

{ #category : 'Converting' }
ByteArray >> asByteArray [
  ^ self 

]

