Extension { #name : 'DecimalFloat' }

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> _exceptionFor: aString [

"Returns exception installed on given symbol, else nil."

| this desired |
(Symbol _existingWithAll: aString) ifNotNil:[ :sym |
  this := AbstractException _staticExceptions.
  desired := ErrorSymbols at: #numErrFltException.

  [this == nil] whileFalse: [
    (this category == GemStoneError and: [
        this number == desired and: [
        this subtype == sym ]]) ifTrue: [ ^this ].
    this := this next].
].
^nil

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> _exceptionKind: aString [

"Arg is a String indicating a type of exception
 (#divideByZero, #inexactResult, #invalidOperation, #overflow,
 #underflow).
 Returns an offset to be used analyzing a DecimalFloat status string."

  | list  |
  (Symbol _existingWithAll: aString) ifNotNil:[ :sym |
    list := self _exceptionList.
    1 to: list size do: [:i | (list at: i) == sym ifTrue:[ ^ i ]].
  ].
  ^ aString _error: #numErrArgNotFltException

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> _exceptionList [

"Returns the list of available exceptions, in order."

^ #(#divideByZero #invalidOperation #overflow #underflow #inexactResult ).

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> _installException: aBlock on: aSymbol [

"Install given block as a static exception."

AbstractException installStaticException:
      [:theException :cat :num :args |
            (args at: 1) = theException subtype ifFalse: [
              theException resignal: cat number: num args: args].
            aBlock value: cat value: num value: args]
        category: GemStoneError
        number: (ErrorSymbols at: #numErrFltException)
        subtype: aSymbol

]

{ #category : 'Private' }
DecimalFloat class >> _oneArgPrim: opcode arg: anArg [

"Private."

"opcode function
      0 new
      1 status:
      2 fromString:
      3 fromStringLocaleC:
"

<primitive: 135>
anArg _validateClass: String .
opcode == 0 ifTrue:[ anArg _error: #numErrArgNotFltStatus ] .
self _primitiveFailed: #_oneArgPrim:arg: args: { opcode . anArg }

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> _raiseInvalidOperationException [

"This method sets the invalid operation exception of the floating point
 processor."

DecimalPlusSignalingNaN + 3.0F0 .
^self

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> _setException: anException to: aBool [

"Turns on (or off) a floating point exception.  Returns the previous state."

|state kind|
kind := self _exceptionKind: anException.
state := self status.
state at: self _exceptionList size * 2 + kind
     put: (aBool ifTrue: [$1] ifFalse: [$0]).
self status: state

]

{ #category : 'Instance Creation' }
DecimalFloat class >> basicNew [

^ self fromString:'0'

]

{ #category : 'Instance Creation' }
DecimalFloat class >> basicNew: aSize [

aSize == 13 ifFalse:[ ArgumentError signal: 'expected 13, arg is ', aSize asString ].
^ self fromString:'0'

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> clearAllExceptions [

"Clears all raised exceptions."
| list |
list := self _exceptionList .
1 to: list size do:[:j ||each | each := list at: j .
  self clearException: each
]

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> clearException: aString [

"Clears the raised exception type defined by aSymbol
 (#divideByZero, #inexactResult, #invalidOperation, #overflow, #underflow).
 If aSymbol is
 not one of these exception types, an error is generated.  Raised exceptions
 are set by GemStone during floating point operations, and must be explicitly
 cleared with this method."

| state kind |
kind := self _exceptionKind: aString .
state := self status.
state at: self _exceptionList size + kind put: $0.
self status: state.

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> enabledExceptions [

"Returns a list of all raised exceptions."

|result list|

result := { } .
list := self _exceptionList .
1 to: list size do:[:j ||each | each := list at: j .
  (self trapEnabled: each) ifTrue: [
    result add: each
  ]
].
^result

]

{ #category : 'Instance Creation' }
DecimalFloat class >> fromStream: aStream [

"Generates a DecimalFloat from aStream.  Generates an error if an attempt is
 made to read beyond the end of the stream.

 The Stream must contain a legal DecimalFloat, as defined by the following BNF
 construction:

 DecimalFloat = ( Integer '.' Digit {Digit} [ E Integer ] ) |
         ( Integer E Integer )
 Integer = [ ('+' | '-') ] Digit {Digit}
 Point = ( '.' | ',' ) depending on Locale
 E = ( 'F' | 'f')

 Note that the syntax does not allow certain valid DecimalFloats (such as
 DecimalPlusInfinity and MinusInfinity) to be read."

| ch s getDigits getChar getSign getExp |

self _checkReadStream: aStream forClass: CharacterCollection.
ch := aStream next.
[ ch isEquivalent: $ ] whileTrue: [ ch := aStream next ].
aStream skip: -1.
s := aStream contents class new.

getDigits := [ | c cnt |
               cnt := 0 .
               [ (c := aStream peek) ~~ nil and: [ c isDigit ] ]
               whileTrue:
                 [ s add: aStream next . cnt := cnt + 1 ].
               cnt == 0 ifTrue:
                 [ self _errIncorrectFormat: aStream ].
             ].

getChar := [ :c |
             (aStream peek isEquivalent: c) ifTrue: [ s add: aStream next ] ].
getSign := [ (getChar value: $-) ifNil:[ getChar value: $+ ] ].
getExp := [ | c |
            c := aStream peek .
            ((c isEquivalent: $e) or:[ c isEquivalent: $f]) ifTrue:[
               s add: aStream next ] ].
getSign value.
getDigits value.
(getChar value: (Locale decimalPoint at: 1))  ifNotNil:[ getDigits value ].
(getExp value) ifNotNil:[ getSign value.  getDigits value.  ].
^ self fromString: s

]

{ #category : 'Converting' }
DecimalFloat class >> fromString: aString [

"Returns an instance of DecimalFloat, constructed from aString.  The String
 must contain only Characters representing the object to be created, although
 leading and trailing blanks are permitted.

 The exponent notation, if present may start with any one of $e, $E, $f, or $F."

^ self _oneArgPrim: 2 arg: aString

]

{ #category : 'Converting' }
DecimalFloat class >> fromStringLocaleC: aString [

"Same as fromString: except that  $.  is always used as the decimal point."

^ self _oneArgPrim: 3 arg: aString

]

{ #category : 'Storing and Loading' }
DecimalFloat class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst |

inst := self fromStringLocaleC:  passiveObj upToSeparator .
passiveObj hasRead: inst.
^inst

]

{ #category : 'Instance Creation' }
DecimalFloat class >> new [

"Returns a PlusSignalingNaN.  You can use this method to
 define a DecimalFloat without specifying its value."

^ self _oneArgPrim: 0 arg: nil

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> operationException: aString [

"Returns true if the specified exception has occurred in the current operation.
 Otherwise, returns false.  The argument aString defines the exception type
 (#divideByZero, #inexactResult, #invalidOperation, #overflow,
 #underflow).
 If aSymbol is not one of these, an error is generated."

| status kind |
status := DecimalFloat status.
kind := DecimalFloat _exceptionKind: aString .
^(status at: kind) == $1

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> operationExceptions [

"Returns a list of all exceptions raised by the last floating point operation."

|result list|

result := { } .
list := self _exceptionList .
1 to: list size do:[:j ||each | each := list at: j .
  (self operationException: each) ifTrue: [
    result add: each
  ]
].
^result

]

{ #category : 'Arithmetic' }
DecimalFloat class >> pi [

"Returns the value of pi, accurate to twenty decimal places."

^ 3.14159265358979323846F0

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> raisedException: aString [

"Returns true if the specified exception has occurred since the last
 clearException: operation.  Otherwise, returns false.  The argument aSymbol
 defines the exception type
 (#divideByZero, #inexactResult, #invalidOperation, #overflow, #underflow).
 If aSymbol is not one of these, an error is generated. "

| status kind |
status := DecimalFloat status.
kind := DecimalFloat _exceptionKind: aString .
^(status at: self _exceptionList size + kind) == $1

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> raisedExceptions [

"Returns a list of all raised exceptions."

|result list |

result := { } .
list := self _exceptionList .
1 to: list size do:[:j ||each | each := list at: j .
  (self raisedException: each) ifTrue: [
    result add: each
  ]
].
^result

]

{ #category : 'Truncation and Rounding' }
DecimalFloat class >> roundingMode [

"Returns the current rounding mode ('nearestEven', 'towardMinusInfinity',
 'towardPlusInfinity', 'towardZero')."

|state mode|

state := DecimalFloat status.
mode := state at: 16.
mode == $E ifTrue: [ ^ #nearestEven].
mode == $N ifTrue: [ ^ #towardMinusInfinity].
mode == $P ifTrue: [ ^ #towardPlusInfinity].
mode == $Z ifTrue: [ ^ #towardZero].
^ #unknown "should never occur!"

]

{ #category : 'Truncation and Rounding' }
DecimalFloat class >> roundingMode: aString [

"The argument aString defines the rounding mode ('nearestEven',
 'towardMinusInfinity', 'towardPlusInfinity', 'towardZero').  If aString is not
 one of these, an error is generated."

|status newChar |
(Symbol _existingWithAll: aString) ifNotNil:[ :aSymbol |
  newChar := nil.
  aSymbol == #nearestEven
    ifTrue: [newChar := $E]
    ifFalse: [
      aSymbol == #towardMinusInfinity
	ifTrue: [newChar := $N]
	ifFalse: [
	  aSymbol == #towardPlusInfinity
	    ifTrue: [newChar := $P]
	    ifFalse: [
	      aSymbol == #towardZero
		ifTrue: [newChar := $Z]
	      ]
	  ]
      ].
  ].
newChar ifNil: [^ aString _error: #numErrArgNotRoundingMode].
status := DecimalFloat status.
status at: 16 put: newChar.
DecimalFloat status: status.

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> status [

"Returns a six-element Array.  The first element of the Array is a String
 representing the status of the floating point processor, including the
 operation exceptions, raised exceptions, rounding mode, and the enabled traps.
 The next five elements of the Array are the blocks associated with each of the
 enabled traps, in this order: divideByZero, inexactResult, invalidOperation,
 overflow, underflow.

 Any method that modifies the trap handlers should first save the status
 using this method.  After the method has modified the trap handlers, it
 should use status: to restore the status."

<primitive: 134>
^ self _primitiveFailed: #floatStatus

]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> status: aString [

"Restores the status of the floating point processor to the previously saved
 status represented by aSymbol.  The argument aSymbol is the first element of
 the Array that DecimalFloat | status returns."

^ self _oneArgPrim:1 arg: aString


]

{ #category : 'Deprecated Exception Handling' }
DecimalFloat class >> trapEnabled: aString [

"Returns true if a trap handler has been defined for the specified exception
 Otherwise, returns false."

|status kind|

status := DecimalFloat status.
kind := DecimalFloat _exceptionKind: aString .
^(status at: self _exceptionList size * 2 + kind) == $1

]

{ #category : 'Decompiling without Sources' }
DecimalFloat >> _asSource [

^ self asString

]

{ #category : 'Private' }
DecimalFloat >> _asString: exponentChar [

"Same as asString, but prints specified exponent letter"

<primitive: 1018>
exponentChar _validateClass: Character .
exponentChar codePoint <= 127 ifFalse:[ Error signal:'unsupported code point'].
^ self _primitiveFailed: #_asString: args: { exponentChar }

]

{ #category : 'Converting' }
DecimalFloat >> _coerce: aNumber [

"Returns an instance of DecimalFloat with the same value as 'aNumber'."

"This method must be defined by all subclasses of Number."

^aNumber asDecimalFloat

]

{ #category : 'Comparing' }
DecimalFloat >> _compare: aNumber opcode: opcode selector: aSymbol [
	"Private."

	"opcode 0 means = , 1 means ~=, 2 means < , 3 means <= "

	<primitive: 117>
	| sk |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ 
			opcode > 1
				ifTrue: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
			^ opcode == 1 ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ opcode == 1 ].	"NaN"
	sk == 3
		ifTrue: [ ^ self _compareInfinityFor: aSymbol with: aNumber ].
	^ (AbstractFraction _coerce: self) perform: aSymbol with: aNumber
]

{ #category : 'Arithmetic' }
DecimalFloat >> _divide: aNumber round: roundingCharacter selector: aSymbol [

"Returns the result of dividing the receiver by aNumber,
 using specified rounding mode.

roundingCharacter is one of  nil , $E, $N, $P, $Z
specifying respectively
   (current DecimalFloat roundingMode),
   nearestEven, towardMinusInfinity, towardPlusInfinity, towardZero."

<primitive: 127>
roundingCharacter ifNotNil:[
  roundingCharacter class == Character ifFalse:[
    roundingCharacter _validateClass: Character
]].
^ self _retry: aSymbol coercing: aNumber

]

{ #category : 'Converting' }
DecimalFloat >> _generality [

"Returns the integer that represents the ordering of the receiver in the
 generality hierarchy."

"Reimplemented from Number"

^ 100

]

{ #category : 'Accessing' }
DecimalFloat >> _getKind [

"Returns an integer, 1...6, for the kind of the receiver."

<primitive: 133>

^ self _primitiveFailed: #_getKind

]

{ #category : 'Indexing Support' }
DecimalFloat >> _isNaN [

"Returns whether the receiver is quiet NaN or a signaling NaN.
 This method is only to be used by the indexing subsystem."

^ self _getKind > 4

]

{ #category : 'Converting' }
DecimalFloat >> _makeQuietNaN [

"This method returns a quiet NaN with the same sign as the input NaN.  If the
 input is not a NaN, this method returns the receiver."

   "Is the receiver a NaN?"
((self kind = #quietNaN) or: [self kind = #signalingNaN])
   ifFalse: [ ^self ]
   ifTrue: [ ^ PlusQuietNaN ]

]

{ #category : 'Private' }
DecimalFloat >> _zeroArgPrim: opcode [

"Private."

"opcode
  0 = truncated
  1 = sign
  2 = asString
  3 = sqrt
  4 = asStringLocaleC
"

<primitive: 128>
^ self _primitiveFailed: #_zeroArgPrim: args: { opcode }

]

{ #category : 'Arithmetic' }
DecimalFloat >> - aNumber [

"Returns the difference between the receiver and aNumber."

<primitive: 125>
^ self _retry: #- coercing: aNumber

]

{ #category : 'Arithmetic' }
DecimalFloat >> * aNumber [

"Returns the result of multiplying the receiver by aNumber."

<primitive: 126>
^ self _retry: #* coercing: aNumber

]

{ #category : 'Arithmetic' }
DecimalFloat >> / aNumber [

"Returns the result of dividing the receiver by aNumber."

^ self _divide: aNumber round: nil selector: #/

]

{ #category : 'Arithmetic' }
DecimalFloat >> // aNumber [

"Divides the receiver by aNumber.  Returns the integer quotient, with
 truncation toward negative infinity.  For example,

 9//4 = 2
 -9//4 = -3
 -0.9//0.4 = -3

 The selector \\ returns the remainder from this division."

   "If a signaling NaN, raise a floating point exception"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException ].

   "If a NaN, returns correct sign of self"
((self kind = #signalingNaN) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self _makeQuietNaN ].

^ (self _divide: aNumber round: $N selector: #// ) floor

]

{ #category : 'Arithmetic' }
DecimalFloat >> + aNumber [

"Returns the sum of the receiver and aNumber."

<primitive: 124>
^ self _retry: #+ coercing: aNumber

]

{ #category : 'Comparing' }
DecimalFloat >> < aNumber [

"Returns true if the receiver is less than aNumber, and false otherwise."

^ self _compare: aNumber opcode: 2 selector: #<

]

{ #category : 'Comparing' }
DecimalFloat >> <= aNumber [

"Returns true if the receiver is less than or equal to a aNumber,
 and false otherwise."

^ self _compare: aNumber opcode: 3 selector: #<=

]

{ #category : 'Comparing' }
DecimalFloat >> = aNumber [

"Returns true if the receiver is equal to aNumber, and false otherwise."

^ self _compare: aNumber opcode: 0 selector: #=

]

{ #category : 'Comparing' }
DecimalFloat >> >= aNumber [

"Returns true if the receiver is greater than or equal to aNumber,
 and false otherwise."

"Reimplemented from Magnitude to handle NaNs correctly."

^ aNumber <= self

]

{ #category : 'Comparing' }
DecimalFloat >> ~= aNumber [

"Returns true if the receiver is not equal to aNumber, and false otherwise."

^ self _compare: aNumber opcode: 1 selector: #~=

]

{ #category : 'Arithmetic' }
DecimalFloat >> abs [

"Returns a Number that is the absolute value of the receiver."

   "If a NaN, raises invalid operation exception and returns quiet NaN"
((self kind = #signalingNaN) or: [ self kind = #quietNaN ])
   ifTrue: [ DecimalFloat _raiseInvalidOperationException.
             ^self _makeQuietNaN ].

(self sign == -1)
   ifTrue: [^self negated].
^self
]

{ #category : 'Converting' }
DecimalFloat >> asDecimalFloat [

"Returns the receiver."

^ self

]

{ #category : 'Converting' }
DecimalFloat >> asFloat [

"Returns a SmallDouble or Float whose value is represented by the receiver."

 | aString kind |

 kind := self _getKind .
 kind > 2 ifTrue:[
   kind == 4 ifTrue:[ ^ 0.0e0 ] .
   kind == 3 ifTrue:[
     self sign < 0 ifTrue:[ ^ MinusInfinity ]
                  ifFalse:[ ^ PlusInfinity ].
     ].
   kind == 5 ifTrue:[ ^ PlusQuietNaN ].
   ^ PlusSignalingNaN
   ].

 aString := self _asString: $E .
 ^ Float fromString: aString

]

{ #category : 'Converting' }
DecimalFloat >> asFraction [

"Returns a Fraction that represents the receiver.  If the receiver is a NaN,
 or Infinity, returns the receiver."

| num numerator denominator |

   "If an infinite or quiet NaN, returns self"
((self kind = #infinity) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self ].

   "If a signaling NaN, raises a floating point exception & returns self"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException.
             ^self _makeQuietNaN ].

num := self.
denominator := 1.

[ num fractionPart = 0.0F0 ]
whileFalse:
  [ denominator := denominator * 10.
    num := num * 10.0F0 .
  ].

numerator := num truncated.

^ Fraction numerator: numerator denominator: denominator
]

{ #category : 'Formatting' }
DecimalFloat >> asString [

"Returns a String corresponding to the receiver.  Where applicable, returns one
 of the following Strings: 'PlusInfinity', 'MinusInfinity', 'PlusQuietNaN',
 'MinusQuietNaN', 'PlusSignalingNaN', or 'MinusSignalingNaN'.

                                    Note:
 GemStone currently formats DecimalFloats independently.  Specifically, it does
 not adjust the style of representation according to locale (decimal notation
 is not internationalized).  Under some circumstances, string representation of
 DecimalFloats may be inconsistent with those of Floats (which do
 internationalize by adjusting to locale)."

^ self _zeroArgPrim: 2

]

{ #category : 'Formatting' }
DecimalFloat >> asStringLocaleC [

"Same as asString, but result always uses  $.  for decimal point."

^ self _zeroArgPrim: 4

]

{ #category : 'Formatting' }
DecimalFloat >> asStringUsingFormat: anArray [

"Returns a String corresponding to the receiver, using the format specified by
 anArray.  The Array contains three elements: two Integers and a Boolean.
 Generates an error if any element of the Array is missing or is of the wrong
 class.

 The first element of the Array (an Integer) specifies a minimum number of
 Characters in the result String (that is, the 'width' of the string).  If this
 element is positive, the resulting String is padded with blanks to the right
 of the receiver.  If this element is negative, the blanks are added to the
 left of the receiver.  If the value of this element is not large enough to
 completely represent the DecimalFloat, a longer String will be generated.

 The second element of the Array (a positive Integer less than or equal to 
 1000) specifies the number of digits to display to the right of the decimal 
 point.  If the value of this element exceeds the number of digits required to 
 specify the DecimalFloat, the result is right-padded with 0 to the required 
 width.  If the value of this element is insufficient to completely specify 
 the DecimalFloat, the value of the DecimalFloat is rounded (see #rounded).

 The third element of the Array (a Boolean) indicates whether or not to display
 the magnitude using exponential notation.  (The value true indicates
 exponential notation and false indicates decimal notation.)

 For example, the number 12.3456 displayed with two different format Arrays
 would appear as follows:

 Format          Output
 #(10 5 true)    ' 1.23456E1'
 #(10 2 false)   '12.34'

                                    Note:
 GemStone currently formats DecimalFloats independently.  Specifically, it does
 not adjust the style of representation according to locale (decimal notation
 is not internationalized).  Under some circumstances, string representation of
 DecimalFloats may be inconsistent with those of Floats (which do
 internationalize by adjusting to locale)."

| value tempString count result width decimalWidth scientific
  decimalIndex expIndex intVal fractVal expVal fractString |

anArray _isArray ifFalse:[ anArray _validateClass: Array ].
(anArray size == 3) ifFalse: [ ^ self _error: #rtErrBadFormatSpec args: { anArray } ].
(width := anArray at: 1) _isInteger ifFalse:[ width _validateClass: Integer ].

( #( #infinity #signalingNaN #quietNaN) includesIdentical: self kind) ifTrue:[
   result := self asString 
] ifFalse: [ 
    decimalWidth := anArray at: 2.
    decimalWidth _isInteger ifFalse:[ decimalWidth _validateClass: Integer ].
    scientific := anArray at: 3.
    scientific class == Boolean ifFalse:[ scientific _validateClass: Boolean ].
    value := self abs.
    result := String new.
    self negative ifTrue: [ result add: $- ].
    scientific ifTrue:[
        tempString := value asString.
        decimalIndex := tempString indexOf: $.  startingAt: 1.
        expIndex := tempString indexOf: $F startingAt: decimalIndex.
        intVal := Integer fromString:
                (tempString copyFrom: 1 to: decimalIndex - 1).
        fractVal := DecimalFloat fromString:
                '1.' , (tempString copyFrom: decimalIndex + 1 to: expIndex - 1).
        expVal := Integer fromString:
                (tempString copyFrom: expIndex + 1 to: tempString size).
    ] ifFalse: [
        intVal := value truncated .
        fractVal := 1.0F0 + value fractionPart.
        expVal := 0.
    ].
    count := 0.
    [ (count < decimalWidth) and:[fractVal fractionPart ~= 0.0F0 ] ] whileTrue:[ 
       fractVal := fractVal * 10.0F0 .
        count := count + 1.
    ].
    fractVal := fractVal + 0.5F0 .
    fractString := fractVal truncated asString.
    ((fractString at: 1) == $1 ) ifFalse:[ intVal := intVal + 1 ].
    (scientific and: [ intVal = 10 ]) ifTrue: [ intVal := 1.  expVal := expVal + 1.  ].

    fractString removeFrom: 1 to: 1.
    [ fractString size < decimalWidth ] whileTrue:[ fractString add: $0 ].
    result addAll: intVal asString;
           add: Locale decimalPoint;
           addAll: fractString.
    scientific ifTrue:[ result add: $F; addAll: expVal asString ].
  ].
  ^ result width: width

]

{ #category : 'Accessing' }
DecimalFloat >> at: anIndex put: aValue [

"Disallowed.  You may not change the value of a DecimalFloat."

self shouldNotImplement: #at:put:

]

{ #category : 'Truncation and Rounding' }
DecimalFloat >> ceiling [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as positive infinity."

   "If a signaling NaN, raise a floating point exception"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException ].

   "If a NaN, returns correct sign of self"
((self kind = #signalingNaN) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self _makeQuietNaN ].

(self <= 0.0F0)
   ifTrue: [^self truncated]
   ifFalse: [^self negated floor negated truncated]

]

{ #category : 'Accessing' }
DecimalFloat >> denominator [

"Returns the denominator of a Fraction representing the receiver."

   "If an infinite or quiet NaN, returns self"
((self kind = #infinity) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self ].

   "If a signaling NaN, raises a floating point exception & returns self"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException.
             ^self _makeQuietNaN ].

^ (self asFraction) denominator

]

{ #category : 'Testing' }
DecimalFloat >> even [

"Returns true if the receiver is an even integer, false otherwise."

 ^ (self \\ 2.0F0 ) = 0.0F0
"Unlike for #odd, rounding does not result in incorrect answers for even,
 so no additional test is necessary."

]

{ #category : 'Arithmetic' }
DecimalFloat >> factorial [

"Returns the factorial of the integer part of the receiver.
 Returns 1 if the receiver is less than or equal to 1."

| x result |
result := 1.0F0  .
x := result .
self asInteger timesRepeat:[ result := result * x .  x := x + 1.0F0 ] .
^ result .

]

{ #category : 'Truncation and Rounding' }
DecimalFloat >> floor [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as negative infinity."

|result|

   "If a signaling NaN, raise a floating point exception"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException ].

   "If a NaN, returns correct sign of self"
((self kind = #signalingNaN) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self _makeQuietNaN ].

result := self truncated.
(self >= 0)
   ifTrue: [^result].
(self = result)
   ifTrue: [^result]
   ifFalse: [^ (result - 1) truncated]

]

{ #category : 'Truncation and Rounding' }
DecimalFloat >> integerPart [

"Returns an integer representing the receiver truncated toward zero."

^ self truncated asDecimalFloat

]

{ #category : 'Testing' }
DecimalFloat >> isZero [
  ^ self = 0.0F0
]

{ #category : 'Arithmetic' }
DecimalFloat >> negated [

"Returns a Number that is the negation of the receiver."

^ (-0.0F0 - self)

]

{ #category : 'Accessing' }
DecimalFloat >> numerator [

"Returns the numerator of a Fraction representing the receiver."

   "If an infinite or quiet NaN, returns self"
((self kind = #infinity) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self ].

   "If a signaling NaN, raises a floating point exception & returns self"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException.
             ^self _makeQuietNaN ].

^ (self asFraction) numerator

]

{ #category : 'Testing' }
DecimalFloat >> odd [

"Returns true if the receiver is an odd integer, false otherwise."

self = self truncated ifFalse: [^false].
"Compare to self truncated to reject values that are not quite integers,
 such as 9.99999999999999f-1, which due to rounding would answer true below."
 ^ (self \\ 2.0F0 ) = 1.0F0

]

{ #category : 'Arithmetic' }
DecimalFloat >> quo: aNumber [

"Divides the receiver by aNumber.  Returns the integer quotient, with
 truncation toward zero.  For example,

 -9 quo: 4 = -2
 -0.9 quo: 0.4 = -2

 The selector rem: returns the remainder from this division."

^ (self _divide: aNumber round: $Z selector: #quo: ) truncated

]

{ #category : 'Arithmetic' }
DecimalFloat >> rem: aNumber [

"Returns the integer remainder defined in terms of quo: (division of the
 receiver by aNumber, with truncation toward zero)."

  "x rem: y | x=infinity or y=0 are invalid floating point
   operations and returns quiet NaNs"

(aNumber sign = 0) "0.0/0.0 is also invalid"
   ifTrue: [ ^ (aNumber asDecimalFloat) / (aNumber asDecimalFloat)].
(self kind = #infinity) "infinity/infinity is also invalid"
   ifTrue: [ ^ self / self ].
^ super rem: aNumber

]

{ #category : 'Truncation and Rounding' }
DecimalFloat >> rounded [

"Returns the integer nearest in value to the receiver."

   "If a signaling NaN, raise a floating point exception"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException ].

   "If a NaN, returns correct sign of self"
((self kind = #signalingNaN) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self _makeQuietNaN ].

self >= 0.0F0 ifTrue: [^(self + 0.5F0) truncated]
            ifFalse:[^(self - 0.5F0) truncated]

]

{ #category : 'Truncation and Rounding' }
DecimalFloat >> roundTo: aNumber [

"Returns the multiple of aNumber that is nearest in value to the receiver."

   "If a signaling NaN, raise a floating point exception"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException ].

   "If a NaN, returns correct sign of self"
((self kind = #signalingNaN) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self _makeQuietNaN ].

aNumber = 0
  ifTrue: [^0]
  ifFalse: [^(self / aNumber) rounded * aNumber]

]

{ #category : 'Accessing' }
DecimalFloat >> sign [

"Returns 1 if the receiver is greater than zero, -1 if the receiver is
 less than zero, and zero if the receiver is zero."

^ self _zeroArgPrim: 1

]

{ #category : 'Accessing' }
DecimalFloat >> size: anInteger [

"Disallowed.  You may not change the size of a DecimalFloat."

self shouldNotImplement: #size:

]

{ #category : 'Arithmetic' }
DecimalFloat >> sqrt [

"Returns the square root of the receiver."

"Reimplemented from Number"

^ self _zeroArgPrim: 3

]

{ #category : 'Truncation and Rounding' }
DecimalFloat >> truncated [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located.  In particular, returns the receiver
 if the receiver is an integer."

^ self _zeroArgPrim: 0

]

{ #category : 'Truncation and Rounding' }
DecimalFloat >> truncateTo: aNumber [

"Returns the multiple of aNumber that is closest to the receiver, on the
 same side of the receiver as zero is located.  In particular, returns
 the receiver if the receiver is a multiple of aNumber."

   "If a signaling NaN, raise a floating point exception"
(self kind = #signalingNaN)
   ifTrue: [ DecimalFloat _raiseInvalidOperationException ].

   "If a NaN, returns correct sign of self"
((self kind = #signalingNaN) or: [ self kind = #quietNaN ])
   ifTrue: [ ^self _makeQuietNaN ].

aNumber = 0
  ifTrue: [^0]
  ifFalse: [^(self quo: aNumber) * aNumber]

]

{ #category : 'Storing and Loading' }
DecimalFloat >> writeTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

passiveObj writeClass: self class;
      nextPutAll: self asStringLocaleC;
      space

]
