Class {
	#name : 'GsPackage',
	#superclass : 'Object',
	#instVars : [
		'sessionMethods',
		'prereqs',
		'symbolDict'
	],
	#category : nil
}

{ #category : 'Accessing' }
GsPackage class >> globalName [

  ^#GsPackage_Current

]

{ #category : 'Instance Creation' }
GsPackage class >> installIn: aSymbolDictionary [

  | package |
  package := self new.
  aSymbolDictionary at: self globalName put: package.
    package symbolDict: aSymbolDictionary.
  ^package

]

{ #category : 'Instance Creation' }
GsPackage class >> new [

  ^self basicNew initialize

]

{ #category : 'Categories' }
GsPackage >> addCategory: aSymbol for: aBehavior [
  "returns the newly added category (a SymbolSet) or nil if not newly added"
  | ar catDict |
  ar := self sessionInfoFor: aBehavior.
  catDict := (ar at: 2).
  (catDict includesKey: aSymbol)
    ifFalse: [ | aSet |
      aSet := SymbolSet new .
      catDict add: (SymbolAssociation newWithKey: aSymbol value: aSet ).
      ^aSet].
  ^ nil

]

{ #category : 'Accessing' }
GsPackage >> addPrereq: aSymbolDict [

    prereqs add: aSymbolDict

]

{ #category : 'Categories' }
GsPackage >> addSelector: aSelector method: aGsNMethod toCategory: categoryName for: aBehavior [
  "Adds an already compiled method to specified class and category"
  | arr catSet mDict catDict |
  arr := self sessionInfoFor: aBehavior .
  mDict := arr at: 1 .
  mDict at: aSelector put: aGsNMethod .
  catDict := arr at: 2 .
  catSet := catDict at: categoryName ifAbsent:[ self addCategory: categoryName for: aBehavior ].
  catSet add: aSelector .

]

{ #category : 'Categories' }
GsPackage >> addSelector: aSelector toCategory: categoryName for: aBehavior [
    "new category is expected to exist"
  | ar catDict |
  ar := self sessionMethods at: aBehavior otherwise: nil .
  ar ifNil:[ ^ self ].
  catDict := (ar at: 2).
    (catDict at: categoryName) add: aSelector.

]

{ #category : 'Enumerating' }
GsPackage >> behaviorAndMethodDictDo: aBlock [
  self enabled ifTrue:[
    self sessionMethods keysAndValuesDo: [:beh :ar |
        aBlock value: beh value: (ar at: 1)
    ].
  ].

]

{ #category : 'Categories' }
GsPackage >> categoryNamesFor: aBehavior into: anArray [

    | ar catDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ self ].
    catDict := (ar at: 2).
    catDict keysAndValuesDo: [:aKey :aValue |
      anArray add: aKey
    ].

]

{ #category : 'Methods' }
GsPackage >> categoryOfSelector: aSymbol for: aBehavior [

    | ar catDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ nil ].
    catDict := (ar at: 2).
    catDict keysAndValuesDo: [:aKey :aValue |
      (aValue includesIdentical: aSymbol ) ifTrue:[ ^ aKey ].
    ].
    ^ nil

]

{ #category : 'Categories' }
GsPackage >> categorysDo: aBlock for: aBehavior [
  "evaluates aBlock for each method category of receiver
   in specified environment. Returns the receiver.

   aBlock should be a two argument block
   expecting the args  categoryNameSymbol ,   selectorsSet.
   aBlock may be invoked more than once for each category name.
   The iteration is done directly over the receiver's categories."

    | ar |
  ar := self sessionMethods at: aBehavior otherwise: nil .
  ar ifNil:[ ^ self ].
  (ar at: 2) keysAndValuesDo: aBlock

]

{ #category : 'Methods' }
GsPackage >> compiledMethodAt: aSymbol for: aBehavior [

    | ar methDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ nil ].
    methDict := (ar at: 1).
    ^methDict at: aSymbol otherwise: nil

]

{ #category : 'Methods' }
GsPackage >> copyCategoryDictFor: aBehavior into: aGsMethodDictionary [

    | ar catDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ self ].
    catDict := (ar at: 2).
    catDict keysAndValuesDo: [:key :values | | coll |
        coll := aGsMethodDictionary at: key otherwise: nil.
        coll ifNil:[
            coll := SymbolSet new.
            aGsMethodDictionary at: key put: coll
        ].
        coll canBeWritten ifFalse: [
            coll := coll copy.
            aGsMethodDictionary at: key put: coll
        ].
        coll addAll: values.
    ].

]

{ #category : 'Methods' }
GsPackage >> copyMethodDictFor: aBehavior into: aGsMethodDictionary [

    | ar methDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ self ].
    methDict := (ar at: 1).
    methDict keysAndValuesDo: [:key :value |
        aGsMethodDictionary at: key put: value
    ].

]

{ #category : 'Control' }
GsPackage >> disable [
  ^ self dynamicInstVarAt: #enabled put: false

]

{ #category : 'Control' }
GsPackage >> enable [
  ^ self dynamicInstVarAt: #enabled put: true

]

{ #category : 'Control' }
GsPackage >> enabled [
  "enabled by default, i.e. if never previously disabled"
  ^ (self dynamicInstVarAt: #enabled) ifNil:[ true ]

]

{ #category : 'Methods' }
GsPackage >> includesSelector: aSymbol for: aBehavior [

    | ar methDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ false ].
    methDict := (ar at: 1).
    ^methDict includesKey: aSymbol

]

{ #category : 'Initialize-Release' }
GsPackage >> initialize [
  sessionMethods := IdentityKeyValueDictionary new.
  prereqs := { } .

]

{ #category : 'Compiling' }
GsPackage >> methodAndCategoryDictionaryFor: aBehavior [

  "Returns a 2 element array { methodDict . categoryDict }"
  | arr |
  arr := self sessionInfoFor: aBehavior.
  ^ { arr at: 1 . arr at: 2 }

]

{ #category : 'Compiling' }
GsPackage >> methodPragmaDictFor: aBehavior [

    | ar |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ nil ].
    ar size < 4 ifTrue: [ar at: 4 put: IdentityKeyValueDictionary new].
   ^(ar at: 4) at: aBehavior otherwise: nil

]

{ #category : 'Reporting' }
GsPackage >> methodsReport [
  | coll str |
  str := String new .
  coll := SortedCollection sortBlock:[:a :b | a key name <= b key name ].
  sessionMethods associationsDo:[:assoc | coll add: assoc ].
  coll do:[:assoc | | sels |
    str add: '--- '; add: assoc key name ; lf.
    sels := assoc value ifNotNil:[:arr | (arr at: 1) keys ] ifNil:[ #() ].
    (SortedCollection withAll: sels ) do:[:aSelector |
      str add: aSelector ; lf .
    ].
  ].
  ^ str .

]

{ #category : 'Compiling' }
GsPackage >> methodStampDictFor: aBehavior [

    | ar |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ nil ].
   ^(ar at: 3) at: aBehavior otherwise: nil

]

{ #category : 'Categories' }
GsPackage >> moveSelector: aSelector toCategory: newCat for: aBehavior [
  "Okay if new category not already present"

  self removeSelector: aSelector fromCategoriesFor: aBehavior.
  self
    addCategory: newCat for: aBehavior;
    addSelector: aSelector toCategory: newCat for: aBehavior

]

{ #category : 'Printing' }
GsPackage >> name [
^  symbolDict name

]

{ #category : 'Accessing' }
GsPackage >> prereqs [

    ^prereqs

]

{ #category : 'Printing' }
GsPackage >> printOn: aStream [
| str |
str := 'a', self class name.
symbolDict name ifNotNil:[:nam |  str addAll:' for ' ; addAll: nam].
aStream nextPutAll: str .

]

{ #category : 'Methods' }
GsPackage >> removeAllMethods [
  | clss |
  clss := self sessionMethods keys .
  clss do:[:aClass |
    self sessionMethods removeKey: aClass otherwise: nil .
  ].

]

{ #category : 'Methods' }
GsPackage >> removeAllMethodsFor: aBehavior [
    self sessionMethods removeKey: aBehavior otherwise: nil .

]

{ #category : 'Methods' }
GsPackage >> removeAllSubclassCodeFor: aBehavior [
    | ar methDict |
  "GsFile gciLogServer: self printString ."
  "self pause."
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ self ].
    methDict := (ar at: 1).
    methDict keysAndValuesDo: [ :aKey :aMethod |
      methDict at: aKey put: ((methDict at: aKey) _copyToForceRecompilation).
    ].

]

{ #category : 'Categories' }
GsPackage >> removeCategory: aSymbol for: aBehavior [

    | ar catDict |
  ar := self sessionMethods at: aBehavior otherwise: nil .
  ar ifNil:[ ^ self ].
  catDict := (ar at: 2).
  catDict removeKey: aSymbol otherwise: nil

]

{ #category : 'Methods' }
GsPackage >> removeMethodAt: aSymbol for: aBehavior [

  | ar methDict meth catDict stampDict pragmaDict sel |
  ar := self sessionMethods at: aBehavior otherwise: nil .
  ar ifNil: [ ^nil ].
  methDict := (ar at: 1).
  meth := methDict removeKey: aSymbol otherwise: nil .
  meth ifNil: [ ^nil ].
  stampDict := (ar at: 3).
  stampDict removeKey: aSymbol otherwise: nil .
  ar size == 4 ifTrue: [
      pragmaDict := (ar at: 4).
      pragmaDict removeKey: aSymbol otherwise: nil .
  ].
  catDict := (ar at: 2).
  sel := meth selector .
  catDict keysAndValuesDo: [:aKey :aValue |
    (aValue remove: sel otherwise: nil ) ifNotNil:[ ^ meth ].
  ].
  ^meth

]

{ #category : 'Categories' }
GsPackage >> removeSelector: aSelector fromCategoriesFor: aBehavior [
  | ar catDict removed |
  ar := self sessionMethods at: aBehavior otherwise: nil .
  ar ifNil:[ ^ false ].
  catDict := (ar at: 2).
  removed := false .
  catDict keysAndValuesDo: [:aKey :aValue |
    (aValue remove: aSelector otherwise: nil ) ifNotNil:[ removed := true ].
  ].
  ^ removed

]

{ #category : 'Methods' }
GsPackage >> selectorsFor: aBehavior into: anArray [

    | ar methDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ self ].
    methDict := (ar at: 1).
    anArray addAll: methDict keys.

]

{ #category : 'Categories' }
GsPackage >> selectorsIn: categoryName for: aBehavior into: anArray [

    | ar catDict |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ self ].
    catDict := (ar at: 2).
    (Symbol _existingWithAll: categoryName) ifNotNil:[ :sym |
      anArray addAll: (catDict at: sym otherwise: #() ).
    ]

]

{ #category : 'Methods' }
GsPackage >> sessionInfoFor: aBehavior [
	^ self
		sessionInfoFor: aBehavior
		ifAbsent: [ 
			| ar methodDict categoryDict stampDict pragmaDict |
			methodDict := GsMethodDictionary new.
			categoryDict := GsMethodDictionary new.
			stampDict := IdentityKeyValueDictionary new.
			pragmaDict := IdentityKeyValueDictionary new.
			ar := {methodDict.
			categoryDict.
			stampDict.
			pragmaDict}.
			self sessionMethods at: aBehavior put: ar.
			ar ]
]

{ #category : 'Methods' }
GsPackage >> sessionInfoFor: aBehavior ifAbsent: absentBlock [
	^ self sessionMethods at: aBehavior ifAbsent: absentBlock
]

{ #category : 'Accessing' }
GsPackage >> sessionMethods [

  ^ sessionMethods

]

{ #category : 'Compiling' }
GsPackage >> setPragmas: pragmaArrayOrNil
forBehavior: aBehavior
forMethod: selector [

  | ar methodPragmaDict |
  ar := self sessionMethods at: aBehavior otherwise: nil .
  ar ifNil:[ ^ nil ].
  ar size < 4 ifTrue: [ar at: 4 put: IdentityKeyValueDictionary new].
  methodPragmaDict := (ar at: 4) at: aBehavior otherwise: nil.
  methodPragmaDict ifNil:[
		methodPragmaDict := IdentityKeyValueDictionary new.
		(ar at: 4) at: aBehavior put: methodPragmaDict.
	].
  ^ pragmaArrayOrNil
         ifNil:[ methodPragmaDict removeKey: selector asSymbol otherwise: nil ]
         ifNotNil:[ methodPragmaDict at: selector asSymbol put: pragmaArrayOrNil ]

]

{ #category : 'Compiling' }
GsPackage >> setStamp: aStampOrNil forBehavior: aBehavior forMethod: selector [
	| ar methodStampDict |
	ar := aStampOrNil
		ifNil: [ 
			"nil means we will remove session data for selector"
			self
				sessionInfoFor: aBehavior
				ifAbsent: [ 
					"no session info present for aBehavior ... we're done (see 49169)"
					^ nil ] ]
		ifNotNil: [ ar := self sessionInfoFor: aBehavior ].
	methodStampDict := (ar at: 3) at: aBehavior otherwise: nil.
	methodStampDict
		ifNil: [ 
			methodStampDict := IdentityKeyValueDictionary new.
			(ar at: 3) at: aBehavior put: methodStampDict ].
	^ aStampOrNil
		ifNil: [ methodStampDict removeKey: selector asSymbol otherwise: nil ]
		ifNotNil: [ methodStampDict at: selector asSymbol put: aStampOrNil ]
]

{ #category : 'Accessing' }
GsPackage >> symbolDict [

    ^symbolDict

]

{ #category : 'Accessing' }
GsPackage >> symbolDict: aSymDict [

    symbolDict := aSymDict

]

{ #category : 'Methods' }
GsPackage >> recompileFor: aBehavior [
    | ar methDict sels cats |
    ar := self sessionMethods at: aBehavior otherwise: nil .
    ar ifNil:[ ^ nil ].
    methDict := ar at: 1. cats := ar at: 2 .
    sels := methDict keys .
    sels do:[:aSel | | meth nm |
      meth := methDict at: aSel .
      nm := meth recompileIntoMethodDict: methDict intoCategories: cats .
      false ifTrue:[ GsFile gciLogServer:' recompiled ', nm printString ,' oop ', nm asOop asString].
    ].
]

