"
A GsPipe is an implementation of a FIFO queue that has no concurrency
conflicts when used in an environment with a single producer (a user who 
adds elements to the pipe) and a single consumer (a user who may remove 
items from the pipe).  In this environment, the producer and the consumer 
are guaranteed not to conflict with each other.

This implemementation is based on US Patent Number 6,360,219 
         ""Object queues with concurrent updating"".

The ""head"" and ""tail"" objects described in the patent are implemented as a
GsPipeElement where the next field references another GsPipeElement and the 
value is a SmallInteger.  For the head GsPipe element, the value indicates 
the number of removes performed and in the tail the number of additions 
performed.  

When the GsPipe is empty the head and tail next fields both refer to the same 
GsPipeElement.  A removal, if the GsPipe is not empty only modifies the head
GsPipeElement.  An add creates a new GsPipeElement object with the value specified
and links it into the list by storing the new GsPipeElement into the next field
of the GsPipeElement referenced by the tail and then updates the 
tail to reference the new element.  Since adds and removes never update the
same objects, there can be NO concurrency conflicts with these operations.

Multiple concurrent producers (sessions adding to the GsPipe) will experience
concurrency conflicts.  Users needing to have concurrent producers run without
conflict should consider using an RcPipe or an RcQueue.

Values in the GsPipe are always removed in the order that they were committed.

"
Class {
	#name : 'GsPipe',
	#superclass : 'Collection',
	#instVars : [
		'head',
		'tail'
	],
	#category : nil
}

{ #category : 'Instance Creation' }
GsPipe class >> new [

"Returns a new GsPipe."

^super new _initialize

]

{ #category : 'Private' }
GsPipe >> _initialize [

"initialize the sub-components"
| element |

element := GsPipeElement newWithNext: nil value: nil.
head := GsPipeElement newWithNext: element value: 0.
tail := GsPipeElement newWithNext: element value: 0.

]

{ #category : 'Adding' }
GsPipe >> add: aValue [

"Adds aValue to the GsPipe and returns aValue."

| element |
element := GsPipeElement newWithNext: nil value: aValue.

tail next next: element.
tail next: element.
tail value: (tail value + 1).
^ aValue

]

{ #category : 'Converting' }
GsPipe >> asArray [

"Returns an Array with the contents of the receiver."

| result element |

result := Array new.
element := head next next.
[element == nil] whileFalse: [
  result add: (element value).
  element := element next
].
^ result

]

{ #category : 'Copying' }
GsPipe >> copy [

"Returns a new instance of GsPipe with the elements that are in the receiver."

| contents |

contents := self asArray.
^ (GsPipe withAll: contents)

]

{ #category : 'Enumerating' }
GsPipe >> do: aBlock [

"Evaluates aBlock with each of the current elements of the GsPipe as
 the argument. The argument aBlock must be a one-argument block.  This
 method traverses the pipe elements in order.  Returns the receiver."

| element |

element := head next next.
[element == nil] whileFalse: [
  aBlock value: (element value).
  element := element next
].
^ self

]

{ #category : 'Accessing' }
GsPipe >> head [

^head

]

{ #category : 'Testing' }
GsPipe >> isEmpty [

"Returns true if the queue is empty, and false otherwise."

^self size == 0

]

{ #category : 'Updating' }
GsPipe >> objectSecurityPolicy: anObjectSecurityPolicy [

"Assigns the receiver and subcomponents to the given security policy."

| element |
super objectSecurityPolicy: anObjectSecurityPolicy.
head objectSecurityPolicy: anObjectSecurityPolicy.
tail objectSecurityPolicy: anObjectSecurityPolicy.

element := head next.
[element == nil] whileFalse: [
  element objectSecurityPolicy: anObjectSecurityPolicy.
  element := element next
].
^ self

]

{ #category : 'Removing' }
GsPipe >> peek [

"Returns the leading element from the receiver without removing it.
 If the receiver is empty, returns nil."

| element |
self size == 0 ifTrue: [ ^ nil ].

element := head next next.
^ element value

]

{ #category : 'Removing' }
GsPipe >> remove [

"Removes the first element from the receiver and returns that element.
 If the receiver is empty, returns nil."

| element aValue |
self size == 0 ifTrue: [ ^ nil ].

element := head next next.
aValue := element value.
head next: element.
head value: (head value + 1).
^aValue

]

{ #category : 'Removing' }
GsPipe >> removeAll [

"Removes all entries from the GsPipe, and returns an Array that contains
 those entries, in order."

| anArray |

anArray := self asArray.
head next: (tail next).
head value: (tail value).
^ anArray

]

{ #category : 'Accessing' }
GsPipe >> size [

"Returns the number of valid entries in the GsPipe."

^(tail value) - (head value)

]

{ #category : 'Accessing' }
GsPipe >> tail [

^tail

]
{ #category : 'Removing' }
GsPipe >> remove: oldObject ifAbsent: anExceptionBlock [
  ^ self shouldNotImplement: #remove:ifAbsent: 
]
{ #category : 'Removing' }
GsPipe >> removeIdentical: oldObject ifAbsent: anExceptionBlock [
  ^ self shouldNotImplement: #removeIdentical:ifAbsent: 
]

