"
The class RedoLog implements only GemStone internals.  That is, it provides 
only functionality required by GemStone itself.  It is not intended for 
customer use, by creating instances or by subclassing.

"
Class {
	#name : 'RedoLog',
	#superclass : 'Object',
	#instVars : [
		'conflictObjects',
		'redoObjects'
	],
	#gs_reservedoop : '94977',
	#category : nil
}

{ #category : 'Instance Creation' }
RedoLog class >> new [

"Create a new initialized instance."

^ self basicNew initialize

]

{ #category : 'Private' }
RedoLog >> _commitNested: aRedoLog [
  "Used during System class >> _commitNestedTransaction .
  self is the redo log for the parent transaction level.
  aRedoLog is the redo log for the nested transaction being committed.
  Merge contents of aRedoLog into self ."

  aRedoLog conflictObjects keysAndValuesDo:[:argKey :argVal |
    (conflictObjects at: argKey otherwise:nil) ifNotNil:[:aVal |
      argVal == aVal ifFalse:[
        System disableCommitsUntilAbortWithReason:'commitNested RedoLog merge failure'.
        Error signal: 'RedoLog conflict objects merge failure, key oop ',
           argKey asOop asString,' value oop ', aVal asOop asString,
           ' != nested value oop ', argVal asOop asString .
      ]
    ] ifNil:[ conflictObjects at: argKey put: argVal ].
  ].
  aRedoLog redoObjects keysAndValuesDo:[:argKey :argVal |
    (redoObjects at: argKey otherwise:nil) ifNotNil:[:arr |
      arr addAll: argVal .
    ] ifNil:[
      redoObjects at: argKey put: argVal
    ]
  ].

]

{ #category : 'Commit Processing' }
RedoLog >> _redoOperationsForEntries: logEntries [
" caller must ensure logEntries is non-nil "
1 to: logEntries size do: [ :i |
    (logEntries at: i) redo ifFalse: [ ^ false ]
].
^ true

]

{ #category : 'Logging' }
RedoLog >> addConflictObject: aConflictObject for: aRedoObject [

"Add an entry in the conflictObjects dictionary, mapping the given
 conflictObject to the given redo object."
| conflicts |
((conflicts := conflictObjects) at: aConflictObject otherwise: nil) == nil
    ifTrue: [ conflicts at: aConflictObject put: aRedoObject ].

]

{ #category : 'Logging' }
RedoLog >> addLargeConflictObject: aConflictObject for: aRedoObject [

"Add an entry in the conflictObjects dictionary, mapping the given
 conflictObject (and all internal nodes for large objects) to the given redo object."

| ar conflicts |
aConflictObject _isLarge
  ifFalse: [ ^ self addConflictObject: aConflictObject for: aRedoObject ].

"Get all internal nodes for aConflictObject (including aConflictObject) and add aRedoObject for each of them"
ar := aConflictObject _getInternalNodes.
conflicts := conflictObjects .
1 to: ar size do: [:j | | each |
  each := ar at: j .
  (conflicts at: each otherwise: nil)
    ifNil: [ conflicts at: each put: aRedoObject ].
].

]

{ #category : 'Logging' }
RedoLog >> addLogEntry: aLogEntry [

  "Add the given log entry to the log.  Update the redo object dictionary."
  | redoObj redos |
  redoObj := aLogEntry receiver.
  ((redos := redoObjects) at: redoObj otherwise: nil) 
    ifNotNil:[ :logArray | logArray add: aLogEntry ]
       ifNil:[  redos at: redoObj put: { aLogEntry } ].
]

{ #category : 'Logging' }
RedoLog >> addLogEntry: aLogEntry for: redoObj forConflictObject: aConflictObject [
  "Add the given log entry to the log.  Update the conflict objects and redo
   object dictionary."
  | conflicts redos |
  ((conflicts := conflictObjects) at: aConflictObject otherwise: nil) ifNil:[
    conflicts at: aConflictObject put: redoObj 
  ].
  ((redos := redoObjects) at: redoObj otherwise: nil) 
    ifNotNil:[ :logArray | logArray add: aLogEntry ]
       ifNil:[ redos at: redoObj put: { aLogEntry } ].
]

{ #category : 'Logging' }
RedoLog >> addLogEntry: aLogEntry for: redoObj conflictObject: obj1 conflictObject: obj2 [
  "Add the given log entry to the log.  Update the conflict objects and redo
   object dictionary."
  | conflicts redos |
  ((conflicts := conflictObjects) at: obj1 otherwise: nil) ifNil:[
    conflicts at: obj1 put: redoObj 
  ].
  (conflicts at: obj2 otherwise: nil) ifNil:[
    conflicts at: obj2 put: redoObj 
  ].
  ((redos := redoObjects) at: redoObj otherwise: nil) 
    ifNotNil:[ :logArray | logArray add: aLogEntry ]
       ifNil:[ redos at: redoObj put: { aLogEntry } ].
]

{ #category : 'Logging' }
RedoLog >> addFirstLogEntry: aLogEntry for: redoObj forConflictObject: aConflictObject [
  "Make the given log entry the first entry for redoObj.  
   Update the conflict objects and redo object dictionary."
  | conflicts redos |
  ((conflicts := conflictObjects) at: aConflictObject otherwise: nil) ifNil:[
    conflicts at: aConflictObject put: redoObj 
  ].
  ((redos := redoObjects) at: redoObj otherwise: nil) 
    ifNotNil:[ :logArray | logArray insertAll: { aLogEntry } at: 1 ]
       ifNil:[ redos at: redoObj put: { aLogEntry } ].
]

{ #category : 'Logging' }
RedoLog >> addLogEntry: aLogEntry for: redoObj forLargeConflictObject: aLargeConflictObject [

"Add the given log entry to the log.  Update the conflict objects and redo
 object dictionary."

| ar conflicts redos |
aLargeConflictObject _isLarge
  ifFalse: [ ^ self addLogEntry: aLogEntry for: redoObj forConflictObject: aLargeConflictObject ].

"Get all internal nodes for aLargeConflictObject (including aConflictObject) and add aRedoObject for each of them"
ar := aLargeConflictObject _getInternalNodes.
conflicts := conflictObjects .
1 to: ar size do: [:j | | aConflictObject |
  aConflictObject := ar at: j .
  (conflicts at: aConflictObject otherwise: nil)
    ifNil: [ conflicts at: aConflictObject put: redoObj ] 
].

((redos := redoObjects) at: redoObj otherwise: nil) 
  ifNotNil:[ :logArray | logArray add: aLogEntry ]
     ifNil:[ redos at: redoObj put: { aLogEntry } ].

]

{ #category : 'Accessing' }
RedoLog >> conflictObjects [

"Returns the value of the instance variable 'conflictObjects'."

^conflictObjects

]

{ #category : 'Updating' }
RedoLog >> conflictObjects: newValue [

"Modify the value of the instance variable 'conflictObjects'."

conflictObjects := newValue

]

{ #category : 'Accessing' }
RedoLog >> conflictObjectsSet [

"Returns a set of all conflict objects that were written.  These are the
 objects of possible conflict with other sessions."

^ conflictObjects keys

]

{ #category : 'Testing' }
RedoLog >> getLogEntriesFor: aRedoObject [

"Returns the Array of log entries for the give redo object.  If there
 are none, returns nil."

^ redoObjects at: aRedoObject otherwise: nil.

]

{ #category : 'Commit Processing' }
RedoLog >> getRedoObjectForConflictingObject: aConflictObject [

"Returns the redo object that contains the given conflict object that
 conflicted on a commit operation."

| obj |
" first see if a mapping is located in the conflictObjects dictionary "
obj := conflictObjects at: aConflictObject otherwise: nil.
obj ifNil: [
  "if there is a mapping in the redoObjects dictionary,
   returns the argument "
   (redoObjects at: aConflictObject otherwise: nil) ifNotNil: [ ^ aConflictObject ]
].
^ obj

]

{ #category : 'Initializing' }
RedoLog >> initialize [

"Initialize the conflictObjects and redoObjects instance variables to new
 Dictionaries."

conflictObjects := IdentityKeyValueDictionary new: 53.
conflictObjects collisionLimit: 1152921504606846975 "SmallInteger maximumValue".
redoObjects := IdentityKeyValueDictionary new: 53.
redoObjects collisionLimit: 1152921504606846975 .

]

{ #category : 'Formatting' }
RedoLog >> printConflictObjects [
  | str |
  str := String new .
  conflictObjects keysDo:[:aKey |
    str add: aKey asOop asString ;
       add: '(a'; add: aKey class name ; add: ') '.
  ].
  ^ str

]

{ #category : 'Formatting' }
RedoLog >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

aStream nextPutAll: 'Conflicts ('.
aStream cr.
conflictObjects keysAndValuesDo: [:conflictObj :redoObj |
    aStream nextPutAll: '[', conflictObj class printString, ', ', conflictObj asOop printString, ']'.
    aStream nextPutAll: '->'.
    aStream nextPutAll: '[', redoObj class printString, ', ', redoObj asOop printString, ']'.
  ].
aStream nextPutAll: ')'; cr.
aStream nextPutAll: 'Redo ('.
aStream cr.
redoObjects keysAndValuesDo: [:redoObj :logEntries|
    aStream nextPutAll: '[', redoObj class printString, ', ', redoObj asOop printString, ']'.
    aStream cr.
    logEntries do: [:logEntry |
      aStream nextPutAll: logEntry printString
    ].
  ].
aStream nextPutAll: ')'.

]

{ #category : 'Accessing' }
RedoLog >> redoObjects [

"Returns the value of the instance variable 'redoObjects'."

^redoObjects

]

{ #category : 'Updating' }
RedoLog >> redoObjects: newValue [

"Modify the value of the instance variable 'redoObjects'."

redoObjects := newValue

]
