Extension { #name : 'ScaledDecimal' }

{ #category : 'Private' }
ScaledDecimal class >> __mantissa: m scale: s [
 "for testing of passivate/activate, returns a ScaledDecimal."
 ^ (self _basicNew _mantissa: m scale: s) immediateInvariant
]

{ #category : 'Instance Creation' }
ScaledDecimal class >> _fromString: aString decimalPoint: dp [

"Given aString such as '34.23', returns an instance of the receiver with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point.  Characters in aString after
 the first character which is neither a digit or decimal point are ignored.
 Signals an OutOfRange if the scale of the result would exceed 30000.
 The specified specified decimal point character is used ,
 if dp == nil, the session's locale state is used."

<primitive: 1115>
aString _validateClasses:{ String }.
dp ifNotNil:[ dp _validateClass: Character ].
self _errIncorrectFormat: aString .
self _primitiveFailed: #_fromString:decimalPoint: args: { aString . dp } .

]

{ #category : 'Instance Creation' }
ScaledDecimal class >> for: aNumber scale: s [
	"Returns an instance of the receiver, having the specified scale
 and value as close to aNumber as that scale allows."

	| m |
	s _isSmallInteger ifFalse: [s _validateClass: SmallInteger].
	s > MaxScale
		ifTrue:
			[(OutOfRange new)
				name: '' max: MaxScale actual: s;
				details: 'invalid scale';
				signal].
	m := (aNumber * (10 raisedToInteger: s)) roundedHalfToEven.
	^ self mantissa: m scale: s 

]

{ #category : 'Instance Creation' }
ScaledDecimal class >> fromString: aString [

"Given aString such as '34.23', returns an instance of the receiver with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point.  Characters in aString after
 the first character which is neither a digit or decimal point are ignored.
 Signals an OutOfRange if the scale of the result would exceed 30000.
 The session's locale state is used for the expected decimal point."

^ self _fromString: aString decimalPoint: nil

]

{ #category : 'Instance Creation' }
ScaledDecimal class >> fromStringLocaleC: aString [
"Given aString such as '34.23', returns an instance of the receiver with
 appropriate numerator and denominator, and with scale equal to the number
 of digits to the right of the decimal point.  Characters in aString after
 the first character which is neither a digit or decimal point are ignored.
 Signals an OutOfRange if the scale of the result would exceed 30000.
 The expected decimal point character is $.  "

^ self _fromString: aString decimalPoint: $.

]

{ #category : 'Storing and Loading' }
ScaledDecimal class >> loadFrom: passiveObj [

"Reads from passiveObj the passive form of an object.  Converts the object to
 its active form by loading the information into a new instance of the receiver.
 Returns the new instance."

| inst m sc |
inst := self _basicNew .
passiveObj hasRead: inst .
passiveObj readNamedIV.
m := passiveObj ivValue.
passiveObj readNamedIV.
sc := passiveObj ivValue.
passiveObj skipNamedInstVars.
inst  _mantissa: m scale: sc ; immediateInvariant .
^ inst

]

{ #category : 'Instance Creation' }
ScaledDecimal class >> mantissa: m scale: s [
 "if primitive succeeds, returns a SmallScaledDecimal"
 <primitive: 1116>
 "primitive failed, return a ScaledDecimal"
 ^ (self basicNew _mantissa: m scale: s) immediateInvariant

]

{ #category : 'Private' }
ScaledDecimal class >> new [

self shouldNotImplement: #new .
^ nil

]

{ #category : 'Compatibility' }
ScaledDecimal class >> numerator: numer denominator: denom scale: scale [

"Returns an instance of ScaledDecimal.
 Signals an OutOfRange if the argument scale > 30000.  "
 | n d m |
 denom = 0 ifTrue: [ ^ numer   _errorDivideByZero ].
 scale _isSmallInteger ifFalse: [ scale _validateClass: SmallInteger ].
(scale < 0 or: [scale > MaxScale])
	ifTrue:
		[(OutOfRange new)
			name: 'scale' min: 0 max: MaxScale actual: scale;
      details: 'invalid scale';
			signal].
 n := numer .
 d := denom .
 d < 0 ifTrue:[  d := 0 - d . n := 0 - n ] .
 m := ( (n * (10 raisedToInteger: scale )) / d ) roundedHalfToEven .
 ^ self mantissa: m scale: scale 

]

{ #category : 'Private' }
ScaledDecimal >> _asString: dpChar [

| str mstr sc sz len isNeg m nDigits |
(m := self mantissa) ifNil:[ ^ 'ScaledDecimalNaN' ].
mstr := m asString .
len := mstr size .
sz := len .
sc := self scale.
nDigits := sz .
(isNeg := (mstr at: 1) == $-) ifTrue:[ nDigits := nDigits - 1 ].
str := String new .
nDigits <= sc  ifTrue:[ | zeros zerosLen zcount idx srcIdx |
  isNeg ifTrue:[ str add: $- ] .
  str add: $0 ; add: dpChar .
  zeros := '00000000000000000000000000000000000000000000000000000000000000000' .
  zerosLen := zeros size .
  zcount := sc - nDigits .
  [ zcount >= zerosLen ] whileTrue:[
    str add: zeros .
    zcount := zcount - zerosLen .
  ].
  zcount > 0 ifTrue:[
    "zeros copyFrom: 1 to: zcount into: str startingAt: str size + 1"
    idx := str size .
    str replaceFrom: idx + 1 to: idx + zcount with: zeros startingAt: 1 .
  ].
  "mstr copyFrom: len - nDigits + 1 to: len into: str startingAt: str size + 1"
  idx := str size + 1 .
  srcIdx := len - nDigits + 1 .
  str replaceFrom: idx  to: idx + nDigits - 1  with: mstr startingAt: srcIdx
] ifFalse:[
  str := mstr .
  str insertAll: dpChar at: (sz + 1 - sc ) .
].
^ str

]

{ #category : 'Converting' }
ScaledDecimal >> _coerce: aNumber [

"Reimplemented from Number."

^ aNumber asScaledDecimal: self scale

]

{ #category : 'Private' }
ScaledDecimal >> _generality [

"Returns an Integer representing the ordering of the receiver in
 the generality hierarchy."

^ 60

]

{ #category : 'Testing' }
ScaledDecimal >> _getKind [
 | m |
 (m := self mantissa) ifNil:[ ^ 5 ]. "nan, should never occur"
 m == 0 ifTrue:[ ^ 4 ]. "zero"
 ^ 1 "normal"
]

{ #category : 'Private' }
ScaledDecimal >> _mantissa: m scale: s [
  m _isInteger ifFalse:[ m _validateClass: Integer ].
  s _isSmallInteger ifFalse:[ s _validateClass: SmallInteger ].
  s > MaxScale ifTrue:[
    OutOfRange new name: '' max: MaxScale actual: s ;
	details: 'invalid scale' ; signal
  ].
  s < 0 ifTrue:[ OutOfRange new name: 'scale' min: 0 actual: s ;
                    details: 'scale must be >= 0' ; signal ].
  mantissa := m .
  scale := s

]

{ #category : 'Arithmetic' }
ScaledDecimal >> - aNumber [
  "Returns the difference between the receiver and aNumber.
   If aNumber is an Integer or a ScaledDecimal, the result will be
   an instance of the receiver's class with a scale equal to
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

aNumber _isScaledDecimal
	ifTrue:
		[^self
			+ (ScaledDecimal mantissa: 0 - aNumber mantissa scale: aNumber scale)].
^self + (0 - aNumber)

]

{ #category : 'Arithmetic' }
ScaledDecimal >> * aNumber [
  "Returns the result of multiplying the receiver by aNumber.
   If aNumber is an Integer or a ScaledDecimal, the result will be
   an instance of the receiver's class with a scale equal to
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

| osc om |
aNumber _isScaledDecimal
	ifTrue:
		[om := aNumber mantissa.
		osc := aNumber scale]
	ifFalse:
		[aNumber _isInteger
			ifTrue:
				[om := aNumber.
				osc := 0]].
om
	ifNotNil:
		[| m pten resultScale excessScale myScale |
		m := self mantissa * om.
		resultScale := (myScale := self scale) max: osc.
		excessScale := myScale + osc - resultScale.
		excessScale > 0
			ifTrue:
				["reduce mantissa to match result scale"
				pten := 10 raisedTo: excessScale.
				m := (m / pten) roundedHalfToEven].
		^self class mantissa: m scale: resultScale].
^self _retry: #* coercing: aNumber

]

{ #category : 'Arithmetic' }
ScaledDecimal >> / aNumber [
  "Returns the result of dividing the receiver by aNumber.
   If aNumber is an Integer or a ScaledDecimal, the result will be
   an instance of the receiver's class with a scale equal to
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

| divisorScale om |
aNumber _isScaledDecimal
	ifTrue:
		[om := aNumber mantissa.
		divisorScale := aNumber scale]
	ifFalse:
		[aNumber _isInteger
			ifTrue:
				[om := aNumber.
				divisorScale := 0]].
om
	ifNotNil:
		[| resultScale adjustFactor m myScale |
		om == 0 ifTrue: [^self _errorDivideByZero].
		resultScale := (myScale := self scale) max: divisorScale.
		adjustFactor := 10 raisedToInteger: divisorScale - myScale + resultScale.
		m := (self mantissa / om * adjustFactor) roundedHalfToEven.
		^self class mantissa: m scale: resultScale].
^self _retry: #/ coercing: aNumber

]

{ #category : 'Arithmetic' }
ScaledDecimal >> // aNumber [
  "Returns an integer that is the result of dividing the receiver by aNumber,
   then rounding towards negative infinity."

| divisorScale om |
aNumber _isScaledDecimal
	ifTrue:
		[om := aNumber mantissa.
		divisorScale := aNumber scale]
	ifFalse:
		[aNumber _isInteger
			ifTrue:
				[om := aNumber.
				divisorScale := 0]].
om
	ifNotNil:
		[| exactResult |
		om == 0 ifTrue: [^self _errorDivideByZero].
		exactResult := self mantissa / om * (10 raisedToInteger: divisorScale - self scale).
		^ exactResult floor].
^self _retry: #// coercing: aNumber

]

{ #category : 'Arithmetic' }
ScaledDecimal >> + aNumber [
  "Returns the sum of the receiver and aNumber.
   If aNumber is an Integer or a ScaledDecimal, the result will be
   an instance of the receiver's class with a scale equal to
   the greater of the receiver's scale and aNumber's scale.
   The scale of an Integer is considered to be zero."

| m sc |
aNumber _isScaledDecimal
	ifTrue:
		[| osc pten myScale |
		osc := aNumber scale.
		(myScale := self scale) == osc
			ifTrue:
				[sc := myScale.
				 m := self mantissa + aNumber mantissa]
			ifFalse:
				[sc := myScale max: osc.
				sc > myScale
					ifTrue:
						[pten := 10 raisedToInteger: sc - myScale.
						m := self mantissa * pten + aNumber mantissa]
					ifFalse:
						[pten := 10 raisedToInteger: sc - osc.
						m := self mantissa + (pten * aNumber mantissa)]].
		^self class mantissa: m scale: sc].
aNumber _isInteger
	ifTrue:
		[ | myScale |
      m := self mantissa + (aNumber * (10 raisedToInteger: (myScale := self scale))).
		  ^ self class mantissa: m scale: myScale].
^self _retry: #+ coercing: aNumber

]

{ #category : 'Comparing' }
ScaledDecimal >> < aNumber [
	| sk osc om m |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	m := self mantissa.
	aNumber _isScaledDecimal
		ifTrue: [ 
			osc := aNumber scale.
			om := aNumber mantissa.
			om ifNil: [ ^ false ] ]
		ifFalse: [ 
			aNumber _isInteger
				ifTrue: [ 
					osc := 0.
					om := aNumber ] ].
	om
		ifNotNil: [ 
			| sc |
			sc := self scale.
			sc == osc
				ifFalse: [ 
					sc > osc
						ifTrue: [ om := om * (10 raisedTo: sc - osc) ]
						ifFalse: [ m := m * (10 raisedTo: osc - sc) ] ].
			^ m < om ].
	^ (AbstractFraction _coerce: self) < aNumber
]

{ #category : 'Comparing' }
ScaledDecimal >> <= aNumber [
	| sk osc om m |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	m := self mantissa.
	aNumber _isScaledDecimal
		ifTrue: [ 
			osc := aNumber scale.
			om := aNumber mantissa.
			om ifNil: [ ^ false ] ]
		ifFalse: [ 
			aNumber _isInteger
				ifTrue: [ 
					osc := 0.
					om := aNumber ] ].
	om
		ifNotNil: [ 
			| sc |
			sc := self scale.
			sc == osc
				ifFalse: [ 
					sc > osc
						ifTrue: [ om := om * (10 raisedTo: sc - osc) ]
						ifFalse: [ m := m * (10 raisedTo: osc - sc) ] ].
			^ m <= om ].
	^ (AbstractFraction _coerce: self) <= aNumber
]

{ #category : 'Comparing' }
ScaledDecimal >> = aNumber [
	| osc om m |
	self == aNumber
		ifTrue: [ ^ true ].
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ false ].
	(m := self mantissa) ifNil: [ ^ false	"self is a NaN" ].
	aNumber _isScaledDecimal
		ifTrue: [ 
			osc := aNumber scale.
			om := aNumber mantissa.
			om ifNil: [ ^ false ] ]
		ifFalse: [ 
			aNumber _isInteger
				ifTrue: [ 
					osc := 0.
					om := aNumber ] ].
	om
		ifNotNil: [ 
			| sc |
			sc := self scale.
			sc == osc
				ifFalse: [ 
					sc > osc
						ifTrue: [ om := om * (10 raisedTo: sc - osc) ]
						ifFalse: [ m := m * (10 raisedTo: osc - sc) ] ].
			^ m = om ].
	^ (AbstractFraction _coerce: self) = aNumber
]

{ #category : 'Comparing' }
ScaledDecimal >> > aNumber [
	| sk osc om m |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	m := self mantissa.
	aNumber _isScaledDecimal
		ifTrue: [ 
			osc := aNumber scale.
			om := aNumber mantissa.
			om ifNil: [ ^ false ] ]
		ifFalse: [ 
			aNumber _isInteger
				ifTrue: [ 
					osc := 0.
					om := aNumber ] ].
	om
		ifNotNil: [ 
			| sc |
			sc := self scale.
			sc == osc
				ifFalse: [ 
					sc > osc
						ifTrue: [ om := om * (10 raisedTo: sc - osc) ]
						ifFalse: [ m := m * (10 raisedTo: osc - sc) ] ].
			^ m > om ].
	^ (AbstractFraction _coerce: self) > aNumber
]

{ #category : 'Comparing' }
ScaledDecimal >> >= aNumber [
	| sk osc om m |
	(aNumber _isNumber or: [ aNumber isNumber ])
		ifFalse: [ ^ ArgumentTypeError signal: 'Expected a Number' ].
	sk := self _getKind.
	sk > 4
		ifTrue: [ ^ false ].	"NaN"
	m := self mantissa.
	aNumber _isScaledDecimal
		ifTrue: [ 
			osc := aNumber scale.
			om := aNumber mantissa.
			om ifNil: [ ^ false ] ]
		ifFalse: [ 
			aNumber _isInteger
				ifTrue: [ 
					osc := 0.
					om := aNumber ] ].
	om
		ifNotNil: [ 
			| sc |
			sc := self scale.
			sc == osc
				ifFalse: [ 
					sc > osc
						ifTrue: [ om := om * (10 raisedTo: sc - osc) ]
						ifFalse: [ m := m * (10 raisedTo: osc - sc) ] ].
			^ m >= om ].
	^ (AbstractFraction _coerce: self) >= aNumber
]

{ #category : 'Converting' }
ScaledDecimal >> asCanonicalForm [
	"Answer self, or, if I am a ScaledDecimal with an equivalent
	SmallScaledDecimal, answer that SmallScaledDecimal."

	| cls res |
	(cls := self class) ~~ SmallScaledDecimal
		ifTrue: [ 
			cls == ScaledDecimal
				ifTrue: [ 
					res := cls mantissa: self mantissa scale: self scale.
					res class == SmallScaledDecimal
						ifTrue: [ ^ res ] ] ].
	^ self
]

{ #category : 'Converting' }
ScaledDecimal >> asDecimalFloat [
	"Answer an instance of DecimalFloat that has a numeric value as close
	as possible to the value of the receiver, rounding if necessary"
	"DecimalFloats have a 20-decimal-digit precision, so the basic approach is to round the mantissa to 20 digits and then scale. In cases where the result should be a subnormal float, however, the 20-digit precision includes the leading zeroes, so we must round the mantissa to fewer than 20 digits. "

	| m mantissaDigits roundedMantissaDigits excessMantissaDigits roundedMantissa adjustmentPower 
    myScale |
	mantissaDigits := (m := self mantissa) _decimalDigitsLength: false.
	roundedMantissaDigits := mantissaDigits - (myScale := self scale) + 15019 min: 20.
	excessMantissaDigits := mantissaDigits - roundedMantissaDigits max: 0.
	roundedMantissa := (m / (10 raisedToInteger: excessMantissaDigits)) roundedHalfToEven.
	adjustmentPower := excessMantissaDigits - myScale.
	"DecimalFloats with very small magnitude must be computed a bit more indirectly to avoid underflow to 0."
	^adjustmentPower < -15000
		ifFalse: [roundedMantissa asDecimalFloat * (10.0f0 raisedToInteger: adjustmentPower)]
		ifTrue: [roundedMantissa asDecimalFloat / 1.0f20 * (10.0f0 raisedToInteger: adjustmentPower + 20)]

]

{ #category : 'Converting' }
ScaledDecimal >> asFloat [
	"Answer a Float that closely approximates the value of the receiver.
	This implementation will answer the closest floating point number to the receiver.
	In case of a tie, it will use the IEEE 754 round to nearest even mode.
	In case of overflow, it will answer +/- Float infinity."

	^ self asFraction asFloat
]

{ #category : 'Converting' }
ScaledDecimal >> asFraction [

"Returns a Fraction that represents the receiver."

^ Fraction numerator: self mantissa denominator: (10 raisedTo: self scale)

]

{ #category : 'Converting' }
ScaledDecimal >> asScaledDecimal: newScale [
"Returns a new instance with the new scale, and a numeric value
 as close to that of the receiver as possible."

| pten m myScale |
(myScale := self scale) == newScale ifTrue: [^self].
newScale _isSmallInteger ifFalse: [newScale _validateClass: SmallInteger].
(newScale < 0 or: [newScale > MaxScale])
	ifTrue:
		[(OutOfRange new)
			name: 'scale' min: 0 max: MaxScale actual: newScale;
      details: 'invalid scale';
			signal].
pten := 10 raisedToInteger: (myScale - newScale) abs.
myScale < newScale
	ifTrue: [m := self mantissa * pten]
	ifFalse: [m := (self mantissa / pten) roundedHalfToEven].
^self class mantissa: m scale: newScale

]

{ #category : 'Formatting' }
ScaledDecimal >> asString [
  "Returns a String of the form '123.56 for a number with scale = 2  ,
   where the decimal point character in the result is per the current Locale."

  ^ self _asString: Locale decimalPoint

]

{ #category : 'Formatting' }
ScaledDecimal >> asStringLocaleC [

  "Returns a String of the form '123.56 for a number with scale = 2.
   Does not use Locale , decimal point character is always $.  "

  ^ self _asString: $.

]

{ #category : 'Truncation and Rounding' }
ScaledDecimal >> ceiling [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as positive infinity."

 | f |
 f := self floor.
^ self = f
	ifTrue: [f]
	ifFalse: [f + 1]

]

{ #category : 'Compatibility' }
ScaledDecimal >> denominator [
  ^ self asFraction denominator

]

{ #category : 'Testing' }
ScaledDecimal >> even [

"Returns true if the receiver is an even integer, false otherwise."
| frac intPart m sc |
m := self mantissa .
(sc := self scale) == 0 ifTrue:[ ^ m even ].
sc := 10 raisedTo: sc .
intPart := m // sc .
frac := m - (intPart * sc) .
frac ~~ 0 ifTrue:[ ^ false ] .
^ intPart even

]

{ #category : 'Truncation and Rounding' }
ScaledDecimal >> floor [
"Returns the integer that is closest to the receiver, on the same side
 of the receiver as negative infinity."

^ self mantissa // (10 raisedTo: self scale)

]

{ #category : 'Comparing' }
ScaledDecimal >> hash [

^ self asFloat hash

]

{ #category : 'Truncation and Rounding' }
ScaledDecimal >> integerPart [
  ^ self class for: self truncated scale:  self scale

]

{ #category : 'Testing' }
ScaledDecimal >> isZero [

"Returns true if the receiver is zero."

^ self mantissa == 0 .

]

{ #category : 'Accessing' }
ScaledDecimal >> mantissa [

^ mantissa

]

{ #category : 'Arithmetic' }
ScaledDecimal >> negated [
  "Returns 0 minus self. Result has same scale as receiver. "
^ self class mantissa: 0 - self mantissa scale: self scale

]

{ #category : 'Compatibility' }
ScaledDecimal >> numerator [
  ^ self asFraction numerator

]

{ #category : 'Testing' }
ScaledDecimal >> odd [

"Returns true if the receiver is an odd integer, false otherwise."

| frac intPart m sc |
m := self mantissa .
(sc := self scale) == 0 ifTrue:[ ^ m odd ].
sc := 10 raisedTo: sc .
intPart := m // sc .
frac := m - (intPart * sc) .
frac ~~ 0 ifTrue:[ ^ false ] .
^ intPart odd

]

{ #category : 'Copying' }
ScaledDecimal >> postCopy [
  ^ self immediateInvariant

]

{ #category : 'Compatibility' }
ScaledDecimal >> reduced [

 ^ self

]

{ #category : 'Truncation and Rounding' }
ScaledDecimal >> roundAndCoerceTo: aNumber [

"Returns the multiple of aNumber that is nearest in value to the receiver.
  If the receiver is exactly between two multiples of aNumber,
  answer the one that is even number times aNumber. Example:
    4.125s roundTo: 0.75s --> 4.500s
    4.875s roundTo: 0.75s --> 4.500s
  because 4.5 = 6 * 0.75

 If aNumber is a kind of Integer, or AbstractFraction,
 the result will be an instance of the class of aNumber .
"
aNumber = 0 ifTrue:[ ^ 0 ].
^  (self / aNumber) roundedHalfToEven * aNumber  .
]

{ #category : 'Truncation and Rounding' }
ScaledDecimal >> rounded [
 "Returns the integer nearest in value to the receiver."
  | r myScale |
  r := self + (self class numerator: self sign denominator: 2 scale: (myScale :=self scale)) .
  ^ r mantissa quo: (10 raisedTo: myScale) .

]

{ #category : 'Truncation and Rounding' }
ScaledDecimal >> roundTo: aNumber [

"Returns the multiple of aNumber that is nearest in value to the receiver.
  If the receiver is exactly between two multiples of aNumber,
  answer the one that is even number times aNumber. Example:
    4.125s roundTo: 0.75s --> 4.500s
    4.875s roundTo: 0.75s --> 4.500s
  because 4.5 = 6 * 0.75 "

| r |
aNumber = 0 ifTrue:[ ^ 0 ].
r := (self / aNumber) roundedHalfToEven * aNumber  .
r _isInteger ifTrue:[ ^ self class numerator: r denominator: 1 scale: self scale].
^ r
]

{ #category : 'Accessing' }
ScaledDecimal >> scale [

^ scale

]

{ #category : 'Accessing' }
ScaledDecimal >> sign [
  | m |
  ((m := self mantissa) > 0) ifTrue:[ ^ 1 ].
  (m < 0) ifTrue:[ ^ -1 ].
  ^0

]

{ #category : 'Converting' }
ScaledDecimal >> truncateAndCoerceTo: aNumber [

 "Returns the multiple of aNumber that is closest to the receiver, on
 the same side of the receiver as zero is located.  In particular,
 returns the receiver if the receiver is a multiple of aNumber."

  aNumber = 0 ifTrue:[ ^ 0 ].

  ^ (self quo: aNumber) * aNumber
]

{ #category : 'Truncation and Rounding' }
ScaledDecimal >> truncated [

"Returns the integer that is closest to the receiver, on the same side
 of the receiver as zero is located."

  | f |
  f := self floor.
  (self mantissa < 0 and: [f ~= self])
      ifTrue:[ f := f + 1 ].
  ^ f

]

{ #category : 'Converting' }
ScaledDecimal >> truncateTo: aNumber [
  | arg |
  (arg := aNumber) _isInteger ifTrue:[
    arg := self class for: aNumber scale: self scale.
  ].
  ^super truncateTo: arg

]

{ #category : 'Compatibility' }
ScaledDecimal >> withScale: newScale [
"Returns a new instance with the new scale, and a numeric value
 as close to that of the receiver as possible."

	^self asScaledDecimal: newScale

]

{ #category : 'Storing and Loading' }
ScaledDecimal >> writeTo: passiveObj [

"Converts the receiver to its passive form and writes that information on
 passiveObj."

"Reimplemented from Number since the receiver has a non-literal representation."

^super basicWriteTo: passiveObj

]
