Extension { #name : 'Time' }

{ #category : 'Instance Creation' }
Time class >> __now [

"Creates and returns an instance of SmallTime from the system clock on the
 machine that is running the Gem process, which is assumed to represent the
 current time of day."

<primitive: 317>
^ self _primitiveFailed: #_now

]

{ #category : 'Instance Creation' }
Time class >> _now [
  | t |
  t := self __now .
  self == Time
    ifTrue:[ ^ t ]
    ifFalse:[ ^ self basicNew _setMilliseconds: t asMilliseconds]  

]

{ #category : 'Private' }
Time class >> dateTimeClass [

"Returns the DateTime class used internally in this class."

^ DateTime.

]

{ #category : 'Instance Creation' }
Time class >> fromMicroseconds: anInteger [
  self == Time ifTrue:[ ^ SmallTime fromMicroseconds: anInteger ]
              ifFalse:[ ^ super new _setMilliseconds: anInteger // 1000 ] 
]

{ #category : 'Instance Creation' }
Time class >> fromMilliseconds: anInteger [

"Creates and returns a kind of the receiver from the specified value,
 which expresses local time."

self == Time ifTrue:[ ^ SmallTime fromMilliseconds: anInteger ]
            ifFalse:[ ^ super new _setMilliseconds: anInteger ] 

]

{ #category : 'Instance Creation' }
Time class >> fromSeconds: anInteger [

"Creates and returns a kind of the receiver from the specified value,
 which expresses local time."

^ self fromMilliseconds: (anInteger * 1000).

]

{ #category : 'Instance Creation' }
Time class >> fromSecondsGmt: anInteger [

"Creates and returns an instance of the receiver from the specified value,
 which expresses Greenwich Mean Time."

^ self fromMicroseconds: ((anInteger - self gmtOffsetSeconds) * 1000000).

]

{ #category : 'Instance Creation' }
Time class >> fromStream: aStream [

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses local time in the default format (HH:MM:SS).
 Generates an error if the String does not conform to the format."

^ self fromStream: aStream usingFormat: #($: true false).

]

{ #category : 'Instance Creation' }
Time class >> fromStream: aStream usingFormat: anArray [

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses Greenwich Mean Time in the format specified by
 anArray.  The expression is terminated either by a space Character or by the
 end of the Stream.  Generates an error if the String does not conform to the
 format, or if anArray contains an incorrect formatting specification.

 See the class documentation of Time for a complete description of the
 String-formatting specification Array."

| hourInt minInt secInt timeDelim ampm ampmPresent secondsPresent parseField
  totalSeconds |

"This block returns a string up from the input stream up to the specified
 delimiter.  If also allows an end-of-file if that parameter is set true.
 It then skips over the delimiter if it is found.
"
parseField := [ :delim :allowEof | | str |
                str := aStream contents class new.
                [ ((aStream peek isEquivalent: delim) not) and:[aStream atEnd not] ]
                whileTrue:
                  [ str add: aStream next ].
                (aStream atEnd)
                ifTrue:
                  [ allowEof
                    ifFalse:
                      [ self _error: #rtErrBadFormat args: { aStream } ]
                  ]
                ifFalse:
                  [ aStream next "skip over delimiter" ].
                str
             ].

self _checkReadStream: aStream forClass: CharacterCollection.

timeDelim := anArray at: 1.
secondsPresent := anArray at: 2.
ampmPresent := anArray at: 3.
hourInt := Integer fromCompleteString: (parseField value: timeDelim value: false).
minInt := Integer fromCompleteString:
                  (parseField value: (secondsPresent ifTrue: [timeDelim] ifFalse: [$ ])
			      value: (secondsPresent not and:[ ampmPresent not])).
secondsPresent
  ifTrue: [
    secInt := Integer fromCompleteString: (parseField value: $  value: ampmPresent not)]
  ifFalse:
    [ secInt := 0 ].

ampmPresent
  ifTrue: [
    hourInt < 0 ifTrue: [ self _error: #rtErrBadFormat args: { aStream }].
    hourInt > 12 ifTrue: [ self _error: #rtErrBadFormat args: { aStream }].
    ampm := String new.
    ampm add: (aStream next); add: aStream next.
    (ampm isEquivalent: 'PM')
      ifTrue: [
        hourInt := hourInt + 12.
        hourInt == 24 ifTrue: [ hourInt := 12].
        ]
      ifFalse: [
        (ampm isEquivalent: 'AM') ifFalse: [
        self _error: #rtErrBadFormat args: { aStream } ].
        hourInt == 12 ifTrue: [ hourInt := 0].
        ].
    ].

totalSeconds := (hourInt * 3600) + (minInt * 60) + secInt.
^ self fromSeconds: totalSeconds .

]

{ #category : 'Instance Creation' }
Time class >> fromStreamGmt: aStream [

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses Greenwich Mean Time in the default format
 (HH:MM:SS).  Generates an error if the String does not conform to the format."

^ self fromStreamGmt: aStream usingFormat: #($: true false).

]

{ #category : 'Instance Creation' }
Time class >> fromStreamGmt: aStream usingFormat: anArray [

"Creates and returns an instance of the receiver by reading a String from
 aStream.  The String expresses local time in the format specified by
 anArray.  The expression is terminated either by a space Character or by the
 end of the Stream.  Generates an error if the String does not conform to the
 format, or if anArray contains an incorrect formatting specification.

 See the class documentation of Time for a complete description of the
 String-formatting specification Array."

^ (self fromStream: aStream usingFormat: anArray )
        subtractSeconds: self gmtOffsetSeconds.

]

{ #category : 'Instance Creation' }
Time class >> fromString: aString [

"Creates and returns an instance of the receiver from the String aString.
 The String expresses local time in the default format (HH:MM:SS).
 Generates an error if the String does not conform to the format."

^ self fromString: aString usingFormat: #($: true false).

]

{ #category : 'Instance Creation' }
Time class >> fromString: aString usingFormat: anArray [

"Creates and returns an instance of the receiver from the String aString.
 The String expresses local time in the format specified by anArray.
 The expression is terminated either by a space Character or by the end of the
 String.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of Time for a complete description of the
 String-formatting specification Array."

| s result |

s := ReadStreamPortable on: aString.
result := self fromStream: s usingFormat: anArray.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result

]

{ #category : 'Instance Creation' }
Time class >> fromStringGmt: aString [

"Creates and returns an instance of the receiver from the String aString.
 The String expresses Greenwich Mean Time in the default format (HH:MM:SS).
 Generates an error if the String does not conform to the format."

^ self fromStringGmt: aString usingFormat: #($: true false).

]

{ #category : 'Instance Creation' }
Time class >> fromStringGmt: aString usingFormat: anArray [

"Creates and returns an instance of the receiver from the String aString.
 The String expresses Greenwich Mean Time in the format specified by anArray.
 The expression is terminated either by a space Character or by the end of the
 String.  Generates an error if the String does not conform to the format,
 or if anArray contains an incorrect formatting specification.

 See the class documentation of Time for a complete description of the
 String-formatting specification Array."

| s result |

s := ReadStreamPortable on: aString.
result := self fromStreamGmt: s usingFormat: anArray.
[ s atEnd ]
whileFalse:
  [ (s next == $  )
    ifFalse:
      [ self _errIncorrectFormat: aString ]
  ].
^ result

]

{ #category : 'Adjusting' }
Time class >> gmtOffsetSeconds [

"Returns a SmallInteger that gives the offset in seconds of the local time zone,
 its difference with respect to Greenwich Mean Time.

 A positive number corresponds to west of Greenwich, a negative number to east
 of Greenwich.  For example, the offset for the Pacific Standard Time zone is
 28800."

^ self dateTimeClass now _localOffset negated.

]

{ #category : 'Storing and Loading' }
Time class >> loadFrom: passiveObj [

"Creates and returns an active instance of the receiver from the passive form
 of the object, which expresses itself in Greenwich Mean Time."
| inst val |
val := passiveObj readObject .
val _isSmallInteger ifFalse:[ Error signal:'unexpected ', val class name ].
passiveObj version >= 620 ifTrue:[
  inst := self fromMicroseconds: val .
] ifFalse:[
 passiveObj version >= 510
  ifTrue: [ inst := self fromMilliseconds: val ]
  ifFalse: [ inst := self fromSecondsGmt: val ].
].
self == SmallTime ifTrue:[
  inst isSpecial ifFalse:[ Error signal:'a SmallTime should be special' ].
] ifFalse:[ | ms |
  ms := inst asMilliseconds .
  inst := self _basicNew _setMilliseconds: ms  .
  passiveObj hasRead: inst .
].
^inst.

]

{ #category : 'Instance Creation' }
Time class >> migrateNew [

"Override default migrateNew behavior with #_basicNew."

^ self _basicNew

]

{ #category : 'Measuring' }
Time class >> millisecondClockValue [

"Returns a SmallInteger representing the current time in milliseconds.
 The result is a SmallInteger equivalent to

    (System _timeGmtFloat * 1000) asInteger

 The result is computed locally in the session process, using the offset
 from the Gem's time that was cached in the session at login.

 Gs64 v2.2, changed to no longer rollover to zero after 524287999 "

<primitive: 651>
^ self _primitiveFailed: #millisecondClockValue

]

{ #category : 'Measuring' }
Time class >> millisecondsElapsedTime: aBlock [

"Returns the elapsed time in milliseconds aBlock takes to return its value.
 The argument aBlock must be a zero-argument block."

^ ((self secondsElapsedTime: aBlock) * 1000) asInteger

]

{ #category : 'Instance Creation' }
Time class >> new [

"Disallowed.  To create a new Time, use another instance creation method."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
Time class >> new: anInteger [

"Disallowed.  To create a new Time, use another instance creation method."

self shouldNotImplement: #new:

]

{ #category : 'Instance Creation' }
Time class >> now [
  "Beginning with Gs64 v3.0, returns an instance of the receiver representing
   the current time, taking into consideration the current smalltalk TimeZone ,
   per class DateAndTime .

   See Time(C)>>_now for previous implementation that gets time directly from the OS."

   | parts secondsSinceMidnight |
   parts := DateAndTime now asFloatParts.
     "parts is { year. dayOfYear. monthIndex. dayOfMonth. hour. minute. second } "
   secondsSinceMidnight := (parts at: 5) * 60 + (parts at: 6) * 60 + (parts at: 7).
   ^self fromMicroseconds: (secondsSinceMidnight * 1000000) roundedNoFpe .

]

{ #category : 'Measuring' }
Time class >> secondsElapsedTime: aBlock [

"Returns the elapsed time in seconds aBlock takes to return its value.
 The argument aBlock must be a zero-argument block.
 The result is a Float with microsecond resolution "

| startTime endTime systm |

systm := System .
startTime :=  systm _timeGmtFloat.
aBlock value.
endTime := systm _timeGmtFloat.

^ endTime - startTime

]

{ #category : 'New Indexing Comparison' }
Time >> _classSortOrdinal [

^ 40

]

{ #category : 'Private' }
Time >> _setMilliseconds: anInteger [

"Private. Initialize and make the receiver invariant. Returns the receiver."

milliseconds := anInteger \\ 86400000.
self immediateInvariant

]

{ #category : 'Private' }
Time >> _topazAsString [
 ^ self asStringMs
]

{ #category : 'Comparing' }
Time >> < aTime [

"Returns true if the receiver represents a time of day before that of the
 argument, and false if it doesn't.  Generates an error if the argument is not
 a Time."

^ self asMicroseconds < aTime asMicroseconds.

]

{ #category : 'Comparing' }
Time >> = aTime [
	"Returns true if the receiver represents the same time of day as that of the
	argument, and false if it doesn't."

	self == aTime
		ifTrue: [ ^ true ].
	(aTime isKindOf: Time)
		ifFalse: [ ^ false ].
	^ self asMicroseconds == aTime asMicroseconds
]

{ #category : 'Comparing' }
Time >> > aTime [

"Returns true if the receiver represents a time of day after that of the
 argument, and false if it doesn't.  Generates an error if the argument is not
 a Time."

^ self asMicroseconds > aTime asMicroseconds

]

{ #category : 'Arithmetic' }
Time >> addMicroseconds: anInteger [

"Returns a Time that describes a time of day anInteger microseconds
 later than that of the receiver."

^ self class fromMicroseconds: (self asMicroseconds + anInteger ).
]

{ #category : 'Arithmetic' }
Time >> addMilliseconds: anInteger [

"Returns a Time that describes a time of day anInteger milliseconds
 later than that of the receiver."

^ self class fromMicroseconds: (self asMicroseconds + (anInteger * 1000) ).
]

{ #category : 'Arithmetic' }
Time >> addSeconds: anInteger [

"Returns a Time that describes a time of day anInteger seconds
 later than that of the receiver."

^ self class fromMicroseconds: (self asMicroseconds + (anInteger * 1000000)).

]

{ #category : 'Arithmetic' }
Time >> addTime: aTime [

"Returns a Time that describes a time of day that is aTime
 later than that of the receiver. aTime represents the duration since
 midnight local time."

^ self class fromMicroseconds: (self asMicroseconds + aTime asMicroseconds).

]

{ #category : 'Converting' }
Time >> asCanonicalForm [
	"Answer self, or, if I am a Time with an equivalent
	SmallTime, answer that SmallTime."

	| cls res |
	(cls := self class) ~~ SmallTime
		ifTrue: [ 
			cls == Time
				ifTrue: [ 
					res := SmallTime fromMicroseconds: self asMicroseconds.
					res class == SmallTime
						ifTrue: [ ^ res ] ] ].
	^ self
]

{ #category : 'Converting' }
Time >> asMicroseconds [
  ^ milliseconds * 1000 
]

{ #category : 'Converting' }
Time >> asMilliseconds [

"Returns a SmallInteger that represents the receiver in units of milliseconds since
 midnight, local time."

^ milliseconds.
]

{ #category : 'Converting' }
Time >> asMillisecondsGmt [

"Returns an SmallInteger that represents the receiver in units of milliseconds since
 midnight, Greenwich Mean Time."

^ (self asMilliseconds + ((self class gmtOffsetSeconds) * 1000)) \\ 86400000 .

]

{ #category : 'Converting' }
Time >> asSeconds [

"Returns an SmallInteger that represents the receiver in units of seconds since
 midnight, local time."

^ self asMilliseconds // 1000.

]

{ #category : 'Converting' }
Time >> asSecondsGmt [

"Returns an SmallInteger that represents the receiver in units of seconds since
 midnight, Greenwich Mean Time."

^ ((self asMilliseconds // 1000) + self class gmtOffsetSeconds) \\ 86400

]

{ #category : 'Formatting' }
Time >> asString [

"Returns a String that expresses the receiver in local time
 in the default format (HH:MM:SS)."

| result ms |

"Check for seconds being nil, for graceful printing during error handling."
ms := self asMilliseconds .
ms ifNil:[ ^ '(nil):(nil):(nil)' ].
result := (ms // 3600000) _digitsAsString.
result addAll: $:; addAll: ((ms \\ 3600000) // 60000) _digitsAsString.
result addAll: $:; addAll: (ms // 1000 \\ 60) truncated _digitsAsString.
^ result.

]

{ #category : 'Formatting' }
Time >> asStringGmt [

"Returns a String that expresses the receiver in Greenwich Mean Time
 in the default format (HH:MM:SS)."

^ (self addSeconds: self class gmtOffsetSeconds) asString

]

{ #category : 'Formatting' }
Time >> asStringGmtUsingFormat: anArray [

"Returns a String that expresses the receiver in Greenwich Mean Time
 in the format defined by anArray.  Generates an error if anArray
 contains an incorrect formatting specification.

 See the class documentation of Time for a complete description of the
 String-formatting specification Array."

^ (self addSeconds: self class gmtOffsetSeconds) asStringUsingFormat: anArray.

]

{ #category : 'Formatting' }
Time >> asStringMs [

"Returns a String that expresses the receiver in local time
 in the format  HH:MM:SS.sss   where  sss are milliseconds."

| msStr result ms |
"Check for seconds being nil, for graceful printing during error handling."
ms := self asMilliseconds .
ms ifNil:[ ^ '(nil):(nil):(nil)' ].
result := (ms // 3600000) _digitsAsString.
result add: $: ; addAll: ((ms \\ 3600000) // 60000) _digitsAsString.
result add: $: ; addAll: (ms // 1000 \\ 60) _digitsAsString.
msStr := (ms \\ 1000) _digitsAsString.
result add: $. .
msStr size < 3 ifTrue:[ result add: $0 ].
result addAll: msStr .
^ result.
]

{ #category : 'Formatting' }
Time >> asStringUs [

"Returns a String that expresses the receiver in local time
 in the format  HH:MM:SS.ssssss  where ssssss are microseconds."

| usStr result ms us |
"Check for seconds being nil, for graceful printing during error handling."
us := self asMicroseconds .
us ifNil:[ ^ '(nil):(nil):(nil)' ].
ms := us // 1000 .
result := (ms // 3600000) _digitsAsString.
result add: $: ; addAll: ((ms \\ 3600000) // 60000) _digitsAsString.
result add: $: ; addAll: (ms // 1000 \\ 60) _digitsAsString.
result add: $. .
usStr := (us \\ 1000000) asString .
6 - usStr size timesRepeat:[ result add: $0 ].
result addAll: usStr .
^ result.
]

{ #category : 'Formatting' }
Time >> asStringUsingFormat: anArray [

"Returns a String that expresses the receiver in Greenwich Mean Time
 in the format defined by anArray.  Generates an error if anArray
 contains an incorrect formatting specification.

 See the class documentation of Time for a complete description of the
 String-formatting specification Array."

| timeSeparator  hourInt hour min sec aString ms |
ms := self asMilliseconds .
timeSeparator := anArray at: 1.
hourInt := ms // 3600000.
hour := hourInt  _digitsAsString.
min := (ms \\ 3600000 // 60000) _digitsAsString.
sec := (ms // 1000 \\ 60) _digitsAsString.

aString := String new.
(anArray at: 3)
  ifTrue: [ "12-hour format"
    (hourInt > 12)
      ifTrue: [
        aString addAll: (hourInt - 12) _digitsAsString;
        addAll: timeSeparator;
        addAll: min.

        (anArray at: 2)
          ifTrue: [ aString addAll: timeSeparator; addAll: sec ].
        ]
      ifFalse: [
        aString addAll: (hourInt == 0 ifTrue: ['12'] ifFalse: [hour]);
        addAll: timeSeparator;
        addAll: min.

        (anArray at: 2)
          ifTrue: [ aString addAll: timeSeparator; addAll: sec].
        ].

    aString addAll: (hourInt >= 12 ifTrue: [' PM'] ifFalse: [' AM']).
    ]
  ifFalse: [
    aString addAll: hour;
            addAll: timeSeparator;
            addAll: min.

    (anArray at: 2)
      ifTrue: [ aString addAll: timeSeparator; addAll: sec].
    ].

^ aString

]

{ #category : 'Accessing' }
Time >> at: anIndex put: aValue [

"Disallowed.  You may not change the value of a Time."

self shouldNotImplement: #at:put:

]

{ #category : 'Comparing' }
Time >> hash [

"Returns an Integer hash code for the receiver."

^ self asSeconds hash.

]

{ #category : 'Accessing' }
Time >> hours [

"Returns a SmallInteger (between zero and 23 inclusive) that gives the number of
 hours represented by the receiver since midnight, local time."

^ self asMilliseconds // 3600000

]

{ #category : 'Accessing' }
Time >> hoursGmt [

"Returns a SmallInteger (between zero and 23 inclusive) that gives the number of
 hours represented by the receiver since midnight, Greenwich Mean Time."

^ (self asMilliseconds // 1000 + self class gmtOffsetSeconds \\ 86400) // 3600

]

{ #category : 'Accessing' }
Time >> minutes [

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 minutes represented by the receiver since the previous hour, local time."

^ (self asMilliseconds \\ 3600000) // 60000

]

{ #category : 'Accessing' }
Time >> minutesGmt [

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 minutes represented by the receiver since the previous hour, Greenwich Mean
 Time."

^ ((self asMilliseconds // 1000 + self class gmtOffsetSeconds) \\ 3600) // 60

]

{ #category : 'Formatting' }
Time >> printOn: aStream [

"Puts a displayable representation of the receiver, expressed in
 local time, on aStream."

aStream nextPutAll: self asString .

]

{ #category : 'Formatting' }
Time >> printJsonOn: aStream [
  (self asStringUsingFormat: #( $: true false )) printJsonOn: aStream
]

{ #category : 'Accessing' }
Time >> seconds [

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 seconds represented by the receiver since the previous hour, local time."

^ self asMilliseconds // 1000 \\ 60

]

{ #category : 'Accessing' }
Time >> secondsGmt [

"Returns a SmallInteger (between zero and 59 inclusive) that gives the number of
 seconds represented by the receiver since the previous hour,  Greenwich Mean
 Time."

^ (self asMilliseconds // 1000 + self class gmtOffsetSeconds) \\ 60

]

{ #category : 'Arithmetic' }
Time >> subtractMicroseconds: anInteger [

"Returns a Time that describes a time of day anInteger milliseconds
 earlier than that of the receiver."

^ self addMicroseconds: (0 - anInteger).

]

{ #category : 'Arithmetic' }
Time >> subtractMilliseconds: anInteger [

"Returns a Time that describes a time of day anInteger milliseconds
 earlier than that of the receiver."

^ self addMilliseconds: (0 - anInteger).

]

{ #category : 'Arithmetic' }
Time >> subtractSeconds: anInteger [

"Returns a Time that describes a time of day anInteger seconds
 earlier than that of the receiver."

^ self addSeconds: (0 - anInteger).

]

{ #category : 'Arithmetic' }
Time >> subtractTime: aTime [

"Returns a Time that describes a time of day that is aTime
 earlier than that of the receiver. aTime represents the duration
 since midnight local time."

^ self class fromMicroseconds: (self asMicroseconds - aTime asMicroseconds).

]

{ #category : 'Deprecated' }
Time >> timeAsSeconds [

"Returns a SmallInteger (between zero and 86399 inclusive) that gives
 the number of seconds represented by the receiver since midnight, local time."

self deprecated: 'Time>>timeAsSeconds deprecated before v3.0, use #asSeconds'.
^ self asMilliseconds // 1000.

]

{ #category : 'Deprecated' }
Time >> timeAsSecondsGmt [

"Returns a SmallInteger (between zero and 86399 inclusive) that gives
 the number of seconds represented by the receiver since midnight, Greenwich
 Mean Time."

self deprecated: 'Time>>timeAsSecondsGmt deprecated before v3.0, use #asSecondsGmt'.
^ self asSecondsGmt.

]

{ #category : 'Storing and Loading' }
Time >> writeTo: passiveObj [

"Writes the passive form of the receiver into passiveObj, expressed in
 Greenwich Mean Time."

passiveObj writeClass: self class.
self asMicroseconds writeTo: passiveObj.
passiveObj space

]
