Extension { #name : 'Lag1MwcRandom' }

{ #category : 'initialization' }
Lag1MwcRandom >> initialize [

"The exact choice of these constants is crucial to obtaining the desired result.
Refer to literature on the multiply-with-carry algorithm for the relationship between the constants.
Note that the implementation of this class also implies the constant b := 2**32."

	multiplier := 698769069.
	lag := 1.

]

{ #category : 'public' }
Lag1MwcRandom >> integer [
	"Answer a random nonnegative 32-bit integer."

	| newCarryAndSeed newSeed |
	index == nil ifTrue: [self setSeedFromHost].
	newCarryAndSeed := (seeds at: index) * multiplier + carry.
	newSeed := newCarryAndSeed bitAnd: 16rFFFFFFFF.
	carry := newCarryAndSeed bitShift: -32.
	index == lag ifTrue: [index := 0].
	index := index + 1.
	seeds at: index put: newSeed.
	^newSeed


]

{ #category : 'public' }
Lag1MwcRandom >> seed: newSeed [

	"Sets the seed from the given SmallInteger."

	| upperBits lowerBits |
	newSeed _isSmallInteger ifFalse: [ArgumentError signal: 'The seed must be a SmallInteger'].

	"A SmallInteger has 61 significant bits, including the sign bit. We will use the lower 32 bits
  for the initial seed and the upper 29, including the sign, to generate the initial carry.
  The legal range for carry is [0, multiplier), which in this class is more than 2**29 but
  less than 2**30. Carry and seed cannot both be zero, so we ensure that the initial carry
  is non-zero by inverting it within its range."

	upperBits := (newSeed bitAnd: 16r1FFFFFFFFFFFFFFF) bitShift: -32.
	lowerBits := newSeed bitAnd: 16rFFFFFFFF.
	seeds := {lowerBits} .
	carry := multiplier - 1 - upperBits.
	index := 1

]

{ #category : 'private' }
Lag1MwcRandom >> setSeedFromHost [

	| random |
	random := HostRandom new.
	seeds := {random integer}.
	carry := random integer \\ multiplier.
	index := 1

]

{ #category : 'private' }
Lag1MwcRandom >> validateFullState: fullState [

	| newIndex newCarry seedArray haveNonZero haveNonMaximal |
	fullState size = 3 ifFalse: [ArgumentError signal: 'expected an Array of size 3'].
	newIndex := fullState at: 1.
	newCarry := fullState at: 2.
	seedArray := fullState at: 3.
	haveNonZero := newCarry ~= 0.
	haveNonMaximal := newCarry ~= (multiplier - 1).
	newIndex _isSmallInteger ifFalse: [ArgumentError signal: 'Index must be a SmallInteger'].
	(newIndex between: 1 and: lag) ifFalse:  [OutOfRange new
												name: 'index' min: 1 max: lag actual: newIndex;
															signal: 'out of range'].
	newCarry _isSmallInteger ifFalse: [ArgumentError signal: 'Carry must be a SmallInteger'].
	(newCarry between: 0 and: multiplier - 1) ifFalse: [OutOfRange new
															name: 'carry' min: 0 max: multiplier - 1 actual: newCarry;
															signal: 'out of range'].
	seedArray size = lag ifFalse: [ArgumentError signal: 'Expected seed array of size ' , lag printString].
	seedArray do: [:seed |
					seed _isSmallInteger ifFalse: [ArgumentError signal: 'Each seed must be a SmallInteger'].
					seed = 0 ifFalse: [haveNonZero := true].
					seed = 16rFFFFFFFF ifFalse: [haveNonMaximal := true].
					(seed between: 0 and: 16rFFFFFFFF) ifFalse: [OutOfRange new
																		name: 'seed' min: 0 max: 16rFFFFFFFF actual: seed;
																		signal: 'out of range']].
	haveNonZero ifFalse: [ArgumentError signal: 'Carry and seeds should not all be zero.'].
	haveNonMaximal ifFalse: [ArgumentError signal: 'Carry and seeds should not all be their maximal legal value.']

]
