Extension { #name : 'ReadStreamLegacy' }

{ #category : 'Instance Creation' }
ReadStreamLegacy class >> new [

"Disallowed.  To create a new ReadStream, use the class method on: instead."

self shouldNotImplement: #new

]

{ #category : 'Passivation - Support' }
ReadStreamLegacy >> _fastNext [
  "Returns the next item on the input stream without end of stream checks.
 This may result in a bad access error instead of an end of stream error."

  | res |
  res := itsCollection at: position.
  position := position + 1.
  ^ res

]

{ #category : 'Passivation - Support' }
ReadStreamLegacy >> _fastPosition [
  "Returns the receiver's current position reference for accessing the sequence
 of objects.  The position is actually the next element of the collection to be
 read or written; the position is incremented by each read or write.  In
 general, to start reading or writing at the nth element of a collection, the
 position must be set to n."

  "See bug 42596: Compatible with Legacy Stream positioning"

  ^ position

]

{ #category : 'Passivation - Support' }
ReadStreamLegacy >> _fastPosition: pos [
  "Sets the position of the receiver without checking the validity of the
 argument 'pos'.  This may result in bad index errors instead of end
 of stream errors in subsequent access of the stream."

  "See bug 42596: Compatible with Legacy Stream positioning"

  position := pos

]

{ #category : 'Accessing' }
ReadStreamLegacy >> next [

"Returns the next object that the receiver can access for reading.  Generates
 an error if an attempt is made to read beyond the end of the stream."
| pos coll |
pos := position .
pos > (coll := itsCollection) size ifTrue:[ ^ EndOfStream signal ]. "inline atEnd check"
position := pos + 1 .
^ itsCollection at: pos

]

{ #category : 'Accessing' }
ReadStreamLegacy >> nextElements: count into: anObject [

"Stores the next count elements that the receiver can access for reading
 into anObject.  The receiver's collection and anObject must be compatible
 SequenceableCollections.  Returns the count of elements read.

 Generates an error if an attempt is made to read beyond the end of the stream."

"Used by PassiveObject."

count <= 0 ifTrue:[ ^ 0 ].
(self atEnd)
   ifTrue: [ ^ EndOfStream signal ].

anObject replaceFrom: 1 to: 1 + count - 1 with: itsCollection startingAt: position .
position := position + count.
^ count

]

{ #category : 'Private' }
ReadStreamLegacy >> nextBytes: count addTo: anObject [
"Used by PassiveObject, avoid use of replaceFrom primitives to avoid
  promotion of a String to Unicode16 for codepoints above 127 
  when (Globals at: #StringConfiguration) == Unicode16."
count <= 0 ifTrue:[ ^ 0 ].
(self atEnd)
   ifTrue: [ ^ EndOfStream signal ].
anObject addAllBytes: (itsCollection copyFrom: position to: position + count - 1).
position := position + count.
^ count


]

{ #category : 'Testing' }
ReadStreamLegacy >> nextMatchFor: anObject [

	"The first object is removed from the receiver's future sequence value and appended to the end of
	the receiver's past sequence values. The value that would result from sending #= to the object with
	anObject as the argument is returned.
	The results are undefined if there are no future sequence values in the receiver.

	ANSI 5.9.2.6"

	^self next = anObject.

]

{ #category : 'Accessing' }
ReadStreamLegacy >> nextOrNil [
 "Returns the next object that the receiver can access for reading.
  Returns nil if an attempt is made to read beyond the end of the stream.
  For use with Streams whose collections do not contain nil .
"
 | pos coll res |
 pos := position .
 pos > (coll := itsCollection) size ifTrue:[ ^ nil ]. "inline atEnd check"
 res := coll at: pos .
 position := pos + 1 .
 ^ res

]

{ #category : 'Adding' }
ReadStreamLegacy >> nextPut: anObject [

"Disallowed.  You cannot write to a ReadStream."

self shouldNotImplement: #nextPut:

]

{ #category : 'Adding' }
ReadStreamLegacy >> upTo: anObject [
| start end |

start := position.
end := itsCollection indexOf: anObject startingAt: start .
end == 0 ifTrue:[ ^self upToEnd ].

"skip to the end and return the data passed over"
position := end + 1.
^itsCollection copyFrom: start to: (end - 1)

]
