Extension { #name : 'ReadStreamPortable' }

{ #category : 'Instance Creation' }
ReadStreamPortable class >> new [

"Disallowed.  To create a new ReadStream, use the class method on: instead."

self shouldNotImplement: #new

]

{ #category : 'Instance Creation' }
ReadStreamPortable class >> on: aCollection from: firstIndex to: lastIndex [
"Answer with a new instance streaming over a copy of aCollection from
firstIndex to lastIndex."

^self basicNew
	on: aCollection
	from: firstIndex
	to: lastIndex

]

{ #category : 'Passivation - Support' }
ReadStreamPortable >> _fastNext [
  "Returns the next item on the input stream without end of stream checks.
 This may result in a bad access error instead of an end of stream error."

  ^ collection at: (position := position + 1)

]

{ #category : 'Passivation - Support' }
ReadStreamPortable >> _fastPosition [
  "Returns the receiver's current position reference for accessing the sequence
 of objects.  The position is actually the next element of the collection to be
 read or written; the position is incremented by each read or write.  In
 general, to start reading or writing at the nth element of a collection, the
 position must be set to n."

  "See bug 42596: Compatible with Legacy Stream positioning"

  ^ position + 1

]

{ #category : 'Passivation - Support' }
ReadStreamPortable >> _fastPosition: pos [
  "Sets the position of the receiver without checking the validity of the
 argument 'pos'.  This may result in bad index errors instead of end
 of stream errors in subsequent access of the stream."

  "See bug 42596: Compatible with Legacy Stream positioning"

  position := pos - 1

]

{ #category : 'Accessing' }
ReadStreamPortable >> next [

"Returns the next object that the receiver can access for reading.
 Returns nil if an attempt is made to read beyond the end of the stream."
| pos |
(pos := position) >= readLimit
       ifTrue: [^nil]
       ifFalse: [^collection at: (position := pos + 1)]

]

{ #category : 'Accessing' }
ReadStreamPortable >> next: amount [
"Answer the next amount elements of my collection.  overriden for efficiency"

| ans endPosition pos |
amount < 0 ifTrue: [ ^self error: 'amount may not be less than 0' ].

endPosition := (pos := position) + amount  min:  readLimit.
ans := collection copyFrom: pos+1 to: endPosition.
position := endPosition.
^ans

]

{ #category : 'Accessing' }
ReadStreamPortable >> next: n into: aCollection startingAt: startIndex [
"Read n objects into the given collection.
Return aCollection or a partial copy if less than
n elements have been read."
| max pos |
max := (readLimit - (pos := position)) min: n.
aCollection
	replaceFrom: startIndex
	to: startIndex+max-1
	with: collection
	startingAt: pos +1.
position := pos + max.
max = n
	ifTrue:[^aCollection]
	ifFalse:[^aCollection copyFrom: 1 to: startIndex+max-1]

]

{ #category : 'Passivation - Support' }
ReadStreamPortable >> nextElements: count into: aCollection [
  "Stores the next count elements that the receiver can access for reading
 into aCollection.  The receiver's collection and aCollection must be compatible
 SequenceableCollections.  Returns the count of elements read.

 Generates an error if an attempt is made to read beyond the end of the stream."

  "Used by PassiveObject."

  ^ self next: count into: aCollection startingAt: 1

]

{ #category : 'Passivation - Support' }
ReadStreamPortable >> nextBytes: count addTo: anObject [
  | pos |
  count <= 0 ifTrue:[ ^ anObject ].
  (self atEnd)
   ifTrue: [ ^ EndOfStream signal ].
  anObject addAllBytes: (collection copyFrom: (pos := position) + 1 to: pos + count ).
  position := pos + count .
  ^ anObject .

]


{ #category : 'Testing' }
ReadStreamPortable >> nextMatchFor: anObject [

	"The first object is removed from the receiver's future sequence value and appended to the end of
	the receiver's past sequence values. The value that would result from sending #= to the object with
	anObject as the argument is returned.
	The results are undefined if there are no future sequence values in the receiver.

	ANSI 5.9.2.6"

	^self next = anObject.

]

{ #category : 'Accessing' }
ReadStreamPortable >> nextOrNil [

"Returns the next object that the receiver can access for reading.
 Returns nil if an attempt is made to read beyond the end of the stream."
| pos |
(pos := position) >= readLimit
	ifTrue: [^nil]
	ifFalse: [^collection at: (position := pos + 1)]

]

{ #category : 'Adding' }
ReadStreamPortable >> nextPut: anObject [

"Disallowed.  You cannot write to a ReadStream."

self shouldNotImplement: #nextPut:

]

{ #category : 'Private' }
ReadStreamPortable >> on: aCollection from: firstIndex to: lastIndex [

| len |
collection := aCollection .
readLimit :=  lastIndex > (len := aCollection size)
	ifTrue: [len]
	ifFalse: [lastIndex].
position := firstIndex <= 1
	ifTrue: [0]
	ifFalse: [firstIndex - 1]

]

{ #category : 'Accessing' }
ReadStreamPortable >> peek [
  "Answer what would be returned if the message next were sent to the
  receiver. If the receiver is at the end, answer nil. 
  Reimplemented as an optimization"

  position >= readLimit"self atEnd" ifTrue: [^nil].
  "nextObject :=self next.  position := position - 1. ^nextObject"
  ^ collection at: position + 1 
]

{ #category : 'Accessing' }
ReadStreamPortable >> peekN: anInteger [
  "Peek returning the next N characters of collection."
  | end start |
  position >= readLimit ifTrue:[ ^ collection class new ].
  start := position + 1 .
  end := start + anInteger - 1.
  end >= readLimit ifTrue:[ end := readLimit - 1 ].
  ^ collection copyFrom: start to: end

]

{ #category : 'Accessing' }
ReadStreamPortable >> size [
"Compatibility with other streams"
^readLimit

]

{ #category : 'Accessing' }
ReadStreamPortable >> skip: anInteger [
"Set the receiver's position to be the current position+anInteger. Do not
 throw error if skipAmount would exceed collection bounds - ANSI compliance. "

self position: ((position + anInteger max: 0) min: readLimit)

]

{ #category : 'Accessing' }
ReadStreamPortable >> upTo: anObject [
"fast version using indexOf:"
| start end |

start := position+1.
end := collection indexOf: anObject startingAt: start .

"not present--return rest of the collection"
(end == 0 or: [end > readLimit]) ifTrue: [ ^self upToEnd ].

"skip to the end and return the data passed over"
position := end.
^collection copyFrom: start to: (end-1)

]

{ #category : 'Accessing' }
ReadStreamPortable >> upToEnd [
| start pos |
start := 1 + position .
position := (pos := readLimit) .
^collection copyFrom: start to: pos

]
