"
GemStoneParameters is used by the class GciInterface.
Instances are used as arguments to the login methods in
GciInterface.

Constraints:
	gemStoneName: String  (the stone name)
	username: String
	password: String
	hostUsername: String
	hostPassword: String
	gemService: String
"
Class {
	#name : 'GemStoneParameters',
	#superclass : 'Object',
	#instVars : [
		'gemStoneName',
		'username',
		'password',
		'hostUsername',
		'hostPassword',
		'gemService'
	],
	#gs_reservedoop : '117505',
	#category : nil
}

{ #category : 'Instance creation' }
GemStoneParameters class >> new [

"Create a new initialized instance."

^ self basicNew initialize.

]

{ #category : 'Instance creation' }
GemStoneParameters class >> newDefault [
  ^ self new 
     gemService: GsNetworkResourceString defaultGemNRSFromCurrent asString ;
     gemStoneName: GsNetworkResourceString defaultStoneNRSFromCurrent asString ;
     username: System myUserProfile userId ;
     password: 'swordfish';
     hostUsername: System effectiveUserIdName ; 
     yourself
]

{ #category : 'Updating' }
GemStoneParameters >> _encrypt: aString [

"Return a new, encrypted string from the given string. ."

^ self _oneArgPrim: 24 with: aString

]

{ #category : 'Error Handling' }
GemStoneParameters >> _errorBadPassword [

"Raise an error because could encrypt and store a password."

self _error: #rtErrPasswordNotVariant args: #()

]

{ #category : 'Private' }
GemStoneParameters >> _oneArgPrim: opcode with: arg [

"Primitive dispatch. Legal values:
    24 GemStoneParameters | _encrypt:
"

<primitive: 198>

^self _primitiveFailed: #_oneArgPrim: args: { opcode . arg }

]


{ #category : 'Accessing' }
GemStoneParameters >> defaultLoginFlags [
^ 16r11  "GCI_LOGIN_PW_ENCRYPTED | GCI_LOGIN_QUIET"

]

{ #category : 'Accessing' }
GemStoneParameters >> gemService [

   "Return the value of the instance variable 'gemService'."
   ^gemService

]

{ #category : 'Updating' }
GemStoneParameters >> gemService: newValue [

   "Modify the value of the instance variable 'gemService'."
   newValue _isOneByteString ifFalse:[ newValue _validateClass: String ].
   gemService := newValue

]

{ #category : 'Accessing' }
GemStoneParameters >> gemStoneName [
  "Return the value of the instance variable 'gemStoneName'."
  ^ gemStoneName
]

{ #category : 'Updating' }
GemStoneParameters >> gemStoneName: aName [
aName ifNotNil:[
  aName _isOneByteString ifFalse:[ aName _validateClass: String ].
].
gemStoneName := aName.

]

{ #category : 'Accessing' }
GemStoneParameters >> stoneName [
  ^ gemStoneName
]

{ #category : 'Updating' }
GemStoneParameters >> stoneName: aName [
  ^ self gemStoneName: aName
]

{ #category : 'Accessing' }
GemStoneParameters >> hostPassword [

   "Return the value of the instance variable 'hostPassword'."
   ^hostPassword

]

{ #category : 'Updating' }
GemStoneParameters >> hostPassword: pword [

"Set the host password for the receiver.  The password is encrypted.
The given password's size is set to zero."

pword _isOneByteString ifFalse:[ pword _validateClass: String ].
hostPassword := self _encrypt: pword.
hostPassword immediateInvariant.
pword isInvariant ifFalse:[ pword size: 0 ].
^ pword

]

{ #category : 'Accessing' }
GemStoneParameters >> hostUsername [

   "Return the value of the instance variable 'hostUsername'."
   ^hostUsername

]

{ #category : 'Updating' }
GemStoneParameters >> hostUsername: newValue [

   "Modify the value of the instance variable 'hostUsername'."
   newValue _isOneByteString ifFalse:[ newValue _validateClass: String ].
   hostUsername := newValue

]

{ #category : 'Initialization' }
GemStoneParameters >> initialize [

"Initialize the instance variables of the receiver."
  ^ self setLoginWithPassword

]

{ #category : 'Accessing' }
GemStoneParameters >> loginFlags [

   "Return the value of the instance variable 'loginFlags'."

^ (self dynamicInstVarAt: #loginFlags) ifNil:[ self defaultLoginFlags ] .

]

{ #category : 'Private' }
GemStoneParameters >> loginFlags: anInt [
"Do not set loginFlags directly unless you know what you are doing."

self dynamicInstVarAt: #loginFlags put: anInt .
^ self

]

{ #category : 'Accessing' }
GemStoneParameters >> password [

   "Return the value of the instance variable 'password'."
   ^password

]

{ #category : 'Updating' }
GemStoneParameters >> password: pword [

"Set the password for the receiver.  The password is encrypted.
The given password's size is set to zero."

pword _isOneByteString ifFalse:[ pword _validateClass: String ].

pword size == 0
  ifTrue:[ self setLoginPasswordless .
           password := pword copy ]
  ifFalse:[ self setLoginWithPassword .
            password := self _encrypt: pword ].
password immediateInvariant.
pword isInvariant ifFalse:[ pword size: 0 ].
^ pword

]

{ #category : 'Accessing' }
GemStoneParameters >> passwordIsEncryptedAsIntegerBoolean [

"Answer a C boolean (1 or 0) indicating whether the password is encrypted.
 The password will always be encrypted unless the login is passwordless."

^ self loginFlags bitAnd: 1

]

{ #category : 'Accessing' }
GemStoneParameters >> passwordlessLoginFlags [
^ 16r90  "GCI_PASSWORDLESS_LOGIN | GCI_LOGIN_QUIET"

]

{ #category : 'Formatting' }
GemStoneParameters >> printString [
  | res f |
  res := 'stoneNrs:', gemStoneName asString, ' user:', username asString .
  hostUsername ifNotNil:[ res add: ' hostuser:', hostUsername asString ].
  res add: ' gemNrs:', gemService asString .
  (f := self loginFlags) == self defaultLoginFlags ifFalse:[
    res add: ' flags:16r', f asHexString
  ].
  ^ res

]

{ #category : 'Accessing' }
GemStoneParameters >> remoteRepository [

"Return nil to indicate no remote repository."

^ nil

]

{ #category : 'Updating' }
GemStoneParameters >> setLoginPasswordless [

^ self loginFlags: self passwordlessLoginFlags

]

{ #category : 'Updating' }
GemStoneParameters >> setSoloLogin [
 
^  self loginFlags: (self loginFlags bitOr: self soloLoginFlag)

]

{ #category : 'Updating' }
GemStoneParameters >> clearQuietLogin [

^  self loginFlags: (self loginFlags bitAnd: (16r10"GCI_LOGIN_QUIET" bitInvert))

]

{ #category : 'Updating' }
GemStoneParameters >> setLoginWithPassword [

^  self loginFlags: self defaultLoginFlags

]

{ #category : 'Accessing' }
GemStoneParameters >> soloLoginFlag [
^ 16r100 "GCI_LOGIN_SOLO"

]

{ #category : 'Accessing' }
GemStoneParameters >> username [

   "Return the value of the instance variable 'username'."
   ^username

]

{ #category : 'Updating' }
GemStoneParameters >> username: newValue [

   "Modify the value of the instance variable 'username'."
   newValue _isOneByteString ifFalse:[ newValue _validateClass: String ].
   username := newValue

]

{ #category : 'Accessing' }
GemStoneParameters >> onetimePasswordFlag [
^ 16r400 "GCI_ONETIME_PASSWORD"

]

{ #category : 'Accessing' }
GemStoneParameters >> onetimePasswordLoginFlags [
^ 16r410  "GCI_ONETIME_PASSWORD | GCI_LOGIN_QUIET"

]

{ #category : 'Constants' }
GemStoneParameters class >> onetimePasswordSize [
"OneTimeLoginToken::TOKEN_BASE64_REAL_SIZE"
^ 64

]

{ #category : 'Updating' }
GemStoneParameters >> setLoginWithOnetimePassword [

^  self loginFlags: self onetimePasswordLoginFlags

]

{ #category : 'Updating' }
GemStoneParameters >> onetimePassword: pword [

"Set the onetime password for the receiver. "
pword _isOneByteString ifFalse:[ pword _validateClass: String ].
pword size == self class onetimePasswordSize
	ifFalse:[ pword _error: #rtErrBadSize args: { self class onetimePasswordSize . pword size } ] .
self setLoginWithOnetimePassword .
password := pword copy.
password immediateInvariant.
pword isInvariant ifFalse:[ pword size: 0 ].
^ pword
]
