Extension { #name : 'BtreePlusLeafKeyNode' }

{ #category : 'Constants' }
BtreePlusLeafKeyNode class >> entrySize [
  "Returns the size of an entry with no encryption."

  "in-lined on instance-side for performance"

  ^ 1

]

{ #category : 'Constants' }
BtreePlusLeafKeyNode class >> maxNumberOfElements [
  "Returns the number of entries that are allowed in an interior node."

  ^ 2028

]

{ #category : 'Updating' }
BtreePlusLeafKeyNode >> _insertKey: aKey atIndex: insertionIndex [

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

| endIndex entrySize |

" see if any existing entries have to be moved "
entrySize := self entrySize .
endIndex := (numElements * entrySize) .

self _insertAt: insertionIndex
     value: aKey
     numToMoveDown: (endIndex - insertionIndex + 1) .

"no prefix encoding"

numElements := numElements + 1

]

{ #category : 'Updating' }
BtreePlusLeafKeyNode >> _insertKey: aKey value: ignoredValue atIndex: insertionIndex [
  "Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey atIndex: insertionIndex


]

{ #category : 'Updating' }
BtreePlusLeafKeyNode >> _insertKey: aKey value: ignoredValue root: ignoredRoot atIndex: insertionIndex [

"Insert the key/value pair in the receiver.  The sender of this message must
verify that the entry will fit in the receiver and provide the insertion
index."

self _insertKey: aKey atIndex: insertionIndex

]

{ #category : 'Updating' }
BtreePlusLeafKeyNode >> _insertKey: aKey value: ignoredValue root: ignoredRoot atIndex: insertionIndex selectiveAbortSet: ignored [

"Compat with BtreePlustInteriorNode api ... inserts do not require selective abort logic"

self _insertKey: aKey value: ignoredValue root: ignoredRoot atIndex: insertionIndex

]

{ #category : 'Updating' }
BtreePlusLeafKeyNode >> at: aKey put: aValue root: ignoredRoot [
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue  ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey ].
      self _insertKey: aKey atIndex: index ]

]

{ #category : 'Updating' }
BtreePlusLeafKeyNode >> at: aKey put: aValue root: ignoredRoot selectiveAbortSet: selectiveAbortSetOrNil [
  "Adds the key/value pair to the node.  If the node is full, performs a 'split'
 on the parent.  Returns the new sibling if a split is performed, otherwise
 returns the receiver."

  | index |
  " determine if a split is needed "
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  numElements == self maxNumberOfElements
    ifTrue: [ " performs the split and returns the new sibling "
      ^ self _splitUsingKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil ]
    ifFalse: [ " entry will fit in leaf node "
      numElements == 0
        ifTrue: [ index := 1 ]
        ifFalse: [ index := self binarySearchCoveringKey: aKey ].
      self _insertKey: aKey atIndex: index ]

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyNode >> compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: ignoredRootObject [
  "Performs a = comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  ^ self compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: ignoredRootObject [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the comparison."

  ^self compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  "value/key/root use same index"

  ^ aValue asOop > (self at: keyIndex) asOop

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex root: ignored [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index. "

  ^ self compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  "value/key/root use same index"

  ^ aValue asOop <= (self at: keyIndex) asOop

]

{ #category : 'Comparison Operators' }
BtreePlusLeafKeyNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex root: ignored [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  ^ self compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex

]

{ #category : 'Constants' }
BtreePlusLeafKeyNode >> entrySize [
  "Returns the size of an entry."

  "in-line of `self class entrySize` for performance"

  ^ 1

]

{ #category : 'Accessing' }
BtreePlusLeafKeyNode >> keyIndexOffset [
  "Answer the offset from entry index for the key. Typical order is value/key/root.
   The value, key and root objects are the same for the receiver."

  ^ 0

]

{ #category : 'Constants' }
BtreePlusLeafKeyNode >> parentNodeClass [
  "Returns the class of node to be created as the parent when a split occurs."

  ^ BtreePlusInteriorKeyValueNode

]

{ #category : 'Accessing' }
BtreePlusLeafKeyNode >> rootIndexOffset [
  "Answer the offset from entry index for the root object. Typical order is value/key/root.
   The value, key and root objects are the same for the receiver."

  ^ 0

]
