Extension { #name : 'BtreePlusLeafNode' }

{ #category : 'Query Support' }
BtreePlusLeafNode >> _addRootObjects: aQuerySpec start: startIndex end: endIndex into: collection [
  ^ (aQuerySpec canUseOptimizedComparison and: [ self rootNode indexObject optimizingComparison ])
    ifTrue: [
      self
        _primAddRootObjects: aQuerySpec
        start: startIndex + self rootIndexOffset
        end: endIndex + self rootIndexOffset
        into: collection
        entrySize: self entrySize ]
    ifFalse: [
      super
        _addRootObjects: aQuerySpec
        start: startIndex
        end: endIndex
        into: collection ]

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> _compare: aComparison key: aKey equalToEntryAt: index [

"Performs a = comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey equalTo: (self _at: index)

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> _compare: aComparison key: aKey lessThanOrEqualToEntryAt: index [

"Perform a <= comparison between aKey and the entry whose key is at the given
index.  The default implementation uses no encryption."

^ aComparison compareKey: aKey lessThanOrEqualTo: (self at: index)

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _createSibling [
  "Creates a new sibling node and move the second half of the receiver's entries into the
sibling.  Returns the new sibling."

  | newSibling nLeaf |
  newSibling := super _createSibling.
  nLeaf := self nextLeaf.
  newSibling
    previousLeaf: self;
    nextLeaf: nLeaf.
  nLeaf ifNotNil: [ nLeaf previousLeaf: newSibling ].
  self nextLeaf: newSibling.
  ^ newSibling

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _createSibling: selectiveAbortSetOrNil [
  "Creates a new sibling node and moves half of the receiver's entries into the
sibling.  Returns the new sibling."

  | newSibling nLeaf |
  newSibling := super _createSibling.
  nLeaf := self nextLeaf.
  newSibling
    previousLeaf: self;
    nextLeaf: nLeaf.
  nLeaf
    ifNotNil: [
      self _selectiveAbort: nLeaf ifNotIn: selectiveAbortSetOrNil.
      nLeaf previousLeaf: newSibling.
      self rootNode _addBtreePlusNodeToRcReadSet: nLeaf ].
  self nextLeaf: newSibling.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ newSibling

]

{ #category : 'Searching' }
BtreePlusLeafNode >> _findAllValuesForIdenticalKey: aKey into: aCollection using: aComparison [

"Finds all the values for the given identical key, placing them in the
 collection.  For leaf nodes, must first find the appropriate entry, then
 iterate through successive entries until we reach the end or a key does
 not match.  Return whether the last key was equal or not (so the caller
 knows it must check the next leaf node)."

| index eSize maxIndex keyIndexOffset |
" find first entry that might contain the key "
index := self _findCoveringIndexForKey: aKey totalOrder: false.
index == 0 ifTrue: [ ^ false ].

eSize := self entrySize.
maxIndex := self _lastIndex.

" now scan entries until reach the end or no match using sort ="
keyIndexOffset := self keyIndexOffset.
[ (index < maxIndex) and:
[ aComparison compareKey: aKey equalTo: (self at: index + keyIndexOffset) ] ] whileTrue: [
  (aKey == (self at: index + keyIndexOffset))
    ifTrue: [ aCollection add: (self at: index) ].
  index := index + eSize
].
^ index >= maxIndex

]

{ #category : 'Searching Support' }
BtreePlusLeafNode >> _findAllValuesGreaterThanKey: aKey into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is greater
 than the given key.  Returns true if a value was found, false otherwise."

  | entryIndex eSize lastIndex keyIndexOffset comparisonForSort |
  numElements == 0
    ifTrue: [ ^ false ].
  lastIndex := self _lastKeyIndex.	" find index where the key would be inserted "
  comparisonForSort := self comparisonForSort.
  entryIndex := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  entryIndex == 0
    ifTrue: [ ^ false ].	"skip keys that sort ="
  eSize := self entrySize.
  keyIndexOffset := self keyIndexOffset.
  [ comparisonForSort
    compareKey: aKey
    equalTo: (self at: entryIndex + keyIndexOffset) ]
    whileTrue: [ entryIndex := entryIndex + eSize.
      entryIndex > lastIndex
        ifTrue: [ ^ false ] ].
  aKey ~~ nil
    ifTrue: [ "bugfix 33805 - verify that aKey < index entry"
      (comparisonForSort
        compareKey: aKey
        lessThan: (self at: entryIndex + keyIndexOffset))
        ifFalse: [ ^ false ] ].
  anArray addLast: entryIndex.
  anArray addLast: self.
  ^ true

]

{ #category : 'Searching Support' }
BtreePlusLeafNode >> _findAllValuesLessThanKey: aKey andEquals: aBoolean into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key.  Returns true
 if a value was found, false otherwise."

  numElements == 0
    ifTrue: [ ^ false ].
  aBoolean
    ifFalse: [ ^ self _findAllValuesLessThanKey: aKey into: anArray using: aComparison ].
  (self _findAllValuesGreaterThanKey: aKey into: anArray using: aComparison)
    ifTrue: [ | eSize index |
      eSize := self entrySize.
      index := anArray at: 1.
      index > eSize
        ifTrue: [ "move index pointer for first greater than entry to first less than or equal to entry"
          index := index - eSize.
          anArray at: 1 put: index ]
        ifFalse: [ "first entry is greater than, so no entries are less than or equal"
          anArray size: 0.
          ^ false ].
      ^ true ].
  (aKey == nil or: [ aComparison
        compareKey: aKey
        greaterThanOrEqualTo: (self at: self _lastKeyIndex) ])
    ifTrue: [ " if so, put the rightmost leaf entry in the Array "
      self _putLastIndexOfLastChildInto: anArray.
      ^ true ].
  ^ false

]

{ #category : 'Searching Support' }
BtreePlusLeafNode >> _findAllValuesLessThanKey: aKey into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is less than
 the given key.  Returns true if a value was found, false otherwise."

  | entryIndex eSize lastIndex lastKeyIndex keyIndex comparisonForSort |
  numElements == 0
    ifTrue: [ ^ false ].
  lastIndex := self _lastIndex.
  " first see if all entries are > aKey "
  lastKeyIndex := self _lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (comparisonForSort
    compareKey: aKey
    lessThanOrEqualTo: (self at: lastKeyIndex))
    ifFalse: [ "bugfix 33805 - verify that aKey is > index entry"
      (aKey == nil
        or: [ aComparison compareKey: aKey greaterThan: (self at: lastKeyIndex) ])
        ifTrue: [ " if so, put the rightmost leaf entry in the Array "
          self _putLastIndexOfLastChildInto: anArray.
          ^ true ] ].
  " find index where the key would be inserted "
  entryIndex := self _findCoveringIndexForKey: aKey totalOrder: false  using: comparisonForSort.	" if index == 0, found no entryes"
  entryIndex == 0
    ifTrue: [ ^ false ].
  " if index == 1, check if any are < aKey "
  entryIndex == 1
    ifTrue: [ " if the first entry is greater than or equal to aKey, none satisfy "
      | keyIndexOffset |
      keyIndexOffset := self keyIndexOffset.
      ((comparisonForSort compareKey: aKey lessThan: (self at: entryIndex + keyIndexOffset))
        or: [ comparisonForSort compareKey: aKey equalTo: (self at: entryIndex + keyIndexOffset) ])
        ifTrue: [ ^ false ] ].
  eSize := self entrySize.
  entryIndex > 1
    ifTrue: [ " subtract an entry to point to a valid 'less than' entry "
      entryIndex := entryIndex - eSize ].
  keyIndex := entryIndex + self keyIndexOffset.
  aKey ~~ nil
    ifTrue: [ "bugfix 33805 - verify that index aKey is > entry"
      (comparisonForSort compareKey: aKey greaterThan: (self at: keyIndex))
        ifFalse: [ (aComparison compareKey: aKey equalTo: (self at: keyIndex))
            ifTrue: [ ^ false ].
          (aComparison compareKey: aKey lessThan: (self at: keyIndex))
            ifFalse: [ ^ false ] ] ].
  anArray addLast: entryIndex.
  anArray addLast: self.
  ^ true

]

{ #category : 'Searching' }
BtreePlusLeafNode >> _findAndCountKey: aKey value: aValue root: rootObj into: col [

  | index eSize maxIndex keyIndexOffset comparisonForSort rootIndexOffset |
  comparisonForSort := self comparisonForSort.
  index := self _findCoveringIndexForKey: aKey totalOrder: false.
  index == 0
    ifTrue: [ ^ false].
  eSize := self entrySize.
  maxIndex := self _lastEntryIndex.
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  " now scan entries until reach the end or no match "
  [ index <= maxIndex
    and: [ self _compare: comparisonForSort key: aKey equalToEntryAt: index + keyIndexOffset] ]
    whileTrue: [
      (((self at: index + keyIndexOffset) == aKey) and: [((self at: index) == aValue) and: [(self at: index + rootIndexOffset) == rootObj]]) ifTrue: [ col add: rootObj ].
      index := index + eSize ].
  ^ index >= maxIndex

]

{ #category : 'Searching' }
BtreePlusLeafNode >> _findFirstRootForKey: aKey using: aComparison [
  "Returns the first root object associated with the given key.  For leaf nodes, finds
 the appropriate entry and makes the comparison.  Follow nextLeaf if we run off end of
 recevier before finding a match. If one is not found, returns the incomplete path
 traversal object."

  | index eSize maxIndex comparisonForSort keyIndexOffset |
  comparisonForSort := self comparisonForSort.
  index := self _findCoveringIndexForKey: aKey totalOrder: false using: comparisonForSort.
  index == 0
    ifTrue: [ ^ #'_incompletePathTraversal' ].
  eSize := self entrySize.
  maxIndex := self _lastEntryIndex.
  keyIndexOffset := self keyIndexOffset.
  " now scan entries until reach the end or no match "
  [ index <= maxIndex
    and: [ self _compare: comparisonForSort key: aKey equalToEntryAt: index + keyIndexOffset ] ]
    whileTrue: [ (self _compare: aComparison key: aKey equalToEntryAt: index + keyIndexOffset)
        ifTrue: [ ^ self rootObjectAt: index ].
      index := index + eSize ].
  (index > maxIndex and: [ self nextLeaf notNil ])
    ifTrue: [
      "continue the search with the next leaf node .. if nextLeaf not nil"
      ^ self nextLeaf _findFirstRootForKey: aKey using: aComparison ].
  ^ #'_incompletePathTraversal'

]

{ #category : 'Searching Support' }
BtreePlusLeafNode >> _lastRootObjectsNotIdenticalTo: aKey into: array [
  ""

  1 to: numElements * self entrySize by: self entrySize do: [ :i |
    (self keyAt: i) == aKey
      ifFalse: [ array add: (self rootObjectAt: i) ] ].


]

{ #category : 'Enumerating' }
BtreePlusLeafNode >> _leafKeysValuesAndRootsDo: aBlock [
  "Execute the two-argument block with the key and value of each entry in the
 receiver."

  | keyIndexOffset rootIndexOffset |
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  1 to: numElements * self entrySize by: self entrySize do: [ :i |
    aBlock
      value: (self at: i + keyIndexOffset)
      value: (self at: i)
      value: (self at: i + rootIndexOffset) ]

]

{ #category : 'Enumerating' }
BtreePlusLeafNode >> _preOrderDo: aBlock [
  aBlock value: self

]

{ #category : 'Query Support' }
BtreePlusLeafNode >> _primAddRootObjects: aQuerySpec start: startIndex end: endIndex into: collection entrySize: entrySize [
  <primitive: 1038>
  ^ super
    _addRootObjects: aQuerySpec
    start: startIndex
    end: endIndex
    into: collection

]

{ #category : 'Sorting Support' }
BtreePlusLeafNode >> _putAscendingRootObjectsInto: array startingAt: offset [

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  array at: j put: (self rootObjectAt: i).
  j := j + 1
].
^ numElements

]

{ #category : 'Sorting Support' }
BtreePlusLeafNode >> _putDescendingRootObjectsInto: array startingAt: offset [

"Returns the number of elements."

| j |
j := offset.
1 to: (numElements * self entrySize) by: self entrySize do: [ :i |
  array at: j put: (self rootObjectAt: i).
  j := j - 1
].
^ numElements

]

{ #category : 'Searching Support' }
BtreePlusLeafNode >> _putFirstIndexOfFirstChildInto: anArray [

"Puts the receiver and the first index of the receiver into the Array."

numElements == 0
    ifTrue: [ anArray addLast: 0]
    ifFalse: [ anArray addLast: 1 ].
anArray addLast: self

]

{ #category : 'Searching Support' }
BtreePlusLeafNode >> _putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison [

"If first entry is greater than or equal to aKey, Puts the receiver
 and the first index of the receiver into the Array.

 Because of NANs we need to validate the assumption of the caller
 (i.e., that all entries are greater than aKey). See bug33805."

"make sure that key is less than or equal to first entry"
(aKey == nil or: [ numElements == 0
   or: [ aComparison compareKey: aKey lessThanOrEqualTo: (self at: 1 + self keyIndexOffset) ] ])
  ifTrue: [
    numElements == 0
      ifTrue: [ anArray addLast: 0]
      ifFalse: [ anArray addLast: 1 ].
  ]
  ifFalse: [
    anArray addLast: 0.
  ].
  anArray addLast: self

]

{ #category : 'Searching Support' }
BtreePlusLeafNode >> _putLastIndexOfLastChildInto: anArray [
  "Puts the receiver and the last index of the receiver into the Array."

  anArray addLast: self _lastKeyIndex - self keyIndexOffset.
  anArray addLast: self

]

{ #category : 'Enumerating' }
BtreePlusLeafNode >> _scanForRootObjectMapsStartingAt: startIndex end: endIndex key: aKey value: aValue valueOop: valueOop using: comparisonForCompare do: aBlock [
  "Given the fact that the keys from startIndex through endIndex are all #=, we know that
   all equal keys are sorted by the oop of the value.

     If the key at endIndex is equal to aKey
       If the oop of the value at endIndex is less than the oop of aValue we can skip
         the elements of the entire leafNode.
     If the key at startIndex is equal to aKey
       If the oop of the value at startIndex is greater than the oop of aValue we can skip
         the elements of the entire leafNode.
"

  | index eSize keyIndexOffset rootIndexOffset firstKey lastKey |
  keyIndexOffset := self keyIndexOffset.
  firstKey := self at: startIndex + keyIndexOffset.
  (comparisonForCompare compareKey: aKey equalTo: firstKey)
    ifTrue: [ (self at: startIndex) asOop > valueOop ifTrue: [ ^ self ] ].
  lastKey := self at: endIndex + keyIndexOffset.
  (comparisonForCompare compareKey: aKey equalTo: lastKey)
    ifTrue: [ (self at: endIndex) asOop < valueOop ifTrue: [ ^ self ] ].
  index := startIndex.
  eSize := self entrySize.
  rootIndexOffset := self rootIndexOffset.
  [ index <= endIndex ]
    whileTrue: [
      ((self at: index) == aValue and: [ (self at: index + keyIndexOffset) == aKey ])
        ifTrue: [ aBlock value: (self at: index + rootIndexOffset) ].
      index := index + eSize ]

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _splitUsingKey: aKey value: aValue [
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := super at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: nil)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue.
      self
        _insertKey: aKey
        value: aValue
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue.
      newSibling
        _insertKey: aKey
        value: aValue
        atIndex: index ].
  ^ newSibling

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _splitUsingKey: aKey value: aValue root: rootObject [
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value/rootObject into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := super at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: rootObject)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      self
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      newSibling
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ].
  ^ newSibling

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _splitUsingKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value/rootObject into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling: selectiveAbortSetOrNil.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := self at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: rootObject)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      self
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue
        root: rootObject.
      newSibling
        _insertKey: aKey
        value: aValue
        root: rootObject
        atIndex: index ].
  ^ newSibling

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _splitUsingKey: aKey value: aValue selectiveAbortSet: selectiveAbortSetOrNil [
  "Creates a new sibling node, moving the second half of the receiver's entries into the
 sibling.  Then inserts a new entry with the given key/value into either the
 receiver or sibling.  Returns the new sibling."

  | index newSibling lastKeyIndex lastKey comparisonForSort |
  newSibling := self _createSibling: selectiveAbortSetOrNil.
  lastKeyIndex := self _lastKeyIndex.
  lastKey := super at: lastKeyIndex.
  comparisonForSort := self comparisonForSort.
  (self
    compare: comparisonForSort
    key: aKey
    value: aValue
    lessThanOrEqualToEntryAt: lastKeyIndex
    root: nil)
    ifTrue: [ " adding to the first half "
      index := self
        binarySearchCoveringKey: aKey
        value: aValue.
      self
        _insertKey: aKey
        value: aValue
        atIndex: index ]
    ifFalse: [ " adding to the second half "
      index := newSibling
        binarySearchCoveringKey: aKey
        value: aValue.
      newSibling
        _insertKey: aKey
        value: aValue
        atIndex: index ].
  ^ newSibling

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> _totalElementsFrom: start toEndNode: endNode endIndex: endIndex [
  "Determine the number of entries in the receiver counting from start.
   If the receiver is the endNode, then count to endIndex, otherwise count
   to the last index of the receiver."

  | end delta |
  end := self == endNode
    ifTrue: [ endIndex ]
    ifFalse: [ end := self _lastIndex ].
  delta := end - start.
  delta < 0
    ifTrue: [ delta := delta negated ].
  ^ (delta // self entrySize) + 1

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> _totalElementsReversedFrom: start toEndNode: endNode endIndex: endIndex [
  "Determine the number of entries in the receiver counting backwards from start.
   If the receiver is the endNode, then count backwards to endIndex, otherwise count
   backwards to 1."

  | end delta |
  end := self == endNode
    ifTrue: [ endIndex ]
    ifFalse: [ 1 ].
  delta := start - end.
  delta < 0
    ifTrue: [ delta := delta negated ].
  ^ (delta // self entrySize) + 1

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _updateLastValue [

"For leaf nodes, the last value is derived, so nothing is required."


]

{ #category : 'Updating' }
BtreePlusLeafNode >> _updateLastValue: selectiveAbortSetOrNil [

"For leaf nodes, the last value is derived, so nothing is required."


]

{ #category : 'Updating' }
BtreePlusLeafNode >> _updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf [
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  | originalPrevious  |
  originalPrevious := self previousLeaf.
  originalPrevious ~~ newPreviousLeaf
    ifTrue: [
      newPreviousLeaf
        previousLeaf: originalPrevious;
        nextLeaf: self.
      originalPrevious ifNotNil: [  originalPrevious nextLeaf: newPreviousLeaf ].
      self
        previousLeaf: newPreviousLeaf;
        nextLeaf: newNextLeaf.
      newNextLeaf
        ifNotNil: [ newNextLeaf previousLeaf: self ] ].

]

{ #category : 'Updating' }
BtreePlusLeafNode >> _updateLeafNodeLinksForNewPreviousLeaf: newPreviousLeaf newNextLeaf: newNextLeaf  selectiveAbortSet: selectiveAbortSetOrNil [
  " newPreviousNode is being inserted in before of receiver, update the previous/next leaf values for both nodes"

  | originalPrevious  |
  originalPrevious := self previousLeaf.
  originalPrevious ~~ newPreviousLeaf
    ifTrue: [
      originalPrevious ifNotNil: [
        self _selectiveAbort: originalPrevious ifNotIn: selectiveAbortSetOrNil ].
      newPreviousLeaf ifNotNil: [
        self _selectiveAbort: newPreviousLeaf ifNotIn: selectiveAbortSetOrNil ].
      newNextLeaf  ifNotNil: [
        self _selectiveAbort: newNextLeaf ifNotIn: selectiveAbortSetOrNil ].
      newPreviousLeaf
        previousLeaf: originalPrevious;
        nextLeaf: self.
      self rootNode _addBtreePlusNodeToRcReadSet: newPreviousLeaf.
      originalPrevious ifNotNil: [
        originalPrevious nextLeaf: newPreviousLeaf.
        self rootNode _addBtreePlusNodeToRcReadSet: originalPrevious ].
      self
        previousLeaf: newPreviousLeaf;
        nextLeaf: newNextLeaf.
      self rootNode _addBtreePlusNodeToRcReadSet: self.
      newNextLeaf
        ifNotNil: [
          newNextLeaf previousLeaf: self.
          self rootNode _addBtreePlusNodeToRcReadSet: newNextLeaf ] ].

]

{ #category : 'Audit' }
BtreePlusLeafNode >> auditForPathTerm: pathTerm using: auditor parentNode: parentNode previousChild: previousChild nextChild: nextChild do: aBlock [
  "Execute the three-argument block with the key, value, root of each entry in the receiver."

  | keyIndexOffset rootIndexOffset |
  keyIndexOffset := self keyIndexOffset.
  rootIndexOffset := self rootIndexOffset.
  previousChild == previousLeaf
    ifFalse: [
      auditor
        btree: self
        parentNode: parentNode
        previousLeafMismatch: pathTerm
        previousLeaf: previousLeaf
        parentPreviousNode: previousChild ].
  nextChild == nextLeaf
    ifFalse: [
      auditor
        btree: self
        parentNode: parentNode
        nextLeafMismatch: pathTerm
        nextLeaf: nextLeaf
        parentNextNode: nextChild ].
  1 to: numElements * self entrySize by: self entrySize do: [ :i | | v k r |
    v := self at: i.
    k := self at: i + keyIndexOffset.
    r := self at: i + rootIndexOffset.
    aBlock value: self value: k value: v value: r ]

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex [
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison."

  | entryIndex |
  entryIndex := keyIndex - self keyIndexOffset.
  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [ aValue == (self at: entryIndex) ]

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> compare: comparisonForSort key: aKey value: aValue equalToEntryAt: keyIndex root: rootObject [
  "Performs a = comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, use the OOP of the value as the next basis for
 comparison. If the value oops are equeal, use the oop of the rootObject as the
 next basis for comparison."

  ^ (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    and: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      aValue == (self at: entryIndex)
        and: [ rootObject == (self at: entryIndex + self rootIndexOffset) ] ]

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index. If the keys are equal, then use the OOP of the value for the
 comparison."

  (comparisonForSort compareKey: aKey lessThan: (self at: keyIndex))
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    ifTrue: [ | entryIndex |
      entryIndex := keyIndex - self keyIndexOffset.
      ^aValue asOop <= (self at: entryIndex) asOop ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject [
  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the
 comparison."

  (comparisonForSort compareKey: aKey lessThan: (self at: keyIndex))
    ifTrue: [ ^ true ].
  (comparisonForSort compareKey: aKey equalTo: (self at: keyIndex))
    ifTrue: [ | entryIndex entryIndexValue |
      entryIndex := keyIndex - self keyIndexOffset.
      entryIndexValue := self at: entryIndex.
      "aValue asOop = (entryIndexValue) asOop"
      aValue == entryIndexValue
        ifTrue: [ ^ rootObject asOop <= (self at: entryIndex + self rootIndexOffset) asOop ].
      ^ aValue asOop < entryIndexValue asOop ].
  ^ false

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [

  "Performs a <= comparison between aKey and the entry whose key is at the given
 index.  If the keys are equal, then use the OOP of the value for the
 comparison. If the value oops are equal, use the OOP of the rootObject for the
 comparison."

 ""

^ self compare: comparisonForSort key: aKey value: aValue lessThanOrEqualToEntryAt: keyIndex root: rootObject

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> compareValueOop: aValue greaterThanEntryValueOopAt: keyIndex [
  "Perform a > comparison between the oop of aValue and oop of the entry whose value is
   at the given index."


  ^ aValue asOop > (self at: keyIndex - self keyIndexOffset) asOop

]

{ #category : 'Comparison Operators' }
BtreePlusLeafNode >> compareValueOop: aValue lessThanOrEqualToEntryValueOopAt: keyIndex [
  "Perform a <= comparison between the oop of aValue and oop of the entry whose value is
   at the given index."

  ^ aValue asOop <= (self at: keyIndex - self keyIndexOffset) asOop

]

{ #category : 'Testing' }
BtreePlusLeafNode >> isLeaf [

"Returns whether the B-tree node is a leaf node."

^ true

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> keyAt: anIndex [
  "answer the key for the given index"

  ^ self at: anIndex + self keyIndexOffset

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> lastKey [

"Returns the key for the last entry in the receiver."

^ self at: (self _lastEntryIndex + self keyIndexOffset)

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> lastRoot [

"Returns the root object for the last entry in the receiver."

^ self at: (self _lastEntryIndex + self rootIndexOffset)

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> lastValue [

"Returns the value for the last entry in the receiver."

^ self at: self _lastEntryIndex

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> nextLeaf [
  "Returns the value of the instance variable 'nextLeaf'."

  ^ nextLeaf

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> nextLeaf: anObject [
  "Modify the value of the instance variable 'nextLeaf'."

  nextLeaf := anObject

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> previousLeaf [
  "Returns the value of the instance variable 'previousLeaf'."

  ^ previousLeaf

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> previousLeaf: anObject [
  "Modify the value of the instance variable 'previousLeaf'."

  previousLeaf := anObject

]

{ #category : 'Printing' }
BtreePlusLeafNode >> printOn: aStream withIndentationLevel: anInteger [

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

anInteger timesRepeat: [ aStream nextPut: Character space ].

"aStream nextPutAll: '  { '; nextPutAll: self _alias asString; nextPutAll: ' } '."

2 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    aStream nextPutAll: (super at: i) asString;
        nextPut: $(;
        nextPutAll: (super at: i - 1) asString;
        nextPutAll: ') '
].
aStream nextPut: Character lf

]

{ #category : 'Printing' }
BtreePlusLeafNode >> printOn: aStream withIndentationLevel: anInteger includingValues: aBoolean [

"Puts a displayable representation of the keys of the receiver on the given
 stream, indenting the given number of spaces."

| valueIndexOffset |
valueIndexOffset := self valueIndexOffset.

anInteger timesRepeat: [ aStream nextPut: Character space ].

2 to: (numElements * self entrySize) by: self entrySize do: [ :i |
    aStream nextPutAll: (self at: i) asString.

    aBoolean
        ifTrue: [
            aStream nextPut: $(;
                nextPutAll: (self at: i + valueIndexOffset) asString;
                nextPutAll: ') '
        ]
        ifFalse: [ aStream nextPutAll: ' ' ]
].
aStream nextPut: Character lf

]

{ #category : 'Removing' }
BtreePlusLeafNode >> removeKey: aKey value: aValue root: rootObject [
  "Removes the key/value pair.  Finds the index for the given key and then shifts
 any following entries to the left."

  | valueIndex eSize |
  valueIndex := self binarySearchForKey: aKey value: aValue root: rootObject.
  valueIndex == 0
    ifTrue: [ ^ false ].
  eSize := self entrySize.
  self _deleteNoShrinkFrom: valueIndex to: valueIndex + eSize - 1.
  numElements := numElements - 1.
  ^ true

]

{ #category : 'Rc Removing' }
BtreePlusLeafNode >> removeKey: aKey value: aValue root: rootObject selectiveAbortSet: selectiveAbortSetOrNil [
  "Removes the key/value pair.  Finds the index for the given key and then shifts
 any following entries to the left."

  | wasRemoved |
  self _selectiveAbort: self ifNotIn: selectiveAbortSetOrNil.
  wasRemoved := self removeKey: aKey value: aValue root: rootObject.
  self rootNode _addBtreePlusNodeToRcReadSet: self.
  ^ wasRemoved

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> rootObjectAt: anIndex [
  "answer the root object for the given index"

  ^ self at: anIndex + self rootIndexOffset

]

{ #category : 'Accessing' }
BtreePlusLeafNode >> totalElements [

"Return the number of elements."

^ numElements

]
