Extension { #name : 'BtreePlusRoot' }

{ #category : 'Rc compatibility' }
BtreePlusRoot >> _addBtreePlusNodeToRcReadSet: aBtreePlusNode [
  "noop for non-rc btrees"

]

{ #category : 'Testing' }
BtreePlusRoot >> _canCompare: aKey withClass: aClass [
  "Returns whether the receiver can make comparisons with the given key."

  ^ btreeRootNode _canCompare: aKey withClass: aClass

]

{ #category : 'Accessing' }
BtreePlusRoot >> _createNewRootNodeForSplit: returnNode [
  "Create a new root node and put the old root and the new split node into the new root."
  " returnNode is the second half of the split "

  |  node |
  " create the new parent node "
  node := btreeRootNode parentNodeClass new
    rootNode: self;
    objectSecurityPolicy: self objectSecurityPolicy;
    lastValue: returnNode lastValue;
    lastRoot: returnNode lastRoot.
  " insert the first half (the original root) "
  node
     _insertKey: (btreeRootNode _at: btreeRootNode _lastKeyIndex)
     value: btreeRootNode
     root: (btreeRootNode _at: btreeRootNode _lastRootIndex)
     atIndex: 1.
  " insert the second half "
  node
    _insertKey: (returnNode _at: returnNode _lastKeyIndex)
    value: returnNode
    root: (returnNode _at: returnNode _lastRootIndex)
    atIndex: node entrySize + 1.
  ^ node

]

{ #category : 'Searching Support' }
BtreePlusRoot >> _findAllValuesForIdenticalKey: aKey into: aCollection [
  "Finds all the values for the given identical key, placing them in the
 collection.  Return whether the last key was equal or not (so the caller
 knows it must check the next leaf node)."

  ^ self btreeRootNode _findAllValuesForIdenticalKey: aKey into: aCollection

]

{ #category : 'Searching Support' }
BtreePlusRoot >> _findAllValuesGreaterThanKey: aKey into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is greater
 than the given key.  Returns true if a value was found, false otherwise."

  ^ self btreeRootNode _findAllValuesGreaterThanKey: aKey into: anArray using: aComparison

]

{ #category : 'Searching Support' }
BtreePlusRoot >> _findAllValuesLessThanKey: aKey andEquals: aBoolean into: anArray using: aComparison [
  "Puts into anArray the index and the leaf node of the entry that is less than
 (or less than and equal to if aBoolean is true) the given key.  Returns true
 if a value was found, false otherwise."

  ^ self btreeRootNode
    _findAllValuesLessThanKey: aKey
    andEquals: aBoolean
    into: anArray
    using: aComparison

]

{ #category : 'Searching' }
BtreePlusRoot >> _findFirstRootForKey: aKey using: aComparison [
"Returns the first root object associated with the given key.  For interior nodes,
 find the appropriate child node and ask it to find the first root."

^ self btreeRootNode _findFirstRootForKey: aKey using: aComparison

]

{ #category : 'Searching Support' }
BtreePlusRoot >> _lastRootObjectsNotIdenticalTo: aKey into: array [

  self btreeRootNode _lastRootObjectsNotIdenticalTo: aKey into: array

]

{ #category : 'Enumerating' }
BtreePlusRoot >> _leafKeysValuesAndRootsDo: aBlock [

"Execute the two-argument block for each child node."

btreeRootNode _leafKeysValuesAndRootsDo: aBlock

]

{ #category : 'Enumerating' }
BtreePlusRoot >> _preOrderDo: aBlock [
  btreeRootNode _preOrderDo: aBlock

]

{ #category : 'Sorting Support' }
BtreePlusRoot >> _putAscendingRootObjectsInto: array startingAt: offset [

"Returns the number of elements."

^ self btreeRootNode _putAscendingRootObjectsInto: array startingAt: offset

]

{ #category : 'Sorting Support' }
BtreePlusRoot >> _putDescendingRootObjectsInto: array startingAt: offset [

"Returns the number of elements."

^ self btreeRootNode _putDescendingRootObjectsInto: array startingAt: offset

]

{ #category : 'Searching Support' }
BtreePlusRoot >> _putFirstIndexOfFirstChildInto: anArray [
  "Puts the receiver and the first index of the receiver into the Array."

  self btreeRootNode _putFirstIndexOfFirstChildInto: anArray

]

{ #category : 'Searching Support' }
BtreePlusRoot >> _putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison [

"If first entry is greater than or equal to aKey, Puts the receiver
 and the first index of the receiver into the Array.

 Because of NANs we need to validate the assumption of the caller
 (i.e., that all entries are greater than aKey). See bug33805."

self btreeRootNode _putFirstIndexOfFirstChildInto: anArray ifGreaterThanOrEqualTo: aKey using: aComparison

]

{ #category : 'Searching Support' }
BtreePlusRoot >> _putLastIndexOfLastChildInto: anArray [
  "Asks the last child in the receiver to
 putLastIndexOfLastChildInto: anArray ."

  self btreeRootNode _putLastIndexOfLastChildInto: anArray

]

{ #category : 'Audit' }
BtreePlusRoot >> auditNsc: nsc for: pathTerm offset: offset on: aString [
  "Verifies that the btree structure consistent relative to the given pathTerm."

  | auditor |
  auditor := BtreePlusNodeAuditor new auditResultString: aString; yourself.
  ^self btreeRootNode auditNsc: nsc for: pathTerm offset: offset using: auditor

]

{ #category : 'Accessing' }
BtreePlusRoot >> btreeAt: aKey put: aValue root: rootObject [
  "Insert the key/value/rootObject tuple into the root B-tree node.  Must check to see if a
 split occurred.  If so, create a new root node and put the old root and the
 new split node into the new root."

  | returnNode |
  returnNode := self btreeRootNode at: aKey put: aValue root: rootObject.
  returnNode == btreeRootNode
    ifFalse: [
      " create the new root node "
      btreeRootNode := self _createNewRootNodeForSplit: returnNode ]

]

{ #category : 'Accessing' }
BtreePlusRoot >> btreeRootNode [
  ^ btreeRootNode
    ifNil: [ btreeRootNode := indexObject btreePlusLeafNodeClass new
        rootNode: self;
        objectSecurityPolicy: self objectSecurityPolicy;
        yourself ]

]

{ #category : 'Accessing' }
BtreePlusRoot >> btreeRootNode: aBtreePlusNode [
  aBtreePlusNode rootNode: self.
  btreeRootNode := aBtreePlusNode

]

{ #category : 'Accessing' }
BtreePlusRoot >> collator [
  ^ self indexObject collator

]

{ #category : 'Comparison Operators' }
BtreePlusRoot >> comparisonForSort [
  "Answer the comparison object to be used for this index"

  ^ self indexObject comparisonForSort

]

{ #category : 'Accessing' }
BtreePlusRoot >> indexObject [
  "Answer the GsAbstractIndex that associated with this btree"

  ^ indexObject

]

{ #category : 'Accessing' }
BtreePlusRoot >> indexObject: aGsAbstractIndex [
  "Set the GsAbstractIndex that associated with this btree"

  indexObject := aGsAbstractIndex

]

{ #category : 'Testing' }
BtreePlusRoot >> isIdentityIndex [
  ^ self indexObject isIdentityIndex

]

{ #category : 'Accessing' }
BtreePlusRoot >> maxNumberOfElements [
  ^ self btreeRootNode maxNumberOfElements

]

{ #category : 'Printing' }
BtreePlusRoot >> prettyPrint [

""

^self btreeRootNode prettyPrint

]

{ #category : 'Removing' }
BtreePlusRoot >> removeKey: aKey value: aValue root: rootObject [
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  (btreeRootNode removeKey: aKey value: aValue root: rootObject)
    ifFalse: [ ^ false ].
  self updateBtreeRootNodeOnRemoval.
  ^ true

]

{ #category : 'Statistics' }
BtreePlusRoot >> report [
  | strm info leafCount totalElements |
  strm := AppendStream on: String new.
  info := {}.
  self btreeRootNode _putFirstIndexOfFirstChildInto: info.
  totalElements := leafCount := 0.
  (info at: 1) > 0
    ifTrue: [ | leaf |
      leaf := info at: 2.
      [ leaf notNil ] whileTrue: [ leafCount := leafCount + 1.
          totalElements := totalElements + leaf numElements.
          leaf := leaf nextLeaf ] ].
  leafCount == 0
    ifTrue: [ strm nextPutAll: 'The btree is empty' ]
    ifFalse: [ strm nextPutAll: 'There are ' , leafCount printString , ' leaf nodes with '
            , totalElements printString , ' elements ('
            , (totalElements / leafCount asScaledDecimal: 2) printString
            , ' elements/node).' ].
  ^ strm contents

]

{ #category : 'Accessing' }
BtreePlusRoot >> totalElements [

"Return the number of elements."

^ btreeRootNode totalElements

]

{ #category : 'Removing' }
BtreePlusRoot >> updateBtreeRootNodeOnRemoval [
  "If the current root node is an interior node of size one, replace the current root node with its
   single child"

  (btreeRootNode numElements == 1 and: [ btreeRootNode isLeaf not ])
    ifTrue: [ btreeRootNode := btreeRootNode
        at: btreeRootNode _lastKeyIndex - btreeRootNode keyIndexOffset ].

]
