"
This class is part of the FFI implementation and represents a C header file. 
It is typically used to generate a Smalltalk class that wraps a function or structure.
Usage examples can be found in the following instance methods:
	#wrapperForLibraryAt:
	#wrapperNamed:forLibraryAt:select:
	#wrapperForTypeNamed:
	#wrapperNamed:forStruct:

"
Class {
	#name : 'CHeader',
	#superclass : 'Object',
	#instVars : [
		'stream',
		'types',
		'structs',
		'unions',
		'functions',
		'enums',
		'enumTags',
		'storage',
		'declarations',
		'preprocessor',
		'source'
	],
	#category : 'FFI'
}

{ #category : 'Reporting' }
CHeader class >> _gciStructReport [
^ self _gciStructsReport: '$GEMSTONE/include/gci.hf' for:
  {  "gcilegacy.ht" 'GciObjInfoSType' . 'GciFetchObjInfoArgsSType' . 'GciJmpBufSType' .
     "gci.ht" 'GciErrSType' .  'GciObjRepHdrSType' .  'GciObjRepSType' . 'GciX509LoginArg' .
     "gci.hf" 'GciCTimeStringType' .
     "gcicmn.ht" 'GciTravBufType' .  'GciTravBufHolder' .  'GciClampedTravArgsSType' .
         'GciStoreTravDoArgsSType' .  'GciDateTimeSType' }

]

{ #category : 'Reporting' }
CHeader class >> _gciStructsReport: headerPath for: typeNames [
  | h rpt |
  h := CHeader path: headerPath .
  rpt := String new .
  typeNames do:[ :tName | (h _typeAt: tName ) _addToReport: rpt .  rpt lf ].
  ^ rpt

]

{ #category : 'Reporting' }
CHeader class >> _gciTsStructReport [
 ^ self _gciStructsReport: '$GEMSTONE/include/gcits.hf' for:
   { "gci.ht" 'GciErrSType' .  'GciObjRepHdrSType' .  'GciObjRepSType' . "Add with TS x509 support is done: 'GciX509LoginArg' . "
     "gcicmn.ht" 'GciTravBufType' .  'GciTravBufHolder' .  'GciClampedTravArgsSType' .
         'GciStoreTravDoArgsSType' .  'GciDateTimeSType' .
     "gcits.hf"  'GciTsObjInfo' }

]

{ #category : 'Class Membership' }
CHeader class >> cPreprocessorSpecies [
"Answer the class to be used for CPreprocessor objects. Subclasses may
 overload this method as needed."
^ CPreprocessor

]

{ #category : 'other' }
CHeader class >> fetchGciLinkedLibraryName [

"Answer a String containing a path to the linked GCI shared library
 in a GemStone product tree, relative to the value of the GEMSTONE
 environment variable."

| result |
result := String withAll: '$GEMSTONE/lib/' .
result addAll: System fetchLinkedGciLibraryName .
^ result

]

{ #category : 'other' }
CHeader class >> fetchGciRpcLibraryName [

"Answer a String containing a path to the 64-bit RPC GCI shared library
 in a GemStone product tree, relative to the value of the GEMSTONE
 environment variable."

| result |
result := String withAll: '$GEMSTONE/lib/' .
result addAll: System fetchRpcGciLibraryName .
^ result

]

{ #category : 'other' }
CHeader class >> path: aString [
  ^ self path: aString searchPaths: { }

]

{ #category : 'other' }
CHeader class >> path: aString searchPath: searchPath [
	"Parse the given file path aString, searching the absolute path searchPath before the system include directories.
	Example:
	CHeader path: 'openssl/ssl.h' searchPath: '/home/me/where/i/put/private/builds/include/'.
	This will find /home/me/where/i/put/private/builds/include/openssl/ssl.h, and includes within
	that file of the form
	#include <openssl/someFile.h>
	will be found in /home/me/where/i/put/private/builds/include/openssl."

	^self path: aString searchPaths: {searchPath}

]

{ #category : 'other' }
CHeader class >> path: aString searchPaths: anArray [
	"Parse the given file path aString, searching the absolute paths in anArray before the system include directories.
	Example:
	CHeader path: 'openssl/ssl.h' searchPaths: #('/home/me/where/i/put/private/builds/include/').
	This will find /home/me/where/i/put/private/builds/include/openssl/ssl.h, and includes within
	that file of the form
	#include <openssl/someFile.h>
	will be found in /home/me/where/i/put/private/builds/include/openssl."

	| preprocessor fullPath |
	preprocessor := self cPreprocessorSpecies new.
	anArray reverseDo: [:path | preprocessor insertSearchPath: path].
	fullPath := preprocessor searchForInclude: aString excluding: nil.
	fullPath
		ifNil: [^UserDefinedError signal: 'Include file ' , aString , ' not found!'].
	preprocessor includePath: fullPath.
	^self preprocessor: preprocessor

]

{ #category : 'other' }
CHeader class >> preprocessor: aCPreprocessor [

	^self new
		initialize: aCPreprocessor;
		yourself.

]

{ #category : 'other' }
CHeader class >> string: aString [
	^ self preprocessor: (self cPreprocessorSpecies parseString: aString).

]

{ #category : 'other' }
CHeader class >> string: aString ignoreWarnings: aBoolean [
	^ self preprocessor: (self cPreprocessorSpecies parseString: aString ignoreWarnings: aBoolean).

]

{ #category : 'Reporting' }
CHeader >> _typeAt: aName [

^ (types at: aName otherwise: nil) ifNil:[
     (structs at: aName otherwise: nil) ifNil:[
       (unions at: aName otherwise: nil) ifNil:[ Error signal: aName , '  not found' ]]].

]

{ #category : 'CLibrary' }
CHeader >> add: aCDeclaration to: aClass [
  | cvn symList |
	aClass addClassVarName: (cvn := aCDeclaration asClassVarName) .
	aClass class
		compileMethod: aCDeclaration createFunctionInitializerCode
		dictionaries: (symList := GsCurrentSession currentSession symbolList )
		category: 'Initializing - private' environmentId: 0 .
	aClass
		compileMethod: aCDeclaration invokeFunctionCode
		dictionaries: symList 
		category: 'Functions' environmentId: 0.
]

{ #category : 'Accessing' }
CHeader >> atEnd [

	^preprocessor atEnd.

]

{ #category : 'Class Membership' }
CHeader >> cByteArraySpecies [
"Answer the class to be used for CByteArray objects. Subclasses may
 overload this method as needed."
^ CByteArray

]

{ #category : 'Class Membership' }
CHeader >> cDeclarationSpecies [
"Answer the class to be used for C Declarations. Subclasses may
 overload this method as needed."
^ CDeclaration

]

{ #category : 'CLibrary' }
CHeader >> createDefinesInClass: aClass [
	"preprocessor createDefinesInClass: aClass"

]

{ #category : 'CLibrary' }
CHeader >> createFunctionsInClass: aClass select: aBlock [

	| localFunctions |
	localFunctions := aBlock
		ifNil: [functions]
		ifNotNil: [functions select: aBlock].
	localFunctions do: [:each |
		self
			add: each
			to: aClass .
	].
	self
		createInitializerInClass: aClass
		forFunctions: localFunctions
]

{ #category : 'CLibrary' }
CHeader >> createFunctionsInClass: aClass libraryPathExpressionString: aString select: aBlock [

	| localFunctions |
	localFunctions := aBlock
		ifNil: [functions]
		ifNotNil: [functions select: aBlock].
	localFunctions do: [:each |
		self
			add: each
			to: aClass .
	].
	self
		createInitializerInClass: aClass
		forFunctions: localFunctions
		libraryPathExpressionString: aString.
]

{ #category : 'Accessing' }
CHeader >> declarations [

   ^declarations

]

{ #category : 'Accessing' }
CHeader >> definitions [

	^preprocessor definitions.

]

{ #category : 'Accessing' }
CHeader >> enums [

   ^enums

]

{ #category : 'Accessing' }
CHeader >> enumTags [

   ^enumTags

]

{ #category : 'Accessing' }
CHeader >> fieldsForStruct: aString [

   ^(structs at: aString) fields.

]

{ #category : 'Accessing' }
CHeader >> fullPath [
  "If not at completion of parsing, the filename may be a nested include file."
  ^ preprocessor path

]

{ #category : 'Accessing' }
CHeader >> functions [

   ^functions

]

{ #category : 'other' }
CHeader >> initialize: aCPreprocessor [

	preprocessor := aCPreprocessor.
	types := KeyValueDictionary new.
	structs := KeyValueDictionary new.
	unions := KeyValueDictionary new.
	functions := KeyValueDictionary new.
	enums := KeyValueDictionary new.
	enumTags := KeyValueDictionary new.
	storage := KeyValueDictionary new.
	declarations := { } .

	[ source := String new.
	  [
		  self skipEmpty.
		  self atEnd .
	  ] whileFalse: [
		  self readOne.
		  source := String new.
	  ].
  ] ensure:[
	  preprocessor clearTokens. "fix 50182"
  ]
]

{ #category : 'CLibrary' }
CHeader >> createInitializerInClass: aClass forFunctions: someFunctions libraryPathExpressionString: aString [
  "Creates a method     aClass class >> initializeFunctions "
	| str symList |
	(str := String new ) add:
'initializeFunctions
	 | library |
	 library := CLibrary named: ' , aString , ' .
   self initializeFunctions: library . '; lf .

  symList := GsCurrentSession currentSession symbolList .
  aClass class compileMethod: str dictionaries: symList category: 'Initializing - private' environmentId: 0.

  self createInitializerInClass: aClass forFunctions: someFunctions .
]

{ #category : 'CLibrary' }
CHeader >> createInitializerInClass: aClass forFunctions: someFunctions [
  "Creates a method     aClass class >> initializeFunctions: aCLibrary  "
  | str symList |
  str := self initializeFunctionsCodeForFunctions: someFunctions .
  symList := GsCurrentSession currentSession symbolList .
  aClass class compileMethod: str dictionaries: symList category: 'Initializing - private' environmentId: 0.
]

{ #category : 'CLibrary' }
CHeader >> initializeFunctionsCodeForFunctions: someFunctions [
	| str |
	str := String new .
	str add: 'initializeFunctions: aCLibrary ' ; lf;tab;
		  add: 'self '; lf; tab .
	(someFunctions asSortedCollection: [:a :b | a name <= b name]) do: [:aFunction |
		str add: aFunction initializerFunctionName,  ' aCLibrary ;' ; lf; tab . 
	].
	str add: 'yourself.'; lf .
  ^ str
]

{ #category : 'CLibrary' }
CHeader >> newClassNamed: aString [

	^self
		newClassNamed: aString
		instVarNames: #().

]

{ #category : 'CLibrary' }
CHeader >> newClassNamed: aString instVarNames: anArray [

	^Object
		subclass: aString
		instVarNames: anArray
		classVars: #()
		classInstVars: #()
		poolDictionaries: #()
		inDictionary: nil
		options: #().

]

{ #category : 'Accessing' }
CHeader >> next [
	| token src |
	preprocessor atEnd ifTrue:[ ^ nil].
	token := preprocessor next.
	source ifNotNil:[ | tkSrc |
     src := source .
     tkSrc := token source .
     (source size > 0 and:[ source last isAlphaNumeric ]) ifTrue:[
       tkSrc size > 0 ifTrue:[ | tC |
         tC := tkSrc at: 1.
         tC isAlphaNumeric ifTrue:[ source add: $  ]
       ]
     ].
     self addToSource: token source
  ].
	^ token.

]

{ #category : 'private' }
CHeader >> addToSource: aString [
 "(SessionTemps current at: #TraceCpp otherwise: nil) ifNotNil:[:t|
    GsFile gciLogServer:'cpp add: ', aString printString .
    t == 2 ifTrue:[ self pause ].
  ].
 "
  source add: aString .
  ^ aString .
]


{ #category : 'Accessing' }
CHeader >> peek [

	^preprocessor peek.

]

{ #category : 'other' }
CHeader >> readClass [
"                                    .----------------.
                                    V                   |
>>--+--class--identifier--{----member--;--+--}---><
"

	| token classDeclaration oldTypes oldStructs oldUnions oldFunctions oldEnums oldStorage newStorage aName |
	classDeclaration := declarations last.
	(token := self next) isSemicolonToken ifTrue: [
		classDeclaration addToSource: source.
		source := nil.
		^self.
	].
	token isOpenCurlyBracketToken ifFalse: [token error: 'syntax error, expected ''{'' '].
  aName := classDeclaration name .
  CPreprocessor _trapDefinition: aName value: classDeclaration .
	types at: aName put: classDeclaration.
	oldTypes := types copy.
	oldStructs := structs copy.
	oldUnions := unions copy.
	oldFunctions := functions copy.
	oldEnums := enums copy.
	oldStorage := storage copy.
	[
		self skipEmpty.
		(token := self peek) isCloseCurlyBracketToken 
	] whileFalse: [
		(token type == #'identifier' and: [token value = 'public' or: [token value = 'private']]) ifTrue: [
			(token := self next; next) isColonToken ifFalse: [token error: 'Unexpected token'].
		].
		self readOne.
	].
	self next.
	types := oldTypes.
	structs := oldStructs.
	unions := oldUnions.
	functions := oldFunctions.
	enums := oldEnums.
	newStorage := storage.
	storage := oldStorage.
	storage keys do: [:each | newStorage removeKey: each].
	newStorage := newStorage asArray collect: [:each |
		(declarations indexOf: each) -> each.
	].
	newStorage := newStorage asSortedCollection asArray collect: [:each | each value].
	classDeclaration setClassProperties: newStorage.
	self atEnd ifFalse: [
		(token := self next) isSemicolonToken ifFalse: [token error: 'expected '';'' '].
	].
	classDeclaration source: source.

]

{ #category : 'other' }
CHeader >> readExternLinkageDeclarationsUnderStorageRegimeFrom: aCDeclaration [
	"Process declarations of the form:
		extern <linkage_specification> { ... }
	Typically:
		extern 'C' { }
	"

	| priorDeclaration priorSource token |
	(token := self next) isOpenCurlyBracketToken ifFalse: [ token error: 'expected ''{'' '].
	source := String new.
	[
		self skipEmpty.
		self atEnd .
	] whileFalse: [
		self peek isCloseCurlyBracketToken ifTrue: [
			self next.
			^self
		].
		self readOneUnderStorageHandler: [:declaration |
			priorDeclaration := declaration.	"to help with debugging"
			(declaration isEmptyExternalLinkage and: [self peek isOpenCurlyBracketToken]) ifTrue: [
				self readExternLinkageDeclarationsUnderStorageRegimeFrom: declaration.
			] ifFalse: [
				declaration canStorageBeMadeExternal ifTrue: [
					declaration
						storage: aCDeclaration storage
						linkageSpec: aCDeclaration linkageSpec.
				].
			].
		].
		priorSource := source.	"to help with debugging"
		source := String new.
	].
	token error: 'could not find end of ''extern <linkage_specification> {'' '

]

{ #category : 'other' }
CHeader >> readOne [

	self readOneUnderStorageHandler:
			[:declaration |
			(declaration isEmptyExternalLinkage and: [self peek isOpenCurlyBracketToken])
				ifTrue:
					[^self readExternLinkageDeclarationsUnderStorageRegimeFrom: declaration]]

]

{ #category : 'other' }
CHeader >> readOneUnderStorageHandler: aMonadicBlock [
	"Read a declaration and process its storage specification using the
	 specified monadic Block. The Block takes the new declaration as
	 an argument."

	| type declaration token "src" |
	type := self cDeclarationSpecies readTypeFrom: self.
	type source: source.
  "uncomment for tracing"
  "src := source size > 30 ifTrue:[ source copyFrom: 1 to: 30 ] ifFalse:[ source ]. "
  "GsFile gciLogServer:'--x ' , src . "
	declaration := type copy readDeclaration.

	aMonadicBlock value: declaration.

	declaration isEmptyExternalLinkage
		ifFalse:
			[declarations add: declaration.
			declaration includesCode ifTrue: [^self].
			(declaration storage == #typedef and: [declaration type == #class]) ifTrue:[
        ^self readClass
      ].

			[self saveDeclaration: declaration.
			self atEnd ifTrue: [^self].
			(token := self next) isSemicolonToken ifTrue: [^self].
			token isCommaToken ifFalse: [token error: 'Expected comma or semicolon'].
			true]
					whileTrue:
						[declaration := type copy readDeclaration.
						declarations add: declaration]]

]

{ #category : 'other' }
CHeader >> saveDeclaration: aDeclaration [

	| dictionary aName |
	aDeclaration type == #'enum' ifTrue: [
		 aDeclaration enumTag ifNotNil: [
			 aDeclaration enumList ifNil: [
         (enumTags includesKey: aDeclaration enumTag) ifFalse: [
            self error: 'Missing definition of enum ' , aDeclaration enumTag.
         ].
         aDeclaration name ifNil: [
           self error: 'enum type should declare a variable or function'.
         ].
       ] ifNotNil: [
         aDeclaration enumList isEmpty ifTrue: [
           self error: 'enum should define some values'.
         ].
         (enumTags includesKey: aDeclaration enumTag) ifTrue: [
           self error: 'Duplicate definition of enum ' , aDeclaration enumTag.
         ].
         enumTags at: aDeclaration enumTag put: aDeclaration.
       ].
    ].
	].
	(aName := aDeclaration name) ifNil: [^self].
	dictionary := aDeclaration storage == #'typedef' 
                ifTrue:[ types ] 
                ifFalse:[ aDeclaration isFunction ifTrue: [	functions] ifFalse: [ storage]].
  dictionary == types ifTrue:[ CPreprocessor _trapDefinition: aName value: aDeclaration ].
	dictionary at: aName put: aDeclaration.

]

{ #category : 'other' }
CHeader >> skipEmpty [
	[
		self atEnd == false and: [self peek isEmptyToken].
	] whileTrue: [
		| token |
		token := self next.
	].

]

{ #category : 'Accessing' }
CHeader >> storage [

   ^storage

]

{ #category : 'Accessing' }
CHeader >> structs [

   ^structs

]

{ #category : 'Accessing' }
CHeader >> types [

   ^types

]

{ #category : 'Accessing' }
CHeader >> unions [
   ^unions

]

{ #category : 'CLibrary' }
CHeader >> wrapperForLibraryAt: aString [
"
| header class |
header := CHeader path: '$GEMSTONE/include/gci.hf'.
class := header wrapperForLibraryAt: CHeader fetchGciRpcLibraryName .
UserGlobals at: class name put: class.
"
	| name |
	name := (((aString subStrings: $/) last subStrings: $.) first subStrings: $-) first.
	(3 < name size and: [(name copyFrom: 1 to: 3) = 'lib']) ifTrue: [name := name copyFrom: 4 to: name size].
	name first isLowercase ifTrue: [name at: 1 put: name first asUppercase].
	^self
		wrapperNamed: name
		libraryPathExpressionString: aString printString
		select: nil.

]

{ #category : 'CByteArray' }
CHeader >> wrapperForType: aCDeclaration [

	^self
		wrapperNamed: aCDeclaration name
		forType: aCDeclaration.

]

{ #category : 'CByteArray' }
CHeader >> wrapperForTypeNamed: aString [
"
| header class |
UserGlobals removeKey: #'GciErrSType' ifAbsent: [].
header := CHeader path: '$GEMSTONE/include/gci.hf'.
class := header wrapperForTypeNamed: 'GciErrSType'.
UserGlobals at: class name put: class.
"
	^self
		wrapperNamed: aString
		forType: (types at: aString).

]

{ #category : 'CLibrary' }
CHeader >> wrapperNamed: nameString forLibraryAt: pathString select: aBlock [
"
| header class |
UserGlobals removeKey: #'GciLibrary' ifAbsent: [].
header := CHeader path: '$GEMSTONE/include/gci.hf'.
class := header
	wrapperNamed: 'GciLibrary'
	forLibraryAt: self class fetchGciRpcLibraryName
	select: [:each | each name first == $G ].
UserGlobals at: class name put: class.
"
	^self
		wrapperNamed: nameString
		libraryPathExpressionString: pathString printString
		select: aBlock

]

{ #category : 'CByteArray' }
CHeader >> wrapperNamed: nameString forStruct: aCDeclaration [
"  Example""
   | header class |
   UserGlobals removeKey: #'GciErrSType' ifAbsent: [].
   header := CHeader path: '$GEMSTONE/include/gci.hf'.
   class := header
	   wrapperNamed: 'GciErrSType'
	   forType: (header _typeAt:'GciErrSType').
   UserGlobals at: class name put: class.
"
	| class symbolList string writeStream typ cbaSizeString |
  ((typ := aCDeclaration type) == #class or:[ typ == #struct]) ifFalse:[
    Error signal:'expected a class or struct , declaration is a ', typ asString .
  ].
  cbaSizeString := aCDeclaration byteSizeForMalloc asString .
	symbolList := GsCurrentSession currentSession symbolList.
	class := (self cByteArraySpecies)
		subclass: nameString
		instVarNames: #()
		classVars: #()
		classInstVars: #()
		poolDictionaries: #()
		inDictionary: nil
		options: #().
	aCDeclaration addSourceTo: (writeStream := AppendStream on: String new).
	class comment: writeStream contents.
	string := 
'on: aCByteArray
  | res |
	res := self fromRegionOf: aCByteArray offset: 0 numBytes: ', cbaSizeString,' .
  res derivedFrom: aCByteArray .
  ^ res
'.
	class class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Instance Creation'
		environmentId: 0.
	string := 
'new
	^self gcMalloc: ' , cbaSizeString ,' 
'.
	class class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Instance Creation'
		environmentId: 0.
	string := 'initialize
'.
	class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Initialization'
		environmentId: 0.
	string := 'initialize: aCByteArray
  | sz | 
  sz := self size min: aCByteArray size .
  self copyBytesFrom: aCByteArray from: 1 to: sz into: 0 allowCodePointZero: true .
'.
	class
		compileMethod: string  "may signal CompileError or CompileWarning"
		dictionaries: symbolList
		category: 'Initialization'
		environmentId: 0.

  "CByteArray>>_stringFromBytes:  is inherited and in base image"

	aCDeclaration sourceStringsForAccessors do: [:each |
		class
			compileMethod: each  "may signal CompileError or CompileWarning"
			dictionaries: symbolList
			category: 'Accessing' environmentId: 0 .
	].
	aCDeclaration sourceStringsForUpdators do: [:each |
		class
			compileMethod: each  "may signal CompileError or CompileWarning"
			dictionaries: symbolList
			category: 'Updating'
			environmentId: 0.
	].
	^class.

]

{ #category : 'CByteArray' }
CHeader >> wrapperNamed: nameString forType: aCDeclaration [
	| struct typ nam |
	((typ := aCDeclaration type) == #'class' or:[ typ == #struct]) ifTrue: [
		^self
			wrapperNamed: nameString
			forStruct: aCDeclaration.
	].
  nam := aCDeclaration name .
  typ _isSymbol ifFalse:[ 
    struct := aCDeclaration type . 
    typ := struct type .
  ].
	(typ == #struct or:[ typ == #class]) ifFalse:[
		Error signal: nam , ' is an ', typ asString , ' , wrapper creation not supported'.
	].
  struct ifNil:[ Error signal:'logic error in wrapperNamed:forType:' ].
  struct fields ifNil:[ Error signal: struct name , ' is an incomplete type (fields not defined)'].
	^self
		wrapperNamed: nameString
		forStruct: struct.

]

{ #category : 'CLibrary' }
CHeader >> wrapperNamed: nameString libraryPathExpressionString: aString select: aBlock [

	| newClass |
	newClass := self newClassNamed: nameString.
	self
		createFunctionsInClass: newClass libraryPathExpressionString: aString select: aBlock;
		createDefinesInClass: newClass.
	newClass initializeFunctions.
	^newClass.

]
