Extension { #name : 'DependencyList' }

{ #category : 'Private' }
DependencyList class >> _depMap: aBool [

<primitive: 586>
aBool _validateClass: Boolean.
^self _primitiveFailed: #_depMap: args: { aBool }.

]

{ #category : 'Private' }
DependencyList class >> _depMap: aBool toHiddenSet: aHiddenSetId [

<primitive: 587>
aBool _validateClass: Boolean .
aHiddenSetId _validateClass: SmallInteger .
^self _primitiveFailed: #_depMap: args: {aBool . aHiddenSetId}.

]

{ #category : 'Updating' }
DependencyList class >> clearForAll: anArray [
"Sets the dependency entry for all elements of anArray to be nil."

<primitive: 982>
| tl |
(tl := System transactionLevel) ~~ 1 ifTrue:[
  Error signal:( tl == 0 ifTrue:[ 'operation not supported outside of a transaction']
			ifFalse:[ 'operation not supported in a nested transaction']).
].
anArray class ~~ Array ifTrue:[ ArgumentTypeError signal:'argument must be an Array'].
self _primitiveFailed: #clear:forAll: args: { anArray } .
self _uncontinuableError

]

{ #category : 'Updating' }
DependencyList class >> clearForHiddenSet: hiddenSetSpecifier [

"For each object represented in the hidden set, set the dependency entry to nil,
 and remove the object from the hidden set.
 hiddenSetSpecifier should be 2 or  in the range 41..45 inclusive.
 Returns a SmallInteger, the number of objects processed."

<primitive: 1016>
hiddenSetSpecifier _validateClass: SmallInteger.
(hiddenSetSpecifier == 2 or:[ hiddenSetSpecifier >= 41 and:[hiddenSetSpecifier <= 45]])
  ifFalse:[  ArgumentError signal:'invalid hidden set' ].
self _primitiveFailed: #clearForHiddenSet: args: { hiddenSetSpecifier } .
self _uncontinuableError

]

{ #category : 'Accessing' }
DependencyList class >> depMapKeys [

"Return an array of objects that have dependents."

^self _depMap: true

]

{ #category : 'Accessing' }
DependencyList class >> depMapKeysToHiddenSet:  aHiddenSetId [

"Returns the objects that have dependents in the specified hidden set.
 The hiddenSet id must be one of the Reserved for GemStone hidden sets"

^self _depMap: true toHiddenSet: aHiddenSetId

]

{ #category : 'Accessing' }
DependencyList class >> depMapValues [

"Returns array of objects that are dependents."

^self _depMap: false

]

{ #category : 'Accessing' }
DependencyList class >> depMapValuesToHiddenSet: aHiddenSetId [

"Returns the objects that are dependents in the specified hidden set.
 The hiddenSet id must be one of the Reserved for GemStone hidden sets"

^self _depMap: false toHiddenSet: aHiddenSetId

]

{ #category : 'Accessing' }
DependencyList class >> for: anObject [

"Returns the dependency list for the receiver.  If one does not exist,
 returns nil."

<primitive: 585>
self _primitiveFailed: #for: args: { anObject } .
self _uncontinuableError

]

{ #category : 'Copying' }
DependencyList class >> from: aDepList [

"Returns a dependency list that is a copy of the given dependency list."

| dl pathTerm pathTerms |
aDepList == nil
  ifTrue: [ ^ self new ].

dl := aDepList copy.
1 to: dl size by: 2 do: [ :i |
  pathTerm := aDepList at: i.
  pathTerm _isObsoletePathTerm
    ifTrue: [
      " lazy initialization of pathTerms set "
      pathTerms == nil
        ifTrue: [ pathTerms := IdentitySet new ].
      pathTerms add: pathTerm.
    ]
].

" remove obsolete path terms if there are any "
pathTerms == nil
  ifFalse: [
    1 to: pathTerms size do: [ :i |
      dl _removeCompletelyPathTerm: (pathTerms at: i)
    ]
  ].
^ dl

]

{ #category : 'Testing' }
DependencyList class >> needsDepList: anObject [

"Returns whether anObject needs a dependency list if it is the last
 object along an equality index path."

<primitive: 210>
self _primitiveFailed: #needsDepList: args: { anObject } .
self _uncontinuableError

]

{ #category : 'Removing' }
DependencyList class >> removePathTerm: pathTerm for: anObject [

^self removePathTerm: pathTerm for: anObject logging: false

]

{ #category : 'Removing' }
DependencyList class >> removePathTerm: pathTerm for: anObject logging: aBoolean [
  "Removes the entry (an entry has 2 slots) for the given path term.
 If it is the last remaining entry, then the dependency list becomes
 nil."

  | depList |
  depList := self for: anObject.
  depList == nil
    ifTrue: [ ^ self ].
  depList size == 0
    ifTrue: [
      self set: nil for: anObject.
      ^ self ].
  (depList size == 2
    and: [ (depList includesIdentical: pathTerm) and: [ (self needsDepList: anObject) not ] ])
    ifTrue: [
      " if only one entry, remove dependency "
      self set: nil for: anObject.
      ^ self ].
  depList := self from: depList.
  depList _removePathTerm: pathTerm for: anObject.
  self set: (SharedDependencyLists at: depList logging: true) for: anObject

]

{ #category : 'Removing' }
DependencyList class >> removeTracker: tracker for: anObject [
  "Removes the given tracking object from the receiver."

  | depList |
  depList := self for: anObject.
  depList == nil
    ifTrue: [ ^ self ].	" if only one entry, set the dependency tag to be nil "
  (depList size <= 2 and: [ tracker == (depList at: 1) ])
    ifTrue: [
      self set: nil for: anObject.
      ^ self ].
  depList := self from: depList.
  depList _removeTracker: tracker for: anObject.
  self set: (SharedDependencyLists at: depList logging: true) for: anObject

]

{ #category : 'Updating' }
DependencyList class >> set: aDependent for: anObject [

"Sets the dependency entry for anObject to be aDependent."

<primitive: 584>
self _primitiveFailed: #set:for: args: { aDependent . anObject } .
self _uncontinuableError

]

{ #category : 'Searching' }
DependencyList >> _findAllTrackersOfClass: aClass [

"Returns an Array of all tracker objects of the given class."

| result |
result := { } .
2 to: self size by: 2 do: [ :i |
  ( (self at: i) <= 0 and: [ (self at: i - 1) class == aClass ] )
    ifTrue: [ result add: (self at: i - 1) ]
].
^ result

]

{ #category : 'Searching' }
DependencyList >> _findOffsetForPathTerm: pathTerm [

"Returns the offset at which a path term is located in the receiver, or nil
 if it is not found.  Performs a linear search."

1 to: self size by: 2 do: [ :i |
  pathTerm == (self at: i)
    ifTrue: [ ^ i ]
].
^ nil

]

{ #category : 'Searching' }
DependencyList >> _findPathTermForOffset: offset [

"Returns the path term associated with the given instance variable offset if
 it exists; otherwise returns nil. NOTE: there may be more than one pathTerm for
 the given object this method returns only the first."

2 to: self size by: 2 do: [ :i |
  (self at: i) == offset
    ifTrue: [ ^ self at: i - 1 ]
].
^ nil

]

{ #category : 'Searching' }
DependencyList >> _findTrackerOfClass: aClass [

"Returns the first tracker object of the given class.  If none is found,
 returns nil."

2 to: self size by: 2 do: [ :i |
  ( (self at: i) <= 0 and: [ (self at: i - 1) class == aClass ] )
    ifTrue: [ ^ self at: i - 1 ]
].
^ nil

]

{ #category : 'Searching' }
DependencyList >> _getAllTrackers [

"Returns an Array of all tracker objects of the given class."


| result |
result := { } .
2 to: self size by: 2 do: [ :i |
  (self at: i) <= 0
    ifTrue: [ result add: (self at: i - 1) ]
].
^ result

]

{ #category : 'Testing' }
DependencyList >> _hasObsoletePathTerm [

"Returns whether the receiver has an obsolete path term
 (one whose size is zero)."

1 to: self size by: 2 do: [ :i |
  ( (self at: i + 1) > 0 and: [ (self at: i) size == 0 ] )
    ifTrue: [ ^ true ]
].
^ false

]

{ #category : 'Auditing' }
DependencyList >> _needsSorting [

"Return whether all pathterms in the receiver are in the correct order."

| hash prevHash |
prevHash := 0.
1 to: self size by: 2 do: [ :i |
  hash := (self at: i) asOop.
  hash < prevHash
    ifTrue: [ ^ true ]
    ifFalse: [ prevHash := hash ].
].
^ false

]

{ #category : 'Removing' }
DependencyList >> _removeCompletelyPathTerm: pathTerm [
  "Removes the entry with the given path term.  Returns the instance variable
   offset for the given path term
     or
   reference count of last element if iv offset is negative"

  | sz ivOffset |
  sz := self size.
  1 to: sz by: 2 do: [ :i |
    pathTerm == (self at: i)
      ifTrue: [
        ivOffset := self at: i + 1.
        i == (sz - 1)
          ifFalse: [
            " not last entry shift remaining entries "
            self
              replaceFrom: i
              to: sz - 2
              with: self
              startingAt: i + 2 ].
        self size: sz - 2.
        ^ ivOffset ] ].
  ^ 0

]

{ #category : 'Auditing' }
DependencyList >> _removeEntryAtOffset: offset [

"Remove the entry at the given offset."

| sz |
offset odd
  ifFalse: [ ^ self ].

sz := self size.
offset == (sz - 1) ifFalse: [
  "self copyFrom: offset + 2 to: sz into: self startingAt: offset  "
  self replaceFrom: offset to:  sz -  2 with: self startingAt: offset + 2
].
self size: sz - 2.

]

{ #category : 'Removing' }
DependencyList >> _removePathTerm: pathTerm for: anObject [
  "Removes the entry with the given path term."

  | sz ivOffset |
  sz := self size.
  1 to: sz by: 2 do: [ :i |
    pathTerm == (self at: i)
      ifTrue: [
        ivOffset := self at: i + 1.
        ((DependencyList needsDepList: anObject)
          and: [ (anObject class isBytes and: [ ivOffset > 1 ]) or: [ ivOffset < -1 ] ])
          ifTrue: [
            ivOffset > 1
              ifTrue: [
                "Bug 42640 - upgraded indexes may have positive reference count for
                 byte indexable objects"
                self at: i + 1 put: ivOffset - 1 ]
              ifFalse: [
                "reference counts are negative numbers"
                " increment reference count"
                self at: i + 1 put: ivOffset + 1 ] ]
          ifFalse: [
            " see if found in the last entry "
            i == (sz - 1)
              ifFalse: [
                " shift remaining entries "
                self
                  replaceFrom: i
                  to: sz - 2
                  with: self
                  startingAt: i + 2 ].
            self size: sz - 2 ].
        ^ self ] ]

]

{ #category : 'Removing' }
DependencyList >> _removeTracker: tracker for: anObject [

"Removes the given tracking object from the receiver."

| sz |
sz := self size.
1 to: sz by: 2 do: [ :i |
  tracker == (self at: i)
    ifTrue: [
      " see if found in the last entry "
      i == (sz - 1)
        ifFalse: [ " shift remaining entries "
          " self copyFrom: i + 2 to: sz into: self startingAt: i "
          self replaceFrom: i to: sz - 2 with: self startingAt: i + 2
        ].
      self size: sz - 2.
      ^ self
    ]
]

]

{ #category : 'Reduced Conflict Support' }
DependencyList >> _replaceReferencesWith: aDepList [

"Scans the write set, looking for objects that have the receiver for a
 dependency tag.  For each one, replace the dependency tag with aDepList."

| writeSet obj depListClass |
writeSet := (GsBitmap newForHiddenSet: #DepMapWriteSet) asArray.
depListClass := self class .
1 to: writeSet size do: [ :i |
  obj := writeSet _at: i.
  self == (depListClass for: obj)
    ifTrue: [ depListClass set: aDepList for: obj ]
]

]

{ #category : 'Auditing' }
DependencyList >> _sortEntries: aBucket [

"Find entries that are not in order, then reinsert them in the
correct position.  Remove the receiver from the given bucket and
reinsert it in the shared dependency lists table.  Return the number
of entries sorted."

| hash prevHash entriesToMove |
prevHash := 0.
1 to: self size by: 2 do: [ :i |
  hash := (self at: i) asOop.
  hash < prevHash
    ifTrue: [
      entriesToMove == nil
        ifTrue: [ entriesToMove := { }  ].
      entriesToMove add: i; add: (self at: i); add: (self at: i + 1)
    ]
    ifFalse: [ prevHash := hash ].
].
entriesToMove size ~~ 0
  ifTrue: [
    aBucket remove: self.
    1 to: entriesToMove size by: 3 do: [ :i |
      self _removeEntryAtOffset: (entriesToMove at: i).
      self addPathTerm: (entriesToMove at: i + 1)
        withIVOffset: (entriesToMove at: i + 2)
        for: nil.
    ].
    SharedDependencyLists at: self logging: false
  ].
^ entriesToMove size

]

{ #category : 'Comparing' }
DependencyList >> < depList [

"Returns whether the receiver precedes the argument when sorted based upon
 the OOPs of the path terms. "

<primitive: 333>
self _primitiveFailed: #< args: { depList } .
self _uncontinuableError

]

{ #category : 'Comparing' }
DependencyList >> = anotherDependencyList [
    "If the comparison comes down to the element-wise test, only
    compare path terms and use identity.

    N.B. a hash method is not required for this class because its use
    is private to Gemstone. If one were desirable, the technique would
    be to OR the path term identities."

    | size |
    self == anotherDependencyList ifTrue:
        [^true].
    self class == anotherDependencyList class ifFalse:
        [^false].
    size := anotherDependencyList size.
    self size == size ifFalse:
        [^false].
    1 to: size do:
        [ :i |
         (self at: i) == (anotherDependencyList at: i) ifFalse:
             [^false]].
    ^true

]

{ #category : 'Comparing' }
DependencyList >> > depList [

"Returns whether the receiver follows the argument when sorted based upon
 the OOPs of the path terms."

<primitive: 522>
self _primitiveFailed: #> args: { depList } .
self _uncontinuableError

]

{ #category : 'Updating' }
DependencyList >> addPathTerm: pathTerm
withIVOffset: anInteger
for: anObject [

"Adds an entry containing the path term, and instance variable offset to the
 receiver. Primitive will grow receiver if necessary."

<primitive: 523>
self _primitiveFailed: #addPathTerm:withIVOffset:for:
     args: { pathTerm . anInteger . anObject } .
self _uncontinuableError

]

{ #category : 'Testing' }
DependencyList >> containsAnyOf: arrayOfPathTerms [

"Returns whether the receiver contains an entry with any path term in
 the Array of path terms."

1 to: self size by: 2 do: [ :i |
  ( arrayOfPathTerms includesIdentical: (self at: i) )
    ifTrue: [ ^ true ]
].
^ false

]

{ #category : 'Updating' }
DependencyList >> copyAndAddLastElementPathTerm: pathTerm for: object [
  "Makes a copy of the receiver that has an additional entry containing the
 path term, and instance variable offset."

  | sz dl ptOop thisTerm referenceCount |
  referenceCount := -1.	"reference counts are negative numbers"
  ptOop := pathTerm asOop.
  sz := self size.
  1 to: sz by: 2 do: [ :i |
    thisTerm := self at: i.
    pathTerm == thisTerm
      ifTrue: [
        dl := self copy.
        (self class needsDepList: object)
          ifTrue: [
            | existingReferenceCount |
            "bump reference count"
            existingReferenceCount := dl at: i + 1.
            existingReferenceCount := existingReferenceCount > 0
              ifTrue: [
                object class isBytes
                  ifFalse: [
                    "not a reference count, no need to do anything else (bug42640)"
                    ^dl ].
                "upgraded reference counts are positive numbers for byte indexable
                 objects, negate and decrement count ... Bug 42460"
                existingReferenceCount negated - 1 ]
              ifFalse: [
                "reference counts are negative numbers ... just decrement the count ... Bug 42460"
                existingReferenceCount - 1 ].
           dl at: i + 1 put: existingReferenceCount ].
        ^ dl ]
      ifFalse: [
        ptOop < thisTerm asOop
          ifTrue: [
            dl := self class new: sz + 2.
            i == 1
              ifTrue: [
                dl
                  replaceFrom: 3
                  to: 2 + sz
                  with: self
                  startingAt: 1 ]
              ifFalse: [
                dl
                  replaceFrom: 1
                  to: i - 1
                  with: self
                  startingAt: 1.
                dl
                  replaceFrom: i + 2
                  to: 2 + sz
                  with: self
                  startingAt: i ].
            dl at: i put: pathTerm.
            dl at: i + 1 put: referenceCount.
            ^ dl ] ] ].
  dl := self class new: sz + 2.
  sz > 0
    ifTrue: [
      dl
        replaceFrom: 1
        to: sz
        with: self
        startingAt: 1 ].
  dl at: sz + 1 put: pathTerm.
  dl at: sz + 2 put: referenceCount.
  ^ dl

]

{ #category : 'Updating' }
DependencyList >> copyAndAddPathTerm: pathTerm withIVOffset: ivOffset for: object [
  "Makes a copy of the receiver that has an additional entry containing the
 path term, and instance variable offset."

  | sz dl ptOop thisTerm |
  ptOop := pathTerm asOop.
  sz := self size.
  1 to: sz by: 2 do: [ :i |
    thisTerm := self at: i.
    pathTerm == thisTerm
      ifTrue: [ ^ self copy ]
      ifFalse: [
        ptOop < thisTerm asOop
          ifTrue: [
            dl := self class new: sz + 2.
            i == 1
              ifTrue: [
                dl
                  replaceFrom: 3
                  to: 2 + sz
                  with: self
                  startingAt: 1 ]
              ifFalse: [
                dl
                  replaceFrom: 1
                  to: i - 1
                  with: self
                  startingAt: 1.
                dl
                  replaceFrom: i + 2
                  to: 2 + sz
                  with: self
                  startingAt: i ].
            dl at: i put: pathTerm.
            dl at: i + 1 put: ivOffset.
            ^ dl ] ] ].
  dl := self class new: sz + 2.
  sz > 0
    ifTrue: [
      dl
        replaceFrom: 1
        to: sz
        with: self
        startingAt: 1 ].
  dl at: sz + 1 put: pathTerm.
  dl at: sz + 2 put: ivOffset.
  ^ dl

]

{ #category : 'Updating' }
DependencyList >> copyAndAddTracker: tracker withId: anInteger for: object [

"Adds the tracking object and the integer offset to the receiver."

| sz dl trackerOop thisTerm |

trackerOop := tracker asOop.
sz := self size.
1 to: sz by: 2 do: [ :i |
  thisTerm := self at: i.
  (tracker == thisTerm)
    ifTrue: [ ^ self copy ]
    ifFalse: [
      trackerOop < thisTerm asOop
        ifTrue: [
          dl := self class new: sz + 2.
          i == 1
            ifTrue: [
              "self copyFrom: 1 to: sz into: dl startingAt: 3. "
              dl replaceFrom: 3 to: 2 + sz  with: self startingAt: 1 .
            ]
            ifFalse: [
              "self copyFrom: 1 to: i - 1 into: dl startingAt: 1 "
              dl replaceFrom: 1 to: i - 1 with: self startingAt: 1 .
              "self copyFrom: i to: sz into: dl startingAt: i + 2 "
              dl replaceFrom: i + 2 to: 2 + sz "(i + 2) + sz - i" with: self startingAt: i
             ].
          dl at: i put: tracker.
          dl at: i + 1 put: anInteger.
          ^ dl
        ]
    ]
].

dl := self class new: sz + 2.
sz ~~ 0 ifTrue: [
  "self copyFrom: 1 to: sz into: dl startingAt: 1 "
  dl replaceFrom: 1 to: sz with: self startingAt: 1
].
dl at: sz + 1 put: tracker.
dl at: sz + 2 put: anInteger.
^ dl

]

{ #category : 'Searching' }
DependencyList >> getInstVarOffsetWithPathTerm: pathTerm [
  "Returns the instance variable offset for the entry at which a path term is
 located in the receiver, or zero if it is not found.

 Should only be used with non-collection based pathTerms ... "

  "Performs a linear search."

  | i |
  (i := self _findOffsetForPathTerm: pathTerm) == nil
    ifTrue: [ ^ 0 ].
  ^ self at: i + 1

]

{ #category : 'Removing' }
DependencyList >> removeCompletelyPathTerm: pathTerm for: anObject [
  "Removes the entry with the given path term.  Returns the instance variable
 offset for the given path term."

  | depList ivOffset depListClass |
  " if only one entry, set the dependency tag to be nil "
  (self size == 2 and: [ pathTerm == (self at: 1) ])
    ifTrue: [
      DependencyList set: nil for: anObject.
      ^ self at: 2	"may be negative for reference counting of last element dependency list" ].
  depListClass := self class.
  depList := depListClass from: self.
  ivOffset := depList _removeCompletelyPathTerm: pathTerm.	"may be negative for reference counting of last element dependency list"
  depListClass
    set: (SharedDependencyLists at: depList logging: true)
    for: anObject.
  IndexManager current commitIndexMaintenance: nil at: 0.	"fix 43647: call commitIndexMaintenance bug indexObj and progressCount not applicable"
  ^ ivOffset

]

{ #category : 'Removing' }
DependencyList >> removePathTerms: pathTerms for: anObject [

^self removePathTerms: pathTerms for: anObject logging: false

]

{ #category : 'Removing' }
DependencyList >> removePathTerms: pathTerms for: anObject logging: aBoolean [

"Removes the entries for the given collection of path terms.  If they are
 the last remaining entries, then the dependency list becomes nil."

" if it is a byte object "
| depListClass |
depListClass := self class .
1 to: pathTerms size do: [ :j |
  depListClass removePathTerm: (pathTerms at: j) for: anObject logging: aBoolean
]

]
