Extension { #name : 'Gs32DisjunctiveClauseOptimizer' }

{ #category : 'optimizing' }
Gs32DisjunctiveClauseOptimizer >> consolidateEnumerablePredicates [
  "Consolidate predicates that have a path terms that can be combined into a single
	EnumeratedPath query, i.e., a clause of the form:

      (each.firstName = name) | (each.lastName = name)

    and an index with enumerated path term for

      each.firstName|lastName
"

  | primaries |
  primaries := self predicates copy.
  1 to: primaries size do: [ :index |
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self consolidateEnumeratedPredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ]

]

{ #category : 'optimizing' }
Gs32DisjunctiveClauseOptimizer >> consolidateEnumeratedPredicates: primaries with: primaryPredicate [
  | consolidatedPredicate |
  primaries
    do: [ :secondaryPredicate |
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [
          (primaryPredicate canConsolidateEnumeratedWith: secondaryPredicate)
            ifTrue: [
              primaries at: (primaries indexOf: secondaryPredicate) put: nil.
              consolidatedPredicate := primaryPredicate
                consolidateEnumeratedWith: secondaryPredicate.
              primaries
                at: (primaries indexOf: primaryPredicate)
                put: consolidatedPredicate.
              ^ self ] ] ]

]

{ #category : 'accessing' }
Gs32DisjunctiveClauseOptimizer >> predicatesAsFormula [
  | newFormula |
  newFormula := (predicates at: 1) copy.
  2 to: predicates size do: [ :index | newFormula := newFormula | (predicates at: index) copy ].
  ^ (newFormula bindEvaluatorsFor: self nsc collator: self collator)
    immediateInvariant

]

{ #category : 'optimizing' }
Gs32DisjunctiveClauseOptimizer >> removeRedundantDisjunctivePredicates [
  | primaries |
  [
  primaries := self predicates copy.
  1 to: primaries size do: [ :index |
    (primaries at: index)
      ifNotNil: [ :primaryPredicate | self removeRedundantDisjunctivePredicates: primaries with: primaryPredicate ] ].
  predicates := primaries select: [ :each | each ~~ nil  ] ]
    on: GsUnsatisfiableQueryNotification
    do: [ :note |
      "short circuit analysis and replace whole enchilada with a `true` predicate"
      predicates := OrderedCollection with: (GsQueryPredicate constant: true).
      ^ self ]

]

{ #category : 'optimizing' }
Gs32DisjunctiveClauseOptimizer >> removeRedundantDisjunctivePredicates: primaries with: primaryPredicate [
  primaries
    do: [ :secondaryPredicate |
      (secondaryPredicate ~~ nil  and: [ primaryPredicate ~~ secondaryPredicate ])
        ifTrue: [
          (primaryPredicate redundantDisjunctivePredicateBetween: secondaryPredicate)
            ifNotNil: [ :redundantPredicate | primaries at: (primaries indexOf: redundantPredicate) put: nil ] ] ]

]

{ #category : 'optimizing' }
Gs32DisjunctiveClauseOptimizer >> runOptimizations [
  self queryOptions removeRedundantPredicates
    ifTrue: [ self removeRedundantDisjunctivePredicates ].
  self queryOptions consolidateEnumerablePredicates
    ifTrue: [ self consolidateEnumerablePredicates ]

]
