Extension { #name : 'GsAbstractIndex' }

{ #category : 'Testing' }
GsAbstractIndex >> _canCompare: aKey withClass: aClass [
  "Returns whether the receiver can make B-tree comparisons with the given key."


  ^ self btreeRoot _canCompare: aKey withClass: aClass

]

{ #category : 'Testing' }
GsAbstractIndex >> _checkSameIndexOptionsAs: anIndexSpec [
  ^ self termsRequired == anIndexSpec requirePathTerms

]

{ #category : 'Testing' }
GsAbstractIndex >> _checkSameLastElementClassAs: indexObj [
  ^ indexObj _checkSameLastElementClassAsEqualityIndex: self

]

{ #category : 'Updating' }
GsAbstractIndex >> _clear [
  "Assigns nil to important instance variables."

  btreeRoot := nil.
  nscRoot := nil.
  self size: 0

]

{ #category : 'Testing' }
GsAbstractIndex >> _commonPathTermsWith: anEvaluatorOrIndex [
  ^ anEvaluatorOrIndex _commonPathTermsWithIndex: self

]

{ #category : 'Testing' }
GsAbstractIndex >> _commonPathTermsWithIndex: anIndex [
  | commonSize commonTerms |
  commonSize := self size min: anIndex size.
  commonTerms := {}.
  1 to: commonSize do: [ :index |
    | term |
    (term := (self at: index) name) = (anIndex at: index) name
      ifFalse: [ ^ commonTerms ].
    commonTerms add: term ].
  ^ commonTerms

]

{ #category : 'Indexing Support' }
GsAbstractIndex >> _doPutCommonPathTermsForPathArray: pathArray for: anIndexList [

"Find the common path terms that already exist for indexes whose path
 components match pathArray (an Array of Strings).  Adds those path terms to
 the end of receiver.  Do not put a common path term into receiver that
 is the last path term for an existing index."

| firstUnique somePathTerms j minSize sz someSz |
firstUnique := 0.
sz := pathArray size.
1 to: anIndexList size by: 2 do: [ :i |
  j := anIndexList at: i + 1.
  " only consider index objects for which the NSC is the root "
  j == 1
    ifTrue: [
      somePathTerms := anIndexList at: i.
      someSz := somePathTerms size.
      minSize := someSz min: sz.

      [ j <= minSize and:
      [ (somePathTerms at: j) name = (pathArray at: j) ] ] whileTrue: [
        ( j > firstUnique _and:
          [ ( j < someSz )_and: [
              j ~~ sz ] ] )
             ifTrue: [
               firstUnique := j.
               self addLast: (somePathTerms at: j) ].
          j := j + 1 ] ] ].

]

{ #category : 'Searching' }
GsAbstractIndex >> _findFirstUnsharedPathTerm [
  "Returns the first path term that is not shared by any other indexes.  This is
 the first path term whose only index is the receiver - which should always be the last term. "

  | pathTerm sz |
  sz := self size.
  1 to: sz do: [ :i |
    pathTerm := self at: i.
    (pathTerm size == 1 and: [ self == (pathTerm at: 1) ])
      ifTrue: [ ^ pathTerm ] ].
  self error: 'no unshared path term found' "should never get here"

]

{ #category : 'Support' }
GsAbstractIndex >> _findSetValuedPathTermOffset [
  "Returns the offset of the set-valued path term in the receiver.  If none
 exists, returns 0."

  1 to: self size do: [ :i |
    (self at: i) indicatesMultiValue
      ifTrue: [ ^ i ] ].
  ^ 0

]

{ #category : 'Sorting' }
GsAbstractIndex >> _getObjectsWithNilOnPath [

"Returns an Array containing each object that has a nil value along the path."

| array |
" gather the root objects with nil on the path "
array := { } .
self _getObjectsWithNilOnPathInto: array.
^ array


]

{ #category : 'Sorting' }
GsAbstractIndex >> _getObjectsWithNilOnPathInto: anNscBuilderOrArray [

"Returns an Array containing each object that has a nil value along the path."

1 to: self size - 1 do: [ :i |
    (self at: i) nilOnPath ifNotNil: [:nilOnPath | anNscBuilderOrArray addAll: nilOnPath ]
].


]

{ #category : 'Testing' }
GsAbstractIndex >> _isIndexObject [

"Returns true if the receiver is an index object; returns false otherwise."

^ true

]

{ #category : 'Indexing Support' }
GsAbstractIndex >> _partialPathComponentsStringUpTo: offset [

"Returns the path components string up to the given offset."

| str |
str := String new.
1 to: (offset min: self size) do: [ :i |
    str add: (self at: i) name.
    i == offset
        ifFalse: [ str add: $. ]
].
^ str

]

{ #category : 'Index Creation' }
GsAbstractIndex >> _preIndexCreationBuildIndexDictionaryFor: pathArray for: anNsc [
  "afford an opportunity to resize dictionary before adding new elements during index creation"

  "noop"


]

{ #category : 'Index Creation' }
GsAbstractIndex >> _preIndexCreationSort [
  | btreeRootNode |
  true
    ifTrue: [
      "skip pre-sort for now"
      ^ self btreeRoot ].
  btreeRootNode := (PathSorter on: {self} directions: #(true))
    lastNodeOffset: 1;
    yourself.
  ^ self btreeRoot
    btreeRootNode: btreeRootNode;
    yourself

]

{ #category : 'Removing' }
GsAbstractIndex >> _removeBtreeEntriesForKey: aKey [

"Removes all entries in the B-tree that have a key identical to aKey.
 Returns an Array of values corresponding to each entry that was removed."

| stream vals |
" first we need to find all values that have aKey as the key "
stream := self asQueryEvaluator _findAllValuesGreaterThanKey: aKey andEquals: true.
vals := { } .

[ stream _btreeAtEnd not and:
[ stream _peekKey _idxForSortEqualTo: aKey ] ] whileTrue: [
    aKey == stream _peekKey
        ifTrue: [ | val root |
          "remove the entry for the value"
          val := stream _peekValue.
          root := stream _peekRoot.
          vals add: {val. root} ].
    stream _btreeNext
].
" now remove the entry for each value "
vals do: [ :ar |
  (self btreeRemoveKey: aKey value: (ar at: 1) root: (ar at: 2))
].

^ vals

]

{ #category : 'Updating' }
GsAbstractIndex >> _setPathTermState [
  "For the last path term, indicate the need to update the B-tree.
   Indicate if the last object along the path needs a dependency list."

  | lastPathTerm |
  lastPathTerm := self at: self size.
  lastPathTerm updateBtree: self

]

{ #category : 'Statistics' }
GsAbstractIndex >> _statisticsInto: dict [

"Puts statistical information into the given dictionary."

| arr arr1 arr2 |
arr := dict at: #BtreeSpaceUtilization ifAbsent: [
  dict
    at: #BtreeSpaceUtilization
    put:  { { } . { } }
].

arr1 := arr at: 1.
arr2 := arr at: 2.

btreeRoot _preOrderDo: [ :node |
  arr1 add: node numElements.
  arr2 add: node class maxNumberOfElements.
].

]

{ #category : 'Traversing' }
GsAbstractIndex >> _traverseObject: anObject [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversal object."

  | nextObj sz pathTerm |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i |
    pathTerm := self at: i.
    nextObj := pathTerm _getNextObjectForTraversal: nextObj.
    (nil == nextObj and: [ i ~~ sz ])
      ifTrue: [ ^ #'_incompletePathTraversal' ] ].
  ^ nextObj

]

{ #category : 'Traversing' }
GsAbstractIndex >> _traverseObject: anObject incompletesInto: incompleteArray incomplete: incomplete [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, place the object
 into the appropriate Array in the incompleteArray of Arrays."

  | nextObj sz pathTerm |
  nextObj := anObject.
  sz := self size.
  1 to: sz do: [ :i |
    pathTerm := self at: i.
    nextObj := pathTerm _getNextObjectForTraversal: nextObj.
    (nil == nextObj and: [ i ~~ sz ])
      ifTrue: [
        incompleteArray == nil
          ifTrue: [ ^ nil ].
        (incompleteArray at: i + 1) add: anObject.
        ^ incomplete ] ].
  ^ nextObj

]

{ #category : 'Testing' }
GsAbstractIndex >> _validateCanCompareWith: aKey [
  "noop"


]

{ #category : 'Updating Indexes' }
GsAbstractIndex >> addDirectMappingsFor: aBag indexList: iList [
  "Add an entry for an index on the elements of the NSC."

  | anObject firstPathTerm indexMgr |
  indexMgr := self indexManager.
  firstPathTerm := self firstPathTerm.
  indexMgr autoCommit
    ifTrue: [ nscRoot _lockForIndexCreation ].
  self lastPathTerm needsDepList
    ifTrue: [ 1 to: aBag size do: [ :i | " Add an entry for an index on the elements of the NSC. "
        nil ~~ (anObject := aBag _at: i)
          ifTrue: [ " add an entry in the dependency list "
            anObject
              getDepListAndAddLastElementPathTerm: firstPathTerm
              logging: false ].
        self btreeAt: anObject put: anObject root: anObject.
        i \\ 100 == 0
          ifTrue: [ indexMgr commitIndexMaintenance: self at: i ] ] ]
    ifFalse: [ 1 to: aBag size do: [ :i | anObject := aBag _at: i.
        self btreeAt: anObject put: anObject root: anObject.
        i \\ 100 == 0
          ifTrue: [ indexMgr commitIndexMaintenance: self at: i ] ] ]

]

{ #category : 'Converting' }
GsAbstractIndex >> asIndexSpecification [
  ^ self subclassResponsibility: #'asIndexSpecification'

]

{ #category : 'Support' }
GsAbstractIndex >> asPathEvaluator [
  "Returns a PathEvaluator over the same path as the receiver."

  | pathEval holder |
  pathEval := PathEvaluator new.
  1 to: self size do: [ :i | pathEval add: (self at: i) name ].
  pathEval nsc: nscRoot.
  holder := { pathEval } .
  pathEval := nil .
  ^ PathEvaluator asMostSpecificType: holder .

]

{ #category : 'Converting' }
GsAbstractIndex >> asString [
  | str |
  str := '('.
  1 to: self size do: [ :index |
    | pt |
    pt := self at: index.
    str := str , pt name asString.
    index < self size
      ifTrue: [ str := str , '.' ] ].
  ^ super asString , str , ')'

]

{ #category : 'Updating' }
GsAbstractIndex >> btreeAt: aKey put: aValue root: rootObject [
  "Insert the key/value pair into the root B-tree node, validate that aKey can be stored in btree."

  self _validateCanCompareWith: aKey.	"signal error if comparison not valid"
  btreeRoot btreeAt: aKey put: aValue root: rootObject

]

{ #category : 'Stream Accessing' }
GsAbstractIndex >> btreeComparisonQuerySpec [
  ^ BtreePlusComparisonQuerySpec new
    rangeIndex: self;
    yourself

]

{ #category : 'Btree Accessing' }
GsAbstractIndex >> btreePlusLeafNodeClass [
  ^ self isIndexOnRootNsc
      ifTrue: [ BtreePlusLeafKeyNode ]
      ifFalse: [ self isIndexOnFirstTerm
          ifTrue: [ BtreePlusLeafKeyValueNode ]
          ifFalse: [ BtreePlusLeafKeyValueRootObjectNode ] ]

]

{ #category : 'Stream Accessing' }
GsAbstractIndex >> btreeRangeComparisonQuerySpec [
  ^ BtreePlusRangeComparisonQuerySpec new

]

{ #category : 'Stream Accessing' }
GsAbstractIndex >> btreeReadStreamClass [
  "Returns the class of btree read stream to create for query results."

  ^ BtreePlusReadStream

]

{ #category : 'Updating' }
GsAbstractIndex >> btreeRemoveKey: aKey value: aValue root: rootObject [
  "Removes the key and value from the B-tree root node.  Must check to see if
 a merge occurred in the root such that the root only contains a single
 entry.  If so, make the single entry the new root.  Returns whether the
 removal occurred."

  ^ btreeRoot removeKey: aKey value: aValue root: rootObject

]

{ #category : 'Accessing' }
GsAbstractIndex >> btreeRoot [
  ^ btreeRoot
    ifNil: [
      btreeRoot := self options reducedConflict
        ifTrue: [ RcBtreePlusRoot new ]
        ifFalse: [ BtreePlusRoot new ].
      btreeRoot
        indexObject: self;
        yourself ]

]

{ #category : 'Clustering' }
GsAbstractIndex >> clusterDepthFirst [

"Cluster the receiver.  Only need to cluster the path terms."

1 to: self size do: [ :i |
  (self at: i) clusterDepthFirst
]

]

{ #category : 'Accessing' }
GsAbstractIndex >> collator [
  "No IcuCollator associated with this index."

  ^ nil

]

{ #category : 'Comparison Operators' }
GsAbstractIndex >> comparisonForCompare [
  self subclassResponsibility: #comparisonForCompare

]

{ #category : 'Comparison Operators' }
GsAbstractIndex >> comparisonForSort [
  "called during creation of receiver, so initialize unset instance variables"

  self subclassResponsibility: #comparisonForSort

]

{ #category : 'Traversing' }
GsAbstractIndex >> cumulativeFactorFor: anObject upTo: aPathTerm startingAt: anOffset [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term. Return an Array of objects. The index
 may be a collection based index."

  | nextObj sz i thePathTerm theResult factor |
  nil == anObject
    ifTrue: [ ^ 0 ].
  nextObj := anObject.
  sz := self size.
  i := anOffset.
  theResult := { nextObj }.
  factor := 0.
  [ thePathTerm := self at: i.
  thePathTerm ~~ aPathTerm and: [ i <= sz ] ]
    whileTrue: [
      theResult do: [ :nextTraversalObj |
        | enumeratedValues terms |
        thePathTerm indicatesMultiValue
          ifTrue: [
            theResult := {}.
            enumeratedValues := IdentityBag new.
terms := 0.
            thePathTerm
              nextObj: nextTraversalObj
              do: [ :obj :ivOffset |
                  enumeratedValues add: obj.
                  factor := factor + (self
                    cumulativeFactorFor: obj
                    upTo: aPathTerm
                    startingAt: i + 1).
                  terms := terms + 1.
                 ].
            factor := factor + (enumeratedValues size / terms).
            ^ factor ]
          ifFalse: [
            nextObj := thePathTerm _nextObjectFor: nextTraversalObj.
            nil == nextObj
              ifTrue: [ ^ factor  ].
            theResult := { nextObj } ] ].
      i := i + 1 ].
  ^ factor

]

{ #category : 'Searching' }
GsAbstractIndex >> findRootObjectMaps: rootObjectMap pathTerm: pathTerm key: aKey value: aValue [
  | stream theKey |
  stream := self asQueryEvaluator
    _findAllValuesGreaterThanKey: aKey
    andEquals: true.
  [
  stream _btreeAtEnd not
    and: [
      theKey := stream _peekKey.
      theKey _idxForSortEqualTo: aKey ] ]
    whileTrue: [
       "collect key/value/root tuples for entries in the btree whose intermediate parent object
       is reachable from the root object in the entry."
      (theKey == aKey and: [ stream _peekValue == aValue ])
        ifTrue: [
          self
            findRootObjects: rootObjectMap
            rootObject: stream _peekRoot
            pathTerm: pathTerm
            key: aKey
            value: aValue ].
      stream _btreeNextNoValue "safe as long as we don't start using reversed read streams" ]

]

{ #category : 'Searching' }
GsAbstractIndex >> findRootObjects: rootObjectMap rootObject: rootObject pathTerm: pathTerm key: aKey value: aValue [
  | pivotObject pivotPathTerm pivotPathTermOffset rootIsPivot |
  pivotPathTerm := rootObjectMap pivotPathTerm.
  pivotPathTermOffset := pivotPathTerm offset.
  pivotObject := rootObjectMap pivotObject.

  rootIsPivot := false.
  pivotPathTermOffset == 1
    ifTrue: [
      rootIsPivot := pivotObject == rootObject.
      rootIsPivot ifTrue: [ (rootObjectMap traversalMap at: rootObject ifAbsentPut: [ IdentityBag new ]) add: rootObject ] ]
    ifFalse: [ (rootObjectMap roots at: rootObject ifAbsentPut: [ IdentityBag new ]) add: rootObject ].

  pathTerm offset == 1
    ifTrue: [ ^ self ].
  (rootObjectMap traversalMap at: aKey ifAbsentPut: [ IdentityBag new ]) add: aValue.
  pathTerm offset == 2
    ifTrue: [
      (rootObjectMap traversalMap at: aValue ifAbsentPut: [ IdentityBag new ]) add: rootObject.
      ^ self ].
  (pivotPathTermOffset ~~ 1 or: [ rootIsPivot ] )
    ifTrue: [
      self
        traverse: rootObject
        parent: nil
        upTo: pathTerm
        startingAt: 1
        do: [ :term :parent :child | (rootObjectMap traversalMap at: child ifAbsentPut: [ IdentityBag new ]) add: parent ] ].

]

{ #category : 'Searching' }
GsAbstractIndex >> findRootObjectsComparingForKey: aKey value: aValue [
  | stream rootObjects theKey |
  stream := self asQueryEvaluator
    _findAllValuesGreaterThanKey: aKey
    andEquals: true.
  rootObjects := IdentityBag new.
  [ stream _btreeAtEnd not and: [
    theKey := stream _peekKey.
    theKey _idxForSortEqualTo: aKey ] ]
    whileTrue: [
      | rootObject |
      "collect key/value/root tuples for entries in the btree whose intermediate parent object
       is reachable from the root object in the entry."
      ((theKey _idxForCompareEqualTo: aKey) and: [ stream _peekValue == aValue ])
        ifTrue: [ "selec value entries that are identically equal to aValue"
          rootObject := stream _peekRoot.
          rootObjects add: rootObject ].
      stream _btreeNext ].
 ^ rootObjects

]

{ #category : 'Searching' }
GsAbstractIndex >> findRootObjectsForKey: aKey value: aValue [
  | stream rootObjects theKey |
  stream := self asQueryEvaluator
    _findAllValuesGreaterThanKey: aKey
    andEquals: true.
  rootObjects := IdentityBag new.
  [
  stream _btreeAtEnd not
    and: [
      theKey := stream _peekKey.
      theKey _idxForSortEqualTo: aKey ] ]
    whileTrue: [
      | rootObject |
      "collect key/value/root tuples for entries in the btree whose intermediate parent object
       is reachable from the root object in the entry."
      (theKey == aKey and: [ stream _peekValue == aValue ])
        ifTrue: [
          "select value entries that are identically equal to aValue"
          rootObject := stream _peekRoot.
          rootObjects add: rootObject ].
      stream _btreeNext ].
  ^ rootObjects

]

{ #category : 'Accessing' }
GsAbstractIndex >> firstPathTerm [

"Returns the first path term in the path components list."

^ self at: 1

]

{ #category : 'Testing' }
GsAbstractIndex >> hasCollectionBasedTerm [
  "Returns true if the path has an enumerated path term or a term that indicates a
   set-valued instance variable."

  ^ self hasEnumeratedTerm or: [ self hasSetValuedTerm ]

]

{ #category : 'Testing' }
GsAbstractIndex >> hasEnumeratedTerm [
  "Returns true if the path has a term that indicates an enumerated path term."

  1 to: self size do: [ :i |
    (self at: i) isEnumeratedTerm
      ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Testing' }
GsAbstractIndex >> hasIndexDictionary [
  "Answer if receiver has an indexDictionary."

  "indexDictionary not used by any of receiver's subclasses"

  ^ false

]

{ #category : 'Testing' }
GsAbstractIndex >> hasIndexOnPath: pathArray [
  "Returns whether the receiver's path components match the path represented by
 pathArray."

  self size == pathArray size
    ifFalse: [ ^ false ].
  1 to: self size do: [ :i |
    ((self at: i) hasMatchingTerm: (pathArray at: i))
      ifFalse: [ ^ false ] ].
  ^ isComplete

]

{ #category : 'Testing' }
GsAbstractIndex >> hasIndexOnPath: pathArray startingAt: offset [

"Returns whether the receiver's path components match the path represented by
 pathArray."

self size - offset + 1 == pathArray size
  ifFalse: [ ^ false ].

offset to: self size do: [ :i |
  (self at: i) name == (pathArray at: i + 1 - offset)
    ifFalse: [ ^ false ].
].
^ isComplete

]

{ #category : 'Support' }
GsAbstractIndex >> hasSamePathAs: anIndex [

"Returns whether the given index is on the same path as the receiver.
 This can only be true if the index is identical."

^ self == anIndex

]

{ #category : 'Testing' }
GsAbstractIndex >> hasSetValuedTerm [
  "Returns true if the receiver has any path term that indicates a set-valued
 instance variable."

  1 to: self size do: [ :i |
    (self at: i) isSetValuedTerm
      ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Accessing' }
GsAbstractIndex >> indexDictionary [

"GsAbstractIndexes do not have an indexDictionary."

^nil

]

{ #category : 'Accessing' }
GsAbstractIndex >> indexManager [

^IndexManager current

]

{ #category : 'Accessing' }
GsAbstractIndex >> indexType [
  ^ self subclassResponsibility: #'indexType'

]

{ #category : 'Testing' }
GsAbstractIndex >> isComplete [

"Returns the value of the instance variable 'isComplete'."

isComplete == nil
  ifTrue: [ ^ true ].
^ isComplete

]

{ #category : 'Testing' }
GsAbstractIndex >> isIdentityIndex [

"Returns false, subclasses will override as needed."

^ false

]

{ #category : 'Testing' }
GsAbstractIndex >> isIndexOnFirstTerm [

"Returns whether the receiver is an index on the first and only path term and is not on
 a root element of the nsc."

^ self lastPathTerm indicatesIndexOnRootNsc
  ifTrue: [ ^false ]
  ifFalse:[ self size == 1 ]

]

{ #category : 'Testing' }
GsAbstractIndex >> isIndexOnNscElements [

"Returns whether the receiver is an index directly on the elements of an NSC
 (either the root NSC or a set-valued instance variable on the path)."

^ self lastPathTerm indicatesIndexOnNscElements

]

{ #category : 'Testing' }
GsAbstractIndex >> isIndexOnRootNsc [

"Returns whether the receiver is an index directly on the elements of
 the root NSC."

^ self lastPathTerm indicatesIndexOnRootNsc

]

{ #category : 'Testing' }
GsAbstractIndex >> isLegacyIndex [

"Legacy indexes are kinds of IdentityIndex, so return false for a kind of GsAbstractIndex."

^ false

]

{ #category : 'Testing' }
GsAbstractIndex >> isPathEvaluator [

^ false

]

{ #category : 'Testing' }
GsAbstractIndex >> isRangeEqualityIndex [

"Returns false, subclasses will override as needed."

^ false

]

{ #category : 'Testing' }
GsAbstractIndex >> isReducedConflictIndex [
  "answer true if the receiver supports reduced conflict operations"

  ^ self options reducedConflict

]

{ #category : 'Testing' }
GsAbstractIndex >> isStreamable [

"Returns true, subclasses will override as needed."

^ true

]

{ #category : 'Accessing' }
GsAbstractIndex >> lastPathTerm [

"Returns the last path term in the path components list."

^ self at: self size

]

{ #category : 'Modification Tracking Support' }
GsAbstractIndex >> modifiedObject: aKey userData: valArray [
  "Notification that modification of aKey is completed and that the receiver
   should update the btree using value and root from valArray."

  ^ self btreeAt: aKey put: (valArray at: 1) root: (valArray at: 2)

]

{ #category : 'Accessing' }
GsAbstractIndex >> nscRoot [

"Returns the value of the instance variable 'nscRoot'."

^nscRoot

]

{ #category : 'Updating' }
GsAbstractIndex >> nscRoot: newValue [

"Modify the value of the instance variable 'nscRoot'."

nscRoot := newValue.
"force lazy initialization of comparisonForSort iv to avoid commit conflicts later... make it invariant, for good measure."
self comparisonForSort immediateInvariant

]

{ #category : 'Testing' }
GsAbstractIndex >> optimizingComparison [
  "Answer true if comparison operations can be optimized in Btree operations and elsewhere.
   Controlled by optimizedComparison index option."

  ^ self options optimizedComparison

]

{ #category : 'Accessing' }
GsAbstractIndex >> options [
  "index options include: optionalPathTerms, reducedConflict, legacyIndex"

  ^ options

]

{ #category : 'Accessing' }
GsAbstractIndex >> options: aGsIndexOptions [
  "make copy to perserve integrity of the incoming options object"

  options := aGsIndexOptions copy _reify immediateInvariant

]

{ #category : 'Accessing' }
GsAbstractIndex >> pathComponentsString [

"Returns a string of the path components."

^ self pathComponentsStringStartingAt: 1

]

{ #category : 'Accessing' }
GsAbstractIndex >> pathComponentsStringStartingAt: offset [

"Returns a string of the path components."

| str sz |

sz := self size.
str := String new.
offset to: sz do: [ :i |
    str addAll: (self at: i) name.
    i == sz
        ifFalse: [ str add: $.  ]
].
^ str

]

{ #category : 'Updating Indexes' }
GsAbstractIndex >> postIndexCreation: originalBtreeRoot [
  btreeRoot btreeRootNode isBtreePlusNode
    ifFalse: [
      | leafNodes pathSorter |
      "Use a merge sort to create a complete B-tree from a PathSorter
       stored in the btreeRoot instance variable."
      leafNodes := {(originalBtreeRoot btreeRootNode)}.	" create an Array containing an empty B-tree node "
      pathSorter := btreeRoot btreeRootNode.
      btreeRoot := {leafNodes.
      pathSorter}.	"per bug 36147, we need the leafNodes array to be persistent to avoid some out-of-memory conditions during index creation"
      pathSorter sortIntoBtreeNodes: leafNodes.	" produce an Array containing many B-tree leaf nodes, all sorted "
      btreeRoot btreeRootNode: (pathSorter createInteriorNodesFor: leafNodes)	" now build up interior nodes for the leaves " ].
  isComplete := true.
  self indexManager autoCommit
    ifTrue: [ | systm |
      systm := System.
      systm commitTransaction
        ifFalse: [ nscRoot _errorCouldNotCommitDuringIndexCreation ].
      systm transactionMode == #'manualBegin'
        ifTrue: [ systm beginTransaction ] ]

]

{ #category : 'Index Creation' }
GsAbstractIndex >> preIndexCreation [
  "record that index creation has started and "

  isComplete := false.
  progress := 0.
  nscRoot size > self btreeRoot maxNumberOfElements
    ifTrue: [ ^ self _preIndexCreationSort ].
  ^ btreeRoot

]

{ #category : 'Updating Indexes' }
GsAbstractIndex >> preIndexRemoval [

isComplete := false.
progress := 0.

^ nil

]

{ #category : 'Formatting' }
GsAbstractIndex >> printOn: aStream [

"Puts a displayable representation of the receiver on the given stream."

"Copy the implementation from Object so we don't inherit it from Collection."

aStream nextPutAll: self asString

]

{ #category : 'Accessing' }
GsAbstractIndex >> progress [

"Returns the value of the instance variable 'progress'."

^progress

]

{ #category : 'Updating' }
GsAbstractIndex >> progress: newValue [

"Modifies the value of the instance variable 'progress'."

progress := newValue

]

{ #category : 'Stream Accessing' }
GsAbstractIndex >> readStreamClass [
  "Returns the class of read stream to create for query results."

  ^ BtreePlusGsIndexReadStream

]

{ #category : 'Accessing' }
GsAbstractIndex >> sizeForNscBuilder [

"Returns the size to be used when an NscBuilder is collecting objects."

^ nscRoot size

]

{ #category : 'Testing' }
GsAbstractIndex >> termsRequired [
  "Returns true if the receiver has any path term that has termsRequired set."

  1 to: self size do: [ :i |
    (self at: i) termsRequired
      ifTrue: [ ^ true ] ].
  ^ false

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverse: anObject parent: aParent pathTerm: aPathTerm startingAt: anOffset limit: limit do: aBlock [
  ""

  | nextObj thePathTerm nextTraversalObj |
  nil == anObject
    ifTrue: [ ^ self ].
  nextTraversalObj := anObject.
  anOffset to: limit do:
    [:i |
        thePathTerm := self at: i.
        thePathTerm indicatesMultiValue
          ifTrue: [
            thePathTerm
              nextObj: nextTraversalObj
              do: [ :obj |
                aBlock value: thePathTerm value: nextTraversalObj value: obj.
                self
                  traverse: obj
                  parent: nextTraversalObj
                  thru: aPathTerm
                  startingAt: i + 1
                  do: aBlock ].
            ^ self ]
          ifFalse: [
            nextObj := thePathTerm _nextObjectFor: nextTraversalObj.
            aBlock value: thePathTerm value: nextTraversalObj value: nextObj.
            nil == nextObj
              ifTrue: [ ^ self ].
            nextTraversalObj := nextObj ] ]

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverse: anObject parent: aParent thru: aPathTerm startingAt: anOffset do: aBlock [
  ""

  | limit |
  nil == anObject
    ifTrue: [ ^ self ].
  limit := (self indexOfIdentical: aPathTerm).
  ^ self
    traverse: anObject
    parent: aParent
    pathTerm: aPathTerm
    startingAt: anOffset
    limit: limit
    do: aBlock

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverse: anObject parent: aParent upTo: aPathTerm startingAt: anOffset do: aBlock [
  ""

  | limit |
  nil == anObject
    ifTrue: [ ^ self ].
  limit := (self indexOfIdentical: aPathTerm) - 1.
  ^ self
    traverse: anObject
    parent: aParent
    pathTerm: aPathTerm
    startingAt: anOffset
    limit: limit
    do: aBlock

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverse: anObject upTo: aPathTerm [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term.  This method assumes that the index is not
 over a set-valued instance variable.  For that kind of index, use the
 traverse:upTo:startingAt:into: method."

  | nextObj sz i |
  nil == anObject
    ifTrue: [ ^ anObject ].
  nextObj := anObject.
  sz := self size.
  i := 1.
  [ (self at: i) ~~ aPathTerm and: [ i <= sz ] ]
    whileTrue: [
      nextObj := (self at: i) _getNextObjectForTraversal: nextObj.
      nil == nextObj
        ifTrue: [ ^ nextObj ].
      i := i + 1 ].
  ^ nextObj

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverse: anObject upTo: aPathTerm startingAt: anOffset addingIndex: addingIndex do: aBlock [
  "Traverse the sub-objects of anObject, using the path terms of the receiver up
 to the given path term.  Add each object at the end of the traversal to the
 result set.  If addingIndex is true, add an index list entry for the
 receiver to any NSCs encountered along the path; if false, remove an index list entry for the receiver.  Returns the result set."

  self traverse: anObject withRoot: anObject upTo: aPathTerm startingAt: anOffset addingIndex: addingIndex do: aBlock

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverse: anObject withRoot: rootObject upTo: aPathTerm startingAt: anOffset addingIndex: addingIndex do: aBlock [
  "Traverse the sub-objects of anObject, using the path terms of the receiver up
 to the given path term.  Add each object at the end of the traversal to the
 result set.  If addingIndex is true, add an index list entry for the
 receiver to any NSCs encountered along the path; if false, remove an index list entry for the receiver.  Returns the result set."

  | nextObj sz i thePathTerm nscOffset |
  nextObj := anObject.
  sz := self size.
  i := anOffset.
  [ (self at: i) ~~ aPathTerm _and: [ i <= sz _and: [ nil ~~ nextObj ] ] ]
    whileTrue: [
      thePathTerm := self at: i.
      thePathTerm indicatesMultiValue
        ifTrue: [
          " if indexing over a collection-based instance variable "
          thePathTerm
            _updateCollectionBasedIndexFor: self
            on: nextObj
            offset: i
            addingIndex: addingIndex.
          nscOffset := i + 1.
          thePathTerm
            nextObj: nextObj
            do: [ :obj |
              " for each object in the collection-based instance variable make recursive call "
              self
                traverse: obj
                withRoot: rootObject
                upTo: aPathTerm
                startingAt: nscOffset
                addingIndex: addingIndex
                do: aBlock ].
          ^ self ]
        ifFalse: [ nextObj := (self at: i) _getNextObjectForTraversal: nextObj ].
      i := i + 1 ].
  i == 1
    ifTrue: [
      "no traversal"
      aBlock value: #'_incompletePathTraversal' value: rootObject.
      ^ self ].
  (nil ~~ nextObj _and: [ aBlock ~~ nil ])
    ifTrue: [ aBlock value: nextObj value: rootObject ].

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverseAllWithParents: anObject parent: theParent upTo: aPathTerm startingAt: anOffset [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term. Return an Array of objects. The index
 may be a collection based index."

  | nextObj sz i thePathTerm theResult |
  theResult := {{ theParent. anObject }}.
  nil == anObject
    ifTrue: [ ^ theResult ].
  sz := self size.
  i := anOffset.
  [ thePathTerm := self at: i.
  thePathTerm ~~ aPathTerm and: [ i <= sz ] ]
    whileTrue: [
      theResult do: [ :ar | | nextTraversalObj |
        nextTraversalObj := ar at: 2.
        thePathTerm indicatesMultiValue
          ifTrue: [
            theResult := {}.
            thePathTerm
              nextObj: nextTraversalObj
              do: [ :obj | | result |
                " for each object in the collection-based instance variable make recursive call "
                result := self
                  traverseAllWithParents: obj
                  parent: nextTraversalObj
                  upTo: aPathTerm
                  startingAt: i + 1.
                theResult addAll: result ].
            ^ theResult ]
          ifFalse: [
            nextObj := thePathTerm _nextObjectFor: nextTraversalObj.
            theResult := {{ nextTraversalObj. nextObj }}.
            nil == nextObj
              ifTrue: [ ^ theResult ] ] ].
      i := i + 1 ].
  ^ theResult

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverseAllWithParents: anObject upTo: aPathTerm startingAt: anOffset [
  "Traverse the sub-objects of the given object using the path terms of the
 receiver up to the given path term. Return an Array of objects. The index
 may be a collection based index."

 ^ self traverseAllWithParents: anObject parent: nil upTo: aPathTerm startingAt: anOffset

]

{ #category : 'Traversing' }
GsAbstractIndex >> traverseObject: anObject [

"Traverse the sub-objects of the given object using the path terms of the
 receiver.  This method assumes that the index is not over a set-valued
 instance variable.  For that kind of index, use the traverse:startingAt:into:
 method.  If a nil value is reached before the end of the path, returns the
 incompletePathTraversal object."

self isIndexOnNscElements
    ifTrue: [ ^ anObject ].

nil == anObject
    ifTrue: [ ^ #_incompletePathTraversal ].

^ self _traverseObject: anObject

]
