Extension { #name : 'GsCompoundClause' }

{ #category : 'instance creation' }
GsCompoundClause class >> clause: clause1 operator: anOperator clause: clause2 [
  | res |
  (res := self new)
    clause1: clause1;
    operator: anOperator;
    clause2: clause2 .
  ^ res

]

{ #category : 'private' }
GsCompoundClause >> _predicateAndEvaluatorsDo: aBlock [
  "visit all predicates in receiver and provide access to predicate and list of evaluators
   for the predicate"

  super _predicateAndEvaluatorsDo: aBlock.
  self clause1 _predicateAndEvaluatorsDo: aBlock.
  self clause2 _predicateAndEvaluatorsDo: aBlock

]

{ #category : 'testing' }
GsCompoundClause >> _queryUsesIdentityIndex [
  "answer true if the one or more of the clauses has an evaluator that is an range or identity index"

  ^ clause1 _queryUsesIdentityIndex or: [ clause2 _queryUsesIdentityIndex ]

]

{ #category : 'testing' }
GsCompoundClause >> _queryUsesIndex [
  "answer true if the one or more of the clauses has an evaluator that is a range or identity index"

  ^ clause1 _queryUsesIndex or: [ clause2 _queryUsesIndex ]

]

{ #category : 'visiting' }
GsCompoundClause >> acceptVisitor: aFormulaVisitor [
  super acceptVisitor: aFormulaVisitor.
  aFormulaVisitor acceptCompoundClause: self

]

{ #category : 'optimizing' }
GsCompoundClause >> applyDeMorgansNegationTransform [
  "actively apply De Morgan's laws ... operating on copy"

  self clause1: self clause1 applyDeMorgansNegationTransform.
  self operator: self operatorNegated.
  self clause2: self clause2 applyDeMorgansNegationTransform

]

{ #category : 'optimizing' }
GsCompoundClause >> applyDeMorgansTransform [
  "do not transform, but propagate the transform ... operating on copy"

  self clause1: self clause1 applyDeMorgansTransform.
  self clause2: self clause2 applyDeMorgansTransform.
  ^ self

]

{ #category : 'converting' }
GsCompoundClause >> asFormulaWithSelectorParts: selectorParts arguments: arguments [
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ self perform: selector with: arguments first

]

{ #category : 'transforming' }
GsCompoundClause >> bind: variableName to: value [
  | bound |
  bound := self copy.
  bound clause1: (bound clause1 bind: variableName to: value).
  bound clause2: (bound clause2 bind: variableName to: value).
  ^ bound immediateInvariant

]

{ #category : 'private' }
GsCompoundClause >> bindEvaluators [
  clause1 := clause1 bindEvaluatorsFor: nsc collator: self collator.
  clause2 := clause2 bindEvaluatorsFor: nsc collator: self collator.
  super bindEvaluators

]

{ #category : 'private' }
GsCompoundClause >> bruteThreshold [
  "number of elements in result set where brute force query is cheaper than using index."

  ^ 1000

]

{ #category : 'accessing' }
GsCompoundClause >> clause1 [
  ^ clause1

]

{ #category : 'accessing' }
GsCompoundClause >> clause1: newValue [
  clause1 := newValue

]

{ #category : 'accessing' }
GsCompoundClause >> clause2 [
  ^ clause2

]

{ #category : 'accessing' }
GsCompoundClause >> clause2: newValue [
  clause2 := newValue

]

{ #category : 'querying-private' }
GsCompoundClause >> elementValue: anObject [
  "the pathTerms in the query will use anObject as the starting point ... analagous to an object in the nsc that is bound to a query"

  ^ (self clause1 elementValue: anObject)
    perform: self operator
    with: (self clause2 elementValue: anObject)

]

{ #category : 'querying' }
GsCompoundClause >> executeAndDo: aBlock [
  | preparedClause2 |
  preparedClause2 := self clause2 prepareForExecution.
  self operator == #'&'
    ifTrue: [
      self clause1
        executeAndDo: [ :obj |
          (preparedClause2 elementValue: obj)
            ifTrue: [ aBlock value: obj ] ] ]
    ifFalse: [
      | preparedClause1 |
      preparedClause1 := self clause1 prepareForExecution.
      self _nsc
        do: [ :obj |
          (preparedClause1 elementValue: obj)
            ifTrue: [ aBlock value: obj ]
            ifFalse: [
              (preparedClause2 elementValue: obj)
                ifTrue: [ aBlock value: obj ] ] ] ]

]

{ #category : 'querying-private' }
GsCompoundClause >> executeClause [
  ^ self executeClauseUsing: self operator

]

{ #category : 'querying-private' }
GsCompoundClause >> executeClauseNegated [
  | intermediateResult |
  intermediateResult := self clause1 executeClauseNegated.
  intermediateResult isEmpty
    ifTrue: [
      self operatorNegated == #'&'
        ifTrue: [ ^ intermediateResult ]
        ifFalse: [ ^ self clause2 executeClauseNegated ] ].
  ^ (self operatorNegated == #'&'
    and: [
      intermediateResult size < self bruteThreshold
        or: [ self clause2 usesPathEvaluatorForFirstClause ] ])
    ifTrue: [
      "Evaluate clause2 against intermediateResult when performing an #& and intermediateResult size
       is less than 1000. Otherwise it is more efficient to use an index for clause2."
      self clause2 executeClauseNegatedOn: intermediateResult ]
    ifFalse: [
      intermediateResult
        perform: (self resultOperatorFor: self operatorNegated)
        with: self clause2 executeClauseNegated ]

]

{ #category : 'querying-private' }
GsCompoundClause >> executeClauseUsing: anOperator [
  | intermediateResult |
  intermediateResult := self clause1 executeClause.
  intermediateResult isEmpty
    ifTrue: [
      anOperator == #'&'
        ifTrue: [ ^ intermediateResult ]
        ifFalse: [ ^ self clause2 executeClause ] ].
  ^ (anOperator == #'&'
    and: [
      intermediateResult size < self bruteThreshold
        or: [ self clause2 usesPathEvaluatorForFirstClause ] ])
    ifTrue: [
      "Evaluate clause2 against intermediateResult when performing an #& and intermediateResult size
       is less than 1000. Otherwise it is more efficient to use an index for clause2."
      self clause2 executeClauseOn: intermediateResult ]
    ifFalse: [
      intermediateResult
        perform: (self resultOperatorFor: anOperator)
        with: self clause2 executeClause ]

]

{ #category : 'querying-private' }
GsCompoundClause >> executeNegatedAndDo: aBlock [
  | preparedClause2 |
  preparedClause2 := self clause2 prepareForExecution.
  self operatorNegated == #'&'
    ifTrue: [
      self clause1
        executeNegatedAndDo: [ :obj |
          (preparedClause2 elementValue: obj)
            ifFalse: [ aBlock value: obj ] ] ]
    ifFalse: [
      | preparedClause1 |
      preparedClause1 := self clause1 prepareForExecution.
      self _nsc
        do: [ :obj |
          (preparedClause1 elementValue: obj)
            ifTrue: [
              (preparedClause2 elementValue: obj)
                ifFalse: [ aBlock value: obj ] ]
            ifFalse: [ aBlock value: obj ] ] ]

]

{ #category : 'private' }
GsCompoundClause >> immediateInvariant [
  super immediateInvariant.
  clause1 immediateInvariant.
  clause2 immediateInvariant

]

{ #category : 'testing' }
GsCompoundClause >> isCompoundPredicate [
  ^ true

]

{ #category : 'testing' }
GsCompoundClause >> isConjunctiveNormalForm [
  ^ self operator == #'&'
    and: [
      self clause1 isConjunctiveNormalForm
        and: [ self clause2 isConjunctiveNormalForm ] ]

]

{ #category : 'testing' }
GsCompoundClause >> isDisjunctiveClause [
  ^ self operator == #'|'

]

{ #category : 'testing' }
GsCompoundClause >> isDisjunctiveNormalForm [
  ^ self operator == #'|'
    and: [
      self clause1 isDisjunctiveNormalForm
        and: [ self clause2 isDisjunctiveNormalForm ] ]

]

{ #category : 'operators' }
GsCompoundClause >> normalize [
  | normalized |
  normalized := super normalize.
  normalized clause1: clause1 normalize.
  normalized clause2: clause2 normalize

]

{ #category : 'accessing' }
GsCompoundClause >> operator [
  ^operator

]

{ #category : 'accessing' }
GsCompoundClause >> operator: aSymbol [
  operator := aSymbol

]

{ #category : 'accessing' }
GsCompoundClause >> operatorNegated [
  ^ operator == #'&'
    ifTrue: [ #'|' ]
    ifFalse: [ #'&' ]

]

{ #category : 'private' }
GsCompoundClause >> postCopy [
  clause1 := clause1 copy.
  clause2 := clause2 copy.
  super postCopy

]

{ #category : 'printing' }
GsCompoundClause >> printOn: aStream [
  | parenthesizeClause |
  aStream nextPutAll: self clause1 printString.
  aStream nextPutAll: ' ' , self operator asString , ' '.
  parenthesizeClause := self clause2 isCompoundPredicate.
  parenthesizeClause
    ifTrue: [ aStream nextPut: $( ].
  aStream nextPutAll: self clause2 printString.
  parenthesizeClause
    ifTrue: [ aStream nextPut: $) ]

]

{ #category : 'accessing' }
GsCompoundClause >> resultOperatorFor: anOperator [
  ^ anOperator == #'&'
    ifTrue: [
      "intersection"
      #'*' ]
    ifFalse: [
      "minimal union"
      #'_union:' ]

]

{ #category : 'transforming' }
GsCompoundClause >> unbind [
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound clause1: unbound clause1 unbind.
  unbound clause2: unbound clause2 unbind.
  ^ unbound immediateInvariant

]

{ #category : 'testing' }
GsCompoundClause >> usesPathEvaluator [
  ^ self clause1 usesPathEvaluator and: [ self clause2 usesPathEvaluator ]

]

{ #category : 'testing' }
GsCompoundClause >> usesPathEvaluatorForFirstClause [
  ^ self clause1 usesPathEvaluatorForFirstClause

]
