"
A GsDigitalEnvelope is a class used to encrypt data with one or more public 
 encryption keys such that the data may only be decrypted with one of the 
 matching private keys. Only RSA encryption key pairs are supported. 

 GsDigitalEnvelopes are also protected by a cryptographic signature which
 is generated using the sender's signing key. The signature allows the
 receiver to guarantee the envelope was created by someone with access to
 the sender's private signing key. For signing algorithms that require
 a message digest algorithm, the SHA2-256 message digest is used.
 Signatures generated with RSA keys are padded using the
 RSA_PKCS1_PSS_PADDING scheme. The following signing key types are supported:

 -DSA
 -ECDSA
 -RSA
 -EC
 -Ed25519
 -Ed448

 Using different key pairs for signing and encrypting is strongly recommended.
 
 A GsDigitalEnvelope has the following security features:
 
  -Confidentiality - the message is encrypted using a randomly generated
                     AES encryption session key and initialization vector.
		     The session key is then encrypted with the provided
		     public encryption key.
		     
  -Integrity -       authenticated encryption guarantees the cipher text
                     has not been alter. The digital signature
                     guarantees the encrypted key, initialization vector
		     and tag have not been altered.
	       
  -Authentication -  the receiver of the envelope is assured the sender
                     signed the envelope with the private key matching
		     the public key used to successfully verify the
		     signature.
		    
 IMPORTANT - in order to guarantee authentication, the receiver must confirm
             the public verification key actually belongs to the sender.
	     Normally confirmation is done by verifying an X509 certificate
	     containing the sender's public key has been signed by a
	     reputable certificate authority. GsDigitalEnvelope does NOT
	     do this public key/certificate signature verification. It is
	     up to the envelope recipient to ensure the public key or
	     X509 certificate used to verify the signature is trustworthy.

 The message to be encrypted and must be a byte collection. Byte 
 collections with a character size greater than one are accepted and will
 be converted to big endian format if necessary before encryption.

 If the message is intended for more than one recipient, multiple instances
 of GsDigitalEnvelope may be created by a single encryption operation by
 supplying multiple public encryption keys, one for each recipient. A maximum
 of 16 GsDigitalEnvelopes may be created by a single encryption operation.
  
 cipherId determines which AEAD (Authenticated Encryption with Additional
 Data) cipher to use from the following list:
 
 ================================
                        Key Size    
 opCode Cipher   Mode  bits/Bytes
 ================================
   4     AES     OCB     128/16  
   5     AES     OCB     192/24  
   6     AES     OCB     256/32  
   7     AES     GCM     128/16  
   8     AES     GCM     192/24  
   9     AES     GCM     256/32  
  10   CHACHA20 Poly1305 256/32  
 ================================

 AEAD guarantees the cipher text has not been altered. Modes 6 and 10 are
 thought to be the most secure and are recommended for most applications. 

 Constraints:
	publicEncryptionKey:	  GsTlsPublicKey or GsX509Certificate
	encryptedKey:   	  ByteArray
	cipherText:	  	  ByteArray
	initVector:	  	  ByteArray
	cipherId:	  	  SmallInteger
	tag:		  	  ByteArray
	messageClassName: 	  String
	digitalSignature:	  ByteArray or nil

"
Class {
	#name : 'GsDigitalEnvelope',
	#superclass : 'Object',
	#instVars : [
		'publicEncryptionKey',
		'encryptedKey',
		'cipherText',
		'initVector',
		'cipherId',
		'tag',
		'messageClassName',
		'digitalSignature'
	],
	#gs_reservedoop : '250369',
	#category : nil
}

{ #category : 'Private' }
GsDigitalEnvelope class >> _primEncryptMessage: bytes withPublicEncryptionKeys: arrayOfPubKeysOrCerts
cipherId: anInt withPrivateSigningKey: aSigningKey [

"A GsDigitalEnvelope is a class used to encrypt data with one or more public
 encryption keys such that the data may only be decrypted with one of the
 matching private keys. Only RSA public/private key pairs are supported.

 To create a secure digital envelope, a random session key and salt are
 created. The session key is then encrypted with each public key provided.

 bytes is message to be encrypted and must be a byte collection. Byte
 collections with a character size greater than one are accepted and will
 be converted to big endian format if necessary before encryption.

 arrayOfPubKeysOrCerts is an array containing at least 1 and no more than 16
 instances of GsTlsPublicKey or GsX509Certificate.

 cipherId determines which AEAD (Authenticated Encryption with Additional
 Data) cipher to use from the following list:

 ================================
                        Key Size
 opCode Cipher   Mode  bits/Bytes
 ================================
   4     AES     OCB     128/16
   5     AES     OCB     192/24
   6     AES     OCB     256/32
   7     AES     GCM     128/16
   8     AES     GCM     192/24
   9     AES     GCM     256/32
  10   CHACHA20 Poly1305 256/32
 ================================

 AEAD guarantees the message has not been altered. Modes 6 and 10 are thought
 to be the most secure and are recommended for most applications.

 aSigningKey must be an instance of GsTlsPrivateKey valid for generating
 digital signatures.

 Returns an Array of GsDigitalEnvelopes (one for each element of arrayOfKeys)
 upon success or raises an exception on error."

<primitive: 1088>
bytes _validateIsBytes .
arrayOfPubKeysOrCerts _validateClass: Array .
arrayOfPubKeysOrCerts size == 0
  ifTrue:[ arrayOfPubKeysOrCerts _error: #objErrCollectionEmpty].
arrayOfPubKeysOrCerts size > 16
  ifTrue:[ arrayOfPubKeysOrCerts _error: #errArgTooLarge args: {16 } ].
arrayOfPubKeysOrCerts do:[:e|
  e _validateClasses: { GsTlsPublicKey . GsX509Certificate } ; _validateIsRsa ] .
anInt _validateClass: SmallInteger .
((anInt < 4) or:[anInt > 10])
  ifTrue:[ anInt _error: #rtErrArgOutOfRange args:{ 4 . 10 } ] .
aSigningKey _validateClass: GsTlsPrivateKey ;
            _validateCreatesDigitalSignatures .
^ self _primitiveFailed:
  #_primEncryptMessage:withPublicEncryptionKeys:cipherId:withPrivateSigningKey:

]

{ #category : 'Examples' }
GsDigitalEnvelope class >> decryptExampleFromString: aPassiveObjString [

"Takes a String produced by the encryptExample method and decodes the message."
| envelope privateKey msg verifyKey |

privateKey := GsTlsPrivateKey newFromPemString:
'-----BEGIN PRIVATE KEY-----
MIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBAMBXQWaLW7QhuWvO
17X5c8LYDbw65if4QR2ATaNZ0Cq3Q6Q+aCsnmT1Q3AvKwUx0Y3PnASLLl0zNQN94
nbNP81vT/0D4J0Ch1HwGUErZReLDN2DSoUELdnmbrgR10Glriy3HD6wb0q+h/N45
8JpR6sTYhVlLtvnGYAJL2OW5kVRVAgMBAAECgYEAl6My+Hld7wG3gXstLVZhIXfc
PE3jLhfWnj+M9f/U0hhxx4c78OnjMigRk2piQrhvv+ybRKdlvTMEtioNilS58ogV
/I5dRoHsRd2opsUeDMloRdOMcL6HhinjGtPFqY/QXdeKKLLAfR2Mw1GKaro55hQv
DRqRk01Gd/KvWij5roECQQDz+9VW54+qolrDH2iw0BBeeYBog/ELA8vNw7te4OWH
0TrPHUDyvHkJCQ/GSWHLVQ2Rw/WoyKMTn7u/LF8pspp9AkEAydBN8IPdID8m5rk8
JYr1iPceAyoI5ZeUA1cqrFjx4HdtyVAuLGQAvVSY7fJaSzlrBeH8HVa3GlDJ3Qr1
Tt1wuQJAJnbhX14KTEBkRrbA7n8e1YYaNF/4tF/Y1YuyEncqOItH1jcqcho8iqwf
DIetHz09cmmOZRmcfA+GrdD0/8HkkQJANUUxvYHhFYj16MMOWE6Uv0GTf3xR+uCG
5lbU4cdcmUaNCS2L8pW3CELTV0O4h9CxKk1bchcYn+6hSiKBW/7hqQJBAKcucW2t
HDhYqOhZi+Eq+z7mnua7x867COGlijb3yE80rvmqBOOZc3PtJNr2SKQoBjD++B4w
HSedUWX12Gyb0SE=
-----END PRIVATE KEY-----' .

verifyKey := GsTlsPublicKey newFromPemString: '-----BEGIN PUBLIC KEY-----
MIGdMAsGCSqGSIb3DQEBCgOBjQAwgYkCgYEAwPc8s5Ox0/BOYYmH/UGHVXlcvsD2
dKu+9Plm2np4BZfjvitYEsobh+sOPJEauzx1UPxRb9vK4dk7NxBHKuHssjfDdD2A
lPUXoaGd3NzUWUHdznLwkkcbHHTbS/xiPT/FkxYM13yzUz3te0q0rQZiktpzQ8J2
qVo7m9joI7X5xVECAwEAAQ==
-----END PUBLIC KEY-----' .

envelope := (PassiveObject newWithContents: aPassiveObjString) activate.
msg := envelope decryptWithPrivateKey: privateKey
                withPublicVerificationKey: verifyKey .
^ msg

]

{ #category : 'Decrypting' }
GsDigitalEnvelope class >> decryptMessage: aMsg withPrivateDecryptionKey: aGsTlsPrivateKey encryptedKey: aKey
initVector: anIv tag: aTag cipherId: anInt messageClassName: aString
digitalSignature: aSig withPublicVerificationKey: aPublicKey [

"Decrypts the message and stores it in a new instance of the class with name
 aString, which must be a byte object class name.
 See the class comments for more information on GsDigitalEnvelopes."

^ self new encryptedKey: aKey ;
       	   cipherText: aMsg ;
	   initVector: anIv ;
	   cipherId: anInt ;
	   tag: aTag ;
	   messageClassName: aString ;
           digitalSignature: aSig ;
	   decryptWithPrivateKey: aGsTlsPrivateKey
           withPublicVerificationKey: aPublicKey

]

{ #category : 'Examples' }
GsDigitalEnvelope class >> encryptExampleWithMessage: msg [

"Create a GsDigitalEnvelope and passivate it to a string for transmission"

| publicKey envelope stream aSigningKey |

"Create RSA public key"
publicKey := GsTlsPublicKey newFromPemString:
'-----BEGIN PUBLIC KEY-----
MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDAV0Fmi1u0Iblrzte1+XPC2A28
OuYn+EEdgE2jWdAqt0OkPmgrJ5k9UNwLysFMdGNz5wEiy5dMzUDfeJ2zT/Nb0/9A
+CdAodR8BlBK2UXiwzdg0qFBC3Z5m64EddBpa4stxw+sG9KvofzeOfCaUerE2IVZ
S7b5xmACS9jluZFUVQIDAQAB
-----END PUBLIC KEY-----' .

aSigningKey := GsTlsPrivateKey newFromPemString:
'-----BEGIN PRIVATE KEY-----
MIICdQIBADALBgkqhkiG9w0BAQoEggJhMIICXQIBAAKBgQDA9zyzk7HT8E5hiYf9
QYdVeVy+wPZ0q770+WbaengFl+O+K1gSyhuH6w48kRq7PHVQ/FFv28rh2Ts3EEcq
4eyyN8N0PYCU9RehoZ3c3NRZQd3OcvCSRxscdNtL/GI9P8WTFgzXfLNTPe17SrSt
BmKS2nNDwnapWjub2OgjtfnFUQIDAQABAoGAVWbWwa9zO5aWSgrBWe+/gq/EwVPL
f9VnHSqoP7eGGQuhKtAqGZ7DUoNQeLPLveRDE8WoETaYcx5eW79jj/IPfAfKbgol
wrd+AxSGRcsK/Gskh6Drt84kCcAoXmhO/s0nfO8HgJ4cvyXWQw9l/ig8KFIl0bFd
M059mBIApIgKrTECQQDouAH+3oYk2hZ8Nlx/WmulW5JDqLAf7FR6PnclHnkM0/mC
GnYEuHlPzUakWepsaTUY8LCl3GpwfDx9U2NTt4hNAkEA1EUkzLd/YPs+9GzkL2W4
ot0sC/JujzDCWKB2d3RIRVo0pbW2pwUEQTy7f1fFpPo3qzJP1DrP+yLCAp3qxPVz
FQJBAJlEJrxOnZZDs69WthCB4odjCa9Zt7Uulmx0G0/tA9g4+wh+mN9/BxZRoYa4
WTXRDFFCo3R49/jhOY1oj/Ag3bkCQA0kwuSSMCb3J6zG2VI+ADLFcybCOipPoJkQ
RoWbA6aXsU7Zc5ff7aWEdy+pZamTfMLy+JJxmdM5Eb5LvO+5KwkCQQCWTte4IvnU
AkH1J5dEDWvpXXX/9NiTxCvdKFI2PJWxpsq6gSkoDwF2Vghe2a+ca1mQB6/HCftp
Ux5Vxey5pUsm
-----END PRIVATE KEY-----' .

"Create envelope"
envelope := GsDigitalEnvelope encryptMessage: msg
	       			 withPublicEncryptionKey: publicKey
				 cipherId: 10
                                 withPrivateSigningKey: aSigningKey.

stream := WriteStream on: String new.
"Convert to passive object format"
PassiveObject passivate: envelope toStream: stream .

"Ready for transport"
^ stream contents

]

{ #category : 'Encrypting' }
GsDigitalEnvelope class >> encryptMessage: bytes withPublicEncryptionKey: aGsTlsPublicKey cipherId: anInt
withPrivateSigningKey: aSigningKey [

"Encrypts the message in bytes and stores it into an instance of GsDigitalEnvelope.
 Returns the new instance or raises an exception on error.
 See the class comments for more information on GsDigitalEnvelopes."

^ (self encryptMessage: bytes withPublicEncryptionKeys: { aGsTlsPublicKey }
cipherId: anInt withPrivateSigningKey: aSigningKey) first

]

{ #category : 'Encrypting' }
GsDigitalEnvelope class >> encryptMessage: bytes withPublicEncryptionKeys: arrayOfPubKeysOrCerts
cipherId: anInt withPrivateSigningKey: aSigningKey [
"See the class comments for more information on GsDigitalEnvelopes."

^ self _primEncryptMessage: bytes withPublicEncryptionKeys: arrayOfPubKeysOrCerts
       cipherId: anInt withPrivateSigningKey: aSigningKey

]

{ #category : 'Examples' }
GsDigitalEnvelope class >> example [

| msg passiveString decryptedMsg |
msg := String withAll:
'Once upon a midnight dreary, while I pondered, weak and weary,
Over many a quaint and curious volume of forgotten lore-
While I nodded, nearly napping, suddenly there came a tapping,
As of some one gently rapping, rapping at my chamber door.
"Tis some visitor, " I muttered, "tapping at my chamber door-
Only this and nothing more."' .

passiveString := GsDigitalEnvelope encryptExampleWithMessage: msg .
decryptedMsg := GsDigitalEnvelope decryptExampleFromString: passiveString .
^ decryptedMsg = msg

]

{ #category : 'Private' }
GsDigitalEnvelope >> _primDecryptWithPrivateKey: aGsTlsPrivateKey into: aCharColl
withPublicVerificationKey: aPublicKey [

<primitive: 1090>

"We don't care about publicEncryptionKey inst var for decryption"
aGsTlsPrivateKey _validateClass: GsTlsPrivateKey ; _validateIsRsa .
aPublicKey _validateClasses: { GsTlsPublicKey . GsX509Certificate } ;
           _validateValidatesDigitalSignatures .
aCharColl    _validateIsBytes .
encryptedKey _validateClass: ByteArray ; _validateNotEmpty .
cipherText   _validateClass: ByteArray ; _validateNotEmpty .
initVector   _validateClass: ByteArray ; _validateNotEmpty .
tag          _validateClass: ByteArray ; _validateNotEmpty .
digitalSignature _validateClass: ByteArray ; _validateNotEmpty .
cipherId     _validateClass: SmallInteger .
((cipherId < 4) or:[cipherId > 10])
  ifTrue:[ cipherId _error: #rtErrArgOutOfRange args:{ 4 . 10 } ] .
^ self _primitiveFailed:
  #_primDecryptWithPrivateKey:into:withPublicVerificationKey:

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> cipherId [
  ^cipherId

]

{ #category : 'Updating' }
GsDigitalEnvelope >> cipherId: anInt [
  cipherId := anInt

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> cipherText [
  ^cipherText

]

{ #category : 'Updating' }
GsDigitalEnvelope >> cipherText: aByteArray [
  cipherText := aByteArray

]

{ #category : 'Decrypting' }
GsDigitalEnvelope >> decryptWithPrivateKey: aGsTlsPrivateKey withPublicVerificationKey: aPublicKey [
"Decrypts the receiver using aGsTlsPrivateKey and stores it
 a new instance of the class name contained in messageClassName.

 All inst vars except for #publicEncryptionKey must be set correctly before
 invoking this method.

 Returns a new instance of messageClassName on success or raises
 an exception on error."
 | cls |
 cls := GsCurrentSession currentSession  objectNamed: self messageClassName  .
 ^ cls
      ifNil:[ self _error: #rtErrObjNotFound
                      args: { self messageClassName } ]
   ifNotNil:[ self _primDecryptWithPrivateKey: aGsTlsPrivateKey
                   into: cls basicNew
                   withPublicVerificationKey: aPublicKey ]

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> digitalSignature [
  ^digitalSignature

]

{ #category : 'Updating' }
GsDigitalEnvelope >> digitalSignature: aSig [
  digitalSignature := aSig

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> encryptedKey [
 ^encryptedKey

]

{ #category : 'Updating' }
GsDigitalEnvelope >> encryptedKey: aByteArray [
 encryptedKey := aByteArray

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> initVector [
  ^initVector

]

{ #category : 'Updating' }
GsDigitalEnvelope >> initVector: aByteArray [
  initVector := aByteArray

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> messageClassName [
  ^messageClassName

]

{ #category : 'Updating' }
GsDigitalEnvelope >> messageClassName: aString [
  messageClassName := aString

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> publicEncryptionKey [
 ^publicEncryptionKey

]

{ #category : 'Updating' }
GsDigitalEnvelope >> publicEncryptionKey: aGsTlsPublicKey [
  publicEncryptionKey := aGsTlsPublicKey

]

{ #category : 'Accessing' }
GsDigitalEnvelope >> tag [
  ^tag

]

{ #category : 'Updating' }
GsDigitalEnvelope >> tag: aByteArray [
  tag := aByteArray

]
