"
GsFileIn supports filing in from topaz-format GemStone source files into the image, without the use of topaz.
	
	To use, send one of the from* methods.  This also performs the file in.
	  for example,
	     GsFileIn fromServerPath: 'mySourceCode.gs'
	
	Only a subset of topaz commands is supported:
	   doit , printit , run , nbrun, send
	   input
	   category:
	   classmethod , classmethod:
	   method , method:
	   removeallmethods , removeallclassmethods
	   trclassmethod , trclassmethod:
	   trmethod , trmethod:
	   trremoveallmethods , trremoveallclassmethods
	   commit, abort
	   env N
	   fileformat
	   set compile_env N , set class , set category, set trait
 	   set enableremoveall, set package , set project 
     	   set sourcestring class
	     (other set commands are ignored)

Following can be read but are ignored:
	   expectvalue, expecterror, iferr, iferr_clear, iferr_list, errorcount
	   fileout, output
	   display, omit, level, limit, list
	   time, remark, status
	   login, logout

Other topaz commands will error.
"
Class {
	#name : 'GsFileIn',
	#superclass : 'Object',
	#instVars : [
		'session',
		'stream',
		'line',
		'lineNum',
		'path',
		'currentClassName',
		'currentClassObj',
		'currentTraitName',
		'currentTraitObj',
		'remoteGsFileInClassOop',
		'category',
		'compileEnvironment',
		'clearTopazSessionState',
		'fileFormat',
		'sourceStringClass',
		'removeAll',
		'clientFiles'
	],
	#category : 'Kernel'
}

{ #category : 'private' }
GsFileIn class >> _fromPath: aString type: aFileType sourceClass: aClass [
  "Returns an Array { aStream . clientFilesBoolean } "
	| aFile fBytes aStream isClient fmt inst |
  fmt := (aFileType == #clientUtf8File or:[ aFileType == #serverUtf8File])  
         ifTrue:[ #utf8 ] ifFalse:[ #'8bit' ].
  (aFileType == #clientText or:[ aFileType == #clientUtf8File]) ifTrue:[
    isClient := true .
    aFile := GsFile open: aString mode: 'r' onClient: true .
  ].
	(aFileType == #serverText or:[ aFileType == #serverUtf8File]) ifTrue:[
    isClient := false .
		aFile := GsFile open: aString mode: 'r' onClient: false .
  ].
  isClient ifNil:[
     ArgumentError signal:'unknown file type ', aFileType printString .
  ].
  (aClass == Unicode16 or:[ aClass == String]) ifFalse:[
    ArgumentError signal:'source class is neither Unicode16 nor String, ', aClass name .
  ].
  aFile ifNil:[ ^ Error signal:'File open failed, path ', aString printString,
      '; ', GsFile lastErrorString ].
  fBytes := aFile contents .
  fBytes ifNil:[ ^ Error signal:'File read failed, ', aFile lastErrorString ].
  aFile close .
  (aFileType == #clientUtf8File or:[ aFileType == #serverUtf8File]) ifTrue:[
    aClass == Unicode16 ifTrue:[ fBytes := fBytes decodeFromUTF8ToUnicode ]
                       ifFalse:[ fBytes := fBytes decodeFromUTF8ToString ].
  ] ifFalse:[
    aClass == Unicode16 ifTrue:[ 
      fBytes := Unicode7 withAll: fBytes .
      fBytes class ~~ Unicode7 ifTrue:[ 
        Error signal: 'File contains code points > 127, and utf8 not specified'.
      ]. 
    ].
  ].
	aStream := ReadByteStreamPortable on: fBytes .
  (inst := self new)
     path: aString ; stream: aStream ; clientFiles: isClient ;
     sourceStringClass: aClass ;
     fileFormat: fmt .
  ^ inst
]

{ #category : 'private' }
GsFileIn class >> _remoteCompile: srcUtf8 categ: catUtf8 class: aClass unicode: uBool env: envId classMeth: clsMethBool [
  | src cat cls |
  uBool ifTrue:[
    src := srcUtf8 decodeToUnicode .
    cat := catUtf8 decodeToUnicode .
  ] ifFalse:[ 
    src := srcUtf8 decodeToString .
    cat := catUtf8 decodeToString .
  ].
  cls := clsMethBool ifTrue:[ aClass class ] ifFalse:[ aClass ] .
  cls compileMethod: src category: cat environmentId: envId
]

{ #category : 'private' }
GsFileIn class >> _remoteTraitCompile: srcUtf8 categ: catUtf8 trait: aTrait unicode: uBool classMeth: clsMethBool [
  | src cat trait |
  uBool ifTrue:[
    src := srcUtf8 decodeToUnicode .
    cat := catUtf8 decodeToUnicode .
  ] ifFalse:[ 
    src := srcUtf8 decodeToString .
    cat := catUtf8 decodeToString .
  ].
  trait := clsMethBool ifTrue:[ aTrait classTrait ] ifFalse:[ aTrait ] .
  trait compile: src category: cat
]

{ #category : 'private' }
GsFileIn class >> _remoteUnicodeExecute: aUtf8 [
  | src |
  src := aUtf8 decodeToUnicode .
  ^ src evaluate 
]

{ #category : 'private' }
GsFileIn class >> _removeAllClassMethods: aClass [
  aClass class removeAllMethods .
]

{ #category : 'private' }
GsFileIn class >> _resolveClass: aString [
  | assoc sym cls |
  sym := Symbol _existingWithAll: aString . 
  sym ifNil:[ ^ self error: 'invalid class name ', aString printString ].
  assoc := GsCurrentSession currentSession resolveSymbol: sym .
  assoc ifNil:[ ^ self error: 'class name not found ' , aString printString ].
  cls := assoc _value .
  cls isBehavior ifFalse:[ ^ self error: 'value of ', aString printString,' is not a Behavior'].
  cls isMeta ifTrue:[ ^ self error: 'value of ', aString printString,' is a Metaclass'].
  ^ cls
]

{ #category : 'private' }
GsFileIn class >> _resolveTrait: aString [
  | assoc sym trait |
  sym := Symbol _existingWithAll: aString . 
  sym ifNil:[ ^ self error: 'invalid trait name ', aString printString ].
  assoc := GsCurrentSession currentSession resolveSymbol: sym .
  assoc ifNil:[ ^ self error: 'class trait not found ' , aString printString ].
  trait := assoc _value .
  trait isTrait ifFalse:[ ^ self error: 'value of ', aString printString,' is not a Trait'].
  ^ trait
]

{ #category : 'private' }
GsFileIn class >> _traitRemoveAllClassMethods: aTrait [
  aTrait classTrait removeAllMethods .
]

{ #category : 'File in' }
GsFileIn class >> fromClientPath: aString [
"file in from path on session client machine into current session"
	self
		fromGciHostPath: aString
		to: nil.
]

{ #category : 'Deprecated' }
GsFileIn class >> fromClientPath: aString to: anExternalSession [
	self deprecated: 'GsFileIn class >> fromClientPath:to: deprecated v3.7.  Use fromGciHostPath:to:'.
	self
		fromPath: aString
		on: #clientText
		to: anExternalSession.
]

{ #category : 'File in' }
GsFileIn class >> fromGciHostPath: aString [

	self
		fromGciHostPath: aString
		to: nil.
]

{ #category : 'File in' }
GsFileIn class >> fromGciHostPath: aString to: anExternalSession [

	self
		fromPath: aString
		on: #clientText
		to: anExternalSession.
]

{ #category : 'File in' }
GsFileIn class >> fromGemHostPath: aString [

	self
		fromGemHostPath: aString
		to: nil.
]

{ #category : 'File in' }
GsFileIn class >> fromGemHostPath: aString to: anExternalSession [

	self
		fromPath: aString
		on: #serverText
		to: anExternalSession.
]

{ #category : 'File in' }
GsFileIn class >> fromPath: aString on: aFileType to: anExternalSession [
  ^ self fromPath: aString on: aFileType to: anExternalSession 
       sourceClass: StringConfiguration"from Globals" .
]

{ #category : 'File in' }
GsFileIn class >> fromPath: aString on: aFileType to: anExternalSession sourceClass: aClass [
"aString is file path to file containg topaz file-out format of smalltalk code.
type is one of #clientFile #serverFile  #clientUtf8File #serverUtf8File 
to denote relative user action to use.
If anExternalSession is not nil, it must be a GsTsExternalSession and file-in executes 
into that session"

	(self _fromPath: aString type: aFileType sourceClass: aClass)
      		session: anExternalSession; 
		doFileIn	
]

{ #category : 'File in' }
GsFileIn class >> fromServerPath: aString [
	self
		fromGemHostPath: aString
		to: nil.
]

{ #category : 'Deprecated' }
GsFileIn class >> fromServerPath: aString to: anExternalSession [
	self deprecated: 'GsFileIn class >> fromServerPath:to: deprecated v3.7, use fromGemHostPath:to:'.
	self
		fromPath: aString
		on: #serverText
		to: anExternalSession.
]

{ #category : 'File in' }
GsFileIn class >> fromStream: aStream [
  ^ (self newFromStream: aStream) doFileIn

]

{ #category : 'Instance Creation' }
GsFileIn class >> new [
  | inst |
	(inst := super new) initialize .
  ^ inst
]

{ #category : 'Instance Creation' }
GsFileIn class >> newFromStream: aStream [
	| inst srcCls species |
	species := aStream collectionSpecies.
	(species inheritsFrom: CharacterCollection)
		ifTrue: [ srcCls := species ]
		ifFalse: [ 
			species == UndefinedObject
				ifTrue: [ srcCls := StringConfiguration ]
				ifFalse: [ 
					ArgumentError
						signal: 'expected a string or unicode string, got a ' , species name ] ].
	(inst := self new)
		stream: aStream;
		clientFiles: false;
		sourceStringClass: srcCls;
		fileFormat: #'utf8'.
	^ inst
]

{ #category : 'accessors' }
GsFileIn >> _clearTopazSessionState [
	^ clearTopazSessionState == true
]

{ #category : 'accessors' }
GsFileIn >> _clearTopazSessionState: aBool [
  clearTopazSessionState := aBool .
]

{ #category : 'private' }
GsFileIn >> _defaultCategory [
	^ 'as yet unspecified'
]

{ #category : 'private' }
GsFileIn >> _nestedType [
  ^ clientFiles ifTrue:[
     fileFormat == #utf8 ifTrue:[ #clientUtf8File ] ifFalse:[ #clientText ]
   ] ifFalse:[
    fileFormat == #utf8 ifTrue:[ #serverUtf8File ] ifFalse:[ #serverText ]
   ]
]

{ #category : 'private' }
GsFileIn >> _remoteMethod: srcArg classMethod: clsMethBool [
  | remoteSrcOop remoteCategOop sess |
  sess := session .
	remoteSrcOop := sess newUtf8String: srcArg encodeAsUTF8 toUnicode: false. 
	remoteCategOop := sess newUtf8String: category encodeAsUTF8 toUnicode: false. 
	[ sess send: '_remoteCompile:categ:class:unicode:env:classMeth:'
          to: remoteGsFileInClassOop
          withOops: { remoteSrcOop . remoteCategOop . currentClassObj . 
                   (sourceStringClass == Unicode7) asOop .  
                   compileEnvironment asOop . clsMethBool asOop }.
  ] ensure:[
	  sess releaseOop: remoteSrcOop ; releaseOop: remoteCategOop .
  ]
]

{ #category : 'private' }
GsFileIn >> _remoteTraitMethod: srcArg classMethod: clsMethBool [
  | remoteSrcOop remoteCategOop sess |
  sess := session .
	remoteSrcOop := sess newUtf8String: srcArg encodeAsUTF8 toUnicode: false. 
	remoteCategOop := sess newUtf8String: category encodeAsUTF8 toUnicode: false. 
	[ sess send: '_remoteTraitCompile:categ:trait:unicode:classMeth:'
          to: remoteGsFileInClassOop
          withOops: { remoteSrcOop . remoteCategOop . currentTraitObj . 
                   (sourceStringClass == Unicode7) asOop .  
                    clsMethBool asOop }.
  ] ensure:[
	  sess releaseOop: remoteSrcOop ; releaseOop: remoteCategOop .
  ]
]

{ #category : 'private' }
GsFileIn >> _resolveClass: aString [ 
  session ifNil:[  
    ^ self class _resolveClass: aString 
  ] ifNotNil:[:sess| | strOop res |
    strOop := sess newUtf8String: aString encodeAsUTF8 toUnicode: true .
    res := sess send: '_resolveClass:' to: remoteGsFileInClassOop withOops: { strOop }.
    ^ (res at: 1) "anOop"
  ]
]

{ #category : 'private' }
GsFileIn >> _resolveTrait: aString [ 
  session ifNil:[  
    ^ self class _resolveTrait: aString 
  ] ifNotNil:[:sess| | strOop res |
    strOop := sess newUtf8String: aString encodeAsUTF8 toUnicode: true .
    res := sess send: '_resolveTrait:' to: remoteGsFileInClassOop withOops: { strOop }.
    ^ (res at: 1) "anOop"
  ]
]

{ #category : 'private' }
GsFileIn >> _sendIsLegal [
  ^ true

]

{ #category : 'private' }
GsFileIn >> _setClass: aString [
  | cls |
  cls := self _resolveClass: aString .
  currentClassObj := cls . "an object or oop of a remote object"
  self _setCurrentClassName: aString . 
]

{ #category : 'private' }
GsFileIn >> _setCurrentClassName: aString [
	currentClassName := aString
]

{ #category : 'private' }
GsFileIn >> _setCurrentTraitName: aString [
	currentTraitName := aString
]

{ #category : 'private' }
GsFileIn >> _setTrait: aString [
  | trait |
  trait := self _resolveTrait: aString .
  currentTraitObj := trait . "an object or oop of a remote object"
  self _setCurrentTraitName: aString .
]

{ #category : 'processing' }
GsFileIn >> abort [
  ^ self abortTransaction
]

{ #category : 'processing' }
GsFileIn >> abortTransaction [
	session ifNotNil:[:sess | sess abort ] ifNil:[ System abortTransaction ]
]

{ #category : 'processing' }
GsFileIn >> category [
  | cat words |
  words := self words: line .
  ((cat := words atOrNil: 2) codePointCompareTo: ':') == 0 ifTrue:[
    cat := words atOrNil:3 .
  ].
  cat ifNil:[ Error signal:'missing argument to CATEGORY'].
	self currentCategory: cat
]

{ #category : 'processing' }
GsFileIn >> classMethod [
  self parseClassmethodLine ; 
        classMethodBody .
]

{ #category : 'processing' }
GsFileIn >> classMethodBody [
  | src |
	currentClassObj ifNil: [self error: 'current class not defined'].
  src := self nextChunk .
  session ifNil:[
    currentClassObj class compileMethod: src 
       category: category 
       environmentId: compileEnvironment.
  ] ifNotNil:[ 
    self _remoteMethod: src classMethod: true 
  ]
]

{ #category : 'accessors' }
GsFileIn >> clientFiles: aBoolean [
  clientFiles := aBoolean .
]

{ #category : 'processing' }
GsFileIn >> commit [
  ^ self commitTransaction
]

{ #category : 'processing' }
GsFileIn >> commitTransaction [
	session ifNotNil:[:sess | sess commit ] ifNil:[ System commit ]
]

{ #category : 'processing' }
GsFileIn >> compileEnvironment: arg [
  | env |
	arg isDigits ifFalse:[ ^ self error: 'ENV  only accepts integers'].
  env := Integer fromString: arg .
  (env < 0 or:[ env > 255]) ifTrue:[ self error:'arg to ENV must be in range 0..255' ].
  compileEnvironment := env .
]

{ #category : 'processing' }
GsFileIn >> currentCategory: cat [
  "implementation of SET CATEGORY"
  category := cat .
	cat isEmpty ifTrue: [self error: 'category is empty'].
	cat first = $' ifTrue: [
		cat last = $' ifFalse:[
      self error: 'category begins with quote but does not end with quote'
    ].
		category := cat copyFrom: 2 to: category size - 1.
	] ifFalse:[
    cat last = $' ifTrue:[
       self error: 'category ends with quote but does not start with quote'.
    ].
  ].
	(category includes: $') ifTrue:[ category := cat evaluate ].
]

{ #category : 'processing' }
GsFileIn >> currentClass [
  "Returns a String (a class name) "
	^ currentClassName
]

{ #category : 'processing' }
GsFileIn >> currentClass: aClassName [
    "The current category is cleared by the LOGOUT, LOGIN, SET CLASS,
    and SET SESSION commands. "

	self _setClass: aClassName .
	category := self _defaultCategory.
]

{ #category : 'processing' }
GsFileIn >> currentTrait: aTraitName [
	self _setTrait: aTraitName .
]

{ #category : 'processing' }
GsFileIn >> doFileFormat: aString [
  | arg |
  aString ifNil:[ Error signal:'missing argument to FILEFORMAT'].
  arg := aString asLowercase .
  (arg codePointCompareTo: 'utf8') == 0 ifTrue:[
    fileFormat == #utf8 ifFalse:[ | bytes str |
      bytes := stream upToEnd . 
      bytes charSize > 1 ifTrue:[ Error signal:'Stream aleady decoded to ', bytes class name].
      str := sourceStringClass == Unicode7 
              ifTrue:[ bytes decodeFromUTF8ToUnicode ]
              ifFalse:[ bytes decodeFromUTF8ToString ].
      stream := ReadByteStreamPortable on: str .
      fileFormat := #utf8 .
    ].
    ^ self
  ].
  (arg codePointCompareTo: '8bit') == 0 ifTrue:[
    fileFormat == #'8bit' ifFalse:[ | bytes str |
      bytes := stream upToEnd encodeAsUTF8 .
      str := bytes decodeFromUTF8ToString .
      str class == String ifFalse:[
        Error signal:'for FILEFORMAT 8BIT, remaining bytes have codePoint(s) above 255' .
      ].
      stream := ReadByteStreamPortable on: str .
      fileFormat := #'8bit' .
    ].
    ^ self
  ].
  ^ self error:'unrecognized fileformat ', aString printString  .
]

{ #category : 'processing' }
GsFileIn >> doFileIn [
 [ lineNum := 0 .
	 [
		 stream atEnd not.
	 ] whileTrue: [
		 line := stream nextLine trimSeparators.
     lineNum := lineNum + 1 .
		 [
       self processLine
     ] on: Error do:[:ex | | loc |
       loc := ', at line ', lineNum asString .
       path ifNotNil:[ loc add:' file ', path asString ].
       ex messageText ifNil:[ ex messageText: loc ]
                   ifNotNil:[:str | ex messageText:(str copy addAll: loc; yourself)].
       ex pass
     ]
	 ]
 ] ensure: [ 
   "topmost doFileIn should clear the topaz sesion state"
    self _clearTopazSessionState ifTrue: [ 
        self currentProject: nil. 
        self currentPackage: nil 
    ]
  ]
]

{ #category : 'processing' }
GsFileIn >> doit [

	self execute: self nextChunk.
]

{ #category : 'processing' }
GsFileIn >> execute: aString [
  session ifNil:[ 
    aString evaluate .
  ] ifNotNil:[:sess | 
    sourceStringClass == String ifTrue:[
	    sess executeString: aString .
    ] ifFalse:[ | srcOop |
      srcOop := sess newUtf8String: aString encodeAsUTF8 toUnicode: false.
      [ sess send: '_remoteUnicodeExecute:' to: remoteGsFileInClassOop 
               withOops: { srcOop } .
      ] ensure:[ 
        sess releaseOop: srcOop .
      ]
    ]
  ].
]

{ #category : 'processing' }
GsFileIn >> executeString:  string [

	string evaluate.
	^nil

]

{ #category : 'accessors' }
GsFileIn >> fileFormat [
  ^ fileFormat ifNil:[ #'8bit' ].
]

{ #category : 'accessors' }
GsFileIn >> fileFormat: aSymbol [
  (aSymbol == #'8bit' or:[ aSymbol == #utf8 ]) ifFalse:[
     ArgumentError signal:'invalid file format'
  ].
  fileFormat := aSymbol
]

{ #category : 'processing' }
GsFileIn >> ignoreList [

	^#('EXPECTVALUE'
		'EXPECTERROR'
		'ERRORCOUNT'
		'FILEOUT'
		'DISPLAY'
		'LEVEL'
		'LIMIT'
		'LIST'
		'IFERR'
		'IFERR_LIST'
		'IFERR_CLEAR'
		'OMIT'
		'OUTPUT'
		'REMARK'
		'STATUS'
		'TIME'
		'LOGIN'
		'LOGOUT')

]

{ #category : 'initialize' }
GsFileIn >> initialize [

	category := self _defaultCategory.
	compileEnvironment := 0 .
]

{ #category : 'processing' }
GsFileIn >> inputNestedFile: aPath [
	"input nested file"
	| aType |
  aType := self _nestedType .
  (self class _fromPath: aPath type: aType sourceClass: self sourceStringClass)
     session: session ;
     _clearTopazSessionState: false ;
     doFileIn .
]

{ #category : 'processing' }
GsFileIn >> method [
  self parseMethodLine ;
       methodBody
]

{ #category : 'processing' }
GsFileIn >> methodBody [
  | src |
	currentClassObj ifNil: [self error: 'current class not defined'].
  src := self nextChunk .
  session ifNil:[
    currentClassObj compileMethod: src 
       category: category 
       environmentId: compileEnvironment.
  ] ifNotNil:[ 
    self _remoteMethod: src classMethod: false
  ]
]

{ #category : 'processing' }
GsFileIn >> nextChunk [
	| str |
	str := sourceStringClass new. "String or Unicode7"
	[
		stream atEnd not.
	] whileTrue: [
		line := stream nextLine .
    lineNum := lineNum + 1 .
    (line notEmpty and: [line first = $%]) ifTrue: [
			^ str
		].
		str addAll: line; lf.
	].
  ^ str "EOF equivalent to % terminator"
]

{ #category : 'processing' }
GsFileIn >> parseClassmethodLine [
  | words |
  words := self words: line .
  (words atOrNil:2) ifNotNil:[:s |
     (s codePointCompareTo: ':') == 0 ifTrue:[ Error signal:'illegal class specification '':'' '].
     self _setClass: s . 
  ].
]

{ #category : 'processing' }
GsFileIn >> parseMethodLine [
	| words |
  words := self words: line .
  (words atOrNil:2) ifNotNil:[:s |
     (s codePointCompareTo: ':') == 0 ifTrue:[ Error signal:'illegal class specification '':'' ' ].
     self _setClass: s . 
  ].
]

{ #category : 'processing' }
GsFileIn >> parseTraitClassmethodLine [
  | words |
  words := self words: line .
  (words atOrNil:2) ifNotNil:[:s |
     (s codePointCompareTo: ':') == 0 ifTrue:[ Error signal:'illegal trait specification '':'' '].
     self _setTrait: s . 
  ].
]

{ #category : 'processing' }
GsFileIn >> parseTraitmethodLine [
  | words |
  words := self words: line .
  (words atOrNil:2) ifNotNil:[:s |
     (s codePointCompareTo: ':') == 0 ifTrue:[ Error signal:'illegal trait specification '':'' '].
     self _setTrait: s . 
  ].
]

{ #category : 'accessors' }
GsFileIn >> path: aString [
	path := aString .
]

{ #category : 'processing' }
GsFileIn >> processLine [
	| words command firstChar |
	(line size == 0) ifTrue: [^self].
	words := line subStrings . 
	command := (words at:1) asUppercase.
  command isUnicodeString ifTrue:[ command := String withAll: command ].
	(#('DOIT' 'PRINTIT' 'RUN' 'NBRUN') includes: command) ifTrue:[
     ^self doit
  ].
	((firstChar := command at:1) == $! or:[ firstChar == $#])  ifTrue: [^nil].

	firstChar == $S ifTrue:[
	  command = 'SET' ifTrue: [
		  ((words at: 2) equalsNoCase: 'compile_env:') ifTrue:[
				words size == 3 ifFalse:[
					self error:'wrong number of arguments to SET COMPILE_ENV:'
				].
				^self compileEnvironment: (words at: 3)
			].
		  (((words at: 2) equalsNoCase: 'class')
			  or: [(words at: 2) equalsNoCase: 'class:']) ifTrue: [
				  words size == 3 ifFalse:[
					  self error:'wrong number of arguments to SET CLASS'
				  ].
				  ^self currentClass: (words at: 3)
           ].
		  (((words at: 2) equalsNoCase: 'trait')
			  or: [(words at: 2) equalsNoCase: 'trait:']) ifTrue: [
				  words size == 3 ifFalse:[
					  self error:'wrong number of arguments to SET TRAIT'
				  ].
				  ^self currentTrait: (words at: 3)
           ].
		  (((words at: 2) equalsNoCase: 'category')
			  or: [(words at: 2) equalsNoCase: 'category:']) ifTrue: [
				  words size == 3 ifFalse:[
					  self error:'wrong number of arguments to SET CATEGORY'
				  ].
				  ^self currentCategory: (words at: 3)
			  ].
		  ((words at: 2) equalsNoCase: 'enableremoveall') ifTrue: [
				  words size == 3 ifFalse:[
					  self error:'wrong number of arguments to SET ENABLEREMOVEALL'
				  ].
				  ^self setEnableRemoveAll: ((words at: 3) equalsNoCase: 'on')
			  ].
		  (((words at: 2) equalsNoCase: 'project')
			  or: [(words at: 2) equalsNoCase: 'project:']) ifTrue: [
                  words size == 2 ifTrue: [ 
                      "clear current project"
                      ^self currentProject: nil ].
				  words size == 3 ifFalse:[
					  self error:'wrong number of arguments to SET PROJECT'
				  ].
				  ^self currentProject: (words at: 3)
			  ].
		  (((words at: 2) equalsNoCase: 'package')
			  or: [(words at: 2) equalsNoCase: 'package:']) ifTrue: [
                  words size == 2 ifTrue: [ 
                      "clear current package"
                      ^self currentPackage: nil ].
				  words size == 3 ifFalse:[
					  self error:'wrong number of arguments to SET PACKAGE'
				  ].
				  ^self currentPackage: (words at: 3)
			  ].
		  ((words at: 2) equalsNoCase: 'sourcestringclass') ifTrue:[ | cName |
				words size == 3 ifFalse:[
					self error:'wrong number of arguments to SET SOURCESTRINGCLASS'
				].
        cName := String withAll:(words at: 3) .
        cName = 'String' ifTrue:[ self sourceStringClass: String ] ifFalse:[
        cName = 'Unicode16' ifTrue:[ 
           self sourceStringClass: Unicode16 ; doFileFormat: 'utf8' 
        ] ifFalse:[ self error:'arg to SET SOURCESTRINGCLASS must be String or Unicode16'. ]]. 
        ^ self  
      ].
      #( 'comp' 'cla' 'cat' 'enable' 'proj' 'pack' 'so' ) do:[:str |
        ((words at: 2) at: 1 equals: str) ifTrue:[
           self error:'unrecognized SET ', (words at: 2) asString .
        ].
      ].
      GsFile gciLogServer: 'WARNING, unrecognized SET ', (words at: 2) asString.
		  ^nil
    ].
    command = 'SEND' ifTrue:[
      self _sendIsLegal ifFalse:[ self error: 'SEND not supported' ].
      words size > 3 ifTrue:[ self error:'more than 2 args to SEND'].
      ^ self execute: (words at: 2), ' ', (words at: 3)
    ]
  ].
  firstChar == $C ifTrue:[
	  (command = 'CATEGORY:' or: [command = 'CATEGORY']) ifTrue: [^self category].
	  command = 'CLASSMETHOD' ifTrue: [^self classMethod ].
	  command = 'CLASSMETHOD:' ifTrue: [^self classMethod ].
	  command = 'COMMIT' ifTrue: [^self commitTransaction].
  ].
  firstChar == $M ifTrue:[
	  command = 'METHOD' ifTrue: [^self method ].
	  command = 'METHOD:' ifTrue: [^self method ].
  ].
  firstChar == $E ifTrue:[
    command = 'ENV' ifTrue:[
      words size == 2 ifTrue:[ ^ self compileEnvironment: (words at: 2)].
      words size == 1 ifTrue:[ ^ nil "no change to env"].
      self error:'wrong number of arguments to ENV'.
    ].
   ].
	firstChar == $R ifTrue: [
		command = 'REMOVEALLMETHODS' ifTrue: [^self removeAllMethods: (words atOrNil: 2)].
		command = 'REMOVEALLCLASSMETHODS' ifTrue: [^self removeAllClassMethods: (words atOrNil: 2)].
	].
	firstChar == $A ifTrue: [
		command = 'ABORT' ifTrue: [^self abort ].
	].
	firstChar == $I ifTrue: [
		command = 'INPUT' ifTrue: [words size == 1 ifTrue: [self error:'wrong number of arguments to input'].
		  ^self inputNestedFile: (words at: 2)
    ].
	].
  firstChar == $F ifTrue:[
    command = 'FILEFORMAT' ifTrue:[
      ^ self doFileFormat: (words atOrNil: 2).
    ]
  ].
  firstChar == $T ifTrue:[
    command = 'TFILE' ifTrue:[ | reader |  
      reader := (Globals at: #RowanKernel_tonel) at: #RwTopazTonelReader .
      ^ reader perform: #topazReadTonelFile: with: (words atOrNil: 2).
    ].
	command = 'TRCLASSMETHOD' ifTrue: [^self traitClassMethod ].
	command = 'TRCLASSMETHOD:' ifTrue: [^self traitClassMethod ].
	command = 'TRMETHOD' ifTrue: [^self traitMethod ].
	command = 'TRMETHOD:' ifTrue: [^self traitMethod ].
    command = 'TRREMOVEALLMETHODS'
      ifTrue: [ ^ self traitRemoveAllMethods: (words atOrNil: 2) ].
    command = 'TRREMOVEALLCLASSMETHODS'
      ifTrue: [ ^ self traitRemoveAllClassMethods: (words atOrNil: 2) ]  ].
    
  ^(self ignoreList includes: command)
		ifTrue: [nil ]
		ifFalse: [self error:  'unrecognized command: ' , command printString ].
]

{ #category : 'processing' }
GsFileIn >> removeAllClassMethods [

	self removeAllEnabled ifFalse: [ ^ self ].
	currentClassObj ifNil: [self error: 'current class not defined'].
  session ifNil:[
	  currentClassObj  class removeAllMethods
  ] ifNotNil:[
    session send: '_removeAllClassMethods:' to: remoteGsFileInClassOop
         withOops: { currentClassObj }
  ]
]

{ #category : 'processing' }
GsFileIn >> removeAllClassMethods: aClassName [
"removes all  class methods for supplied class. Supplied class becomes current class"

	self removeAllEnabled ifFalse: [ ^ self ].
	aClassName ifNotNil: [ self _setClass: aClassName ].
	currentClassObj ifNil: [self error: 'current class not defined'].
	self removeAllClassMethods
]

{ #category : 'accessors' }
GsFileIn >> removeAllEnabled [
	"if true (default), removeAllMethods is enabled"

	^ removeAll ~~ false
]

{ #category : 'processing' }
GsFileIn >> removeAllMethods [

	self removeAllEnabled ifFalse: [ ^ self ].
	currentClassObj ifNil: [self error: 'current class not defined'].
  session ifNil:[
    currentClassObj removeAllMethods
  ] ifNotNil:[
    session send: 'removeAllMethods' to: currentClassObj withArguments:#() 
  ]
]

{ #category : 'processing' }
GsFileIn >> removeAllMethods: aClassName [
"remove all methods for supplied class. Supplied class becomes current class"

	self removeAllEnabled ifFalse: [ ^ self ].
	aClassName ifNotNil: [ self _setClass: aClassName ].
	currentClassObj ifNil: [self error: 'current class not defined'].
	self removeAllMethods
]

{ #category : 'accessors' }
GsFileIn >> session: anExternalSession [
    "The current category is cleared by the LOGOUT, LOGIN, SET CLASS,
    and SET SESSION commands. "

  (session := anExternalSession) ifNotNil:[:sess|
    remoteGsFileInClassOop := (sess executeString:'GsFileIn') at: 1 "anOop".
  ].
	category := self _defaultCategory.
]

{ #category : 'accessors' }
GsFileIn >> setCurrentClass: aClassName [
    "The current category is cleared by the LOGOUT, LOGIN, SET CLASS,
    and SET SESSION commands. "

	self currentClass: aClassName
]

{ #category : 'accessors' }
GsFileIn >> setCurrentTrait: aTraitName [
	self currentTrait: aTraitName
]

{ #category : 'accessors' }
GsFileIn >> setEnableRemoveAll: aBool [
	"if true (default), removeAllMethods is enabled"
	removeAll := aBool
]

{ #category : 'accessors' }
GsFileIn >> setSession: aSession [
	aSession ifNotNil:[ 
    (aSession isKindOfClass: GsTsExternalSession) ifFalse:[
      ArgumentError signal: 'Expected a GsTsExternalSession, got a ', aSession class name .
    ]
  ].
  session := aSession .
]

{ #category : 'accessors' }
GsFileIn >> sourceStringClass [

  | cls |
  (cls := sourceStringClass) == Unicode7 ifTrue:[ ^ Unicode16].
  ^ cls  "String"
]

{ #category : 'accessors' }
GsFileIn >> sourceStringClass: aClass [
  aClass == String ifTrue:[ 
    sourceStringClass := aClass 
  ] ifFalse:[
    aClass == Unicode16 ifTrue:[
      sourceStringClass := Unicode7 
    ] ifFalse:[
      ArgumentError signal:'argument must be String or Unicode16'.
    ]
  ]
]

{ #category : 'accessors' }
GsFileIn >> stream: aFileStream [
	stream := aFileStream.
]

{ #category : 'processing' }
GsFileIn >> traitClassMethod [
  self parseTraitmethodLine ; 
       traitClassMethodBody .
]

{ #category : 'processing' }
GsFileIn >> traitClassMethodBody [
  | src |
	currentTraitObj ifNil: [self error: 'current trait not defined'].
  src := self nextChunk .
  session ifNil:[
    currentTraitObj classTrait compile: src 
       category: category.
  ] ifNotNil:[ 
    self _remoteTraitMethod: src classMethod: true 
  ]
]

{ #category : 'processing' }
GsFileIn >> traitMethod [
  self parseTraitmethodLine ; 
       traitMethodBody .
]

{ #category : 'processing' }
GsFileIn >> traitMethodBody [
  | src |
	currentTraitObj ifNil: [self error: 'current trait not defined'].
  src := self nextChunk .
  session ifNil:[
    currentTraitObj compile: src 
       category: category.
  ] ifNotNil:[ 
    self _remoteTraitMethod: src classMethod: false
  ]
]

{ #category : 'processing' }
GsFileIn >> traitRemoveAllClassMethods [

	self removeAllEnabled ifFalse: [ ^ self ].
	currentTraitObj ifNil: [self error: 'current trait not defined'].
  session ifNil:[
	  currentTraitObj  classTrait removeAllMethods
  ] ifNotNil:[
    session send: '_traitRemoveAllClassMethods:' to: remoteGsFileInClassOop
         withOops: { currentTraitObj }
  ]
]

{ #category : 'processing' }
GsFileIn >> traitRemoveAllClassMethods: aTraitName [
"removes all  class methods for supplied class. Supplied class becomes current class"

	self removeAllEnabled ifFalse: [ ^ self ].
	aTraitName ifNotNil: [ self _setTrait: aTraitName ].
	currentTraitObj ifNil: [self error: 'current trait not defined'].
	self traitRemoveAllClassMethods
]

{ #category : 'processing' }
GsFileIn >> traitRemoveAllMethods [

	self removeAllEnabled ifFalse: [ ^ self ].
	currentTraitObj ifNil: [self error: 'current trait not defined'].
  session ifNil:[
    currentTraitObj removeAllMethods
  ] ifNotNil:[
    session send: 'removeAllMethods' to: currentTraitObj withArguments:#() 
  ]
]

{ #category : 'processing' }
GsFileIn >> traitRemoveAllMethods: aTraitName [
"remove all methods for supplied trait. Supplied trait becomes current trait"

	self removeAllEnabled ifFalse: [ ^ self ].
	aTraitName ifNotNil: [ self _setTrait: aTraitName ].
	currentTraitObj ifNil: [self error: 'current trait not defined'].
	self traitRemoveAllMethods
]

{ #category : 'private' }
GsFileIn >> words: lineArg [
  | arr n cls aLine aWord |
  aLine := lineArg trimSeparators .
  arr := { } .
  aWord := (cls := aLine class) new . 
  n := 1 .
  [ n <= aLine size ] whileTrue:[ | ch |
    ch := aLine at: n .
    ch isSeparator ifTrue:[ 
      aWord size > 0 ifTrue:[ arr add: aWord . aWord := cls new ].
      n := n + 1 .
      aWord := cls new .
      [ (ch := aLine atOrNil: n) ~~ nil and:[ ch isSeparator]] whileTrue:[ n := n + 1 ].
    ] ifFalse:[
       ch == $' ifTrue:[ | done |
         aWord size > 0 ifTrue:[ arr add: aWord . aWord := cls new ].
         n := n + 1 .
         [ n < aLine size and:[ done == nil] ] whileTrue:[
           ch := aLine at: n .
           ch == $' ifTrue:[
             (aLine atOrNil: n+1) == $' ifTrue:[ n := n + 2 .  aWord add: $' ]
                 ifFalse:[ done := true . arr add: aWord . aWord := cls new ].
           ] ifFalse:[
             aWord add: ch .
           ].
           n := n + 1 .
         ].
       ] ifFalse:[
        aWord add: ch .
        n := n + 1 .
      ] 
    ]. 
  ].
  aWord size > 0 ifTrue:[
    arr add: aWord .
  ].
  ^ arr
]
