Extension { #name : 'GsIndexedQueryEvaluator' }

{ #category : 'Instance Creation' }
GsIndexedQueryEvaluator class >> on: anIndex [

  ^ self new index: anIndex

]

{ #category : 'Support' }
GsIndexedQueryEvaluator >> _allIdenticalTo: anObject in: nsc [

"Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

| result tmp |
result := self _primAllIdenticalTo: anObject in: nsc _asIdentityBag.
result class = nsc species
  ifFalse: [
    tmp := nsc species new.
    tmp addAll: result.
    result := tmp
  ].
^ result

]

{ #category : 'Support' }
GsIndexedQueryEvaluator >> _allIdenticalTo: anObject in: nsc do: aBlock [
  "Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

  | result |
  result := self _primAllIdenticalTo: anObject in: nsc _asIdentityBag.
  result do: aBlock

]

{ #category : 'Error Handling' }
GsIndexedQueryEvaluator >> _errorCannotInvokeRangeOperationOnIdentityIndex [

"An attempt was made to use a range operation (< > = <= >=) on a path
 expression only supported by an identity index."

^ self index _errorCannotInvokeRangeOperationOnIdentityIndex

]

{ #category : 'Searching' }
GsIndexedQueryEvaluator >> _findAllValuesIdenticalTo: aValue [
  "Returns a set of all values that satisfy the query."

  | array1 array2 stream doNext |
  " if can't make the comparison, then none are identical "
  (self index _canCompareWith: aValue)
    ifFalse: [ self _error: #'assocErrBadComparison' ].
  " get boundary entries in the B-tree "
  array1 := {}.
  " >=, perform < and do an 'next' operation later "
  doNext := false.
  (self btreeRoot
    _findAllValuesLessThanKey: aValue
    andEquals: false
    into: array1
    using: self comparisonForCompare)
    ifTrue: [ doNext := true ]
    ifFalse: [ " all were greater than or equal to "
      self btreeRoot
        _putFirstIndexOfFirstChildInto: array1
        ifGreaterThanOrEqualTo: aValue
        using: self comparisonForCompare ].
  array2 := {}.
  stream := self readStreamClass on: self index.
  (self btreeRoot
    _findAllValuesLessThanKey: aValue
    andEquals: true
    into: array2
    using: self comparisonForCompare)
    ifFalse: [ " none were found less than "
       stream currentStack: {0}.
       ^ stream ].
  " create the read stream "
  stream currentStack: array1.
  stream endIndex: (array2 at: 1).
  stream endNode: (array2 at: 2).
  doNext
    ifTrue: [ " advance the B-tree positioning "
      stream _advance ].
  ^ stream

]

{ #category : 'Searching' }
GsIndexedQueryEvaluator >> _findAllValuesThroughSetsNotIdenticalTo: aKey [
  "Returns a set of all values that satisfy the query.  This method works
 when the path is through a set-valued instance variable."

  |  tmpHolder idx |
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  idx := self index.
  idx btreeRoot _lastRootObjectsNotIdenticalTo: aKey into: tmpHolder.
  ^ tmpHolder completeBag

]

{ #category : 'Searching' }
GsIndexedQueryEvaluator >> _findFirstKeyNotIdenticalTo: aValue atTerm: pathTerm [

  ^IndexedQueryEvaluator subclassResponsibility: #_findFirstKeyNotIdenticalTo:atTerm:

]

{ #category : 'Support' }
GsIndexedQueryEvaluator >> _primAllIdenticalTo: anObject in: nsc [

"Returns a new NSC containing the given object with the number of occurrences
 as in the root NSC."

<primitive: 340>

self _primitiveFailed: #_primAllIdenticalTo:in: args: { anObject . nsc } .
self _uncontinuableError

]

{ #category : 'Accessing' }
GsIndexedQueryEvaluator >> btreeComparisonQuerySpec [

  ^self index btreeComparisonQuerySpec

]

{ #category : 'Accessing' }
GsIndexedQueryEvaluator >> collator [
  "Return the IcuCollator to be used in comparing Unicode strings"

  ^ self index collator

]

{ #category : 'Accessing' }
GsIndexedQueryEvaluator >> doBlock [
  ^ doBlock

]

{ #category : 'Accessing' }
GsIndexedQueryEvaluator >> doBlock: aBlock [
  doBlock := aBlock

]

{ #category : 'Query Select' }
GsIndexedQueryEvaluator >> findAllValuesIdenticalTo: aValue [
  "Returns a set of all values that satisfy the query.  Since this is an identity
 comparison on a range index (which supports equality comparison), this
 method creates a B-tree read stream for equal values, then iterates through the
 values making the identity comparison."

  | tmpHolder stream |
  " optimize query for objects in the NSC itself "
  self index isIndexOnRootNsc
    ifTrue: [
      doBlock
        ifNil: [
          " optimize query for objects in the NSC itself "
          ^ self _allIdenticalTo: aValue in: self nscRoot ].
      ^ self _allIdenticalTo: aValue in: self nscRoot do: doBlock ].
  stream := self _findAllValuesIdenticalTo: aValue.
  self doBlock
    ifNotNil: [
      stream atEnd
        ifTrue: [ ^ self ].
      ^ [ stream atEnd ]
           whileFalse: [
             | nextKey next |
             nextKey := stream _peekKey.
             next := stream next.
             nextKey == aValue
             ifTrue: [ self doBlock value: next ] ] ].
  stream atEnd
    ifTrue: [ ^ self nscRoot species new ].
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  stream _valuesAddInto: tmpHolder identicalTo: aValue.
  ^ tmpHolder completeBag

]

{ #category : 'Query Select' }
GsIndexedQueryEvaluator >> findAllValuesNotIdenticalTo: aValue [
  "Returns a set of all values that satisfy the query."

  | tmpHolder res |
  self index isIndexOnRootNsc
    ifTrue: [
      tmpHolder := self nscRoot species new.
      tmpHolder addAll: self nscRoot.
      tmpHolder removeAll: (self _allIdenticalTo: aValue in: self nscRoot).
      doBlock ifNotNil: [ tmpHolder do: doBlock ].
      ^ tmpHolder ].
  self hasCollectionBasedTerm
    ifTrue: [
      res := self _findAllValuesThroughSetsNotIdenticalTo: aValue.
      doBlock ifNotNil: [ res do: doBlock ].
      ^ res ].
  tmpHolder := NscBuilder
    for: self nscRoot species new
    max: self sizeForNscBuilder.
  tmpHolder _addAllNsc: (self _findAllValuesIdenticalTo: aValue) makeNsc.	" get all objects identical to aValue and identical to nil "
  tmpHolder _addAllNsc: (self _allIdenticalTo: nil in: self nscRoot).
  " get all objects with nil along the path "
  self index _getObjectsWithNilOnPathInto: tmpHolder.
  res := self nscRoot _asIdentityBag - tmpHolder completeBag _asIdentityBag.
  doBlock ifNotNil: [ res do: doBlock ].
  res class == self nscRoot species
    ifFalse: [
      | finalRes |
      finalRes := self nscRoot species new.
      finalRes addAll: res.
      ^ finalRes ].
  ^ res

]

{ #category : 'Query Select' }
GsIndexedQueryEvaluator >> findAllValuesNotIdenticalTo: aValue do: aBlock [
  "Iterates over all values that satisfy the query."

  | result |
  result := self findAllValuesNotIdenticalTo: aValue.
  result do: aBlock.
  ^ result

]

{ #category : 'Query Detect' }
GsIndexedQueryEvaluator >> findFirstValueIdenticalTo: aValue [

  ^self subclassResponsibility: #findFirstValueIdenticalTo:

]

{ #category : 'Query Detect' }
GsIndexedQueryEvaluator >> findFirstValueNotIdenticalTo: aValue [

"Returns the first value that satisfies the query."

| key lastPathTerm |

lastPathTerm := self index lastPathTerm.
" optimize query for objects in the NSC itself "
self index isIndexOnRootNsc
  ifTrue: [
    index nscRoot do: [ :element |
      element == aValue
        ifFalse: [ ^ element ] ].
    ^ #_incompletePathTraversal ].

key := self _findFirstKeyNotIdenticalTo: aValue atTerm: lastPathTerm.
(key == #_incompletePathTraversal)
  ifTrue: [ ^ #_incompletePathTraversal ].

^ self findFirstValueIdenticalTo: key

]

{ #category : 'Testing' }
GsIndexedQueryEvaluator >> hasCollectionBasedTerm [
  ^ self index hasCollectionBasedTerm

]

{ #category : 'Testing' }
GsIndexedQueryEvaluator >> hasSetValuedTerm [
  ^ self index hasSetValuedTerm

]

{ #category : 'Accessing' }
GsIndexedQueryEvaluator >> index [

  ^index

]

{ #category : 'Updating' }
GsIndexedQueryEvaluator >> index: anIndex [

  index := anIndex

]

{ #category : 'Accessing' }
GsIndexedQueryEvaluator >> nscRoot [

^self index nscRoot

]

{ #category : 'Accessing' }
GsIndexedQueryEvaluator >> sizeForNscBuilder [

"Returns the size to be used when an NscBuilder is collecting objects."

^ self nscRoot size

]
