Extension { #name : 'GsQueryPredicate' }

{ #category : 'instance creation' }
GsQueryPredicate class >> _createPredicateFromQueryExecuter: aQueryExecuter startingAt: offset [
  | firstOperand secondOperand searchOp predicateType predicate operator |
  firstOperand := aQueryExecuter firstOperandAt: offset.
  searchOp := aQueryExecuter operatorAt: offset.
  secondOperand := aQueryExecuter secondOperandAt: offset.
  predicateType := aQueryExecuter at: offset.
  searchOp == 8
    ifTrue: [
      predicate := (GsQueryPredicate constant: secondOperand) copy.
      predicate _nsc: aQueryExecuter nsc collator: nil.
      ^ predicate ].
  searchOp == 9
    ifTrue: [
      | searchOp1 searchOp2 |
      predicateType ~~ 2
        ifTrue: [ self error: 'invalid predicateType for searchOp ''9''' ].
      searchOp1 := secondOperand at: 1.
      searchOp2 := secondOperand at: 3.
      predicate := (GsQueryPredicate
        constant: (secondOperand at: 2)
        operator:
          (self inverseOperatorFor: (aQueryExecuter operationSelectors at: searchOp1 + 1))
        path: 'each.' , firstOperand pathComponentsString
        operator: (aQueryExecuter operationSelectors at: searchOp2 + 1)
        constant: (secondOperand at: 4)) copy.
      predicate
        _nsc: aQueryExecuter nsc collator: nil;
        evaluator: firstOperand.
      ^ predicate ].
  operator := aQueryExecuter operationSelectors at: searchOp + 1.
  predicateType == 2
    ifTrue: [
      predicate := (GsQueryPredicate
        path: 'each.' , firstOperand pathComponentsString
        operator: operator
        constant: secondOperand) copy.
      predicate
        _nsc: aQueryExecuter nsc collator: nil;
        evaluator1: firstOperand.
      ^ predicate ].
  predicateType == 1
    ifTrue: [
      predicate := (GsQueryPredicate
        constant: firstOperand
        operator: operator
        constant: secondOperand) copy.
      predicate _nsc: aQueryExecuter nsc collator: nil.
      ^ predicate ].
  predicateType == 4
    ifTrue: [
      predicate := (GsQueryPredicate
        path: 'each.' , firstOperand pathComponentsString
        operator: operator
        path: 'each.' , secondOperand pathComponentsString) copy.
      predicate
        _nsc: aQueryExecuter nsc collator: nil;
        evaluator1: firstOperand;
        evaluator2: secondOperand.
      ^ predicate ].
  self error: 'Unknown predicate type: ' , predicateType printString

]

{ #category : 'instance creation' }
GsQueryPredicate class >> _createPredicateFromTermsArray: termsArray startingAt: offset [
  "Return the proper Query Predicate instance.
   Predicates are classified into four types:

 1.  constant-constant (such as 3 > 2)
 2.  path-constant     (such as 'spouse.age' > 30)
 3.  constant-path     (such as 30 <= 'spouse.age')
 4.  path-path         (such as 'spouse.age' < 'spouse.weight')"

  (termsArray at: offset + 4)
    ifTrue: [
      (termsArray at: offset + 1)
        ifTrue: [
          " its path-path "
          ^ GsPathPathPredicate new ]
        ifFalse: [
          " its path-constant "
          ^ GsPathConstantPredicate new ] ]
    ifFalse: [
      (termsArray at: offset + 1)
        ifTrue: [
          " its constant-path "
          ^ GsConstantPathPredicate new ]
        ifFalse: [
          " its constant-constant "
          ^ GsConstantConstantPredicate new ] ]

]

{ #category : 'instance creation' }
GsQueryPredicate class >> constant: operand1 [
  (operand1 _idxValue == true or: [ operand1 _idxValue == false ])
    ifFalse: [ self error: 'Unary constant must be Boolean' ].
  ^ GsConstantConstantPredicate new
    constant1: operand1;
    operator: #'unary';
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> constant: operand1 operator: operator constant: operand2 [
  ^ GsConstantConstantPredicate new
    constant1: operand1;
    operator: operator;
    constant2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> constant: operand1 operator: operator path: operand2 [
  ^ (GsConstantPathPredicate
    constant: operand1
    operator: operator
    path: operand2) immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> constant: constant1 operator: operator1 path: path operator: operator2 constant: constant2 [
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    constant: constant1
    operator: operator1
    path: path
    operator: operator2
    constant: constant2) immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> constant: constant1 operator: operator1 path: path operator: operator2 variable: constant2 [
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    constant: constant1
    operator: operator1
    path: path
    operator: operator2
    variable: constant2) immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> constant: operand1 operator: operator variable: operand2 [
  ^ GsConstantConstantPredicate new
    constant1: operand1;
    operator: operator;
    variable2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> fromQueryStatement: aString [
  ^ (GsQueryParser compileFormula: aString) immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> fromSelectBlock: aSelectBlock [
  "queryBlockArray is a four element Array of Arrays describing a SelectBlock
 in the manner needed to process a query on the receiver:

   1.  Array of the bound variables from the predicate
   2.  Array of the predicate's terms
   3.  Array of the predicate's paths
   4.  Array of Strings of the path names used in the predicate"

  | queryBlockArray varsArray termsArray linksArray pathsArray formula |
  queryBlockArray := aSelectBlock queryBlock value: nil.
  varsArray := queryBlockArray at: 1.
  termsArray := queryBlockArray at: 2.
  linksArray := queryBlockArray at: 3.
  pathsArray := queryBlockArray at: 4.
  formula := nil.
  1 to: termsArray size by: 7 do: [ :i |
    | predicate |
    predicate := self _createPredicateFromTermsArray: termsArray startingAt: i.
    predicate
      _fromArray: varsArray
      terms: termsArray
      links: linksArray
      paths: pathsArray
      startingAt: i.
    formula == nil
      ifTrue: [ formula := predicate ]
      ifFalse: [ formula := formula & predicate ] ].
  ^ formula

]

{ #category : 'instance creation' }
GsQueryPredicate class >> fromSelectBlock: aSelectBlock usingQueryExecuterWith: anNsc [
  "Extract predicates from QueryExecuter using <aSelectBlock>
        and <anNsc>. Return an GsQuery. Note that the predicates are
        based on standard transforms applied by the QueryExecuter."

  | queryExecuter gueryBlockArray queryFormula |
  queryExecuter := QueryExecuter on: anNsc.
  gueryBlockArray := aSelectBlock queryBlock value: nil.
  queryExecuter
    _buildIndexOperationsList: gueryBlockArray;
    optimize.
  queryFormula := nil.
  queryExecuter _getOperationsOrder
    do: [ :offset |
      | predicate |
      predicate := self
        _createPredicateFromQueryExecuter: queryExecuter
        startingAt: offset.
      queryFormula == nil
        ifTrue: [ queryFormula := predicate ]
        ifFalse: [ queryFormula := queryFormula & predicate ] ].
  queryFormula
    ifNil: [
      "unsatisfiable query"
      queryFormula := (self constant: false)
        bindEvaluatorsFor: anNsc
        collator: nil ].
  ^ queryFormula

]

{ #category : 'instance creation' }
GsQueryPredicate class >> operand: operand1 operator: operator operand: operand2 [
  ^ self new
    operand1: operand1;
    operator: operator;
    operand2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> operand: operand1 operator: operator1 path: path operator: operator2 operand: operand2 [
  "e.g.: 4 < a.b.c < 6 or 6 > a.b.c >= 4 "

  ^ (GsRangeQueryPredicate
    operand: operand1
    operator: operator1
    path: path
    operator: operator2
    operand: operand2) immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> path: operand1 [
  ^ GsPathConstantPredicate new
    path1: operand1;
    operator: #'unary';
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> path: operand1 operator: operator constant: operand2 [
  ^ GsPathConstantPredicate new
    path1: operand1;
    operator: operator;
    constant2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> path: operand1 operator: operator path: operand2 [
  ^ GsPathPathPredicate new
    path1: operand1;
    operator: operator;
    path2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> path: operand1 operator: operator variable: operand2 [
  ^ GsPathConstantPredicate new
    path1: operand1;
    operator: operator;
    variable2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> variable: operand1 [
  (operand1 _idxValue == true or: [ operand1 _idxValue == false ])
    ifFalse: [ self error: 'Unary constant must be Boolean' ].
  ^ GsConstantConstantPredicate new
    variable1: operand1;
    operator: #'unary';
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> variable: operand1 operator: operator constant: operand2 [
  ^ GsConstantConstantPredicate new
    variable1: operand1;
    operator: operator;
    constant2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> variable: operand1 operator: operator path: operand2 [
  ^ GsConstantPathPredicate new
    variable1: operand1;
    operator: operator;
    path2: operand2;
    immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> variable: constant1 operator: operator1 path: path operator: operator2 constant: constant2 [
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    variable: constant1
    operator: operator1
    path: path
    operator: operator2
    constant: constant2) immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> variable: constant1 operator: operator1 path: path operator: operator2 variable: constant2 [
  "e.g.: 4 < a.b.c < 6"

  ^ (GsRangeQueryPredicate
    variable: constant1
    operator: operator1
    path: path
    operator: operator2
    variable: constant2) immediateInvariant

]

{ #category : 'instance creation' }
GsQueryPredicate class >> variable: operand1 operator: operator variable: operand2 [
  ^ GsConstantConstantPredicate new
    variable1: operand1;
    operator: operator;
    variable2: operand2;
    immediateInvariant

]

{ #category : 'private' }
GsQueryPredicate >> _check: indexObj compare: compareOp value: searchVal [
  "Raise an error if the comparison operation is not valid for the index object
 and the search value."

  " check for range operation using an identity index "

  ((self isRangeEqualityOperation: compareOp)
    and: [ indexObj isRangeEqualityIndex not ])
    ifTrue: [ ^ indexObj _errorCannotInvokeRangeOperationOnIdentityIndex ]

]

{ #category : 'private' }
GsQueryPredicate >> _evaluators [
  "return list of evaluators associated with predicate"

  ^ {(self evaluator1)}

]

{ #category : 'private' }
GsQueryPredicate >> _fromArray: varsArray terms: termsArray links: linksArray paths: pathsArray startingAt: i [
  self subclassResponsibility: #'_fromArray:terms:links:paths:startingAt:'

]

{ #category : 'testing' }
GsQueryPredicate >> _isValidQueryOperator: queryOp [
  "Returns true if the given search operation is one of == ~~ < > = <= >= or ~=."

  ^ self operationSelectors includes: queryOp

]

{ #category : 'private' }
GsQueryPredicate >> _pathFrom: linksArray paths: pathsArray offset: offset [
  | nextLink path i linkSize term |
  path := 'each'.
  offset == 0
    ifTrue: [ ^ path ].
  i := offset * 2 - 1.
  linkSize := linksArray size.
  [ i < linkSize ]
    whileTrue: [
      nextLink == 0
        ifTrue: [ ^ path ].
      nextLink := linksArray at: i.
      term := (pathsArray at: (linksArray at: i + 1)) asString.
      path := path , '.' , term.
      i := nextLink * 2 - 1 ].
  ^ path

]

{ #category : 'optimizing' }
GsQueryPredicate >> applyDeMorgansNegationTransform [
  "actively apply De Morgan's laws ... operating on copy"

  self operator: self operatorNegated

]

{ #category : 'converting' }
GsQueryPredicate >> asFormulaWithSelectorParts: selectorParts arguments: arguments [
  | selector |
  selector := selectorParts first inputValue asSymbol.
  ^ (self operationSelectors includes: selector)
    ifTrue: [
      | clause2 |
      clause2 := self operand2
        asFormulaWithSelectorParts: selectorParts
        arguments: arguments.
      self collapseToRangePredicateIfPossible: clause2 ]
    ifFalse: [ self perform: selector with: arguments first ]

]

{ #category : 'transforming' }
GsQueryPredicate >> bind: variableName to: value [
  | bound |
  bound := self copy.
  bound operand1: (bound operand1 bind: variableName to: value).
  bound operand2: (bound operand2 bind: variableName to: value).
  ^ bound immediateInvariant

]

{ #category : 'accessing' }
GsQueryPredicate >> compatibleInverseRangeOperatorsFor: anOperator [
  | less greater |
  less := #(#'<' #'<=').
  greater := #(#'>' #'>=').
  ^ ((less includes: anOperator) and: [ greater includes: self operator ])
    or: [ (greater includes: anOperator) and: [ less includes: self operator ] ]

]

{ #category : 'accessing' }
GsQueryPredicate >> compatibleRangeOperatorsFor: anOperator [
  | less greater |
  less := #(#'<' #'<=').
  greater := #(#'>' #'>=').
  ^ ((less includes: anOperator) and: [ less includes: self operator ])
    or: [ (greater includes: anOperator) and: [ greater includes: self operator ] ]

]

{ #category : 'accessing' }
GsQueryPredicate >> constant [
  self subclassResponsibility: #'constant'

]

{ #category : 'accessing' }
GsQueryPredicate >> constant1: anObject [
  operand1 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject

]

{ #category : 'accessing' }
GsQueryPredicate >> constant2: anObject [
  operand2 := GsConstantReferenceAssociation
    newWithKey: anObject asString
    value: anObject

]

{ #category : 'querying' }
GsQueryPredicate >> elementValueEvaluator [
  "companion method to #elementValue:"

  "anObject is the value obtained by traversing path terms to the last value ... analgous to the last element class"

  "not all subclasses are required to implement this selector, if they do not implement this selector, then they must implement #elementValue:"

  ^ (evaluator1 == nil  or: [ evaluator1 isPathEvaluator not ])
    ifTrue: [ self pathEvaluatorFor: self path ]
    ifFalse: [ evaluator1 ]

]

{ #category : 'accessing' }
GsQueryPredicate >> evaluator [
  ^ self evaluator1

]

{ #category : 'accessing' }
GsQueryPredicate >> evaluator1 [
  ^ evaluator1

]

{ #category : 'accessing' }
GsQueryPredicate >> evaluator1: anEvaluator [
  evaluator1 := anEvaluator

]

{ #category : 'private' }
GsQueryPredicate >> immediateInvariant [
  super immediateInvariant.
  operand1 immediateInvariant.
  operand2 immediateInvariant

]

{ #category : 'testing' }
GsQueryPredicate >> isConstantBound [
  self subclassResponsibility: #'isConstantBound'

]

{ #category : 'accessing' }
GsQueryPredicate >> operand1 [
  ^ operand1

]

{ #category : 'accessing' }
GsQueryPredicate >> operand1: anObject [
  operand1 := anObject

]

{ #category : 'testing' }
GsQueryPredicate >> operand1IsPath [
  ^ false

]

{ #category : 'accessing' }
GsQueryPredicate >> operand2 [
  ^ operand2

]

{ #category : 'accessing' }
GsQueryPredicate >> operand2: anObject [
  operand2 := anObject

]

{ #category : 'testing' }
GsQueryPredicate >> operand2IsPath [
  ^ false

]

{ #category : 'accessing' }
GsQueryPredicate >> operator [
  ^ operator

]

{ #category : 'accessing' }
GsQueryPredicate >> operator: queryOp [
  (self _isValidQueryOperator: queryOp)
    ifFalse: [ ^ self error: 'Invalid query operator: ' , queryOp printString ].
  operator := queryOp

]

{ #category : 'accessing' }
GsQueryPredicate >> operatorNegated [
  ^ self negatedOperatorFor: self operator

]

{ #category : 'accessing' }
GsQueryPredicate >> path [
  ^ self rawPath key

]

{ #category : 'accessing' }
GsQueryPredicate >> path1: aPathString [
  operand1 := self pathReferenceFor: aPathString

]

{ #category : 'accessing' }
GsQueryPredicate >> path2: aPathString [
  operand2 := self pathReferenceFor: aPathString

]

{ #category : 'private' }
GsQueryPredicate >> postCopy [
  operand1 := operand1 copy.
  operand2 := operand2 copy.
  super postCopy

]

{ #category : 'printing' }
GsQueryPredicate >> printOn: aStream [
  aStream
    nextPutAll:
      '(' , self operand1 printString , ' ' , self operator asString , ' '
        , self operand2 printString , ')'

]

{ #category : 'accessing' }
GsQueryPredicate >> rawPath [
  self subclassResponsibility: #'rawPath'

]

{ #category : 'querying' }
GsQueryPredicate >> readStream [
  | queryEvaluator theOperator theConstant |
  queryEvaluator := self evaluator1 asQueryEvaluator.
  theOperator := self operator.
  theConstant := self constant.
  self _check: self evaluator1 compare: theOperator value: theConstant.
  (theOperator == #'<' or: [ theOperator == #'<=' ])
    ifTrue: [
      | stream querySpec |
      stream := queryEvaluator
        _findAllValuesGreaterThan: nil
        andEquals: true
        andLessThan: theConstant
        andEquals: theOperator == #'<='
        using: (BtreeComparisonForCompare newForComparison: self collator).
      querySpec := queryEvaluator btreeComparisonQuerySpec
        key: theConstant selector: theOperator;
        yourself.
      stream streamQuerySpec: querySpec.
      ^ stream ].
  (theOperator == #'>' or: [ theOperator == #'>=' ])
    ifTrue: [
      ^ queryEvaluator
        _findAllValuesGreaterThanKey: theConstant
        andEquals: theOperator == #'>='
        using: (BtreeComparisonForCompare newForComparison: self collator) ].
  theOperator == #'='
    ifTrue: [
      ^ queryEvaluator
        _findAllValuesEqualTo: theConstant
        using: (BtreeComparisonForCompare newForComparison: self collator) ].
  theOperator == #'=='
    ifTrue: [
      | querySpec stream |
      stream := queryEvaluator _findAllValuesIdenticalTo: theConstant.
      querySpec := queryEvaluator btreeComparisonQuerySpec
        key: theConstant selector: theOperator;
        yourself.
      stream streamQuerySpec: querySpec.
      ^  stream ]

]

{ #category : 'transforming' }
GsQueryPredicate >> unbind [
  "remove all bindings"

  | unbound |
  unbound := super unbind.
  unbound operand1: unbound operand1 unbind.
  self operator == #'unary'
    ifFalse: [ unbound operand2: unbound operand2 unbind ].
  unbound unbindEvaluators.
  ^ unbound immediateInvariant

]

{ #category : 'private' }
GsQueryPredicate >> unbindEvaluators [
  "remove all bindings"

  evaluator1 := nil

]

{ #category : 'testing' }
GsQueryPredicate >> usesComparisonOperation [
  ^ #(#'<=' #'<' #'>' #'>=') includes: operator

]

{ #category : 'testing' }
GsQueryPredicate >> usesEqualityOperation [
  ^ operator == #'=' or: [ operator == #'~=' ]

]

{ #category : 'testing' }
GsQueryPredicate >> usesEqualOperation [
  ^ operator == #'='

]

{ #category : 'testing' }
GsQueryPredicate >> usesIdenticalToOperation [
  ^ operator == #'=='

]

{ #category : 'testing' }
GsQueryPredicate >> usesIdentityOperation [
  ^ operator == #'==' or: [ operator == #'~~' ]

]

{ #category : 'testing' }
GsQueryPredicate >> usesPathEvaluator [
  self evaluator1 ifNil: [ ^ true ].
  ^ self evaluator1 isPathEvaluator

]

{ #category : 'accessing' }
GsQueryPredicate >> variable1: aVariableName [
  operand1 := self variableReferenceFor: aVariableName

]

{ #category : 'accessing' }
GsQueryPredicate >> variable2: aVariableName [
  operand2 := self variableReferenceFor: aVariableName

]
