Extension { #name : 'GsSetValuedPathTerm' }

{ #category : 'Error Handling' }
GsSetValuedPathTerm >> _errorPathObjectNotAnNsc: anObject [
  "An object traversed along an index path through a set-valued instance
 variable was not an NSC."

  ImproperOperation new
    reason: #'rtErrPathTermObjectNotAnNsc';
    object: anObject;
    details: 'Expected an UnorderedCollection';
    signal

]

{ #category : 'Testing' }
GsSetValuedPathTerm >> _isLastOccurrenceInIndexFor: anNsc [

  ^ (self findReachableRootsFor: anNsc) size <= anNsc size

]

{ #category : 'Indexing Support' }
GsSetValuedPathTerm >> _updateCollectionBasedIndexFor: anIndex on: anNsc offset: anOffset addingIndex: addingIndex [
  "if anNsc already has an index, add/remove anIndex as appropriate"

  anNsc _indexedPaths ~~ nil
    ifTrue: [
      | nscOffset |
      nscOffset := anOffset + 1.
      addingIndex
        ifTrue: [
          anNsc _indexedPaths
            addIndex: anIndex
            withOffset: (anIndex at: anOffset) offset + 1
            nsc: anNsc ]
        ifFalse: [ anNsc _indexedPaths removeIndex: anIndex withOffset: nscOffset for: anNsc ] ]

]

{ #category : 'Adding' }
GsSetValuedPathTerm >> addDirectMappingFor: anObject root: rootObject nsc: anNsc logging: aBoolean [
  " Update the B-tree and the dependency list for anObject."

  updateBtree ~~ nil
    ifTrue: [ updateBtree btreeAt: anObject put: anNsc root: rootObject ].
  needsDepList
    ifTrue: [ " add an entry in the dependency list "
      anObject getDepListAndAddLastElementPathTerm: self logging: aBoolean ]

]

{ #category : 'Updating Indexes' }
GsSetValuedPathTerm >> addMappingsForObject: anNsc root: rootObject logging: aBoolean [
  "Adds index dictionary entries for the objects in anNsc.  Also update the
 NSC's index list."

  | iList sz |
  (nil == anNsc _or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  anNsc class isNsc
    ifFalse: [ ^ self _errorPathObjectNotAnNsc: anNsc ].
  sz := self size.
  iList := anNsc _getIndexList.
  anNsc _putInWriteSet.	" for each index that utilizes this path term "
  1 to: sz do: [ :i | iList addIndex: (self at: i) withOffset: offset + 1 nsc: anNsc ].
  sz := children size.
  anNsc
    do: [ :setElement | " insert dictionary entry mapping set element -> NSC "
      updateBtree ~~ nil
        ifTrue: [
          (self _checkBtreeComparisonWith: setElement)
            ifFalse: [
              ImproperOperation new
                _number:
                    (ErrorSymbols at: #'rtErrRangeEqualityIndexInvalidClassKindForBtree');
                args: { setElement class. self name. updateBtree lastElementClassDescription };
                signal ].
          updateBtree btreeAt: setElement put: anNsc root: rootObject.
          needsDepList
            ifTrue: [ " add an entry in the dependency list "
              setElement getDepListAndAddLastElementPathTerm: self logging: aBoolean ] ].
      1 to: sz do: [ :j | (children at: j) addMappingsForObject: setElement root: rootObject logging: aBoolean ] ]

]

{ #category : 'Audit' }
GsSetValuedPathTerm >> auditDepListFor: obj index: indexObj using: auditor optionalSentinel: optionalSentinel [
  "the set does not have a dependency list"

  self shouldNotImplement: #'auditDepListFor:index:using:optionalSentinel:'

]

{ #category : 'Audit' }
GsSetValuedPathTerm >> auditNscCountsFor: anNsc using: auditor count: btreeCounts [
  "Private. "

  | sz |
  sz := self size.
  (nil == anNsc _or: [ sz == 0 ])
    ifTrue: [ ^ self ].
  anNsc do: [ :setElement | | index |
      index := 1.
      children isEmpty
        ifTrue: [ self auditDirectNscCountsFor: setElement using: auditor count: btreeCounts ]
        ifFalse: [ 1 to: sz do: [ :i | | count indexObj |
            indexObj := self at: i.
            indexObj size == offset
              ifTrue: [ count := btreeCounts at: index.
                count == 0
                  ifTrue: [ auditor pathTermIncorrectNumberOfBtreeEntries: self index: i offset: offset ].
                btreeCounts at: index put: count - 1.
                index := index + 1 ] ].
          1 to: children size do: [ :i | (children at: i)
              auditNscCountsFor: setElement
              using: auditor
              count: (btreeCounts at: index).
            index := index + 1 ] ] ]

]

{ #category : 'Audit' }
GsSetValuedPathTerm >> auditNscForRootObj: anNsc rootObj: rootObj using: auditor [
  "Private. "

  self size == 0
    ifTrue: [ ^ self ].
  self indicatesIndexOnNscElements
    ifTrue: [ "^ self auditDirectNscRootObj: obj using: auditor" self error: 'not yet implemented' ].
  nil == anNsc
    ifTrue: [ ^ self ].
self nextObj: anNsc do: [:nextObj |
  1 to: self size do: [ :i | | indexObj |
    indexObj := self at: i.
    indexObj size == offset
      ifTrue: [ | col found expected |
        col := IdentityBag new.
        indexObj btreeRoot btreeRootNode _findAndCountKey: nextObj value: anNsc root: rootObj into: col.
        (found := col size) == (expected := indexObj nscRoot occurrencesOf: rootObj)
          ifFalse: [
            auditor
              btree: indexObj btreeRoot
              incorrectCountForKeyValueRootPairs: self
              key: nextObj
              value: anNsc
              root: rootObj
              found: found
              expected: expected ] ] ].
  1 to: children size do: [ :i | (children at: i)
      auditNscForRootObj: nextObj
      rootObj: rootObj
      using: auditor ]
]

]

{ #category : 'Audit' }
GsSetValuedPathTerm >> auditUpdateCacheFor: anNsc root: root nilOnPathRoots: nilOnPathRoots occurrences: num using: auditor [
  "verify proper accounting for objects with nil on path"

  | prevObj reachableNilOnPathRoots excessNilOnPathCount |
  prevObj := #_incompletePathTraversal.
  reachableNilOnPathRoots := self findAllReachableNilOnPathRoots.
  nilOnPathRoots ifNotNil: [ reachableNilOnPathRoots addAll: nilOnPathRoots ].
  anNsc do: [ :nextObj |
     updateBtree
       ifNil: [
        nextObj ~~ prevObj
          ifTrue: [ | newNum |
            newNum := num * (anNsc occurrencesOf: nextObj).
            1 to: children size do: [ :i |
              (children at: i)
                auditUpdateCacheFor: nextObj
                root: root
                nilOnPathRoots: reachableNilOnPathRoots
                occurrences: newNum
                using: auditor ] ] ].
    prevObj := nextObj ].
  (((excessNilOnPathCount := reachableNilOnPathRoots occurrencesOf: root) > 0) and: [ nilOnPathRoots isNil ])
    ifTrue: [
      "uppermost set valued term is where the accounts must balance"
      auditor setValuedPathTermExtraNilOnPathCount: self root: root excessCount: excessNilOnPathCount ].

]

{ #category : 'Updating Indexes' }
GsSetValuedPathTerm >> cleanupDependencyListFor: anNsc [
  ""

  | sz iList |
  (nil == anNsc _or: [ self size == 0 ])
    ifTrue: [ ^ self ].
  anNsc class isNsc
    ifFalse: [ ^ self _errorPathObjectNotAnNsc: anNsc ].
  (iList := anNsc _indexedPaths) ~~ nil
    ifTrue: [ iList removeIndexesInPathTerm: self for: anNsc ].
  sz := children size.
  anNsc
    do: [ :setElement |
      | depList |
      needsDepList
        ifTrue: [
          depList := DependencyList for: setElement.
          depList ~~ nil
            ifTrue: [ depList removeCompletelyPathTerm: self for: setElement ] ].
      1 to: sz do: [ :j | (children at: j) cleanupDependencyListFor: setElement ] ]

]

{ #category : 'Collection Based Modification' }
GsSetValuedPathTerm >> findRootObjectMaps: rootObjectMap for:  anNsc [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  anNsc
    do: [ :nextObj |
      "Make a full pass for each of the enumerated instance variables"
      updateBtree ~~ nil
        ifTrue: [
          "btree exists for receiver, collect the root objects, where nextObj is the key and anNsc
           is the value."
          self findRootObjectMaps: rootObjectMap forKey: nextObj value: anNsc ].
      (self children isEmpty)
        ifTrue: [ ^ self ].
      1 to: children size do: [:i |
        (children at: i) findRootObjectMaps: rootObjectMap for:  nextObj] ].

]

{ #category : 'Collection Based Modification' }
GsSetValuedPathTerm >> findRootObjectMaps: rootObjectMap for: anNsc oldValue: oldValue [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  anNsc
    do: [ :nextObj |
      "Make a full pass for each of the enumerated instance variables"
      updateBtree ~~ nil
        ifTrue: [
          "btree exists for receiver, collect the root objects, where nextObj is the key and anNsc
           is the value."
          self findRootObjectMaps: rootObjectMap forKey: nextObj value: anNsc ].
      (self children isEmpty)
        ifTrue: [ ^ self ].
      1 to: children size do: [:i |
        (children at: i) findRootObjectMaps: rootObjectMap for:  nextObj] ].

]

{ #category : 'Collection Based Modification' }
GsSetValuedPathTerm >> findRootObjectsFor: anNsc [
  "Traverse btree nodes from receiver to leaf term. Collect rootObjects for all btree elements
   for which anObject is a key or parent of a key."

  | rootObjects roots |
  rootObjects := IdentityBag new.
  anNsc
    do: [ :nextObj |
      "Make a full pass for each of the enumerated instance variables"
      updateBtree ~~ nil
        ifTrue: [
          "btree exists for receiver, collect the root objects, where nextObj is the key and anNsc
           is the value."
          roots := self findRootObjectsForKey: nextObj value: anNsc.
          rootObjects := rootObjects _union: roots ].
      (self children isEmpty)
        ifTrue: [ ^ rootObjects ].
      1 to: children size do: [:i |
        roots := (children at: i) findRootObjectsFor: nextObj.
        rootObjects := rootObjects _union: roots ] ].
  ^ rootObjects

]

{ #category : 'Modification' }
GsSetValuedPathTerm >> findRootObjectsForKey: aKey value: aValue [
  ^ updateBtree findRootObjectsComparingForKey: aKey value: aValue

]

{ #category : 'Testing' }
GsSetValuedPathTerm >> indicatesIndexOnNscElements [
  "Returns true if the path term indicates that the index is on elements of the
 NSC itself.  This is the case if the path term ends in '*' (that is, this
 path term has no children)."

  ^ children isEmpty

]

{ #category : 'Testing' }
GsSetValuedPathTerm >> indicatesNsc [
  "Returns true, the receiver indicates that the index is on an
 instance variable that is expected to be a collection (i.e., set valued pathterm."

  ^ true

]

{ #category : 'Testing' }
GsSetValuedPathTerm >> isSetValuedTerm [
  "Returns true if the receiver indicates a set-valued instance variable."

  ^ true

]

{ #category : 'Accessing' }
GsSetValuedPathTerm >> nextObj: anNsc do: aBlock [
  anNsc do: aBlock

]

{ #category : 'Removing' }
GsSetValuedPathTerm >> removeDirectMappingFor: anObject root: root nsc: anNsc logging: doLogging [
  "There is a range index directly on the elements of the NSC.  Update the
 B-tree."

  updateBtree ~~ nil
    ifTrue: [ " remove it from the B-tree "
      (updateBtree btreeRemoveKey: anObject value: anNsc root: root)
         ifFalse: [ self _errorObjectNotInBtree: anObject value: anNsc root: root ] ]

]

{ #category : 'Removing' }
GsSetValuedPathTerm >> removeMappingsFor: anNsc root: rootObject lastOne: aBoolean logging: doLogging [
  "Remove entries in the btree and dependency lists for anNsc."

  | depList isLast |
  self size == 0
    ifTrue: [ ^ self ].
  anNsc
    do: [ :setElement |
      updateBtree ~~ nil
        ifTrue: [
          " remove an entry from the B-tree "
          (updateBtree btreeRemoveKey: setElement value: anNsc root: rootObject)
            ifFalse: [ self _errorObjectNotInBtree: setElement value: anNsc root: rootObject ].
          needsDepList
            ifTrue: [
              " remove dependency list entry for next object "
              DependencyList removePathTerm: self for: setElement ] ].
      (nil ~~ setElement and: [ children isEmpty not ])
        ifTrue: [
          aBoolean
            ifTrue: [
              | last |
              last := (children at: 1) _isLastOccurrenceInIndexFor: setElement.
              isLast := aBoolean and: [ last ] ]
            ifFalse: [ isLast := false ].
          " make recursive call to remove mappings "
          1 to: children size do: [ :i |
            (children at: i) removeMappingsFor: setElement root: rootObject lastOne: isLast logging: doLogging ].
          isLast
            ifFalse: [
              "check again in case there are no more references left in btree"
              isLast := ((children at: 1) findReachableRootsFor: setElement) isEmpty ].
          isLast
            ifTrue: [
              " remove dependency list entries for next object "
              (depList := DependencyList for: setElement) ~~ nil
                ifTrue: [ depList removePathTerms: children for: setElement logging: doLogging ] ] ] ]

]
