"
A GsSftpSocket represents a sftp socket connection, allowing you to perform SFTP operations to a
remote node.

To create a GsSfptSocket connection, create a client socket using methods inherited from GsSocket, and 
connect using sshConnect. 

GsSftpSocket provides some file and directory operations. To read and write remote files, create an 
instance of GsSftpRemoteFile.  

instance variables:
  openFiles -- instance of GsSftpRemoteFile that are open on this instance.

For example, the following connects to the host hostnameOrIP and fetches directory contents:
  sshFtpSock := GsSftpSocket newClient.
  sshFtpSock connectToHost: <aHostnameOrIP> timeoutMs: 2000.
  sshFtpSock userId: <userName>.
  sshFtpSock password: <passwordString>. 
  sshFtpSock sshConnect.
  result := sftpSock contentsOfRemoteDirectory: '.'.
  sshFtpSock close
"
Class {
	#name : 'GsSftpSocket',
	#superclass : 'GsSshSocket',
	#instVars : [
		'openFiles'
	],
	#category : nil
}

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample1 [

"This example returns an Array of strings which represent the files in the default directory of the sftp server excluding files that have names that start with a dot (.)."

"GsSftpSocket remoteDirectoryExample1"

| sftpSock result  |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
result := sftpSock contentsOfRemoteDirectoryNoDotFiles: '.' .
sftpSock close.
^result

]

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample2 [

"This example returns an Array of strings which represent the files in the default directory of the sftp server."

"GsSftpSocket remoteDirectoryExample2"

| sftpSock result  |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
result := sftpSock contentsOfRemoteDirectory: '.' .
sftpSock close.
^result

]

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample3 [

"This example returns an Array pairs where the odd elements are strings which represent the files names and the even elements are instances of GsFileStat."

"GsSftpSocket remoteDirectoryExample3"

| sftpSock result  |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
result := sftpSock contentsAndStatDetailsOfRemoteDirectory: '.' .
sftpSock close.
^result

]

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample4 [

"This example returns a StringKeyValueDictionary where the keys are strings which represent the files names and the values instances of GsFileStat."

"GsSftpSocket remoteDirectoryExample4"

| sftpSock result  |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
result := sftpSock contentsOfRemoteDirectoryAsDictionary: '.' .
sftpSock close.
^result

]

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample5 [

"This example creates a remote directory, retrieves its stat info, then removes the remote directory and returns the stat info."

"GsSftpSocket remoteDirectoryExample5"

| sftpSock result dirName |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
dirName := GsUuidV4 new asString.
sftpSock createRemoteDirectory: dirName.
result := sftpSock stat: dirName .
sftpSock removeRemoteDirectory: dirName ; close .
^result

]

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample6 [

"This example creates empty files and searches the remote directory using file name patterns."

"GsSftpSocket remoteDirectoryExample6"

| sftpSock files allFoo allBar myDir myFiles |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
files := { 'foo.bar' . 'bar.foo' . 'a.bar' . 'b.bar' . 'c.bar' }.
"Create zero-size remote files"
myDir := './', GsUuidV4 new asString.
sftpSock createRemoteDirectory: myDir.
myFiles := files collect:[:e| myDir, '/', e].
myFiles do:[:fn| (GsSftpRemoteFile createOrOverwriteRemoteFile: fn withSftpSocket: sftpSock) close ].
allFoo := sftpSock contentsOfRemoteDirectory: myDir matchPattern: '*.foo' .
allBar := sftpSock contentsOfRemoteDirectory: myDir matchPattern: '*.bar' .
myFiles do:[:fn| sftpSock removeRemoteFile: fn  ].
sftpSock removeRemoteDirectory: myDir.
sftpSock close.
^Array with: allFoo with: allBar

]

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample7 [

"This example returns a list of subdirectories in the default directory of the sftp server."

"GsSftpSocket remoteDirectoryExample7"

| sftpSock result  |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
result := sftpSock contentsAndStatDetailsOfRemoteDirectory: '.' .
sftpSock close.
^ result select:[:assoc| assoc value isDirectory]

]

{ #category : 'Examples' }
GsSftpSocket class >> remoteDirectoryExample8 [

"This example returns a list of regular files in the default directory of the sftp server."

"GsSftpSocket remoteDirectoryExample8"

| sftpSock result  |
sftpSock := GsSftpRemoteFile getSftpSocketExample.
result := sftpSock contentsAndStatDetailsOfRemoteDirectory: '.' .
sftpSock close.
^ result select:[:assoc| assoc value isDirectory not]

]

{ #category : 'Private' }
GsSftpSocket >> _addReference: aGsSftpRemoteFile [

openFiles add: aGsSftpRemoteFile .
^self

]

{ #category : 'Private' }
GsSftpSocket >> _removeReference: aGsSftpRemoteFile [

openFiles remove: aGsSftpRemoteFile otherwise: nil .
^self

]

{ #category : 'Directory Operations' }
GsSftpSocket >> contentsAndStatDetailsOfRemoteDirectory: dirname [

"Returns an Array of Associations describing the contents of the given remote directory.
The association keys are the names of files and directories contained in dirname, and the
association values are instances of GsFileStat, which contain details regarding the file named
in the key."

^ self contentsAndStatDetailsOfRemoteDirectory: dirname matchPattern: nil

]

{ #category : 'Directory Operations' }
GsSftpSocket >> contentsAndStatDetailsOfRemoteDirectory: dirname matchPattern: aString [

"Returns an Array of Associations describing the contents of the given remote directory.
The association keys are the names of files and directories contained in dirname which match the pattern
in aString, and the association values are instances of GsFileStat, which contain details regarding the file named
in the key.
File name pattern matching is done using the fnmatch() function. Refer to the man page for fnmatch for further details.
If aString is nil, no matching is done and all file names are returned except for directories '.' and '..' ."

^ self _twoArgSshPrim: 52 with: dirname with: aString

]

{ #category : 'Directory Operations' }
GsSftpSocket >> contentsOfRemoteDirectory: dirname [

"Returns an Array of file names describing the contents of the given directory."

^ self contentsOfRemoteDirectory: dirname matchPattern: nil

]

{ #category : 'Directory Operations' }
GsSftpSocket >> contentsOfRemoteDirectory: dirname matchPattern: aString [

"Returns an Array of file names describing the contents of the given directory which match the search pattern in aString.
 File name matching is done using the fnmatch() function. Refer to the man page for fnmatch for further details.
If aString is nil, no matching is done and all file names are returned except for directories '.' and '..' ."

^ self _twoArgSshPrim: 53 with: dirname with: aString

]

{ #category : 'Directory Operations' }
GsSftpSocket >> contentsOfRemoteDirectoryAsDictionary: dirname [
"Returns StringKeyValueDictionary describing the contents of the given remote directory.
The keys are the names of all files and directories contained in dirname and the values
are instances of GsFileStat, which contains details regarding the file named by the key."

^ self contentsOfRemoteDirectoryAsDictionary: dirname matchPattern: nil

]

{ #category : 'Directory Operations' }
GsSftpSocket >> contentsOfRemoteDirectoryAsDictionary: dirname matchPattern: aString [
"Returns StringKeyValueDictionary describing the contents of the given remote directory which match the pattern in aString.
File name matching is done using the fnmatch() function. Refer to the man page for fnmatch for further details.
If aString is nil, no matching is done and all file names are returned except for directories '.' and '..' .
The keys are the names of all files and directories contained in dirname and the values
are instances of GsFileStat, which contains details regarding the file named by the key."

| result array |
array := self contentsAndStatDetailsOfRemoteDirectory: dirname matchPattern: aString.
result := StringKeyValueDictionary new: array size.
array do:[:assoc| result at: assoc key put: assoc value ].
^ result


]

{ #category : 'Directory Operations' }
GsSftpSocket >> contentsOfRemoteDirectoryNoDotFiles: dirname [

"Same as the contentsOfRemoteDirectory: method except files and directorys which start with a period character (.) are excluded from the result."

^ (self contentsOfRemoteDirectory: dirname) reject:[:each| each first == $. ]

]

{ #category : 'Directory Operations' }
GsSftpSocket >> createRemoteDirectory: dirname [

"Creates the named directory on the server host. Returns the receiver on success or raises an exception on error.
The new directory will have the default permissions of 8r770"

^ self createRemoteDirectory: dirname mode: nil

]

{ #category : 'Directory Operations' }
GsSftpSocket >> createRemoteDirectory: dirname mode: modeInt [

"Creates the named directory on the server host.
The mode modeInt must be a SmallInteger which is a valid directory mode.
 If modeInt == nil, a value of 8r770 will be used.
Returns the receiver on success or raises an exception on error. "

^ self _twoArgSshPrim: 50 with: dirname with: modeInt

]

{ #category : 'Directory Operations' }
GsSftpSocket >> currentRemoteDirectory [

"Returns a string which describes the current remote directory. Raises an exception on error. "

^ self _zeroArgSshPrim: 50

]

{ #category : 'Private' }
GsSftpSocket >> initializeAfterConnect [

openFiles := IdentitySet new.
^super initializeAfterConnect

]

{ #category : 'Private' }
GsSftpSocket >> initializeAfterConnect: host [

openFiles := IdentitySet new.
^super initializeAfterConnect: host

]

{ #category : 'File Operations' }
GsSftpSocket >> lstat: aFileName [

"Functions the same as the stat: method except when the remote file is a symbolic link.
In that case, return an instance of GsFileStat describing the symbolic link rather than the
file referenced by the link."

^ self _oneArgSshPrim: 55 with: aFileName

]

{ #category : 'File Operations' }
GsSftpSocket >> remoteFileExists: aFileOrDirectory [

"Answer a Boolean indicating if the remote file or directory exists. Raises an exception on error."

^ [(self stat: aFileOrDirectory) class == GsFileStat ]
	on: SshSocketError
	do:[:ex| (ex reason findPattern: { 'No such file' } startingAt: 1) ~~ 0
		ifTrue:[ false ]  "file does not exist"
		ifFalse:[ ex signal ] "some other error"
]

]

{ #category : 'Directory Operations' }
GsSftpSocket >> removeRemoteDirectory: dirname [

"Removes the named directory from the server host. Returns the receiver on success or raises an exception on error. "

^ self _oneArgSshPrim: 51 with: dirname

]

{ #category : 'File Operations' }
GsSftpSocket >> removeRemoteFile: filename [

"Removes the named file from the server host. Returns the receiver on success or raises an exception on error. "

^ self _oneArgSshPrim: 53 with: filename

]

{ #category : 'File Operations' }
GsSftpSocket >> renameRemoteFile: oldName to: newName [

"Changes the name of a remote file or directory from oldName to newName.
Raises an exception if the remote file or directory does not exist or if the
rename action is not permitted."

^ self _twoArgSshPrim: 51 with: oldName with: newName

]

{ #category : 'File Operations' }
GsSftpSocket >> stat: aFileName [

"Return an instance of GsFileStat describing the remote file.
Raise an exception if the file does not exist or cannot be accessed."

^ self _oneArgSshPrim: 54 with: aFileName

]

{ #category : 'Private' }
GsSftpSocket >> _sshConnect [

"Connects the receiver to the ssh server.
 Returns true on success, false if the connection is in progress on a nonblocking socket, or raises an exception on error.
If the method returns false, the application should wait for the socket to become read-ready and then send
the message again."

| ret |
(ret := self _zeroArgSshPrim: 3)
	 ifTrue:[ self makeBlocking ]. "Workaround for libssh issue #58"
^ ret
]

{ #category : 'Client Operations (Non-blocking)' }
GsSftpSocket >> nbSshConnect [
  "libssh does not support sftp operations on a non-blocking ssh connection, 
   see  https://gitlab.com/libssh/libssh-mirror/-/issues/58"

  ^ ImproperOperation signal: 'nbSshConnect not supported on a GsSftpSocket'.
]

{ #category : 'Client Operations (Non-blocking)' }
GsSftpSocket >> nbSshConnectTimeout: timeoutMs [
  "libssh does not support sftp operations on a non-blocking ssh connection, 
   see  https://gitlab.com/libssh/libssh-mirror/-/issues/58"

  ^ ImproperOperation signal: 'nbSshConnectTimeout: not supported on a GsSftpSocket'.
]

{ #category : 'Client Operations (Non-blocking)' }
GsSftpSocket >> makeNonBlocking [
  "libssh does not support sftp operations on a non-blocking ssh connection, 
   see  https://gitlab.com/libssh/libssh-mirror/-/issues/58"

  ^ ImproperOperation signal: 'non blocking operations on a GsSftpSocket are not supported'.
]

