"
GsTlsCredential is a class that encapsulates TLS private keys, public keys,
  and X509 certificates. Instances contain a hidden reference to C pointer
  to the OpenSSL representation of the TLS object.
"
Class {
	#name : 'GsTlsCredential',
	#superclass : 'Object',
	#gs_options : [
		'instancesNonPersistent'
	],
	#gs_reservedoop : '249857',
	#category : 'X509'
}

{ #category : 'Storing and Loading' }
GsTlsCredential class >> loadFrom: passiveObj [

"Creates and returns an active instance of the receiver from the passive form
 of the object"

| inst str |
str := passiveObj readObject.
inst := self isOpenSslClass 
	ifTrue:[ self newFromPemString: str ]
	ifFalse:[ self newFromOpenSshString: str ].
passiveObj hasRead: inst.
^inst.
]

{ #category : 'Private' }
GsTlsCredential class >> tls0ArgClassPrim: opCode [

"
  OpCode   Function
=========================================================================
     0      instance: asPemString
     1      instance: hash
     2      instance: algorithm
     3      instance: sslAlgorithm
     4      instance: description
     5      instance: securityBits
     6      instance: supportsDigitalSignatures
     7      instance: asOpenSshString
     8      instance: asOpenSshStringOneLine
=========================================================================
"

<primitive: 1057>
^ self _primitiveFailed: #tls0ArgClassPrim: args: { opCode }

]

{ #category : 'Private' }
GsTlsCredential class >> tls3ArgPrim: opCode with: aString with: pfArg with: type [

"
  OpCode   Function
=========================================================================
     0      New GsTlsCredential from PEM file, pfArg is the passphrase
     1      New GsTlsCredential from PEM string, pfArg is the passphrase
     2      New GsX509CertificateChain from PEM file
     3      New GsX509CertificateChain from PEM file
     4      New GsTlsPublicKey from a GsX509Certificate
     5      New GsTlsCredential from PEM file, pfArg is a file name
     6      New GsTlsCredential from PEM string, pfArg is a file name
     7      New GsTlsCredential from OpenSSH file, pfArg is the passphrase
     8      New GsTlsCredential from OpenSSH string, pfArg is the passphrase
     9      New GsTlsCredential from OpenSSH file, pfArg is a file name
    10      New GsTlsCredential from OpenSSH string, pfArg is a file name
=========================================================================

  Type      Kind
============================
    1       Private Key
    2       Public Key
    3       X509 Certificate
============================
"

<primitive: 1056>
^ self _primitiveFailed: #tls3ArgPrim:with:with:with: args: { opCode . aString . pfArg . type }

]

{ #category : 'Private' }
GsTlsCredential >> _validateAlgorithm: expectedSymbol [

^ self algorithm == expectedSymbol
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal:
  ('Incorrect key algorithm. Expected: ', expectedSymbol asString, ' Actual: ', self algorithm asString)]

]

{ #category : 'Private' }
GsTlsCredential >> _validateCreatesDigitalSignatures [
^ self canCreateDigitalSignatures
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'This key is not valid for creating digital signatures']

]

{ #category : 'Private' }
GsTlsCredential >> _validateIsPrivateKey [
^ self isPrivateKey
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'Incorrect key type: expected a private key']

]

{ #category : 'Private' }
GsTlsCredential >> _validateIsPublicKey [
^ self isPublicKey
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'Incorrect key type: expected a public key']

]

{ #category : 'Private' }
GsTlsCredential >> _validateIsRsa [

^ self algorithm == #RSA
     ifTrue:[ true ]
    ifFalse:[  CryptoError signal: 'An RSA key or certificate is required']

]

{ #category : 'Private' }
GsTlsCredential >> _validateValidatesDigitalSignatures [
^ self canVerifyDigitalSignatures
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'This key is not valid for validating digital signatures']

]

{ #category : 'Testing' }
GsTlsCredential >> = anObj [

^ self tls1ArgInstPrim: 0 with: anObj

]

{ #category : 'Accessing' }
GsTlsCredential >> algorithm [

"Answers a Symbol indicating the type of high-level PKI algorithm the
receiver uses. The high-level PKI algorithms currently supported are:

   #RSA - Rivest-Shamir-Adleman
   #DSA - Data Signature Algorithm
   #EC  - Elliptic Curve Cryptography

 All high-level algorithm have various sub-types. Use the sslAlgorithm
 method to obtain information about the specific PKI algorithm of the
 receiver.

 If the receiver is an instance of GsX509Certificate, the result
 indicates the algorithm of the public key contained therein.

 Returns #UNSUPPORTED if the algorithm could not be determined."

^ self tls0ArgInstPrim: 2

]

{ #category : 'Converting' }
GsTlsCredential >> asPemString [

^ self tls0ArgInstPrim: 0

]

{ #category : 'Testing' }
GsTlsCredential >> canCreateDigitalSignatures [
  self subclassResponsibility: #canCreateDigitalSignatures

]

{ #category : 'Testing' }
GsTlsCredential >> canVerifyDigitalSignatures [
  self subclassResponsibility: #canVerifyDigitalSignatures

]

{ #category : 'Printing' }
GsTlsCredential >> description [
"Answers a String obtained from SSL describing the receiver. The contents
 and format of the string vary depending on the receiver's class."

^ self tls0ArgInstPrim: 4

]

{ #category : 'Hashing' }
GsTlsCredential >> hash [

^ self tls0ArgInstPrim: 1

]

{ #category : 'Testing' }
GsTlsCredential >> isPrivateKey [
  self subclassResponsibility: #isPrivateKey

]

{ #category : 'Testing' }
GsTlsCredential >> isPublicKey [
  self subclassResponsibility: #isPublicKey

]

{ #category : 'Testing' }
GsTlsCredential >> isX509Certificate [
  self subclassResponsibility: #isX509Certificate

]

{ #category : 'Comparing' }
GsTlsCredential >> matches: anotherKey [

"Determines if the receiver and anotherKey match each other as a valid
 public-private key pair. If the receiver is a public key, anotherKey
 is expected to be a private key. If the receiver is a private key,
 another key is expected to be a public key.

 If the receiver or anotherKey is an instance of GsX509Certificate, the
 public key is extracted from the certificate and the comparison is
 performed as described above.

 RSA and DSA key pairs match if both keys use the same modulus.
 Elliptic curve key pairs match if both keys use the same curve and the
 same point on that curve.

 Returns true if the keys match, false if they do not match.
 Raises an error if the receiver and anotherKey are both the same type
 (public or private keys)."

^ self tls1ArgInstPrim: 2 with: anotherKey

]

{ #category : 'Accessing' }
GsTlsCredential >> securityBits [
"Answers a SmallInteger representing the number of security bits of the
 receiver.  Bits of security is defined in NIST SP800-57.  See
   https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-57pt1r4.pdf
 for further information.

 For instances of GsX509Certificate, answers the security bits of the public
 key contained therein."

^ self tls0ArgInstPrim: 5

]

{ #category : 'Accessing' }
GsTlsCredential >> sslAlgorithm [

"Answers a Symbol indicating the SSL type of PKI algorithm the
 receiver uses. The values returned by OpenSSL as of version
 1.1.1 are:

    #EVP_PKEY_NONE
    #EVP_PKEY_RSA
    #EVP_PKEY_RSA2
    #EVP_PKEY_RSA_PSS
    #EVP_PKEY_DSA
    #EVP_PKEY_DSA1
    #EVP_PKEY_DSA2
    #EVP_PKEY_DSA3
    #EVP_PKEY_DSA4
    #EVP_PKEY_DH
    #EVP_PKEY_DHX
    #EVP_PKEY_EC
    #EVP_PKEY_SM2
    #EVP_PKEY_HMAC
    #EVP_PKEY_CMAC
    #EVP_PKEY_SCRYPT
    #EVP_PKEY_TLS1_PRF
    #EVP_PKEY_HKDF
    #EVP_PKEY_POLY1305
    #EVP_PKEY_SIPHASH
    #EVP_PKEY_X25519
    #EVP_PKEY_ED25519
    #EVP_PKEY_X448
    #EVP_PKEY_ED448

 If the recevier is an instance of GsX509Certificate, the result
 indicates the algorithm of the public key contained therein."

^ self tls0ArgInstPrim: 3

]

{ #category : 'Testing' }
GsTlsCredential >> supportsDigitalSignatures [
"Answers a Boolean indicating if the receiver supports digital signatures.
 The type of support offered, either signing or verifying, depends upon the
 class of the receiver."

^ self tls0ArgInstPrim: 6

]

{ #category : 'Private' }
GsTlsCredential >> tls0ArgInstPrim: opCode [

"
  OpCode   Function
=========================================================================
     0      instance: asPemString
     1      instance: hash
     2      instance: algorithm
     3      instance: sslAlgorithm
     4      instance: description
     5      instance: securityBits
     6      instance: supportsDigitalSignatures
     7      instance: asOpenSshString
     8      instance: asOpenSshStringOneLine
     9      instance: subjectName
    10      instance: issuerName
    11      instance: notBeforeTimeGmtSeconds
    12      instance: notAfterTimeGmtSeconds
    13      instance: subjectAlternateNames
=========================================================================
"

<primitive: 1057>
^ self _primitiveFailed: #tls0ArgInstPrim: args: { opCode }

]

{ #category : 'Private' }
GsTlsCredential >> tls1ArgInstPrim: opCode with: anObj [

"
  OpCode   Function
=========================================================================
     0      instance: =
     1      instance: asPublicKey
     2      instance: matches:
     3      instance: encrypt:
     4      instance: decrypt:
=========================================================================
"

<primitive: 1058>
^ self _primitiveFailed: #tls1ArgInstPrim:with: args: { opCode . anObj }

]

{ #category : 'Storing and Loading' }
GsTlsCredential >> writeTo: passiveObj [
"Writes the passive form of the receiver into passiveObj, expressed as
 a PEM string."

| str |
passiveObj writeClass: self class.
str := self isOpenSshKey
	ifTrue:[ self asOpenSshString ]
	ifFalse:[ self asPemString ].
str writeTo: passiveObj .
passiveObj space

]

{ #category : 'Converting' }
GsTlsCredential >> asOpenSshString [

"Returns a String representing the receiver in OpenSSH base 64 format.
 For private keys, base64 text lines are limited to 70 characters."
 
  ^ self tls0ArgInstPrim: 7
]


{ #category : 'Converting' }
GsTlsCredential >> asOpenSshStringOneLine [

"Returns a String representing the receiver in OpenSSH base 64 format.
 Base64 text is placed on a single line"
 
  ^ self tls0ArgInstPrim: 8
]


{ #category : 'Testing' }
GsTlsCredential >> isOpenSshKey [
  ^ self class isOpenSshClass

]

{ #category : 'Testing' }
GsTlsCredential >> isOpenSslKey [
  ^ self class isOpenSslClass

]

{ #category : 'Converting' }
GsTlsCredential >> asOpenSslKey [
  self subclassResponsibility: #asOpenSslKey

]

{ #category : 'Converting' }
GsTlsCredential >> asOpenSshKey [
  self subclassResponsibility: #asOpenSshKey

]

{ #category : 'Class Membership' }
GsTlsCredential class >> speciesForOpenSslPublicKey [
  ^ GsTlsPublicKey

]

{ #category : 'Class Membership' }
GsTlsCredential class >> speciesForOpenSshPublicKey [
  ^ GsSshPublicKey

]

{ #category : 'Class Membership' }
GsTlsCredential class >> speciesForOpenSslPrivateKey [
  ^ GsTlsPrivateKey

]

{ #category : 'Class Membership' }
GsTlsCredential class >> speciesForOpenSshPrivateKey [
  ^ GsSshPrivateKey

]

{ #category : 'Class Membership' }
GsTlsCredential class >> isOpenSshClass [

^ self subclassResponsibility: #isOpenSshClass
]

{ #category : 'Class Membership' }
GsTlsCredential class >> isOpenSslClass [

^ self isOpenSshClass not
]

{ #category : 'Testing' }
GsTlsCredential >> isRsa [

"Answer a Boolean indicating if the receiver uses RSA cryptography"
^ self algorithm == #RSA
]

{ #category : 'Testing' }
GsTlsCredential >> isDsa [

"Answer a Boolean indicating if the receiver uses DSA cryptography"
^ self algorithm == #DSA
]

{ #category : 'Testing' }
GsTlsCredential >> isEllipticCurve [

"Answer a Boolean indicating if the receiver uses elliptic curve cryptography"
^ self algorithm == #EC
]
