"
GsTlsPrivateKey encapsulates a TLS/SSL private key.
"
Class {
	#name : 'GsTlsPrivateKey',
	#superclass : 'GsTlsCredential',
	#category : 'X509'
}

{ #category : 'Instance Creation' }
GsTlsPrivateKey class >> newFromPemFile: fileNameString [
"Reads data from the given file name in PEM format and creates a new instance
 of the receiver. If the PEM file contains multiple keys, only the
 first is read.

 Raises an exception if the file is not in PEM format or if
 the type of object in the file does not match the receiver.

 The private key is assumed to not have a passphrase."

^ self tls3ArgPrim: 0 with: fileNameString with: nil with: 1

]

{ #category : 'Instance Creation' }
GsTlsPrivateKey class >> newFromPemFile: fileNameString withPassphrase: aPf [
"Reads data from the given file name in PEM format and creates a new instance
 of the receiver. If the PEM file contains multiple keys, only the
 first is read.

 Raises an exception if the file is not in PEM format or if
 the type of object in the file does not match the receiver or if the
 passphrase is incorrect."

^ self tls3ArgPrim: 0 with: fileNameString with: aPf with: 1

]

{ #category : 'Instance Creation' }
GsTlsPrivateKey class >> newFromPemFile: fileNameString withPassphraseFile: aPfFile [
"Reads data from the given file name in PEM format and the passphrase from 
 the text file aPfFile, then creates a new instance of the receiver.
 If the PEM file contains multiple keys, only the first key is read.

 Raises an exception if the file is not in PEM format or if
 the type of object in the file does not match the receiver or if 
 aPfFile does not exist or cannot be read."
 
^ self tls3ArgPrim: 5 with: fileNameString with: aPfFile with: 1

]

{ #category : 'Instance Creation' }
GsTlsPrivateKey class >> newFromPemString: aPemString [
"Creates a new instance of the receiver based on the PEM string. If the PEM
 string contains multiple keys, only the first is read.

 Raises an exception if the string is not in PEM format or if the type
 of object in the PEM string does not match the receiver.

 The private key is assumed to not have a passphrase."

^ self tls3ArgPrim: 1 with: aPemString with: nil with: 1

]

{ #category : 'Instance Creation' }
GsTlsPrivateKey class >> newFromPemString: aPemString withPassphrase: aPf [
"Creates a new instance of the receiver based on the PEM string. If the PEM
 string contains multiple keys, only the first is read.

 Raises an exception if the string is not in PEM format or if the type
 of object in the PEM string does not match the receiver or if the
 passphrase is incorrect."

^ self tls3ArgPrim: 1 with: aPemString with: aPf with: 1

]

{ #category : 'Instance Creation' }
GsTlsPrivateKey class >> newFromPemString: aPemString withPassphraseFile: aPfFile [
"Creates a new instance of the receiver based on the PEM string. If the PEM
 string contains multiple keys, only the first is read. Reads the passphrase
 from the file aPfFile.

 Raises an exception if the string is not in PEM format, if the type 
 of object in the PEM string does not match the receiver, if the 
 passphrase is incorrect, or if aPfFile does not exist or cannot
 be read."

^ self tls3ArgPrim: 6 with: aPemString with: aPfFile with: 1

]

{ #category : 'Class Membership' }
GsTlsPrivateKey class >> speciesForPublicKey [
  ^ GsTlsPublicKey

]

{ #category : 'Private' }
GsTlsPrivateKey >> _validateIsNotRsaPss [
^ self sslAlgorithm ~~ #EVP_PKEY_RSA_PSS
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'Incorrect key type: illegal RSS_PSS private key']

]

{ #category : 'Private' }
GsTlsPrivateKey >> _validateIsRsa [
^ self algorithm == #RSA
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'Incorrect key algorithm: expected an RSA private key']

]

{ #category : 'Private' }
GsTlsPrivateKey >> _validateIsRsaPss [
^ self sslAlgorithm == #EVP_PKEY_RSA_PSS
    ifTrue:[ true ]
   ifFalse:[ CryptoError signal: 'Incorrect key type: expected an RSS_PSS private key']

]

{ #category : 'Converting' }
GsTlsPrivateKey >> asPublicKey [
"Answers a new object which represents the public key for the receiver.
 The class of the new instance is the result of the message
 #speciesForPublicKey, which is GsTlsPublicKey by default"

^ self tls1ArgInstPrim: 1 with: self speciesForPublicKey

]

{ #category : 'Testing' }
GsTlsPrivateKey >> canCreateDigitalSignatures [
^ self supportsDigitalSignatures

]

{ #category : 'Testing' }
GsTlsPrivateKey >> canVerifyDigitalSignatures [
"Only public keys and certificates may be used to verify signatures."
^ false

]

{ #category : 'Testing' }
GsTlsPrivateKey >> isPrivateKey [
 ^ true

]

{ #category : 'Testing' }
GsTlsPrivateKey >> isPublicKey [
 ^ false

]

{ #category : 'Testing' }
GsTlsPrivateKey >> isX509Certificate [
 ^ false

]

{ #category : 'Class Membership' }
GsTlsPrivateKey >> speciesForPublicKey [
  ^ self class speciesForPublicKey

]

{ #category : 'Decrypting' }
GsTlsPrivateKey >> decrypt: aByteArray [
"Decrypts aByteArray which was previously encrypted with the receiver's matching 
 public key. Only RSA keys are supported for encrypting and decrypting.

 aByteArray must be an instance of a ByteArray.

 Returns a new instance ByteArray on success.
 Raises an exception on error."

^ self tls1ArgInstPrim: 4 with: aByteArray

]

{ #category : 'Examples' }
GsTlsPrivateKey class >> encryptDecryptExample [

|privKey pubKey poe crypt poe2 |

privKey := GsTlsPrivateKey newFromPemString:
'-----BEGIN PRIVATE KEY-----
MIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBAMBXQWaLW7QhuWvO
17X5c8LYDbw65if4QR2ATaNZ0Cq3Q6Q+aCsnmT1Q3AvKwUx0Y3PnASLLl0zNQN94
nbNP81vT/0D4J0Ch1HwGUErZReLDN2DSoUELdnmbrgR10Glriy3HD6wb0q+h/N45
8JpR6sTYhVlLtvnGYAJL2OW5kVRVAgMBAAECgYEAl6My+Hld7wG3gXstLVZhIXfc
PE3jLhfWnj+M9f/U0hhxx4c78OnjMigRk2piQrhvv+ybRKdlvTMEtioNilS58ogV
/I5dRoHsRd2opsUeDMloRdOMcL6HhinjGtPFqY/QXdeKKLLAfR2Mw1GKaro55hQv
DRqRk01Gd/KvWij5roECQQDz+9VW54+qolrDH2iw0BBeeYBog/ELA8vNw7te4OWH
0TrPHUDyvHkJCQ/GSWHLVQ2Rw/WoyKMTn7u/LF8pspp9AkEAydBN8IPdID8m5rk8
JYr1iPceAyoI5ZeUA1cqrFjx4HdtyVAuLGQAvVSY7fJaSzlrBeH8HVa3GlDJ3Qr1
Tt1wuQJAJnbhX14KTEBkRrbA7n8e1YYaNF/4tF/Y1YuyEncqOItH1jcqcho8iqwf
DIetHz09cmmOZRmcfA+GrdD0/8HkkQJANUUxvYHhFYj16MMOWE6Uv0GTf3xR+uCG
5lbU4cdcmUaNCS2L8pW3CELTV0O4h9CxKk1bchcYn+6hSiKBW/7hqQJBAKcucW2t
HDhYqOhZi+Eq+z7mnua7x867COGlijb3yE80rvmqBOOZc3PtJNr2SKQoBjD++B4w
HSedUWX12Gyb0SE=
-----END PRIVATE KEY-----' .

pubKey := GsTlsPublicKey newFromPemString:
'-----BEGIN PUBLIC KEY-----
MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDAV0Fmi1u0Iblrzte1+XPC2A28
OuYn+EEdgE2jWdAqt0OkPmgrJ5k9UNwLysFMdGNz5wEiy5dMzUDfeJ2zT/Nb0/9A
+CdAodR8BlBK2UXiwzdg0qFBC3Z5m64EddBpa4stxw+sG9KvofzeOfCaUerE2IVZ
S7b5xmACS9jluZFUVQIDAQAB
-----END PUBLIC KEY-----' .

poe := 'Once upon a midnight dreary, while I pondered, weak and weary,'.

"Encrypt using public key. crypt is a ByteArray"
crypt := pubKey encrypt: poe.

"Decrypt using private key and convert ByteArray into String"
poe2 :=  (privKey decrypt: crypt) bytesIntoString .

^ poe = poe2
]


{ #category : 'Examples' }
GsTlsPrivateKey class >> encryptDecryptBase64Example [

|privKey pubKey poe b64 poe2 |

privKey := GsTlsPrivateKey newFromPemString:
'-----BEGIN PRIVATE KEY-----
MIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBAMBXQWaLW7QhuWvO
17X5c8LYDbw65if4QR2ATaNZ0Cq3Q6Q+aCsnmT1Q3AvKwUx0Y3PnASLLl0zNQN94
nbNP81vT/0D4J0Ch1HwGUErZReLDN2DSoUELdnmbrgR10Glriy3HD6wb0q+h/N45
8JpR6sTYhVlLtvnGYAJL2OW5kVRVAgMBAAECgYEAl6My+Hld7wG3gXstLVZhIXfc
PE3jLhfWnj+M9f/U0hhxx4c78OnjMigRk2piQrhvv+ybRKdlvTMEtioNilS58ogV
/I5dRoHsRd2opsUeDMloRdOMcL6HhinjGtPFqY/QXdeKKLLAfR2Mw1GKaro55hQv
DRqRk01Gd/KvWij5roECQQDz+9VW54+qolrDH2iw0BBeeYBog/ELA8vNw7te4OWH
0TrPHUDyvHkJCQ/GSWHLVQ2Rw/WoyKMTn7u/LF8pspp9AkEAydBN8IPdID8m5rk8
JYr1iPceAyoI5ZeUA1cqrFjx4HdtyVAuLGQAvVSY7fJaSzlrBeH8HVa3GlDJ3Qr1
Tt1wuQJAJnbhX14KTEBkRrbA7n8e1YYaNF/4tF/Y1YuyEncqOItH1jcqcho8iqwf
DIetHz09cmmOZRmcfA+GrdD0/8HkkQJANUUxvYHhFYj16MMOWE6Uv0GTf3xR+uCG
5lbU4cdcmUaNCS2L8pW3CELTV0O4h9CxKk1bchcYn+6hSiKBW/7hqQJBAKcucW2t
HDhYqOhZi+Eq+z7mnua7x867COGlijb3yE80rvmqBOOZc3PtJNr2SKQoBjD++B4w
HSedUWX12Gyb0SE=
-----END PRIVATE KEY-----' .

pubKey := GsTlsPublicKey newFromPemString:
'-----BEGIN PUBLIC KEY-----
MIGfMA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDAV0Fmi1u0Iblrzte1+XPC2A28
OuYn+EEdgE2jWdAqt0OkPmgrJ5k9UNwLysFMdGNz5wEiy5dMzUDfeJ2zT/Nb0/9A
+CdAodR8BlBK2UXiwzdg0qFBC3Z5m64EddBpa4stxw+sG9KvofzeOfCaUerE2IVZ
S7b5xmACS9jluZFUVQIDAQAB
-----END PUBLIC KEY-----' .

poe := 'Once upon a midnight dreary, while I pondered, weak and weary,'.

"Encrypt using public key. b64 is a String "
b64 := (pubKey encrypt: poe) asBase64String.

"Decrypt using private key and convert ByteArray into String"
poe2 := (privKey decrypt: (ByteArray fromBase64String: b64)) bytesIntoString .
^ poe = poe2
]

{ #category : 'Class Membership' }
GsTlsPrivateKey class >> isOpenSshClass [

^ false
]

{ #category : 'Converting' }
GsTlsPrivateKey >> asOpenSslKey [
  ^ self
]

{ #category : 'Converting' }
GsTlsPrivateKey >> asOpenSshKey [
  ^ self class speciesForOpenSshPrivateKey newFromOpenSshString: self asOpenSshString
]

