"
Hash tree heap is dbTransient, so contents are not committed.

It's a min-heap sorted by hash value and also keeping an 
index for each hash value.
See e.g. https://en.wikipedia.org/wiki/Heap_(data_structure) 
for explanation of structure and algorithm.

tally is always even, and twice the number of elements in the heap.

HtHeap is used to heap-sort an HtLeafNode being split. This is necessary only in uncommon 
situations where the normal heuristic for splitting a leaf node results in a very uneven split.
"
Class {
	#name : 'HtHeap',
	#superclass : 'Object',
	#type : 'variable',
	#instVars : [
		'tally'
	],
	#gs_options : [
		'dbTransient'
	],
	#category : 'HashTree-Core'
}

{ #category : 'instance creation' }
HtHeap class >> new: count [

	^ (self basicNew: count * 2)
		  initialize;
		  yourself
]

{ #category : 'asserting' }
HtHeap >> assertEmpty [

	tally == 0 ifFalse: [ self error: 'Expected empty heap' ]
]

{ #category : 'auditing' }
HtHeap >> auditEmptyOnto: stream [
	tally = 0
		ifFalse: [ 
			stream
				nextPutAll:
						'Expected heap to be empty with zero tally, but tally is ' , tally printString;
				lf ]
]

{ #category : 'accessing' }
HtHeap >> buildMinHeap [
	"Re-order the heap to satisfy the 'heap property' (i.e. sort it into heap order).
	Note that this implementation allows duplicate key values, so only has to re-order
	elements if they violate the heap constraint.
	
	tally // 4 * 2 is the index of the last value that has children. All past that
	already have the heap property, since they have no children. "

	tally // 4 * 2 to: 2 by: -2 do: [ :i | self minHeapify: i ]
]

{ #category : 'accessing' }
HtHeap >> bulkAddHash: hash index: index [
	"Add the given hash/index pair (both SmallIntegers) without maintaining the heap property."

	tally := tally + 2.
	self
		_basicAt: tally - 1 put: hash;
		_basicAt: tally put: index
]

{ #category : 'enumerating' }
HtHeap >> destructiveKeysAndValuesDo: aBinaryBlock [
	"Remove key/value pairs in ascending key order, evaluating the given block for each.
	This message is not used by TreeDictionary"

	[ tally > 0 ] whileTrue: [
		aBinaryBlock value: (self _basicAt: 1) value: (self _basicAt: 2).
		self _basicAt: 1 put: (self _basicAt: tally - 1).
		self _basicAt: 2 put: (self _basicAt: tally).
		self _basicAt: tally - 1 put: nil.
		self _basicAt: tally put: nil.
		tally := tally - 2.
		self minHeapify: 2 ]
]

{ #category : 'initialization' }
HtHeap >> initialize [

	tally := 0
]

{ #category : 'private' }
HtHeap >> minHeapify: index [
	"index, and the others, point to the value of the cell to be considered.
	The key is one before that."

	| left right smallest |
	left := 2 * index.
	right := left + 2.
	smallest := index.

	(left <= tally and: [
		 (self _basicAt: left - 1) < (self _basicAt: smallest - 1) ]) ifTrue: [
		smallest := left ].

	(right <= tally and: [
		 (self _basicAt: right - 1) < (self _basicAt: smallest - 1) ])
		ifTrue: [ smallest := right ].

	smallest = index ifFalse: [
		| tKey tValue |
		tKey := self _basicAt: index - 1.
		tValue := self _basicAt: index.
		self _basicAt: index - 1 put: (self _basicAt: smallest - 1).
		self _basicAt: index put: (self _basicAt: smallest).
		self _basicAt: smallest - 1 put: tKey.
		self _basicAt: smallest put: tValue.
		self minHeapify: smallest ]
]

{ #category : 'enumerating' }
HtHeap >> minKey [
	"Answer the key that is currently the minimum key, assuming
	that I currently have the heap property."

	^ self _basicAt: 1
]

{ #category : 'enumerating' }
HtHeap >> removeValueWithMinKey [
	"Remove a value with the minimum key, and answer it."

	| result |
	tally > 0 ifFalse: [ LookupError new
							object: self;
							signal: 'Heap is empty'].
	result := self _basicAt: 2.
	self _basicAt: 1 put: (self _basicAt: tally - 1).
	self _basicAt: 2 put: (self _basicAt: tally).
	self _basicAt: tally - 1 put: nil.
	self _basicAt: tally put: nil.
	tally := tally - 2.
	self minHeapify: 2.
	^ result
]
