"
A hash tree set leaf node is an HtLeafNode specialized for use as part of a TreeSet.
See HtLeafNode's class comment for further information.
"
Class {
	#name : 'HtSetLeafNode',
	#superclass : 'HtLeafNode',
	#category : 'HashTree-Core'
}

{ #category : 'instance creation' }
HtSetLeafNode class >> nodeBasicSize [
	"To keep it in a page, limit is 2034 - self instSize
	For us, with seven named instvars, that is 2027.
	2027 is prime, so we'll use that as a table size.
	"

	^ 2027
]

{ #category : 'other leaf access' }
HtSetLeafNode >> add: newKey withHash: hash [
	"Key is never already present.
	Used during node split or join.
	Sender is responsible for ensuring that 
	the key is not one I already contain,
	and that I have room for the new key."

	| keyIndex |
	keyIndex := self indexForKey: newKey withHash: hash.
	(self at: keyIndex) ifNotNil: [ self error: 'Unexpected duplicate key' ].
	self at: keyIndex put: newKey.
	tally := tally + 1.
	tally > fillLine
		ifTrue: [ self error: 'Leaf node is too full.' ]
]

{ #category : 'walker access' }
HtSetLeafNode >> addKey: key atKeyIndex: index [
	index > arraySize ifTrue: [self halt].
	self at: index put: key.
	tally := tally + 1.
	collection incrementTally
]

{ #category : 'auditing' }
HtSetLeafNode >> auditOnto: stream for: aCollection lowestHash: parentLow highestHash: parentHigh [
	" Things to check:
	Are the constant instvars holding the correct constants?
		arraySize
		fillLine
		tableSize
	collection identical to aCollection?
	highestHash and lowestHash equal to parent ones passed in?
	Scan indexed instvars
		Is each key where it should be?
		Is the total number of non-nil key slots equal to tally?
"

	| identifier keyCount |
	identifier := 'LeafNode ' , self asOop printString , ' '.

	self _isLarge
		ifTrue: [ 
			stream
				nextPutAll: identifier , 'is a large object, should fit in one page.';
				lf ].

	arraySize = 2027 & (tableSize = 2027) & (fillLine = 1520)
		ifFalse: [ 
			stream
				nextPutAll:
						identifier
								,
									'one or more constants are incorrect. Expected: arraySize = 2027, tableSize = 2027, fillLine = 1520. Actual: arraySize =  '
								, arraySize printString , ' tableSize = ' , tableSize printString
								, ' fillLine = ' , fillLine printString;
				lf ].

	collection == aCollection
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'collection is ' , collection printString , ' with oop '
								, collection asOop printString , ' should be the collection I belong to';
				lf ].

	highestHash = parentHigh
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'highestHash should be equal to parent hash of '
								, parentHigh printString , ' but is ' , highestHash printString;
				lf ].

	lowestHash = parentLow
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'lowestHash should be equal to parent hash of '
								, parentLow printString , ' but is ' , lowestHash printString;
				lf ].

	highestHash < lowestHash
		ifTrue: [ 
			stream
				nextPutAll:
						identifier , 'lowestHash ' , lowestHash printString
								, ' is greater than highestHash ' , highestHash printString;
				lf ].

	keyCount := 0.

	1 to: arraySize do: [ :i | 
		| key |
		key := self at: i.
		key
			ifNotNil: [ 
				| expectedIndex |
				keyCount := keyCount + 1.
				expectedIndex := self
					indexForKey: key
					withHash: (collection permutedHashOf: key).
				i = expectedIndex
					ifFalse: [ 
						stream
							nextPutAll:
									identifier , 'key at index ' , i printString , ' was expected to be at index '
											, expectedIndex printString;
							lf ] ] ].

	tally = keyCount
		ifFalse: [ 
			stream
				nextPutAll:
						identifier , 'tally is ' , tally printString
								, ' but the number of keys found was ' , keyCount printString;
				lf ].

	^ keyCount
]

{ #category : 'walker access' }
HtSetLeafNode >> copyElementsFrom: otherLeaf [
	"Copy all elements in otherLeaf to myself.
	Sender is responsible for assuring that I have sufficient room.
	otherLeaf is about to be discarded, don't bother to remove
	elements from it."

	1 to: otherLeaf _basicSize do: [ :i | 
		| key |
		key := otherLeaf at: i.
		key ifNotNil: [ self add: key withHash: (collection permutedHashOf: key) ] ]
]

{ #category : 'enumerating' }
HtSetLeafNode >> do: unaryBlock [
	1 to: arraySize do: [ :i | 
		| key |
		key := self at: i.
		key ifNotNil: [ unaryBlock value: key ] ]
]

{ #category : 'walker access' }
HtSetLeafNode >> indexForKey: key withHash: hash [
	"Answer basic index of either where the key is if present, or where it would be inserted if not."

	| currentIndex currentKey |
	currentIndex := hash \\ tableSize + 1.
	[ 
	currentKey := self at: currentIndex.
	currentKey == nil or: [ currentKey = key ] ]
		whileFalse: [ 
			currentIndex := currentIndex + 1.
			currentIndex > arraySize
				ifTrue: [ currentIndex := 1 ] ].
	^ currentIndex
]

{ #category : 'initialization' }
HtSetLeafNode >> initialize [

	super initialize.
	tableSize := arraySize := 2027.
	fillLine := tableSize * 3 // 4.
	tally := 0
]

{ #category : 'private' }
HtSetLeafNode >> loadHeap: heap [
	heap assertEmpty.
	1 to: arraySize do: [ :i | 
		| key |
		key := self at: i.
		key ~~ nil
			ifTrue: [ heap bulkAddHash: (collection permutedHashOf: key) index: i ] ].
	heap buildMinHeap
]

{ #category : 'walker access' }
HtSetLeafNode >> removeKeyAt: index [
	| nilIndex currentIndex currentKey |
	self at: index put: nil.
	tally := tally - 1.
	collection decrementTally.
	nilIndex := currentIndex := index.
	[ 
	currentIndex := currentIndex + 1.
	currentIndex > arraySize
		ifTrue: [ currentIndex := 1 ].
	currentKey := self at: currentIndex.
	currentKey ~~ nil ]
		whileTrue: [ 
			| correctIndex |
			correctIndex := self
				indexForKey: currentKey
				withHash: (collection permutedHashOf: currentKey).
			correctIndex = currentIndex
				ifFalse: [ 
					self
						at: nilIndex put: (self at: currentIndex);
						at: currentIndex put: nil.
					nilIndex := currentIndex ] ]
]

{ #category : 'walker access' }
HtSetLeafNode >> split [
	"Split into two nodes. The left-hand (lower hash) node will be me,
	and the right-hand node will be new. Answer the new node.
	This split is approximate, based on the assumption that the permuted hash
	values of my keys are approximately evenly distributed across the 
	non-negative SmallInteger range. This assumption usually holds,
	but can fail, especially if there are a large number of keys with
	the same hash value. If the split using this method is worse than
	3/4 to 1/4, fall back to the slower split that uses heap sort to 
	achieve a precise split."

	| scratchLeaf newLeaf scratchLeafCount medianHash minAcceptableCount maxAcceptableCount |
	scratchLeaf := collection scratchLeaf.
	scratchLeaf
		initialize;
		collection: collection.	"Might have dropped from memory since last use."

	medianHash := (lowestHash bitShift: -1) + (highestHash bitShift: -1).	"Assumes even distribution. Avoid creating large integers."
	newLeaf := self class
		forCollection: collection
		lowestHash: medianHash
		highestHash: highestHash.
	scratchLeafCount := 0.

	1 to: self size do: [ :i | 
		| key hash |
		key := self at: i.
		key ~~ nil
			ifTrue: [ 
				hash := collection permutedHashOf: key.
				hash <= medianHash
					ifTrue: [ 
						scratchLeafCount := scratchLeafCount + 1.
						scratchLeaf add: key withHash: hash ]
					ifFalse: [ newLeaf add: key withHash: hash ] ] ].
	minAcceptableCount := tally bitShift: -2.	"3/4-1/4 split is the most uneven acceptable."
	maxAcceptableCount := minAcceptableCount * 3.
	^ (scratchLeafCount < minAcceptableCount
		or: [ scratchLeafCount > maxAcceptableCount ])
		ifTrue: [ 
			scratchLeaf removeAll.
			self splitUsingHeap ]
		ifFalse: [ 
			self replaceAllWith: scratchLeaf.
			highestHash := medianHash.
			scratchLeaf removeAll.
			newLeaf ]
]

{ #category : 'private' }
HtSetLeafNode >> splitUsingHeap [
	"Split into two equal-size nodes. The left-hand (lower hash) node will be me,
	and the right-hand node will be new. Answer the new node.
	This method uses a heap sort to be precise about splitting the node, so is
	slower than the approximate split that is usually used."

	| heap scratchLeaf newLeaf firstHalfCount secondHalfCount currentKey currentHash medianHash |
	heap := collection heap.
	heap initialize.	"Might have dropped from memory since last use."
	scratchLeaf := collection scratchLeaf.
	self loadHeap: heap.
	scratchLeaf
		initialize;
		collection: collection.	"Might have dropped from memory since last use."
	firstHalfCount := tally bitShift: -1.	"First half of my contents go to scratchLeaf"
	secondHalfCount := tally - firstHalfCount.
	firstHalfCount
		timesRepeat: [ 
			| keyIndex |
			currentHash := heap minKey.
			keyIndex := heap removeValueWithMinKey.
			currentKey := self at: keyIndex.
			scratchLeaf add: currentKey withHash: currentHash ].
	medianHash := currentHash.
	newLeaf := self class
		forCollection: collection
		lowestHash: medianHash
		highestHash: highestHash.
	secondHalfCount
		timesRepeat: [ 
			| keyIndex |
			currentHash := heap minKey.
			keyIndex := heap removeValueWithMinKey.
			currentKey := self at: keyIndex.
			newLeaf add: currentKey withHash: currentHash ].
	self replaceAllWith: scratchLeaf.
	highestHash := medianHash.
	scratchLeaf removeAll.
	^ newLeaf
]
