Extension { #name : 'IcuCollator' }

{ #category : 'Private' }
IcuCollator class >> _availableCollators [
  "Return a dictionary of IcuLocales for which
   there are predefined Collators in the libicudata.so library "
    | key dict |
  dict := SessionTemps current at: (key := #IcuCollator_AVAILABLE_COLLATORS) otherwise: nil .
  dict ifNil:[
    dict := StringKeyValueDictionary new .
    IcuCollator availableLocales do:[:loc | dict at: loc name put: loc ].
    SessionTemps current at: key put: dict .
  ].
  ^ dict

]

{ #category : 'Private' }
IcuCollator class >> _forLocale: aIcuLocale [
  "private.
   calls Collator::createInstance(const Locale &loc, UErrorCode &err)"
  <primitive: 915>
  aIcuLocale _validateClass: IcuLocale .
  self _primitiveFailed:#forLocale: args: { aIcuLocale }

]

{ #category : 'Accessing' }
IcuCollator class >> availableLocales [
  "calls Collator::getAvailableLocales() in coll.h.
   Returns an Array of IcuLocales, for which there are predefined
   Collators in the libicudata.so shared library."
  ^ IcuLocale _locales: 6 value: nil

]

{ #category : 'Instance Creation' }
IcuCollator class >> default [
  "Returns the session's default collator.
   Note that IcuLocale default: will change the default collators,
   and that IcuLocale(C)>>default  is dependent on operating system
   locale if  IcuLocale(C)>>default: has not executed during this
   session."
  ^ (System __sessionStateAt: 20) ifNil:[ | coll |
      coll := self _forLocale: IcuLocale default .
      self default: coll .
      coll
  ].

]

{ #category : 'Instance Creation' }
IcuCollator class >> default: aIcuCollator [
  aIcuCollator _validateClass: IcuCollator .
  System __sessionStateAt: 20 put: aIcuCollator .
  System __sessionStateAt: 21 put: aIcuCollator _copyCaseInsensitive

]

{ #category : 'Instance Creation' }
IcuCollator class >> defaultCaseInsensitive [
  "Returns the session's default case-insensitive  collator.
   Note that IcuLocale default: will change the default collators,
   and that IcuLocale(C)>>default  is dependent on operating system
   locale if  IcuLocale(C)>>default: has not executed during this
   session."

  ^ (System __sessionStateAt: 21) ifNil:[ | coll |
      coll := self forLocale: IcuLocale default .
      self default: coll .
      System __sessionStateAt: 21
  ].

]

{ #category : 'Instance Creation' }
IcuCollator class >> forLocale: aIcuLocale [
  "Return an instance of IcuCollator built from a predefined Collator
   for the specified IcuLocale . If there is no predefined Collator
   in libicudata.so library for the specified Locale, signals an ArgumentError"

(self _availableCollators at: aIcuLocale name otherwise: nil) ifNil:[
  ArgumentError signal:'No predefined IcuCollator for locale ''' , aIcuLocale displayName , ''' '
].
^ self _forLocale: aIcuLocale

]

{ #category : 'Instance Creation' }
IcuCollator class >> forLocaleNamed: localeName [
  "Return an instance of IcuCollator built from a predefined Collator
   using the IcuLocale for the specified name.
   for the specified IcuLocale . If there is no predefined Collator
   in libicudata.so library for the specified Locale, signals an ArgumentError"

(self _availableCollators at: localeName otherwise: nil ) ifNotNil:[: aLocale|
  ^ self _forLocale: aLocale
] ifNil:[
   ArgumentError signal:'No predefined IcuCollator for locale ''' , localeName, ''' '
]

]

{ #category : 'Accessing' }
IcuCollator class >> libraryVersion [

"Returns a String such as '58.2'  the version of the libicu library
 that is loaded into this session's virtual machine."

<primitive: 568>
self _primitiveFailed: #libraryVersion

]

{ #category : 'Instance Creation' }
IcuCollator class >> new [
  "disallowed"
  self shouldNotImplement: #new

]

{ #category : 'Accessing' }
IcuCollator class >> strengthNames [
  "Returns an array of the allowed args to IcuCollator>>strength:"

^ StrengthNames

]

{ #category : 'Accessing' }
IcuCollator class >> strengths [
  "Returns an array of the allowed args to IcuCollator>>strength:"

^ StrengthArgs

]

{ #category : 'Updating' }
IcuCollator >> _copyCaseInsensitive [
  "Returns a copy of the receiver which will do case-insensitive compares."

 | stren c |
 " #( #PRIMARY #SECONDARY #TERTIARY #QUATERNARY #IDENTICAL ) "
 c := self copy .
 stren := (c _getAttribute: 0) + 1 .
 stren >= 3 ifTrue:[
   c strength: #SECONDARY .
 ].
 ^ c

]

{ #category : 'Private' }
IcuCollator >> _getAttribute: opcode [
  <primitive: 907>

  self _primitiveFailed:#_getAttribute: args: { opcode }

]

{ #category : 'Private' }
IcuCollator >> _setAttribute: opcode value: val [

  <primitive: 916>
  opcode == 0 ifTrue:[
    val _validateClass: SmallInteger .
    (self class strengths includesIdentical: val) ifFalse:[
      OutOfRange new name:'strength' min: PRIMARY max: IDENTICAL actual: val; signal.
    ]
  ] ifFalse:[
    opcode == 1 ifTrue:[
      val _validateClass: String .
      (val = 'off' or:[ val = 'upperFirst' or:[ val = 'lowerFirst']]) ifFalse:[
         ArgumentError signal:'invalid value for caseFirst:'
      ].
    ] ifFalse:[
      val _validateClass: Boolean .
    ]
  ].
  self _primitiveFailed:#_setAttribute:value: args: { opcode . val }

]

{ #category : 'Comparing' }
IcuCollator >> = anIcuCollator [

"Returns true if the argument is equal to the receiver, false otherwise."

<primitive: 959>
anIcuCollator _validateClass: IcuCollator .
self _primitiveFailed: #= args: { anIcuCollator }.

]

{ #category : 'Comparing' }
IcuCollator >> alternateHandling [
  "Returns true or false.
   See UCOL_ALTERNATE_HANDLING under  enum UColAttribute  at icu-project.org"
  ^ self _getAttribute: 4

]

{ #category : 'Updating' }
IcuCollator >> alternateHandling: aBoolean [
  "Sets UCOL_ALTERNATE_HANDLING , aBoolean == true  means set to UCOL_SHIFTED."
  self _setAttribute: 4 value: aBoolean

]

{ #category : 'Comparing' }
IcuCollator >> caseFirst [
  "Returns a String, one of  'off', 'upperFirst' , or 'lowerFirst' .
   See UCOL_CASE_FIRST under  enum UColAttribute  at icu-project.org"
  ^ self _getAttribute: 1

]

{ #category : 'Updating' }
IcuCollator >> caseFirst: aValue [
  "aValue must be a String, one of  'off', 'upperFirst' , or 'lowerFirst' .
   Sets UCOL_CASE_FIRST ."
  self _setAttribute: 1 value: aValue

]

{ #category : 'Comparing' }
IcuCollator >> caseLevel [
  "Returns true (UCOL_ON) or false (UCOL_OFF).
   See UCOL_CASE_LEVEL under  enum UColAttribute  at icu-project.org.

   From i18n/unicode/ucol.h :
      Controls whether an extra case level (positioned before the third
      level) is generated or not. Acceptable values are UCOL_OFF (default),
      when case level is not generated, and UCOL_ON which causes the case
      level to be generated. Contents of the case level are affected by
      the value of UCOL_CASE_FIRST attribute. A simple way to ignore
      accent differences in a string is to set the strength to UCOL_PRIMARY
      and enable case level.

   This is not controlling case-sensitivity.
   "
  ^ self _getAttribute: 2

]

{ #category : 'Updating' }
IcuCollator >> caseLevel: aBoolean [
  "Sets UCOL_CASE_LEVEL,  aBoolean == true  means set to UCOL_ON"
  self _setAttribute: 2 value: aBoolean

]

{ #category : 'Comparing' }
IcuCollator >> frenchCollation [

"Returns true or false.
 See UCOL_FRENCH_COLLATION under  enum UColAttribute  at icu-project.org"
  ^ self _getAttribute: 6

]

{ #category : 'Updating' }
IcuCollator >> frenchCollation: aBoolean [
  "Sets UCOL_FRENCH_COLLATION , aBoolean == true means set to UCOL_ON."
  self _setAttribute: 6 value: aBoolean

]

{ #category : 'Comparing' }
IcuCollator >> hash [

^ locale hash

]

{ #category : 'Comparing' }
IcuCollator >> hiraganaQuarternary [
  "Returns true or false.
   See UCOL_HIRAGANA_QUATERNARY_MODE under  enum UColAttribute  at icu-project.org"
  ^ self _getAttribute: 7

]

{ #category : 'Accessing' }
IcuCollator >> locale [
  ^ locale

]

{ #category : 'Comparing' }
IcuCollator >> normalization [
  "Returns true or false.
   See UCOL_NORMALIZATION_MODE  under  enum UColAttribute  at icu-project.org"
  ^ self _getAttribute: 5

]

{ #category : 'Updating' }
IcuCollator >> normalization: aBoolean [
  "Sets UCOL_NORMALIZATION_MODE aBoolean == true  means set to UCOL_ON"
  self _setAttribute: 5 value: aBoolean

]

{ #category : 'Comparing' }
IcuCollator >> numericCollation [
  "Returns true or false.
   See UCOL_NUMERIC_COLLATION under  enum UColAttribute  at icu-project.org"
  ^ self _getAttribute: 3

]

{ #category : 'Updating' }
IcuCollator >> numericCollation: aBoolean [
  "Sets UCOL_NUMERIC_COLLATION , aBoolean == true  means set to UCOL_ON"
  self _setAttribute: 3 value: aBoolean

]

{ #category : 'Accessing' }
IcuCollator >> strength [
 "Returns the strength of the receiver
  as a Symbol,  one of #PRIMARY, #SECONDARY, #TERTIARY, #QUATERNARY, #IDENTICAL."

 ^ StrengthVals at: (self _getAttribute: 0) + 1

]

{ #category : 'Updating' }
IcuCollator >> strength: aValue [
  "aValue should be one of PRIMARY, SECONDARY, TERTIARY,
     QUATERNARY, IDENTICAL per the class variables of IcuCollator,
     or one of the symbols
     #PRIMARY, #SECONDARY, #TERTIARY, #QUATERNARY, #IDENTICAL"
  | val |
  aValue _isSymbol ifTrue:[ | ofs |
    ofs := StrengthNames indexOfIdentical: aValue .
    ofs == 0 ifTrue:[ ArgumentError signal:
     'arg must be one of #PRIMARY #SECONDARY #TERTIARY #QUATERNARY #IDENTICAL'
    ].
    val := StrengthArgs at: ofs .
  ] ifFalse:[
    val := aValue .
  ].
  self _setAttribute: 0 value: val .  "see also usage in _copyCaseInsensitive"

]
