Extension { #name : 'Object' }

{ #category : 'Object Canonicalization' }
Object >> _canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy [

	^aReferencingObjectPolicy canonicalizeReferencesInObject: self

]

{ #category : 'Repository Conversion' }
Object >> fixReferencesAfterConversionFromDirectory: string [

"Attempt to old objects to special objects as follows:
        SmallFloat ->           SmallDouble
        Float ->                SmallDouble
        ObsLargePositiveInteger ->      SmallInteger
        ObsLargeNegativeInteger ->      SmallInteger

Note: some instances may not be converted for the following reasons:
        Instance is in an IdentitySet or IdentityBag larger than 2024 elements.
        Instances is in an Array or other oop collection larger than 2024 elements.
        Value of the instance is out of range of the new special object class."

| fn aString refs objs2ndPass alreadyFixed |
  refs := GsBitmap newForHiddenSet: #Customer1.
  objs2ndPass := GsBitmap newForHiddenSet: #Customer2.
  alreadyFixed := GsBitmap newForHiddenSet: #Customer5.
  refs removeAll.
  objs2ndPass removeAll.
  alreadyFixed removeAll.

"Load in the 2 bitmaps that contain refs to LargeInts and Floats"
aString := String withAll: string.
(aString last == $/)
        ifFalse:[aString add: '/'].
fn := String new.
fn addAll: aString; addAll: 'AllLrgIntRefs.bm'.
(GsFile existsOnServer: fn)
        ifTrue:[refs readFromFile: fn]
        ifFalse:[self halt: ('Cannot open file ', fn) ].

fn size: 0; addAll: aString; addAll: 'AllFloatRefs.bm'.
"This one is optional because 6.1.5 conversion does not produce AllFloatRefs.bm"
(GsFile existsOnServer: fn)
        ifTrue:[refs readFromFile: fn].

(refs size) == 0
        ifTrue:[^false].

[ | array |
  System transactionMode: #autoBegin.
  System signalAlmostOutOfMemoryThreshold: 70 .

  [ array := refs enumerateWithLimit: 2000 startingAfter: 0.
    array size == 0] whileFalse:[
        1 to: array size do:[:n| |obj|
                obj := array at: n.
                obj fixReferencesInFirstPass
                        ifTrue:[obj fixRefsAfterConversion] "do it now"
                        ifFalse:[objs2ndPass add: obj]. "save it for later"
                refs removeAll: array.
        ].
  ].
  System commitTransaction ifFalse:[ Object halt: 'error: commit failure!'].
  [ array := objs2ndPass enumerateWithLimit: 2000 startingAfter: 0. array size == 0] whileFalse:[
     1 to: array size do:[:n| |obj|
        obj := array at: n.
        obj fixRefsAfterConversion.
        objs2ndPass removeAll: array
     ].
  ].
  System commitTransaction ifFalse:[Object halt: 'error: commit failure!'].
  ^ alreadyFixed size

] onException: AlmostOutOfMemory do:[:ex |
  "do a commit..."
  System commitTransaction
           ifFalse:[Object halt: 'error: commit failure!'].
   "run another markSweep, to see if we can drop below the threshold..."
  System _vmMarkSweep.
  System signalAlmostOutOfMemoryThreshold: 70 .
  System enableAlmostOutOfMemoryError. "re-enable the exception"
  ex resume
].

]

{ #category : 'Repository Conversion' }
Object >> fixRefsAfterConversion [

"Default method for fixing references ObsLargePositiveInteger and
 ObsLargeNegativeInteger instances that can now be represented as
 a SmallInteger and Floats and SmallFloats which can now be represented as
 a SmallDouble."

| convBm |
convBm := (GsBitmap newForHiddenSet: #Conversion).
(convBm includes: self)
        ifTrue:[^false]. "already fixed this one"

self isInternalObject
        ifFalse:[
                self fixInstVarRefsAfterConversion.
                self fixIndexableRefsAfterConversion.
        ].

convBm add: self.
^true

]

{ #category : 'AST-Kernel-Core' }
Object >> value [
  "in bootstrap files in case upgrade stops after base image bootstrap"
  ^self

]

{ #category : 'Accessing' }
Object >> recursiveSize [
  "Returns an Array of the form { size . numberOfObjects }
   where size is approximate physical size on disk of an object and all of the
   objects referenced by it  and numberOfObjects is the count of those objects.
   Excludes referenced objects in the symbolList, 
   referenced Symbols and referenced special objects."
  | terminationSet arr |
  terminationSet := IdentitySet new .
  terminationSet addAll: AllUsers ; add: SystemRepository .
  GsCurrentSession currentSession symbolList do:[:dict |
    terminationSet add: dict values 
  ].
  1 to: 40 do:[:n |
    (System __sessionStateAt: n) ifNotNil:[ :obj | terminationSet add: obj]
  ].
  arr := { terminationSet . 0"size" . 0"obj count" }.
  self _recursiveSize: arr .
  ^ {  arr at: 2 . arr at: 3 }.
]

{ #category : 'Accessing' }
Object >> recursiveSizeReport [
  | arr |
  arr := self recursiveSize .
  ^ (arr at: 2) asString, ' objects ', (arr at: 1) asString, ' bytes'.
]

{ #category : 'Private' }
Object >> _recursiveSize: arr [
 | terminationSet |
 self isSpecial ifTrue:[^ self ].
 self isBehavior ifTrue:[ ^ self ].
 self _isSymbol ifTrue:[ ^ self ].
 ((terminationSet := arr at: 1) includes: self) ifTrue:[ ^ self ].
 terminationSet add: self .
 arr at: 2 put:(arr at:2) + self physicalSizeOnDisk .
 arr at: 3 put:(arr at:3) + 1 .
 1 to: self namedSize do: [:i | (self instVarAt: i) _recursiveSize: arr ].
 self __recursiveSize: arr

]

{ #category : 'Private' }
Object >> __recursiveSize: arr [
  1 to: self _basicSize do: [ :i | (self at: i) _recursiveSize: arr ].

]

{ #category : 'Accessing' }
Object >> recursiveSizeInMemory [
  "Returns an Array of the form { size . numberOfObjects }
   where size is approximate physical size on disk of an object and all of the
   objects referenced by it  and numberOfObjects is the count of those objects.
   Excludes committed objects in memory unless those objects have uncommitted changes.
   Excludes referenced objects in the symbolList, 
   referenced Symbols and referenced special objects."
  | arr terminationSet |
  terminationSet := IdentitySet new .
  arr := { terminationSet . 0"size" . 0 "obj count" }.
  self _recursiveSizeInMemory: arr .
  ^ { arr at: 2 . arr at: 3 }
]

{ #category : 'Accessing' }
Object >> recursiveSizeInMemoryReport [
  | arr |
  arr := self recursiveSizeInMemory .
  ^ (arr at: 2) asString, ' objects ', (arr at: 1) asString, ' bytes'.
]
 
{ #category : 'Private' }
Object >> _recursiveSizeInMemory: arr [
 | terminationSet |
 self isSpecial ifTrue:[^ self ].
 self _isSymbol ifTrue:[^ self ].
 (self isCommitted and:[ self isWritten == false ]) ifTrue:[ ^ self  ].
 ((terminationSet := arr at: 1) includes: self) ifTrue:[ ^ self ].
 terminationSet add: self .
 arr at: 2 put:(arr at:2) + self physicalSizeOnDisk .
 arr at: 3 put:(arr at:3) + 1 .
 1 to: self namedSize do: [:i | (self instVarAt: i) _recursiveSizeInMemory: arr ].
 self __recursiveSizeInMemory: arr

]

{ #category : 'Private' }
Object >> __recursiveSizeInMemory: arr [
  1 to: self _basicSize do: [ :i | (self at: i) _recursiveSizeInMemory: arr ].

]

{ #category : 'Listing References' }
Object >> findReferencePath [
  | finder |
  (finder := GsSingleRefPathFinder newForSearchObjects: { self })
    printToLog: false .
  ^ finder runScan buildResultObjects first
]

{ #category : 'Listing References' }
Object >> findReferencePathString [
  ^ self findReferencePath resultString 
]

