"
A Pragma represents an occurrence of a pragma in a compiled method.  
A pragma is a literal message pattern that occurs between angle brackets at the 
start of a method after any temporaries.  An example:
	<foo: 123 >

In GemStone, the <primitive: nnn> directive looks like a pragma, but is not a pragma,
primitive: inside the first pragma is a reserved word.

Pragmas show up in the Symbol literals for a method; however they are not
message sends.   ClassOrganizer>>literalsReport:  and
ClassOrganizer>>referencesToLiteral: can be used to search for pragmas.
One can query a method for its pragmas by sendng it the pragmas message, 
which answers an Array of instances of me, one for each pragma in the method.

A Pragma can provide information about the defining class, method, its selector, 
as well as the information about the pragma keyword and its arguments. 
See the two 'accessing' protocols for details. 
'accessing-method' provides information about the method the pragma is found in, 
while 'accessing-pragma' is about the pragma itself.

Instances are retrieved using one of the pragma search methods of 
the 'finding' protocol on the class side.
"
Class {
	#name : 'Pragma',
	#superclass : 'Object',
	#instVars : [
		'method',
		'keyword',
		'arguments'
	],
	#category : nil
}

{ #category : 'finding' }
Pragma class >> allNamed: aSymbol from: aSubClass to: aSuperClass [
	"Answer a collection of all pragmas found in methods of all classes between aSubClass and aSuperClass (inclusive) whose keyword is aSymbol."

	| stream |
	stream := WriteStream on: (Array new: 100).
	{aSubClass} , aSubClass allSuperClasses do:
			[:class |
			self withPragmasIn: class
				do: [:pragma | pragma keyword = aSymbol ifTrue: [stream nextPut: pragma]].
			aSuperClass = class ifTrue: [^stream contents]].
	^stream contents

]

{ #category : 'finding' }
Pragma class >> allNamed: aSymbol from: aSubClass to: aSuperClass sortedByArgument: anInteger [
	"Answer a collection of all pragmas found in methods of all classes between aSubClass and aSuperClass (inclusive) whose keyword is aSymbol, sorted according to argument anInteger."

	^ self allNamed: aSymbol from: aSubClass to: aSuperClass sortedUsing: [ :a :b | (a argumentAt: anInteger) < (b argumentAt: anInteger) ].

]

{ #category : 'finding' }
Pragma class >> allNamed: aSymbol from: aSubClass to: aSuperClass sortedUsing: aSortBlock [
	"Answer a collection of all pragmas found in methods of all classes between aSubClass and aSuperClass (inclusive) whose keyword is aSymbol, sorted according to aSortBlock."

	^ (self allNamed: aSymbol from: aSubClass to: aSuperClass) sortWithBlock: aSortBlock.

]

{ #category : 'finding' }
Pragma class >> allNamed: aSymbol in: aClass [
	"Answer a collection of all pragmas found in methods of aClass whose keyword is aSymbol."

	| stream |
	stream := WriteStream on: (Array new: 100).
	self withPragmasIn: aClass
		do: [:pragma | pragma keyword = aSymbol ifTrue: [stream nextPut: pragma]].
	^stream contents

]

{ #category : 'finding' }
Pragma class >> allNamed: aSymbol in: aClass sortedByArgument: anInteger [
	"Answer a collection of all pragmas found in methods of aClass whose keyword is aSymbol, sorted according to argument anInteger."

	^ self allNamed: aSymbol in: aClass sortedUsing: [ :a :b | (a argumentAt: anInteger) < (b argumentAt: anInteger) ].

]

{ #category : 'finding' }
Pragma class >> allNamed: aSymbol in: aClass sortedUsing: aSortBlock [
	"Answer a collection of all pragmas found in methods of aClass whose keyword is aSymbol, sorted according to aSortBlock."

	^ (self allNamed: aSymbol in: aClass) sortWithBlock: aSortBlock.

]

{ #category : 'private' }
Pragma class >> keyword: aSymbol arguments: anArray [
	^ self new
		setKeyword: aSymbol;
		setArguments: anArray;
		yourself.

]

{ #category : 'private' }
Pragma class >> withPragmasIn: aClass do: aBlock [
	(aClass methodDictForEnv: 0) keysAndValuesDo: [ :selector :method | method pragmas do: aBlock ].

]

{ #category : 'testing' }
Pragma >> _refersToLiteral: aLiteral [
  ^ self keyword == aLiteral or: [ self arguments _refersToLiteral: aLiteral ]

]

{ #category : 'accessing-pragma' }
Pragma >> argumentAt: anInteger [
	"Answer one of the arguments of the pragma."

	^ self arguments at: anInteger.

]

{ #category : 'accessing-pragma' }
Pragma >> arguments [
	"Answer the arguments of the receiving pragma. For a pragma defined as <key1: val1 key2: val2> this will answer #(val1 val2)."

	^ arguments

]

{ #category : 'accessing-pragma' }
Pragma >> keyword [
	"Answer the keyword of the receiving pragma. For a pragma defined as <key1: val1 key2: val2> this will answer #key1:key2."

	^ keyword

]

{ #category : 'accessing-pragma' }
Pragma >> message [
	"Answer the message of the receiving pragma."

	^ Message selector: self keyword arguments: self arguments.

]

{ #category : 'accessing-method' }
Pragma >> method [
	"Answer the compiled-method containing the pragma."

	^ method

]

{ #category : 'accessing-method' }
Pragma >> methodClass [
	"Answer the class of the method containing the pragma."

	^self method inClass

]

{ #category : 'accessing-pragma' }
Pragma >> numArgs [
	"Answer the number of arguments in the pragma."

	^ self arguments size.

]

{ #category : 'printing' }
Pragma >> printOn: aStream [
	aStream nextPut: $<.
	self keyword precedence = 1
		ifTrue: [ aStream space; nextPutAll: self keyword ]
		ifFalse: [
			self keyword keywords with: self arguments do: [ :key :arg |
				aStream space; nextPutAll: key; space; print: arg ]].
	aStream space; nextPut: $>.

]

{ #category : 'accessing-method' }
Pragma >> selector [
	"Answer the selector of the method containing the pragma."

	^ method selector.

]

{ #category : 'initialization' }
Pragma >> setArguments: anArray [
	arguments := anArray

]

{ #category : 'initialization' }
Pragma >> setKeyword: aSymbol [
	keyword := aSymbol

]

{ #category : 'initialization' }
Pragma >> setMethod: aCompiledMethod [
	method := aCompiledMethod

]
