Extension { #name : 'ReversedRangeIndexReadStream' }

{ #category : 'Instance Creation' }
ReversedRangeIndexReadStream class >> on: aRangeIndex [
  "Create a stream that can access the entire contents of the B-tree whose
 root node is aBtreeNode."

  | arr1 arr2 newStream btreeNode |
  arr1 := {}.
  arr2 := {}.
  btreeNode := aRangeIndex btreeRoot.
  btreeNode _putFirstIndexOfFirstChildInto: arr2.
  btreeNode _putLastIndexOfLastChildInto: arr1.
  newStream := self new.
  newStream endIndex: (arr2 at: 1).
  newStream endNode: (arr2 at: 2).
  newStream currentStack: arr1.
  newStream rangeIndex: aRangeIndex.
  ^ newStream

]

{ #category : 'Accessing' }
ReversedRangeIndexReadStream >> _btreeNext [
  "Returns the previous value on a stream of B-tree values.  Updates the current
 stack for a subsequent 'next'."

  ^ self _btreePrevious

]

{ #category : 'Accessing' }
ReversedRangeIndexReadStream >> _btreePrevious [
  "Returns the previous value on a stream of B-tree values.  Updates the current
 stack for a subsequent 'next'."

  | val leaf index |
  index := currentStack at: 1.	" get the index into the leaf node and see if it has reached the end "
  index == 0
    ifTrue: [ ^ self _errorEndOfStream ].
  leaf := currentStack at: 2.	" get the leaf and the value within the leaf "
  val := leaf _basicAt: index.
  (leaf == endNode and: [ endIndex == index ])
    ifTrue: [
      " if this is the end of the stream, sets the top index to zero and returns "
      currentStack at: 1 put: 0.
      ^ val ].
  index == 1
    ifTrue: [
      " must look down the stack for the previous leaf node "
      self _previousLeaf ]
    ifFalse: [ currentStack at: 1 put: index - leaf entrySize ].
  ^ val

]

{ #category : 'Accessing' }
ReversedRangeIndexReadStream >> _previousLeaf [
  "Looks down the stack for the previous leaf node and makes it the current node."

  | nodeIndex notPositioned currentNode currentIndex previousIndex node |
  nodeIndex := 4.
  notPositioned := true.
  [ notPositioned ]
    whileTrue: [
      currentIndex := currentStack at: nodeIndex - 1.
      nil == (currentNode := currentStack at: nodeIndex)
        ifTrue: [ self _error: #'rtErrInvalidBtreeReadStream' ].
      " if the current index is the fist in the node "
      currentIndex == 1
        ifTrue: [
          " must look down the stack another level "
          nodeIndex == currentStack size
            ifTrue: [ ^ self _errorEndOfStream ].
          nodeIndex := nodeIndex + 2 ]
        ifFalse: [
          " current node has a previous entry "
          previousIndex := currentIndex - currentNode entrySize.
          currentStack at: nodeIndex - 1 put: previousIndex.	" put the previous entry (index and node) on the stack "
          nil == (node := currentNode _basicAt: previousIndex)
            ifTrue: [ self _error: #'rtErrInvalidBtreeReadStream' ].
          currentStack at: nodeIndex - 2 put: node.
          nodeIndex - 3
            _downTo: 3
            by: 2
            do: [ :i |
              | childNode |
              " put the last child on any remaining levels in the stack "
              childNode := (currentStack at: i + 1) _basicAt: (currentStack at: i + 1) _lastKeyIndex - 1.
              currentStack at: i put: childNode _lastKeyIndex - 1.
              currentStack at: i - 1 put: childNode ].
          currentStack at: 1 put: node _lastKeyIndex - 1.
          notPositioned := false ] ]

]

{ #category : 'Query Support' }
ReversedRangeIndexReadStream >> _uniqueValuesAddInto: collection spec: querySpec [

"Iterates over the unique values of the B-tree, traversing the reverse mappings
 to the root object.  Adds the root to the given NSC."

| leaf index eSize prevVal offset pathTerm |

" get the index into the leaf node and see if it has reached the end "
index := currentStack at: 1.
index == 0
    ifTrue: [ ^ self ].

rangeIndex isIndexOnNscElements
    ifTrue: [ offset := rangeIndex size ]
    ifFalse: [ offset := rangeIndex lastPathComponentsDictionaryOffset max: 1 ].
pathTerm := rangeIndex at: offset.

leaf := currentStack at: 2.
eSize := endNode entrySize.
prevVal := #_incompletePathTraversal.
querySpec rangeIndex: rangeIndex offset: offset pathTerm: pathTerm.
" while current node is not the end node "
[ leaf ~~ endNode ] whileTrue: [
    " iterate through the values of the current node "
  prevVal := querySpec traverseValuesFrom: leaf
                                  start: 1
                                  end: index
                                  previous: prevVal
                                  into: collection.
    " look down the stack for the next leaf node "
    self _previousLeaf.

    leaf := currentStack at: 2.
    index := currentStack at: 1
].
" iterate through the values of the end node "
querySpec traverseValuesFrom: leaf
                 start: endIndex
                 end: index
                 previous: prevVal
                 into: collection.

]

{ #category : 'Query Support' }
ReversedRangeIndexReadStream >> _valuesAddInto: collection spec: querySpec [

"Adds each entry value into the given NSC."

| leaf index eSize |

" get the index into the leaf node and see if it has reached the end "
index := currentStack at: 1.
index == 0
    ifTrue: [ ^ self ].

leaf := currentStack at: 2.
eSize := endNode entrySize.
" while current node is not the end node "
[ endNode ~~ leaf ] whileTrue: [
    " put the values for the current node in the set "
    querySpec addValuesFrom: leaf
                     start: 1
                     end: index
                     into: collection.
    " look down the stack for the next leaf node "
    self _previousLeaf.

    leaf := currentStack at: 2.
    index := currentStack at: 1
].

" put values of end node in the NSC "
querySpec addValuesFrom: leaf
                 start: endIndex
                 end: index
                 into: collection.

]

{ #category : 'Private' }
ReversedRangeIndexReadStream >> reversedRangeIndexClass [
  ^ RangeIndexReadStream

]
