Extension { #name : 'SortedCollection' }

{ #category : 'Repository Conversion' }
SortedCollection class >> _loadBitmap: bm [
| files |
bm removeAll.
files := GsFile contentsOfDirectory: self getSortedCollDir onClient: false .
files := files select:[:e| (e findPattern: (Array with: '.bm') startingAt: 1) ~~ 0].
1 to: files size do:[:n| |fn bm2|
  fn := files at: n.
  bm2 := GsBitmap new.
  bm2 readFromFile: fn.
  bm addAll: bm2.
  bm2 removeAll.
].
^ bm size

]

{ #category : 'Repository Conversion' }
SortedCollection class >> convertAll: anArray usingCache: dict bmForErrors: errBm [

"Convert an Array of SortedCollections by recompiling their sort blocks from
 SimpleBlocks to ExecBlocks.  anArray must contain only SortedCollections.
 dict is used as a cache to canonicalize sort blocks with the same source
 string.

 Caller must handle and catch AlmostOutOfMemory errors and handle commits
 and aborts."

| sbCls errorBlock oldIcuDefaultMth newIcuBlk |
sbCls := Globals at:#SimpleBlock ifAbsent:[ ObsoleteClasses at: #SimpleBlock ] .
newIcuBlk := IcuSortedCollection new _defaultBlock . "a 3 arg block"
oldIcuDefaultMth := Globals at:#IcuSortedCollection_oldDefaultMethod "meth for old 2 arg block"
                    ifAbsent:[ newIcuBlk := nil . nil ].
errorBlock := [ :_sc :_errBm :msg |
     GsFile gciLogServer:'SortedCollection oop ', _sc asOop asString , msg .
     errBm add: _sc.
     System _sessionCacheStatAt: 1 put: (_errBm size)
    ].
1 to: anArray size do:[: ofs | | aSortedColl aBlock cls |
  aSortedColl := anArray at: ofs .
  aBlock := aSortedColl sortBlock .
  cls := aBlock class .
  (aSortedColl isKindOf: IcuSortedCollection) ifTrue:[
    aBlock method == oldIcuDefaultMth ifTrue:[
      aSortedColl _sortBlock: newIcuBlk .
      cls := nil "don't need recompile "
    ] ifFalse:[
      aBlock argumentCount == 3 ifFalse:[
	errorBlock value: aSortedColl value: errBm
                value:' an IcuSortedCollection, 2 arg block must be converted to 3 arg block'.
        cls := nil "disable recompile attempt"
      ].
    ].
  ].
  cls ifNotNil:[
    (cls _subclassOf: ExecBlock) ifTrue:[
      aBlock needsRecompile ifTrue:[
	[ aBlock recompile  .
	  "successful recompilation for block's method,
	   this block will show up as needsRecompile==false if seen again"
	] onSynchronous: Error do:[ :ex|
	  errorBlock value: aSortedColl value: errBm value:
             ' block recompile failed,' , ex asString
	].
      ].
    ] ifFalse:[
      cls == sbCls ifTrue:[ "SimpleBlock from 2.x " | str val |
	str := aBlock _sourceString .
	(val := dict at: str otherwise: nil)
	  ifNotNil:[ aSortedColl _sortBlock: val ]
	  ifNil:[ "A miss on the dictionary, compile it now"
	    [ val := str evaluate .
	      val ifNotNil:[ dict at: str put: val. "Add to cache"
			  aSortedColl _sortBlock: val ]
		  ifNil:[ errorBlock value: aSortedColl value: errBm
			  value: ' evaluate of recompiled block failed' ].
	    ]  onSynchronous: Error do:[:ex |
               errorBlock value: aSortedColl value: errBm
                 value: ' block recompile failed, ' , ex asString
	    ] .
	  ].
      ] ifFalse:[
	errorBlock value: aSortedColl value: errBm value:' block is a v2.x ComplexBlock'.
      ].
    ].
  ].
].

]

{ #category : 'Repository Conversion' }
SortedCollection class >> convertInstancesFromFilesForGem: gemNum of: totalGems [
  "Convert instances of SortedCollection by recompiling sortBlocks"
  "Convert instances of SortedCollection by recompiling sortBlocks"

  | bm errBm ar myShare failed ok unique |
  bm := GsBitmap new.
  errBm := GsBitmap new.
  ar := self
    convertInstancesFromFilesForGem: gemNum
    of: totalGems
    bm: bm
    errBm: errBm
    bmDirectoryPath: self getSortedCollDir
    conversion: [ :array :dict : _errBm | self convertAll: array usingCache: dict bmForErrors: _errBm ].
  myShare := ar at: 1.
  unique := ar at: 2.
  self
    writeFailedConversionObjsToFileForGem: gemNum
    bmForErrors: errBm.	"Now write out the ones that failed conversion to a file, if any."
  failed := errBm size.
  ok := myShare - failed.
  System incrementGlobalSessionStatAt: 0 by: ok.
  System incrementGlobalSessionStatAt: 1 by: failed.
  System incrementGlobalSessionStatAt: 2 by: 1.
  ^ String new
    addAll: 'SortedCollection conversion done. Of ';
    addAll: myShare asString;
    addAll: ' total: ';
    addAll: failed asString;
    addAll: ' failed, ';
    addAll: ok asString;
    addAll: ' successful (';
    addAll: unique asString;
    addAll: ' unique).';
    yourself

]

{ #category : 'Repository Conversion' }
SortedCollection class >> convertInstancesFromFilesForGem: gemNum of: totalGems bm: bm errBm: errBm bmDirectoryPath: bmDirectoryPath conversion: conversionBlock [
  "Session Stat Usage
   0 - num left for this session to process.
   1 - num errors found so far
 Global Shared Counter Usage
   0 - num successfully converted (all sessions).
   1 - num failed conversion (all sessions).
   2 - num sessions completed OK"

  | id sys totalOops sr myShare mod commitBlock errNumAlmostOutOfMemory unique numToSkipAtFront |
  sys := System.
  sr := SystemRepository.
  totalOops := self _loadBitmap: bm fromFilesIn: bmDirectoryPath.
  numToSkipAtFront := 0.
  myShare := totalOops // totalGems.
  mod := totalOops \\ totalGems.
  myShare == 0
    ifTrue: [
      gemNum == 1
        ifTrue: [ myShare := totalOops ]
        ifFalse: [
          "No work for me to do"
          GsFile gciLogServer: 'No oops to convert'.
          ^ #(0 0) ]	]
    ifFalse: [
      numToSkipAtFront := myShare * (gemNum - 1).
      gemNum == totalGems
        ifTrue: [
          "Last guy does the left-overs too."
           myShare := myShare + mod ]	].
  numToSkipAtFront > 0
    ifTrue: [ bm removeFirst: numToSkipAtFront ].
  gemNum == totalGems
    ifFalse: [ bm _truncateToSize: myShare ].
  errNumAlmostOutOfMemory := ErrorSymbols at: #'rtErrSignalAlmostOutOfMemory'.
  commitBlock := [
  sys commitTransaction
    ifFalse: [
      sys abortTransaction.
      TransactionError signal: 'Unexpected commit failure in post conversion' ] ].
  errBm removeAll.
  [
  [
  | numLeft array dict numThisTime |
  numLeft := bm size.
  dict := StringKeyValueDictionary new: 1000.
  sys signalAlmostOutOfMemoryThreshold: 90.
  sys _sessionCacheStatAt: 0 put: numLeft.
  sys _sessionCacheStatAt: 1 put: 0.
  array := bm enumerateWithLimit: 2034 startingAfter: 0.
  numThisTime := array size.
  [ numThisTime > 0 ]
    whileTrue: [
      conversionBlock value: array value: dict value: errBm.
      numLeft := numLeft - numThisTime.
      sys _sessionCacheStatAt: 0 put: numLeft.
      array := bm enumerateWithLimit: 2034 startingAfter: (array last).
      numThisTime := array size ].
  unique := dict size ]
    onException: AlmostOutOfMemory
    do: [ :ex |
      ex number == errNumAlmostOutOfMemory
        ifTrue: [
          commitBlock value.
          sys
            _vmMarkSweep;
            enableAlmostOutOfMemoryError.
          ex resume ]
        ifFalse: [ ex outer ] ] ]
    ensure: [
      "Do the final commit"
      commitBlock value.
      "Disable the AlmostOutOfMemory error."
      sys signalAlmostOutOfMemoryThreshold: -1.
      "No longer valid."
      id := nil	 ].
  ^ {myShare.
  unique}

]

{ #category : 'Repository Conversion' }
SortedCollection class >> createConversionResultFileForTotalGems: totalGems [

"Called by the master gem to consolidate all lists of SortedCollections
 which failed conversion into a single result file."

| uld bm outFile totalObjs |

 bm := GsBitmap new.
 uld := self getUpgradeLogDir .

1 to: totalGems do:[:n| |fn|
  fn := uld copy.
  fn addAll: (self buildFileNameForGem: n) .
  (GsFile existsOnServer: fn)
    ifTrue:[
       bm readFromFile: fn
    ].
].

totalObjs := bm size.
totalObjs == 0
  ifTrue:[ ^ true ].

outFile := uld copy .
outFile addAll: 'AllFailedSortedCollections.bm' .
(GsFile existsOnServer: outFile)
  ifTrue:[ GsFile removeServerFile: outFile ].

bm writeToFile: outFile.
GsFile gciLogServer: ('A total of ', totalObjs asString, ' SortedCollections failed to convert').
^ true

]

{ #category : 'Repository Conversion' }
SortedCollection class >> resortInstancesFromFilesForGem: gemNum of: totalGems [
  "Resort instances of SortedCollection"

  | bm errBm ar myShare failed ok unique |
  bm := GsBitmap new.
  errBm := GsBitmap new.
  ar := self
    convertInstancesFromFilesForGem: gemNum
    of: totalGems
    bm: bm
    errBm: errBm
    bmDirectoryPath: self getSortedCollDir
    conversion: [ :array :ignored :_errBm | self resortAll: array ].
  myShare := ar at: 1.
  unique := ar at: 2.
  self
    writeFailedConversionObjsToFileForGem: gemNum
    bmForErrors: errBm.	"Now write out the ones that failed conversion to a file, if any."
  failed := errBm size.
  ok := myShare - failed.
  System incrementGlobalSessionStatAt: 0 by: ok.
  System incrementGlobalSessionStatAt: 1 by: failed.
  System incrementGlobalSessionStatAt: 2 by: 1.
  ^ String new
    addAll: 'SortedCollection resorting done. Of ';
    addAll: myShare asString;
    addAll: ' total: ';
    addAll: failed asString;
    addAll: ' failed, ';
    addAll: ok asString;
    addAll: ' successful (';
    addAll: unique asString;
    addAll: ' unique).';
    yourself

]

{ #category : 'Object Canonicalization' }
SortedCollection >> _canonicalizeReferencesUsingPolicy: aReferencingObjectPolicy [

	^aReferencingObjectPolicy canonicalizeReferencesInCollection: self

]

{ #category : 'Private' }
SortedCollection >> resort [

"Re-sorts the receiver according to its sortBlock."

| saveArray |
self size > 1 ifTrue: [
  saveArray := Array withAll: self.
  [ self _basicSize: 0 .
    self addAll: saveArray
  ] onException: Error do: [:ex |
    "fix bug 39227/41089"
    System disableCommitsUntilAbortWithReason: 'Error during SortedCollection>>resort'.
    ex pass
  ].
].

]
