Extension { #name : 'GsComSendNode' }

{ #category : 'Class Initialization' }
GsComSendNode class >> initialize [
  "initialize the control op dictionary; these are the selectors for
   which the generator can generate in-line branching or looping code,
   if the receiver and/or arguments to the send meet certain critiera.

   For example:
     If the argument to ifTrue:  is a block , the block can be inlined.

   See GsComSendNode >> optimizeIfPossible for smalltalk implementation
   of the optimization logic; this logic also in src/comparse.c .
  "

  | dict |
  dict := IdentityKeyValueDictionary new .
  self _removeClassVar: #ControlOpDict  ifAbsent:[].
  self _addInvariantClassVar: #ControlOpDict value: dict .

  "all of the following branch constructs will observe the isRuby attribute
   of the current method node .
   for example:  'ifTrue:'  means  'if neither false nor nil' in a Ruby method.
  "
  		"ruby_selector_suffix dependent"
  dict at: #ifTrue:          put: COMPAR__IF_TRUE ;
       at: #ifFalse:         put: COMPAR__IF_FALSE ;
       at: #ifTrue:ifFalse:  put: COMPAR_IF_TRUE_IF_FALSE  ;
       at: #ifFalse:ifTrue:  put: COMPAR_IF_FALSE_IF_TRUE  ;
       at: #ifNil:ifNotNil:  put: COMPAR_IF_NIL_IF_NOTNIL ;
       at: #ifNotNil:ifNil:  put: COMPAR_IF_NOTNIL_IF_NIL ;
       at: #ifNil:           put: COMPAR_IF_NIL ;
       at: #ifNotNil:        put: COMPAR_IF_NOT_NIL ;
       at: #or:              put: COMPAR_OR_SELECTOR  ;
       at: #and:             put: COMPAR_AND_SELECTOR  ;
       at: #whileFalse:      put: COMPAR_WHILE_FALSE  ;
       at: #'whileFalse#1__'  put: COMPAR_WHILE_FALSE  ;

       at: #whileTrue:       put: COMPAR_WHILE_TRUE  ;
       at: #'whileTrue#1__'   put: COMPAR_WHILE_TRUE  ;

       at: #untilFalse:      put: COMPAR_UNTIL_FALS_COLON ;
       at: #'untilFalse#1__'  put: COMPAR_UNTIL_FALS_COLON  ;

       at: #untilTrue:       put: COMPAR_UNTIL_TRU_COLON ;
       at: #'untilTrue#1__'   put: COMPAR_UNTIL_TRU_COLON  ;

       at: #untilFalse       put: COMPAR_UNTIL_FALSE ;
       at: #untilTrue        put: COMPAR_UNTIL_TRUE ;
       at: #whileFalse       put: COMPAR_UNTIL_TRUE ;
       at: #whileTrue        put: COMPAR_UNTIL_FALSE ;
       at: #repeat      put: COMPAR_FOREVER_repeat ;

       at: #to:do:           put: COMPAR_TO_DO ;
       at: #to:by:do:        put: COMPAR_TO_BY_DO ;
       at: #timesRepeat:     put: COMPAR_TIMES_REPEAT ;
       at: #_downTo:do:      put: COMPAR__DOWNTO_DO ;
       at: #_downTo:by:do:   put: COMPAR__DOWNTO_BY_DO .

]

{ #category : 'Instance Creation' }
GsComSendNode class >> new [
  ^ self _basicNew initialize

]

{ #category : 'Instance Initialization' }
GsComSendNode >> appendArgument: aNode [
  aNode ifNil:[ self error:'illegal nil argument'].
  arguments addLast: aNode

]

{ #category : 'Instance Initialization' }
GsComSendNode >> environment [
  ^ envFlags bitAnd: 16rFF

]

{ #category : 'Instance Initialization' }
GsComSendNode >> environment: anInteger [
  envFlags := ((envFlags bitShift: -8) bitShift: 8)
               bitOr: (self validateEnvironment: anInteger)

]

{ #category : 'Instance Initialization' }
GsComSendNode >> initialize [
  arguments := { } .
  controlOp := COMPAR_NO_OPTIMIZATION "a normal send" .
  envFlags := 0 .
  kind := COMPAR_SEND_NODE .

]

{ #category : 'Transformation' }
GsComSendNode >> optimizationPossible [
  "Return true if optimization to an in-line  branch or loop could be possible,
   otherwise return false .
   To be sent after initializing the selector."

  | sel |
  (sel := selLeaf) _isSymbol ifFalse:[ sel := sel selector ].
  ^ (ControlOpDict at: sel otherwise: 0 ) ~~ 0

]

{ #category : 'Transformation' }
GsComSendNode >> optimize [

  "Attempt to optimize the receiver to a special selector.
   Generates an error if receiver is not optimizable .
   Use  optimizationPossible  to determine if  optimize  would succeed."

  | op sel |
  (sel := selLeaf) _isSymbol ifFalse:[ sel := sel selector ].
  op := ControlOpDict at: sel otherwise: 0 .
  op ~~ 0 ifTrue:[
    controlOp := op
  ] ifFalse: [
    self error:'not optimizable'
  ]

]

{ #category : 'Transformation' }
GsComSendNode >> optimizeIfPossible [
  "returns true if optimization performed, false otherwise.
   Does not consider any possible Ruby env1 selectors."
  | sel op argSz args |
  (sel := selLeaf) _isSymbol ifFalse:[ sel := sel selector ].
  op := ControlOpDict at: sel otherwise: 0 .
  op ~~ 0 ifTrue:[ | clsBlkNode |
    clsBlkNode := GsComBlockNode .
    argSz := (args := arguments) size .
    argSz <= 1 ifTrue:[
      argSz == 1 ifTrue:[
        (sel == #ifTrue: or:[ sel == #ifFalse:
          or:[ sel == #ifNil: or:[ sel == #ifNotNil:
          or:[ sel == #or:  or:[ sel == #and:
          or:[ sel == #timesRepeat:          ]]]]]]) ifTrue:[
            (args atOrNil: 1) class == clsBlkNode ifTrue:[
              controlOp := op .
              ^ true
            ].
         ].
         (sel == #whileFalse: or:[ sel == #whileTrue:
          or:[ sel == #untilFalse:  or:[ sel == #untilTrue:  ]]]) ifTrue:[
            ((args atOrNil: 1) class == clsBlkNode
             and: [ rcvr class == clsBlkNode ]) ifTrue:[
              controlOp := op .
              ^ true
             ]
         ].
      ].
      "argSize == 0"
      (sel == #untilFalse or:[ sel == #untilTrue
        or:[ sel == #whileFalse or:[ sel == #whileTrue ]]]) ifTrue:[
          rcvr class == clsBlkNode ifTrue:[
            controlOp := op .
            ^ true
          ].
      ].
      sel == #repeat ifTrue:[
         (rcvr class == clsBlkNode and:[ rcvr numStatements > 0]) ifTrue:[
            controlOp := op .
            ^ true
         ].
      ].
      ^ false .
    ].
    argSz == 2 ifTrue:[
      (sel == #ifTrue:ifFalse:  or:[ sel == #ifFalse:ifTrue:  or:[
       sel == #ifNil:ifNotNil:  or:[ sel == #ifNotNil:ifNil:  ]]]) ifTrue:[
         ((args atOrNil: 1) class == clsBlkNode
          and:[ (args atOrNil: 2) class == clsBlkNode ]) ifTrue:[
            controlOp := op .
            ^ true
          ].
       ].
       (sel == #to:do: or:[ sel == #_downTo:do:])  ifTrue:[
          (args atOrNil: 2) class == clsBlkNode ifTrue:[
            controlOp := op .
            ^ true
          ].
       ].
       ^ false
    ].
    (sel == #to:by:do: or:[ sel ==  #_downTo:by:do: ]) ifTrue:[
      (args atOrNil: 3) class == clsBlkNode  ifTrue:[
         controlOp := op .
         ^ true
      ].
    ].
  ].
  ^ false

]

{ #category : 'Printing' }
GsComSendNode >> printFormattedOn: aStream [
  | sel |
  super printOn: aStream .
  aStream nextPutAll:' selLeaf: '; indentMore ; cr .
  (sel := selLeaf) _isSymbol ifTrue:[
     aStream nextPut: $# ; nextPut: $' ; nextPutAll: sel; nextPut: $' ; nextPut: $  .
  ] ifFalse:[
     sel printFormattedOn: aStream.
  ].
  aStream  indentLess ; cr ;
     print:' controlOp:' int: controlOp ;
     print:' envFlags:' int: envFlags ; cr;
  nextPutAll:' rcvr:' .
    rcvr ~~ nil ifTrue:[ rcvr printFormattedOn: aStream]
              ifFalse:[ aStream nextPutAll:'nil '].
  aStream nextPutAll: ' args:' ; do: arguments ;
  nextPut: $) ; cr .

]

{ #category : 'Instance Initialization' }
GsComSendNode >> rcvr: aGsCompilerIRNode [
  rcvr := aGsCompilerIRNode

]

{ #category : 'Instance Initialization' }
GsComSendNode >> rubySelector: aSymbol [
  " for a send NOT to super"

  selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 1 .
  envFlags ~~ 0 ifTrue:[ self error:'should set flags after selector'].
  envFlags := 1 .

]

{ #category : 'Instance Initialization' }
GsComSendNode >> rubySelector: aSymbol toSuper: superBool [
  superBool ifTrue:[
    selLeaf := aSymbol  "can't optimize send to super"
  ] ifFalse:[
    selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 1
  ].
  envFlags ~~ 0 ifTrue:[ self error:'should set flags after selector'].
  envFlags := 1 .

]

{ #category : 'Instance Initialization' }
GsComSendNode >> rubySelectorInBridge: aSymbol [
  selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 1 .
    " and  comgen.c should generate a SEND_CURRENT"
  envFlags ~~ 0 ifTrue:[ self error:'should set flags after selector'].
  envFlags := 1 .

]

{ #category : 'Instance Initialization' }
GsComSendNode >> setBypassRubyProtection [
  envFlags := envFlags bitOr: BypassProtection_MASK

]

{ #category : 'Instance Initialization' }
GsComSendNode >> setEvalLastArgFirst [
  envFlags := envFlags bitOr: EvalLastArgFirst_MASK

]

{ #category : 'Instance Initialization' }
GsComSendNode >> stSelector: aSymbol [
  selLeaf := GsComSelectorLeaf newSelector: aSymbol env: 0 .
  envFlags := 0 .
  "at this point selLeaf is a Symbol for a non-optimized send
   or a GsComSelectorLeaf for an optimized send of a special selector."

]
