"
A TLS 1.2 server, using libssl 1.1.
Currently expects mutual certificate authentication in all cases.

"
Class {
	#name : 'TlsActor',
	#superclass : 'Object',
	#instVars : Array [
		'certificate',
		'libcFcntl',
		'libCrypto',
		'libSsl',
		'privateKey',
		'ssl',
		'sslCtx',
		'tcpSocket',
		'trustAnchor'
	],
	#category : 'X509-HostAgent'
}

{ #category : 'instance creation' }
TlsActor class >> new [

	^super new initialize
]

{ #category : 'utility' }
TlsActor >> checkForNullReturn: retval from: funcName [
	"For functions that return a pointer or null in case of error"

	retval memoryAddress = 0
		ifTrue: [self error: 'Error: ' , funcName , ' returned ' , retval printString]
]

{ #category : 'utility' }
TlsActor >> checkReturn: retval from: funcName [
	"For libssl functions that return >0 on success and 0 or -1 on failure."

	retval > 0
		ifFalse: 
			[self error: 'Error: ' , funcName , ' returned ' , retval printString]
]

{ #category : 'utility' }
TlsActor >> checkSslReturn: retval from: funcName [
	"For libssl functions that return >0 on success and 0 or -1 on failure."

	retval > 0
		ifFalse: 
			[| resultCode |
			resultCode := libSsl SSL_get_error_: ssl _: retval.
			self
				error: 'Error: ' , funcName , ' returned ' , retval printString
						, ' result code ' , resultCode printString]
]

{ #category : 'error handling' }
TlsActor >> error: errorDescription [

	Error signal: errorDescription
]

{ #category : 'certificate info' }
TlsActor >> getAllValuesOf: oid from: subject [

	| values index |
	values := Array new.
	index := -1.
	
	[index := libCrypto X509_NAME_get_index_by_OBJ_: subject _: oid _: index.
	index = -1]
			whileFalse: 
				[| entry string name |
				entry := libCrypto X509_NAME_get_entry_: subject _: index.
				self checkForNullReturn: entry from: 'X509_NAME_get_entry'.
				string := libCrypto X509_NAME_ENTRY_get_data_: entry.
				self checkForNullReturn: string from: 'X509_NAME_ENTRY_get_data'.
				name := libCrypto ASN1_STRING_get0_data_: string.
				values add: name].
	^values
]

{ #category : 'initialization' }
TlsActor >> initialize [

	libCrypto := LibCrypto new.
	libSsl := LibSsl new.
	libcFcntl := LibcFcntl new.
	^self
]

{ #category : 'certificate info' }
TlsActor >> peerCertificate [
	"This will only give expected results after a successful TLS handshake."

	| peer |
	peer := libSsl SSL_get1_peer_certificate_: ssl.
	self checkForNullReturn: peer from: 'SSL_get1_peer_certificate'.
	^peer
]

{ #category : 'certificate info' }
TlsActor >> peerSubject [
	"This will only give expected results after a successful TLS handshake is completed."

	| cert name |
	cert := self peerCertificate.
	name := libCrypto X509_get_subject_name_: cert.
	libCrypto X509_free_: cert . "Fix 49411"
	self checkForNullReturn: name from: 'X509_get_subject_name'.
	^name
]

{ #category : 'read/write' }
TlsActor >> readLine [
	"Reads one lf-terminated line, of no more than 255 characters, and returns the string *without* the terminating lf.
	It is expected that this line is all that is being sent in this TLS record, so if more characters are received beyond the lf,
	an error is signaled.
	We are currently using blocking sockets. This will get more complicated if we go to non-blocking.
	Also, this method may fail if renegotiation takes place during the read. This is improbable during the short
	interval of the initial login handshake, but should be coded for eventually."

	| maxSize res |
	maxSize := 255.
  res := tcpSocket _readLine: maxSize maxWaitMs: 500 .
  res ifNil:[
    self error: 'Socket failure during readLine' .
  ].
  (res codePointAt: res size ) == 10 ifFalse:[ 
    self error: 'Received line not lf-terminated: ', res .
  ].
  ^ res .
]

{ #category : 'certificate info' }
TlsActor >> remoteIpOfFd: aFileDescriptor [
	"Answer an IPv4 address as a ByteArray."

	| libcSocket addrInfo socklen retval family |
	libcSocket := LibcSocket new.
	addrInfo := StructSockaddrIn new.
	socklen := CByteArray gcMalloc: 4.
	socklen uint32At: 0 put: StructSockaddrIn byteSize.
	retval := libcSocket getpeername_: aFileDescriptor _: addrInfo _: socklen.
	retval = 0
		ifFalse: 
			[self error: 'Could not get peername for file descriptor '
						, aFileDescriptor printString].
	family := addrInfo sin_family.
	family = LibcSocket af_inet
		ifFalse: [self error: 'Socket is not IPv4. Only IPv4 is currently supported.'].
	^addrInfo address
]

{ #category : 'certificate info' }
TlsActor >> remoteIpOfSsl: anSsl [
	"Answer an IPv4 address as a ByteArray."

	| fd |
	fd := libSsl SSL_get_fd_: anSsl.
	"Strictly, a 0 return is valid, but we will not have a socket on stdin in this context."
	self checkReturn: fd from: 'SSL_get_fd'.
	^self remoteIpOfFd: fd
]

